#include "renderer.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

typedef struct {
    SDL_Window* window;
    SDL_Renderer* renderer;
    SDL_Texture* texture;
    uint32_t* framebuffer;
    int width;
    int height;
} Renderer;
static Renderer g_renderer;
static Renderer * const renderer = &g_renderer;

bool renderer_init(const char* title, int width, int height) {
    // Create window
    renderer->window = SDL_CreateWindow(title, width, height, 0);
    if (!renderer->window) {
        fprintf(stderr, "Failed to create window: %s\n", SDL_GetError());
        return false;
    }
    
    // Create renderer
    renderer->renderer = SDL_CreateRenderer(renderer->window, NULL);
    if (!renderer->renderer) {
        fprintf(stderr, "Failed to create renderer: %s\n", SDL_GetError());
        SDL_DestroyWindow(renderer->window);
        return false;
    }
    
    // Create texture for framebuffer
    renderer->texture = SDL_CreateTexture(renderer->renderer, 
        SDL_PIXELFORMAT_ARGB8888, SDL_TEXTUREACCESS_STREAMING, width, height);
    if (!renderer->texture) {
        fprintf(stderr, "Failed to create texture: %s\n", SDL_GetError());
        SDL_DestroyRenderer(renderer->renderer);
        SDL_DestroyWindow(renderer->window);
        return false;
    }
    
    // Allocate framebuffer
    renderer->framebuffer = (uint32_t*)malloc(width * height * sizeof(uint32_t));
    if (!renderer->framebuffer) {
        fprintf(stderr, "Failed to allocate framebuffer\n");
        SDL_DestroyTexture(renderer->texture);
        SDL_DestroyRenderer(renderer->renderer);
        SDL_DestroyWindow(renderer->window);
        return false;
    }
    
    renderer->width = width;
    renderer->height = height;
    
    return true;
}

void renderer_clear(uint32_t color) {
    for (int i = 0; i < renderer->width * renderer->height; i++) {
        renderer->framebuffer[i] = color;
    }
}

void renderer_present(void) {
    // Update texture with framebuffer
    SDL_UpdateTexture(renderer->texture, NULL, renderer->framebuffer, 
        renderer->width * sizeof(uint32_t));
    
    // Copy texture to renderer
    SDL_RenderTexture(renderer->renderer, renderer->texture, NULL, NULL);
    
    // Present
    SDL_RenderPresent(renderer->renderer);
	SDL_Delay(1);
}

void renderer_cleanup(void) {
    if (renderer->framebuffer) {
        free(renderer->framebuffer);
    }
    if (renderer->texture) {
        SDL_DestroyTexture(renderer->texture);
    }
    if (renderer->renderer) {
        SDL_DestroyRenderer(renderer->renderer);
    }
    if (renderer->window) {
        SDL_DestroyWindow(renderer->window);
    }
}

void draw_pixel(int x, int y, uint32_t color) {
    if (x >= 0 && x < renderer->width && y >= 0 && y < renderer->height) {
        renderer->framebuffer[y * renderer->width + x] = color;
    }
}

void draw_rect(int x, int y, int w, int h, uint32_t color) {
    // Top and bottom edges
    for (int i = 0; i < w; i++) {
        draw_pixel(x + i, y, color);
        draw_pixel(x + i, y + h - 1, color);
    }
    
    // Left and right edges
    for (int i = 1; i < h - 1; i++) {
        draw_pixel(x, y + i, color);
        draw_pixel(x + w - 1, y + i, color);
    }
}

void draw_filled_rect(int x, int y, int w, int h, uint32_t color) {
    for (int j = 0; j < h; j++) {
        for (int i = 0; i < w; i++) {
            draw_pixel(x + i, y + j, color);
        }
    }
}

void draw_sprite(Sprite* sprite, int x, int y) {
    if (!sprite || !sprite->pixels) return;
    
    for (int j = 0; j < sprite->height; j++) {
        for (int i = 0; i < sprite->width; i++) {
            uint32_t pixel = sprite->pixels[j * sprite->width + i];
            // Only draw non-transparent pixels (alpha > 0)
            if ((pixel & 0xFF000000) > 0) {
                draw_pixel(x + i, y + j, pixel);
            }
        }
    }
}

void draw_sprite_scaled(Sprite* sprite, int x, int y, float scale) {
    if (!sprite || !sprite->pixels || scale <= 0) return;
    
    int scaledWidth = (int)(sprite->width * scale);
    int scaledHeight = (int)(sprite->height * scale);
    
    for (int j = 0; j < scaledHeight; j++) {
        for (int i = 0; i < scaledWidth; i++) {
            int srcX = (int)(i / scale);
            int srcY = (int)(j / scale);
            
            if (srcX >= 0 && srcX < sprite->width && srcY >= 0 && srcY < sprite->height) {
                uint32_t pixel = sprite->pixels[srcY * sprite->width + srcX];
                // Only draw non-transparent pixels (alpha > 0)
                if ((pixel & 0xFF000000) > 0) {
                    draw_pixel(x + i, y + j, pixel);
                }
            }
        }
    }
}

// Simple bitmap font (8x8 characters)
static const uint8_t font_data[95][8] = {
    // ASCII 32 (space) to 126
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, // Space
    {0x18, 0x3C, 0x3C, 0x18, 0x18, 0x00, 0x18, 0x00}, // !
    {0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, // "
    {0x36, 0x36, 0x7F, 0x36, 0x7F, 0x36, 0x36, 0x00}, // #
    {0x0C, 0x3E, 0x03, 0x1E, 0x30, 0x1F, 0x0C, 0x00}, // $
    {0x00, 0x63, 0x33, 0x18, 0x0C, 0x66, 0x63, 0x00}, // %
    {0x1C, 0x36, 0x1C, 0x6E, 0x3B, 0x33, 0x6E, 0x00}, // &
    {0x06, 0x06, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00}, // '
    {0x18, 0x0C, 0x06, 0x06, 0x06, 0x0C, 0x18, 0x00}, // (
    {0x06, 0x0C, 0x18, 0x18, 0x18, 0x0C, 0x06, 0x00}, // )
    {0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00}, // *
    {0x00, 0x0C, 0x0C, 0x3F, 0x0C, 0x0C, 0x00, 0x00}, // +
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x06}, // ,
    {0x00, 0x00, 0x00, 0x3F, 0x00, 0x00, 0x00, 0x00}, // -
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00}, // .
    {0x60, 0x30, 0x18, 0x0C, 0x06, 0x03, 0x01, 0x00}, // /
    {0x3E, 0x63, 0x73, 0x7B, 0x6F, 0x67, 0x3E, 0x00}, // 0
    {0x0C, 0x0E, 0x0C, 0x0C, 0x0C, 0x0C, 0x3F, 0x00}, // 1
    {0x1E, 0x33, 0x30, 0x1C, 0x06, 0x33, 0x3F, 0x00}, // 2
    {0x1E, 0x33, 0x30, 0x1C, 0x30, 0x33, 0x1E, 0x00}, // 3
    {0x38, 0x3C, 0x36, 0x33, 0x7F, 0x30, 0x78, 0x00}, // 4
    {0x3F, 0x03, 0x1F, 0x30, 0x30, 0x33, 0x1E, 0x00}, // 5
    {0x1C, 0x06, 0x03, 0x1F, 0x33, 0x33, 0x1E, 0x00}, // 6
    {0x3F, 0x33, 0x30, 0x18, 0x0C, 0x0C, 0x0C, 0x00}, // 7
    {0x1E, 0x33, 0x33, 0x1E, 0x33, 0x33, 0x1E, 0x00}, // 8
    {0x1E, 0x33, 0x33, 0x3E, 0x30, 0x18, 0x0E, 0x00}, // 9
    {0x00, 0x00, 0x0C, 0x0C, 0x00, 0x0C, 0x0C, 0x00}, // :
    {0x00, 0x00, 0x0C, 0x0C, 0x00, 0x0C, 0x0C, 0x06}, // ;
    {0x18, 0x0C, 0x06, 0x03, 0x06, 0x0C, 0x18, 0x00}, // <
    {0x00, 0x00, 0x3F, 0x00, 0x00, 0x3F, 0x00, 0x00}, // =
    {0x06, 0x0C, 0x18, 0x30, 0x18, 0x0C, 0x06, 0x00}, // >
    {0x1E, 0x33, 0x30, 0x18, 0x0C, 0x00, 0x0C, 0x00}, // ?
    {0x3E, 0x63, 0x7B, 0x7B, 0x7B, 0x03, 0x1E, 0x00}, // @
    {0x18, 0x24, 0x42, 0x42, 0x7E, 0x42, 0x42, 0x00}, // A
    {0x3E, 0x42, 0x42, 0x3E, 0x42, 0x42, 0x3E, 0x00}, // B
    {0x3C, 0x42, 0x02, 0x02, 0x02, 0x42, 0x3C, 0x00}, // C
    {0x1E, 0x22, 0x42, 0x42, 0x42, 0x22, 0x1E, 0x00}, // D
    {0x7E, 0x02, 0x02, 0x3E, 0x02, 0x02, 0x7E, 0x00}, // E
    {0x7E, 0x02, 0x02, 0x3E, 0x02, 0x02, 0x02, 0x00}, // F
    {0x3C, 0x42, 0x02, 0x02, 0x72, 0x42, 0x3C, 0x00}, // G
    {0x42, 0x42, 0x42, 0x7E, 0x42, 0x42, 0x42, 0x00}, // H
    {0x3E, 0x08, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00}, // I
    {0x70, 0x20, 0x20, 0x20, 0x22, 0x22, 0x1C, 0x00}, // J
    {0x42, 0x22, 0x12, 0x0E, 0x12, 0x22, 0x42, 0x00}, // K
    {0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x7E, 0x00}, // L
    {0x42, 0x66, 0x5A, 0x5A, 0x42, 0x42, 0x42, 0x00}, // M
    {0x42, 0x46, 0x4A, 0x52, 0x62, 0x42, 0x42, 0x00}, // N
    {0x18, 0x24, 0x42, 0x42, 0x42, 0x24, 0x18, 0x00}, // O
    {0x3E, 0x42, 0x42, 0x3E, 0x02, 0x02, 0x02, 0x00}, // P
    {0x18, 0x24, 0x42, 0x42, 0x52, 0x24, 0x58, 0x00}, // Q
    {0x3E, 0x42, 0x42, 0x3E, 0x12, 0x22, 0x42, 0x00}, // R
    {0x3C, 0x42, 0x02, 0x1C, 0x20, 0x42, 0x3C, 0x00}, // S
    {0x7F, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00}, // T
    {0x42, 0x42, 0x42, 0x42, 0x42, 0x42, 0x3C, 0x00}, // U
    {0x42, 0x42, 0x42, 0x42, 0x24, 0x18, 0x00, 0x00}, // V
    {0x42, 0x42, 0x42, 0x5A, 0x5A, 0x66, 0x42, 0x00}, // W
    {0x42, 0x24, 0x18, 0x18, 0x24, 0x42, 0x00, 0x00}, // X
    {0x41, 0x22, 0x14, 0x08, 0x08, 0x08, 0x08, 0x00}, // Y
    {0x7E, 0x20, 0x10, 0x08, 0x04, 0x02, 0x7E, 0x00}, // Z
};

void draw_text(const char* text, int x, int y, uint32_t color) {
    int startX = x;
    
    while (*text) {
        if (*text == '\n') {
            y += 10;
            x = startX;
        } else {
            char c = *text;
            if (c >= 32 && c <= 126) {
                const uint8_t* charData = font_data[c - 32];
                
                // Standard text rendering - no flips at all
                for (int row = 0; row < 8; row++) {
                    uint8_t rowData = charData[row];  // No vertical flip
                    for (int col = 0; col < 8; col++) {
                        if (rowData & (1 << col)) {  // No horizontal flip either
                            draw_pixel(x + col, y + row, color);
                        }
                    }
                }
                x += 8;
            }
        }
        text++;
    }
}
