//trigfunc.h
//Fixed-point trigonometry
//Bryan E. Topp <betopp@betopp.com> 2025

#include "trigfunc.h"

//Quarter-circle sine function (domain 0...pi/2 in 2^8 steps) as 0.24 fraction (range 0...1 in 2^24 steps)
static const int32_t trigfunc_sintab24[256] = 
{
	0x00000000, 0x0001921F, 0x0003243A, 0x0004B64D, 0x00064855, 0x0007DA4D, 0x00096C32, 0x000AFE00, 0x000C8FB2, 0x000E2146, 0x000FB2B7, 0x00114401, 0x0012D520, 0x00146611, 0x0015F6D0, 0x00178758, 
	0x001917A6, 0x001AA7B7, 0x001C3785, 0x001DC70E, 0x001F564E, 0x0020E540, 0x002273E1, 0x0024022D, 0x00259020, 0x00271DB7, 0x0028AAED, 0x002A37BF, 0x002BC428, 0x002D5026, 0x002EDBB3, 0x003066CD, 
	0x0031F170, 0x00337B97, 0x00350540, 0x00368E65, 0x00381704, 0x00399F19, 0x003B269F, 0x003CAD94, 0x003E33F2, 0x003FB9B8, 0x00413EE0, 0x0042C367, 0x00444749, 0x0045CA83, 0x00474D10, 0x0048CEEE, 
	0x004A5018, 0x004BD08B, 0x004D5043, 0x004ECF3B, 0x00504D72, 0x0051CAE2, 0x00534789, 0x0054C362, 0x00563E69, 0x0057B89C, 0x005931F7, 0x005AAA75, 0x005C2214, 0x005D98D0, 0x005F0EA4, 0x0060838E, 
	0x0061F78A, 0x00636A94, 0x0064DCA9, 0x00664DC5, 0x0067BDE5, 0x00692D04, 0x006A9B20, 0x006C0835, 0x006D7440, 0x006EDF3C, 0x00704927, 0x0071B1FD, 0x007319BA, 0x0074805B, 0x0075E5DD, 0x00774A3C, 
	0x0078AD74, 0x007A0F83, 0x007B7065, 0x007CD016, 0x007E2E93, 0x007F8BD9, 0x0080E7E4, 0x008242B1, 0x00839C3C, 0x0084F483, 0x00864B82, 0x0087A135, 0x0088F59A, 0x008A48AD, 0x008B9A6B, 0x008CEAD0, 
	0x008E39D9, 0x008F8784, 0x0090D3CC, 0x00921EAF, 0x0093682A, 0x0094B039, 0x0095F6D9, 0x00973C07, 0x00987FBF, 0x0099C200, 0x009B02C5, 0x009C420C, 0x009D7FD1, 0x009EBC11, 0x009FF6CA, 0x00A12FF8, 
	0x00A26799, 0x00A39DA8, 0x00A4D224, 0x00A6050A, 0x00A73655, 0x00A86604, 0x00A99414, 0x00AAC081, 0x00ABEB49, 0x00AD1469, 0x00AE3BDD, 0x00AF61A4, 0x00B085BA, 0x00B1A81D, 0x00B2C8C9, 0x00B3E7BC, 
	0x00B504F3, 0x00B6206B, 0x00B73A22, 0x00B85215, 0x00B96841, 0x00BA7CA4, 0x00BB8F3A, 0x00BCA002, 0x00BDAEF9, 0x00BEBC1B, 0x00BFC767, 0x00C0D0D9, 0x00C1D870, 0x00C2DE28, 0x00C3E200, 0x00C4E3F4, 
	0x00C5E403, 0x00C6E229, 0x00C7DE65, 0x00C8D8B3, 0x00C9D112, 0x00CAC77F, 0x00CBBBF7, 0x00CCAE79, 0x00CD9F02, 0x00CE8D8F, 0x00CF7A1F, 0x00D064AF, 0x00D14D3D, 0x00D233C6, 0x00D31848, 0x00D3FAC2, 
	0x00D4DB31, 0x00D5B992, 0x00D695E4, 0x00D77025, 0x00D84852, 0x00D91E6A, 0x00D9F269, 0x00DAC44F, 0x00DB941A, 0x00DC61C6, 0x00DD2D53, 0x00DDF6BE, 0x00DEBE05, 0x00DF8327, 0x00E04621, 0x00E106F1, 
	0x00E1C597, 0x00E28210, 0x00E33C59, 0x00E3F472, 0x00E4AA59, 0x00E55E0B, 0x00E60F87, 0x00E6BECC, 0x00E76BD7, 0x00E816A7, 0x00E8BF3B, 0x00E96591, 0x00EA09A6, 0x00EAAB7A, 0x00EB4B0B, 0x00EBE858, 
	0x00EC835E, 0x00ED1C1D, 0x00EDB293, 0x00EE46BE, 0x00EED89D, 0x00EF682F, 0x00EFF573, 0x00F08066, 0x00F10908, 0x00F18F57, 0x00F21352, 0x00F294F8, 0x00F31447, 0x00F3913E, 0x00F40BDD, 0x00F48421, 
	0x00F4FA0A, 0x00F56D97, 0x00F5DEC6, 0x00F64D96, 0x00F6BA07, 0x00F72417, 0x00F78BC5, 0x00F7F110, 0x00F853F7, 0x00F8B47A, 0x00F91297, 0x00F96E4E, 0x00F9C79D, 0x00FA1E84, 0x00FA7301, 0x00FAC515, 
	0x00FB14BE, 0x00FB61FB, 0x00FBACCD, 0x00FBF531, 0x00FC3B27, 0x00FC7EAF, 0x00FCBFC9, 0x00FCFE72, 0x00FD3AAB, 0x00FD7474, 0x00FDABCB, 0x00FDE0B0, 0x00FE1323, 0x00FE4323, 0x00FE70AF, 0x00FE9BC8, 
	0x00FEC46D, 0x00FEEA9C, 0x00FF0E57, 0x00FF2F9D, 0x00FF4E6D, 0x00FF6AC7, 0x00FF84AB, 0x00FF9C18, 0x00FFB10F, 0x00FFC38E, 0x00FFD397, 0x00FFE128, 0x00FFEC43, 0x00FFF4E5, 0x00FFFB10, 0x00FFFEC4,		
};

//Eighth-circle arctan function (domain 0...1 slope in 2^8 steps) as 0.16 angle (range 0...pi/4 in 2^13 steps)
static const int16_t trigfunc_atantab[256] = 
{
	0x0000, 0x0028, 0x0051, 0x007A, 0x00A2, 0x00CB, 0x00F4, 0x011D, 0x0145, 0x016E, 0x0197, 0x01BF, 0x01E8, 0x0211, 0x0239, 0x0262, 
	0x028B, 0x02B3, 0x02DC, 0x0304, 0x032D, 0x0355, 0x037E, 0x03A6, 0x03CE, 0x03F7, 0x041F, 0x0448, 0x0470, 0x0498, 0x04C0, 0x04E8, 
	0x0511, 0x0539, 0x0561, 0x0589, 0x05B1, 0x05D9, 0x0601, 0x0628, 0x0650, 0x0678, 0x06A0, 0x06C7, 0x06EF, 0x0716, 0x073E, 0x0765, 
	0x078D, 0x07B4, 0x07DB, 0x0803, 0x082A, 0x0851, 0x0878, 0x089F, 0x08C6, 0x08ED, 0x0913, 0x093A, 0x0961, 0x0987, 0x09AE, 0x09D4, 
	0x09FB, 0x0A21, 0x0A47, 0x0A6D, 0x0A94, 0x0ABA, 0x0AE0, 0x0B05, 0x0B2B, 0x0B51, 0x0B77, 0x0B9C, 0x0BC2, 0x0BE7, 0x0C0C, 0x0C32, 
	0x0C57, 0x0C7C, 0x0CA1, 0x0CC6, 0x0CEB, 0x0D0F, 0x0D34, 0x0D58, 0x0D7D, 0x0DA1, 0x0DC6, 0x0DEA, 0x0E0E, 0x0E32, 0x0E56, 0x0E7A, 
	0x0E9E, 0x0EC1, 0x0EE5, 0x0F08, 0x0F2C, 0x0F4F, 0x0F72, 0x0F95, 0x0FB8, 0x0FDB, 0x0FFE, 0x1021, 0x1044, 0x1066, 0x1089, 0x10AB, 
	0x10CD, 0x10EF, 0x1111, 0x1133, 0x1155, 0x1177, 0x1199, 0x11BA, 0x11DC, 0x11FD, 0x121E, 0x123F, 0x1260, 0x1281, 0x12A2, 0x12C3, 
	0x12E4, 0x1304, 0x1325, 0x1345, 0x1365, 0x1385, 0x13A5, 0x13C5, 0x13E5, 0x1405, 0x1424, 0x1444, 0x1463, 0x1483, 0x14A2, 0x14C1, 
	0x14E0, 0x14FF, 0x151E, 0x153C, 0x155B, 0x1579, 0x1598, 0x15B6, 0x15D4, 0x15F2, 0x1610, 0x162E, 0x164C, 0x166A, 0x1687, 0x16A5, 
	0x16C2, 0x16DF, 0x16FC, 0x1719, 0x1736, 0x1753, 0x1770, 0x178C, 0x17A9, 0x17C5, 0x17E2, 0x17FE, 0x181A, 0x1836, 0x1852, 0x186E, 
	0x188A, 0x18A5, 0x18C1, 0x18DC, 0x18F7, 0x1913, 0x192E, 0x1949, 0x1964, 0x197F, 0x1999, 0x19B4, 0x19CE, 0x19E9, 0x1A03, 0x1A1D, 
	0x1A37, 0x1A51, 0x1A6B, 0x1A85, 0x1A9F, 0x1AB9, 0x1AD2, 0x1AEC, 0x1B05, 0x1B1E, 0x1B37, 0x1B50, 0x1B69, 0x1B82, 0x1B9B, 0x1BB4, 
	0x1BCC, 0x1BE5, 0x1BFD, 0x1C16, 0x1C2E, 0x1C46, 0x1C5E, 0x1C76, 0x1C8E, 0x1CA5, 0x1CBD, 0x1CD5, 0x1CEC, 0x1D04, 0x1D1B, 0x1D32, 
	0x1D49, 0x1D60, 0x1D77, 0x1D8E, 0x1DA5, 0x1DBB, 0x1DD2, 0x1DE9, 0x1DFF, 0x1E15, 0x1E2C, 0x1E42, 0x1E58, 0x1E6E, 0x1E84, 0x1E99, 
	0x1EAF, 0x1EC5, 0x1EDA, 0x1EF0, 0x1F05, 0x1F1B, 0x1F30, 0x1F45, 0x1F5A, 0x1F6F, 0x1F84, 0x1F99, 0x1FAD, 0x1FC2, 0x1FD7, 0x1FEB,
};

int32_t trigfunc_sin8(int32_t input_ang16)
{
	//Todo - could make this interpolate between sine table entries using the extra precision of them
	if(input_ang16 < 0)
	{
		//Grumble grumble... handle negative modulos
		input_ang16 *= -1;
		input_ang16 &= 0xFFFF;
		input_ang16 *= -1;
		input_ang16 += 65536;
	}
	input_ang16 &= 0xFFFF;
	
	//Use the sine table flipped around for each quadrant
	if(input_ang16 < 16384*1)
	{
		return  (trigfunc_sintab24[ (input_ang16 -     0) / 64] / 65536);
	}
	else if(input_ang16 < 16384*2)
	{
		return  (trigfunc_sintab24[ (32767 - input_ang16) / 64 ] / 65536);
	}
	else if(input_ang16 < 16384*3)
	{
		return -(trigfunc_sintab24[ (input_ang16 - 32768) / 64 ] / 65536);
	}
	else
	{
		return -(trigfunc_sintab24[ (65535 - input_ang16) / 64 ] / 65536);
	}
}

int32_t trigfunc_cos8(int32_t input_ang16)
{
	return trigfunc_sin8(input_ang16 + 16384);
}

int32_t trigfunc_atan2(int32_t y, int32_t x)
{
	//Handle trivial cases upfront to avoid bad divides...
	
	if(y == 0 && x == 0) //Badly defined
		return 0;
	
	if(y == 0 && x > 0) //Straight east
		return 0;
	if(y == 0 && x < 0) //Straight west
		return 32768 - 65536;
	if(x == 0 && y > 0) //Straight north
		return 16384;
	if(x == 0 && y < 0) //Straight south
		return 49152 - 65536;
	
	//Use arctan table flipped around for each octant
	if(x > 0)
	{
		if(y > 0)
		{
			if(y > x)
			{
				return 16384 - trigfunc_atantab[256 * x / y];
			}
			else if(y < x)
			{
				return 0 + trigfunc_atantab[256 * y / x];
			}
			else
			{
				return 8192; //Straight northeast
			}
		}
		else
		{
			y *= -1;
			if(y > x)
			{
				return 49152 + trigfunc_atantab[256 * x / y] - 65536;
			}
			else if(y < x)
			{
				return 65536 - trigfunc_atantab[256 * y / x] - 65536;
			}
			else
			{
				return 57344 - 65536; //Straight southeast
			}
		}
	}
	else
	{
		x *= -1;
		if(y > 0)
		{
			if(y > x)
			{
				return 16384 + trigfunc_atantab[256 * x / y];
			}
			else if(y < x)
			{
				return 32768 - trigfunc_atantab[256 * y / x];
			}
			else
			{
				return 24576; //Straight northwest
			}
		}
		else
		{
			y *= -1;
			if(y > x)
			{
				return 49152 - trigfunc_atantab[256 * x / y] - 65536;
			}
			else if(y < x)
			{
				return 32768 + trigfunc_atantab[256 * y / x] - 65536;
			}
			else
			{
				return 40960 - 65536; //Straight southwest
			}
		}
	}
	
}
