#ifndef _LCR_ASSETS_H
#define _LCR_ASSETS_H

/** @file assets.h

  Licar: assets

  This file holds assets and other bigger pieces of data that are to be compiled
  right into the executable. This is so that for a minimal version of the game a
  filesystem is not required, all data will be embedded in the program. On
  platforms supporting files additional data (maps, replays, ...) can still be
  supplied via the external data file.

  NOTES:

  - All images are 64x64, stored in an indexed mode (8bits per pixel), the image
    starts with a 256 color palette in RGB565 format (i.e. the palette takes 512
    bytes, then 64x64 bytes follow).
  - Each sky (background) image is composed of 4 regular images so they have
    higher resolution. The partial images are stored in the above format, one
    sky part consists of interlaced lines, so it effectively stores a 128x32
    strip (instead of 64x64 square) -- this is convenient for drawing the sky.
*/

#include "general.h"
#include "map.h"

#ifdef LCR_MODULE_NAME
  #undef LCR_MODULE_NAME
#endif

#define LCR_MODULE_NAME "asset"

#define LCR_VERSION "1.0"

static const char *LCR_texts[] =
{
#define LCR_TEXTS_VERSION 0
  LCR_VERSION, // version string
#define LCR_TEXTS_TABS 1
  "main menu",
  "play map",
  "view repl",
  "race repl",
#define LCR_TEXTS_MAIN_MENU 5
  "camera",
  "music",
  "sound",
  "save repl",
  "exit",
#define LCR_TEXTS_LOADING 10
  "loading",
#define LCR_TEXTS_SAVED 11
  "saved",
#define LCR_TEXTS_FAIL 12
  "failed"
};

/// String that can be used by frontends to offer help about CLI arguments.
#define LCR_ARG_HELP_STR \
  "  -cN  set camera mode to N\n" \
  "  -s0  turn sound off\n" \
  "  -m0  turn music off\n" \
  "  -M   load the last map in the data file\n" \
  "  -R   load the last replay in the data file\n"

static const char LCR_internalDataFile[] =
{
// data generated by make_internal_data_file.c:
#if !LCR_SETTING_ONLY_SMALL_MAPS
0x4d,0x4c,0x43,0x31,0x3b,0x34,0x38,0x37,0x20,0x30,0x20,0x3a,0x2a,0x43,0x32,0x6d,
0x4c,0x3a,0x21,0x78,0x36,0x47,0x3a,0x2b,0x4c,0x32,0x48,0x3a,0x2b,0x44,0x33,0x38,
0x3a,0x6e,0x77,0x30,0x77,0x32,0x4c,0x3a,0x66,0x31,0x35,0x31,0x3a,0x6d,0x31,0x35,
0x31,0x3a,0x6e,0x7a,0x30,0x77,0x32,0x4c,0x3a,0x66,0x31,0x35,0x31,0x3a,0x6d,0x31,
0x35,0x31,0x3a,0x6e,0x77,0x30,0x48,0x32,0x4c,0x3a,0x66,0x31,0x35,0x31,0x3a,0x6d,
0x31,0x35,0x31,0x3a,0x6e,0x7a,0x30,0x48,0x32,0x4c,0x3a,0x66,0x31,0x35,0x31,0x3a,
0x6d,0x31,0x35,0x31,0x3a,0x3d,0x7a,0x30,0x39,0x3a,0x66,0x38,0x33,0x44,0x3a,0x3d,
0x48,0x30,0x6b,0x3a,0x66,0x61,0x31,0x73,0x3a,0x3d,0x48,0x30,0x44,0x3a,0x66,0x61,
0x32,0x39,0x3a,0x2d,0x41,0x32,0x39,0x3a,0x66,0x36,0x31,0x43,0x3a,0x27,0x41,0x32,
0x70,0x49,0x3a,0x66,0x36,0x31,0x31,0x3a,0x27,0x42,0x32,0x6a,0x3a,0x66,0x35,0x31,
0x31,0x3a,0x3d,0x43,0x30,0x6a,0x3a,0x66,0x33,0x34,0x37,0x3a,0x3d,0x48,0x30,0x71,
0x32,0x3a,0x66,0x34,0x32,0x63,0x3a,0x78,0x42,0x32,0x6b,0x3a,0x66,0x36,0x32,0x35,
0x3a,0x75,0x42,0x32,0x6f,0x3a,0x75,0x46,0x32,0x6f,0x4a,0x3a,0x75,0x42,0x32,0x6b,
0x4c,0x3a,0x75,0x46,0x32,0x6b,0x49,0x3a,0x5d,0x47,0x32,0x6b,0x4c,0x3a,0x5d,0x47,
0x32,0x6f,0x4c,0x3a,0x2d,0x77,0x32,0x39,0x3a,0x66,0x34,0x31,0x65,0x3a,0x78,0x48,
0x30,0x72,0x3a,0x66,0x33,0x35,0x62,0x3a,0x3d,0x47,0x30,0x43,0x3a,0x66,0x35,0x36,
0x31,0x3a,0x5e,0x47,0x35,0x43,0x4a,0x3a,0x5e,0x4b,0x35,0x43,0x4c,0x3a,0x3d,0x48,
0x30,0x6f,0x31,0x3a,0x66,0x34,0x31,0x32,0x3a,0x3d,0x48,0x31,0x4c,0x31,0x3a,0x66,
0x61,0x31,0x31,0x3a,0x3b,0x52,0x30,0x71,0x3a,0x66,0x31,0x36,0x6c,0x3a,0x3b,0x45,
0x30,0x4d,0x4c,0x3a,0x66,0x63,0x36,0x31,0x3a,0x29,0x51,0x31,0x72,0x4a,0x3a,0x66,
0x31,0x31,0x63,0x3a,0x29,0x51,0x32,0x44,0x4a,0x3a,0x66,0x31,0x31,0x38,0x3a,0x75,
0x51,0x32,0x4c,0x7c,0x3a,0x66,0x31,0x34,0x31,0x3a,0x5c,0x52,0x30,0x4c,0x49,0x7c,
0x3a,0x66,0x31,0x36,0x31,0x3a,0x5c,0x51,0x30,0x4d,0x4c,0x3a,0x66,0x31,0x36,0x31,
0x3a,0x3d,0x76,0x30,0x39,0x3a,0x66,0x63,0x32,0x67,0x3a,0x5e,0x7a,0x33,0x73,0x3a,
0x5e,0x7a,0x33,0x74,0x49,0x3a,0x3b,0x52,0x30,0x6c,0x3a,0x66,0x31,0x32,0x35,0x3a,
0x3b,0x4c,0x30,0x6a,0x4a,0x3a,0x66,0x35,0x32,0x31,0x3a,0x75,0x51,0x31,0x6b,0x49,
0x3a,0x5c,0x52,0x30,0x6b,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x51,0x30,0x6a,0x4c,
0x7c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,0x47,0x30,0x66,0x31,0x3a,0x66,0x35,0x34,
0x35,0x3a,0x41,0x4b,0x30,0x66,0x31,0x3a,0x66,0x31,0x34,0x31,0x3a,0x3d,0x41,0x32,
0x61,0x32,0x3a,0x66,0x31,0x32,0x66,0x3a,0x5e,0x41,0x33,0x61,0x32,0x3a,0x3d,0x77,
0x30,0x6f,0x3a,0x66,0x33,0x33,0x31,0x3a,0x3d,0x76,0x30,0x37,0x32,0x3a,0x66,0x31,
0x34,0x69,0x3a,0x3d,0x47,0x30,0x37,0x32,0x3a,0x66,0x31,0x34,0x64,0x3a,0x27,0x47,
0x31,0x6c,0x4c,0x3a,0x66,0x31,0x31,0x33,0x3a,0x3c,0x48,0x31,0x6c,0x4c,0x3a,0x66,
0x31,0x31,0x33,0x3a,0x2f,0x4d,0x31,0x7a,0x3a,0x66,0x33,0x31,0x31,0x3a,0x3c,0x47,
0x32,0x44,0x4c,0x3a,0x66,0x31,0x31,0x38,0x3a,0x3d,0x76,0x34,0x69,0x3a,0x66,0x36,
0x31,0x33,0x3a,0x6e,0x76,0x34,0x69,0x7c,0x3a,0x6e,0x41,0x34,0x69,0x3a,0x27,0x77,
0x32,0x6e,0x3a,0x66,0x34,0x31,0x31,0x3a,0x5e,0x78,0x33,0x6f,0x3a,0x66,0x32,0x31,
0x31,0x3a,0x3d,0x76,0x30,0x36,0x32,0x3a,0x66,0x63,0x33,0x33,0x3a,0x3d,0x76,0x30,
0x35,0x32,0x3a,0x66,0x63,0x37,0x31,0x3a,0x3d,0x76,0x33,0x36,0x32,0x3a,0x66,0x63,
0x31,0x31,0x3a,0x5e,0x76,0x34,0x36,0x32,0x49,0x3a,0x5e,0x47,0x34,0x36,0x32,0x49,
0x3a,0x27,0x77,0x32,0x38,0x49,0x3a,0x66,0x61,0x31,0x31,0x3a,0x5e,0x77,0x33,0x37,
0x49,0x3a,0x66,0x61,0x31,0x31,0x3a,0x5e,0x77,0x34,0x36,0x49,0x3a,0x66,0x61,0x31,
0x31,0x3a,0x27,0x43,0x32,0x71,0x49,0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x43,0x33,
0x70,0x49,0x3a,0x66,0x33,0x31,0x31,0x3a,0x3c,0x43,0x34,0x6f,0x49,0x3a,0x66,0x33,
0x31,0x31,0x3a,0x2d,0x43,0x34,0x6c,0x3a,0x66,0x33,0x31,0x33,0x3a,0x3c,0x43,0x34,
0x6b,0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x43,0x33,0x6a,0x3a,0x66,0x33,0x31,0x31,
0x3a,0x27,0x43,0x32,0x69,0x3a,0x66,0x33,0x31,0x31,0x3a,0x29,0x43,0x33,0x6f,0x2d,
0x3a,0x66,0x33,0x31,0x31,0x3a,0x29,0x43,0x33,0x6b,0x2d,0x49,0x3a,0x66,0x33,0x31,
0x31,0x3a,0x2d,0x77,0x35,0x75,0x3a,0x66,0x33,0x31,0x64,0x3a,0x27,0x77,0x35,0x47,
0x3a,0x66,0x33,0x31,0x31,0x23,0x4d,0x4c,0x43,0x32,0x3b,0x38,0x30,0x39,0x20,0x31,
0x3a,0x2d,0x76,0x38,0x6b,0x3a,0x66,0x32,0x31,0x33,0x3a,0x27,0x76,0x38,0x6e,0x3a,
0x66,0x32,0x31,0x31,0x3a,0x49,0x76,0x30,0x6d,0x32,0x7c,0x3a,0x66,0x31,0x38,0x31,
0x3a,0x2e,0x76,0x32,0x6c,0x32,0x49,0x3a,0x2e,0x76,0x34,0x6c,0x32,0x49,0x3a,0x7c,
0x78,0x38,0x6b,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x34,0x3a,0x78,0x76,0x30,0x6b,0x3a,
0x6d,0x33,0x61,0x34,0x3a,0x3d,0x41,0x33,0x42,0x32,0x3a,0x66,0x31,0x31,0x33,0x3a,
0x5e,0x41,0x34,0x42,0x32,0x4c,0x3a,0x66,0x31,0x31,0x33,0x3a,0x5e,0x42,0x33,0x42,
0x32,0x4c,0x3a,0x66,0x31,0x31,0x33,0x3a,0x3b,0x42,0x31,0x42,0x3a,0x66,0x31,0x32,
0x32,0x3a,0x3b,0x41,0x31,0x44,0x4c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x49,0x42,0x31,
0x44,0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,0x41,0x31,0x42,0x3a,0x66,0x31,0x32,0x32,
0x3a,0x6d,0x32,0x34,0x33,0x3a,0x5c,0x6d,0x33,0x6f,0x7c,0x49,0x3a,0x5c,0x6d,0x33,
0x6e,0x3a,0x5c,0x6c,0x33,0x6e,0x49,0x3a,0x5c,0x6c,0x33,0x6f,0x4a,0x3a,0x5c,0x6c,
0x33,0x70,0x4c,0x3a,0x5c,0x6d,0x33,0x6d,0x7c,0x49,0x3a,0x5c,0x6c,0x33,0x6d,0x7c,
0x3a,0x6d,0x32,0x31,0x34,0x3a,0x78,0x6d,0x33,0x6a,0x3a,0x66,0x31,0x31,0x38,0x3a,
0x78,0x6c,0x33,0x6b,0x3a,0x66,0x31,0x31,0x34,0x3a,0x3d,0x66,0x30,0x6e,0x32,0x3a,
0x66,0x7a,0x31,0x74,0x3a,0x3d,0x66,0x31,0x49,0x32,0x3a,0x66,0x7a,0x31,0x38,0x3a,
0x3d,0x44,0x30,0x77,0x33,0x3a,0x66,0x61,0x31,0x63,0x3a,0x3d,0x42,0x30,0x77,0x31,
0x3a,0x66,0x32,0x31,0x65,0x3a,0x78,0x41,0x31,0x49,0x3a,0x66,0x39,0x31,0x32,0x3a,
0x78,0x66,0x30,0x6e,0x3a,0x66,0x37,0x31,0x35,0x3a,0x78,0x66,0x30,0x73,0x3a,0x66,
0x63,0x31,0x61,0x3a,0x78,0x43,0x30,0x42,0x3a,0x66,0x34,0x31,0x39,0x3a,0x78,0x4c,
0x30,0x6e,0x3a,0x66,0x33,0x32,0x63,0x3a,0x78,0x66,0x30,0x4e,0x3a,0x66,0x67,0x32,
0x33,0x3a,0x2d,0x72,0x31,0x73,0x3a,0x66,0x34,0x31,0x38,0x3a,0x3d,0x69,0x30,0x68,
0x32,0x3a,0x66,0x36,0x33,0x61,0x3a,0x5e,0x6e,0x32,0x6d,0x32,0x4c,0x3a,0x66,0x31,
0x31,0x34,0x3a,0x5e,0x6c,0x32,0x71,0x32,0x49,0x3a,0x66,0x32,0x31,0x31,0x3a,0x76,
0x6e,0x32,0x71,0x32,0x4a,0x3a,0x41,0x6e,0x30,0x71,0x32,0x4c,0x3a,0x66,0x31,0x32,
0x31,0x3a,0x3d,0x6c,0x30,0x6d,0x3a,0x6d,0x33,0x33,0x35,0x3a,0x3d,0x6e,0x30,0x71,
0x32,0x3a,0x27,0x6e,0x30,0x43,0x32,0x4c,0x3a,0x66,0x31,0x31,0x36,0x3a,0x27,0x6f,
0x30,0x43,0x32,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x27,0x6c,0x31,0x49,0x32,0x4c,
0x3a,0x66,0x31,0x31,0x35,0x3a,0x2d,0x6d,0x31,0x49,0x32,0x3a,0x66,0x32,0x31,0x35,
0x3a,0x27,0x6f,0x31,0x49,0x32,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x41,0x75,0x30,
0x4e,0x32,0x49,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x76,0x30,0x4f,0x32,0x49,0x3a,
0x66,0x31,0x32,0x31,0x3a,0x41,0x78,0x30,0x50,0x32,0x49,0x3a,0x66,0x31,0x32,0x31,
0x3a,0x78,0x76,0x30,0x50,0x3a,0x66,0x32,0x32,0x31,0x3a,0x3d,0x71,0x31,0x48,0x32,
0x3a,0x66,0x39,0x31,0x31,0x3a,0x41,0x70,0x31,0x48,0x32,0x4a,0x3a,0x41,0x75,0x31,
0x48,0x32,0x4a,0x3a,0x5c,0x74,0x31,0x48,0x32,0x3a,0x41,0x7a,0x31,0x48,0x32,0x3a,
0x41,0x41,0x31,0x4a,0x32,0x3a,0x5c,0x71,0x31,0x47,0x32,0x4a,0x3a,0x5c,0x72,0x31,
0x47,0x32,0x7c,0x4c,0x3a,0x5c,0x76,0x31,0x47,0x32,0x4a,0x3a,0x5c,0x77,0x31,0x47,
0x32,0x7c,0x4c,0x3a,0x5c,0x44,0x31,0x4a,0x32,0x4a,0x3a,0x5c,0x45,0x31,0x4a,0x32,
0x7c,0x4c,0x3a,0x41,0x46,0x31,0x4b,0x32,0x3a,0x41,0x47,0x31,0x4b,0x32,0x7c,0x3a,
0x41,0x49,0x31,0x4a,0x32,0x7c,0x3a,0x41,0x71,0x30,0x42,0x32,0x4a,0x3a,0x41,0x71,
0x30,0x73,0x32,0x49,0x3a,0x41,0x6c,0x30,0x72,0x32,0x49,0x3a,0x41,0x43,0x30,0x4a,
0x31,0x3a,0x41,0x46,0x30,0x4a,0x32,0x7c,0x3a,0x41,0x43,0x30,0x42,0x31,0x4c,0x3a,
0x41,0x46,0x30,0x42,0x33,0x49,0x3a,0x41,0x66,0x30,0x43,0x32,0x4a,0x3a,0x41,0x66,
0x30,0x4d,0x32,0x49,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x4e,0x30,0x50,0x32,0x4c,
0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x4e,0x30,0x7a,0x32,0x3a,0x3d,0x48,0x30,0x41,
0x3a,0x3d,0x4a,0x30,0x41,0x3a,0x3d,0x4c,0x30,0x41,0x3a,0x3d,0x47,0x30,0x42,0x3a,
0x3d,0x49,0x30,0x42,0x3a,0x3d,0x4b,0x30,0x42,0x3a,0x3d,0x4d,0x30,0x42,0x3a,0x3d,
0x48,0x30,0x43,0x3a,0x3d,0x4a,0x30,0x43,0x3a,0x3d,0x4c,0x30,0x43,0x3a,0x76,0x4a,
0x32,0x75,0x31,0x3a,0x76,0x4b,0x32,0x75,0x31,0x7c,0x3a,0x5e,0x4a,0x32,0x74,0x31,
0x4a,0x3a,0x5e,0x4b,0x32,0x74,0x31,0x4c,0x3a,0x76,0x4a,0x32,0x73,0x31,0x4c,0x3a,
0x76,0x4b,0x32,0x73,0x31,0x49,0x3a,0x7e,0x48,0x32,0x75,0x31,0x3a,0x76,0x49,0x32,
0x74,0x31,0x4a,0x3a,0x76,0x4c,0x31,0x76,0x31,0x4a,0x3a,0x5e,0x4c,0x31,0x73,0x31,
0x4c,0x3a,0x66,0x31,0x31,0x33,0x3a,0x5e,0x47,0x31,0x76,0x31,0x49,0x3a,0x66,0x35,
0x31,0x31,0x3a,0x5e,0x47,0x32,0x74,0x31,0x49,0x3a,0x66,0x32,0x31,0x31,0x3a,0x5e,
0x49,0x32,0x73,0x31,0x4c,0x3a,0x3d,0x47,0x31,0x73,0x31,0x3a,0x66,0x35,0x31,0x33,
0x3a,0x3d,0x47,0x30,0x73,0x31,0x3a,0x66,0x32,0x33,0x31,0x3a,0x6d,0x36,0x33,0x34,
0x3a,0x7e,0x48,0x33,0x72,0x31,0x3a,0x78,0x41,0x31,0x6e,0x3a,0x66,0x34,0x33,0x35,
0x3a,0x2d,0x46,0x31,0x6a,0x3a,0x66,0x32,0x31,0x36,0x3a,0x3d,0x46,0x30,0x6a,0x32,
0x3a,0x66,0x32,0x31,0x32,0x3a,0x2d,0x72,0x31,0x6a,0x3a,0x66,0x67,0x31,0x32,0x3a,
0x3d,0x72,0x30,0x6a,0x32,0x3a,0x66,0x32,0x31,0x32,0x32,0x3a,0x27,0x72,0x31,0x6a,
0x4c,0x3a,0x66,0x31,0x31,0x32,0x3a,0x28,0x45,0x31,0x49,0x49,0x3a,0x66,0x33,0x31,
0x31,0x3a,0x6d,0x33,0x31,0x31,0x3a,0x3d,0x42,0x30,0x48,0x32,0x3a,0x66,0x31,0x31,
0x32,0x3a,0x3d,0x47,0x30,0x48,0x32,0x3a,0x66,0x31,0x31,0x32,0x3a,0x3d,0x76,0x30,
0x72,0x3a,0x66,0x31,0x36,0x31,0x3a,0x3c,0x77,0x35,0x72,0x4c,0x2d,0x3a,0x5d,0x77,
0x36,0x72,0x4a,0x3a,0x7c,0x77,0x37,0x72,0x4a,0x3a,0x7c,0x78,0x36,0x72,0x4c,0x2d,
0x3a,0x5d,0x78,0x37,0x72,0x4c,0x2d,0x3a,0x3d,0x78,0x38,0x72,0x3a,0x2d,0x76,0x62,
0x72,0x3a,0x3c,0x77,0x62,0x72,0x4c,0x3a,0x5d,0x77,0x61,0x72,0x4a,0x2d,0x3a,0x7c,
0x77,0x39,0x72,0x4a,0x2d,0x3a,0x7c,0x78,0x61,0x72,0x4c,0x3a,0x5d,0x78,0x39,0x72,
0x4c,0x3a,0x3c,0x75,0x62,0x72,0x4a,0x3a,0x5d,0x75,0x61,0x72,0x4c,0x2d,0x3a,0x7c,
0x75,0x39,0x72,0x4c,0x2d,0x3a,0x7c,0x74,0x61,0x72,0x4a,0x3a,0x5d,0x74,0x39,0x72,
0x4a,0x3a,0x3d,0x74,0x38,0x72,0x3a,0x3c,0x75,0x35,0x72,0x4a,0x2d,0x3a,0x5d,0x75,
0x36,0x72,0x4c,0x3a,0x7c,0x75,0x37,0x72,0x4c,0x3a,0x7c,0x74,0x36,0x72,0x4a,0x2d,
0x3a,0x5d,0x74,0x37,0x72,0x4a,0x2d,0x3a,0x3c,0x67,0x31,0x47,0x32,0x3a,0x66,0x35,
0x31,0x31,0x3a,0x27,0x67,0x31,0x48,0x32,0x3a,0x66,0x35,0x31,0x31,0x3a,0x3c,0x4a,
0x31,0x47,0x32,0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x4a,0x31,0x48,0x32,0x3a,0x66,
0x35,0x31,0x31,0x3a,0x76,0x77,0x31,0x46,0x31,0x3a,0x66,0x31,0x33,0x31,0x3a,0x76,
0x78,0x31,0x46,0x31,0x7c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x76,0x77,0x31,0x45,0x31,
0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x76,0x78,0x31,0x45,0x31,0x7c,0x4a,0x3a,0x66,
0x31,0x33,0x31,0x3a,0x7e,0x6d,0x31,0x44,0x32,0x3a,0x7e,0x71,0x31,0x45,0x32,0x3a,
0x7e,0x7a,0x32,0x49,0x32,0x3a,0x7e,0x76,0x32,0x4d,0x32,0x3a,0x7e,0x45,0x32,0x4b,
0x32,0x3a,0x7e,0x71,0x31,0x6f,0x32,0x3a,0x7e,0x75,0x31,0x70,0x32,0x3a,0x7e,0x47,
0x31,0x7a,0x3a,0x7e,0x48,0x31,0x45,0x3a,0x3b,0x72,0x31,0x72,0x4a,0x3a,0x66,0x64,
0x31,0x31,0x3a,0x3d,0x75,0x31,0x72,0x3a,0x66,0x33,0x31,0x31,0x3a,0x29,0x75,0x32,
0x72,0x4a,0x3a,0x29,0x77,0x32,0x72,0x4c,0x3a,0x3b,0x7a,0x31,0x73,0x3a,0x66,0x31,
0x31,0x37,0x3a,0x3d,0x4c,0x30,0x6e,0x3a,0x66,0x31,0x32,0x63,0x3a,0x3b,0x42,0x31,
0x72,0x4a,0x3a,0x66,0x33,0x32,0x31,0x3a,0x28,0x42,0x33,0x72,0x49,0x3a,0x66,0x33,
0x31,0x31,0x3a,0x27,0x42,0x33,0x71,0x3a,0x66,0x33,0x31,0x31,0x3a,0x2d,0x42,0x33,
0x70,0x3a,0x66,0x33,0x31,0x31,0x3a,0x27,0x42,0x33,0x6f,0x49,0x3a,0x66,0x33,0x31,
0x31,0x3a,0x28,0x42,0x33,0x6e,0x3a,0x66,0x33,0x31,0x31,0x3a,0x3b,0x42,0x31,0x6e,
0x4c,0x3a,0x66,0x33,0x32,0x31,0x3a,0x6f,0x43,0x30,0x70,0x3a,0x2a,0x76,0x39,0x68,
0x3a,0x2b,0x76,0x38,0x72,0x3a,0x2b,0x6a,0x32,0x4b,0x3a,0x2b,0x46,0x33,0x72,0x3a,
0x2b,0x6c,0x33,0x6d,0x3a,0x21,0x43,0x31,0x70,0x23,0x4d,0x4c,0x43,0x33,0x3b,0x35,
0x36,0x30,0x20,0x32,0x3a,0x2a,0x71,0x72,0x74,0x3a,0x21,0x6f,0x31,0x79,0x3a,0x21,
0x6f,0x31,0x7a,0x3a,0x2b,0x57,0x73,0x7a,0x3a,0x3d,0x6d,0x71,0x72,0x3a,0x66,0x39,
0x36,0x64,0x3a,0x78,0x6e,0x72,0x73,0x3a,0x66,0x37,0x34,0x62,0x3a,0x5e,0x6d,0x76,
0x72,0x4a,0x3a,0x66,0x31,0x31,0x64,0x3a,0x5e,0x75,0x76,0x72,0x4c,0x3a,0x66,0x31,
0x31,0x64,0x3a,0x7e,0x6f,0x75,0x7a,0x33,0x2d,0x3a,0x7e,0x73,0x75,0x7a,0x33,0x2d,
0x3a,0x7e,0x6f,0x75,0x77,0x33,0x2d,0x3a,0x7e,0x73,0x75,0x77,0x33,0x2d,0x3a,0x5e,
0x75,0x74,0x75,0x49,0x2d,0x3a,0x5e,0x75,0x74,0x76,0x2d,0x3a,0x5e,0x75,0x74,0x7a,
0x49,0x2d,0x3a,0x5e,0x75,0x74,0x41,0x2d,0x3a,0x27,0x75,0x73,0x74,0x49,0x3a,0x2d,
0x75,0x73,0x75,0x3a,0x2d,0x75,0x73,0x76,0x3a,0x27,0x75,0x73,0x77,0x3a,0x27,0x75,
0x73,0x79,0x49,0x3a,0x2d,0x75,0x73,0x7a,0x3a,0x2d,0x75,0x73,0x41,0x3a,0x27,0x75,
0x73,0x42,0x3a,0x78,0x70,0x76,0x41,0x3a,0x66,0x33,0x31,0x33,0x3a,0x75,0x70,0x76,
0x41,0x4c,0x3a,0x75,0x72,0x76,0x41,0x49,0x3a,0x75,0x72,0x76,0x43,0x4a,0x3a,0x75,
0x70,0x76,0x43,0x3a,0x6f,0x70,0x71,0x7a,0x3a,0x66,0x33,0x31,0x34,0x3a,0x5e,0x70,
0x77,0x41,0x49,0x2d,0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x70,0x77,0x7a,0x3a,0x66,
0x33,0x31,0x31,0x3a,0x5e,0x70,0x78,0x42,0x49,0x2d,0x3a,0x66,0x33,0x31,0x31,0x3a,
0x5e,0x70,0x78,0x41,0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x70,0x79,0x43,0x49,0x2d,
0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x70,0x79,0x42,0x3a,0x66,0x33,0x31,0x31,0x3a,
0x5e,0x70,0x7a,0x44,0x49,0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x70,0x7a,0x43,0x3a,
0x66,0x33,0x31,0x31,0x3a,0x27,0x6f,0x76,0x45,0x49,0x3a,0x66,0x35,0x31,0x31,0x3a,
0x3c,0x6f,0x75,0x45,0x2d,0x3a,0x66,0x35,0x31,0x31,0x3a,0x3c,0x6f,0x76,0x46,0x49,
0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x6f,0x75,0x46,0x2d,0x3a,0x66,0x35,0x31,0x31,
0x3a,0x5e,0x6f,0x75,0x47,0x49,0x3a,0x66,0x35,0x31,0x31,0x3a,0x5e,0x6f,0x74,0x47,
0x2d,0x3a,0x66,0x35,0x31,0x31,0x3a,0x5e,0x6f,0x74,0x48,0x49,0x3a,0x66,0x35,0x31,
0x31,0x3a,0x5e,0x6f,0x73,0x48,0x2d,0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x6f,0x73,
0x49,0x49,0x3a,0x66,0x35,0x31,0x31,0x3a,0x3c,0x6f,0x72,0x49,0x2d,0x3a,0x66,0x35,
0x31,0x31,0x3a,0x3c,0x6f,0x73,0x4a,0x49,0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x6f,
0x72,0x4a,0x2d,0x3a,0x66,0x35,0x31,0x31,0x3a,0x3d,0x6f,0x72,0x4b,0x3a,0x66,0x42,
0x31,0x36,0x3a,0x3e,0x49,0x72,0x4c,0x3a,0x66,0x35,0x31,0x34,0x3a,0x5e,0x6f,0x72,
0x50,0x49,0x2d,0x3a,0x66,0x42,0x31,0x31,0x3a,0x41,0x4b,0x74,0x4c,0x32,0x7c,0x49,
0x3a,0x6d,0x31,0x31,0x31,0x3a,0x41,0x53,0x74,0x50,0x32,0x7c,0x49,0x3a,0x6d,0x31,
0x31,0x31,0x3a,0x49,0x50,0x73,0x4c,0x32,0x3a,0x49,0x56,0x73,0x4e,0x32,0x3a,0x2d,
0x55,0x72,0x43,0x33,0x2d,0x3a,0x66,0x35,0x31,0x38,0x3a,0x27,0x55,0x72,0x4a,0x33,
0x2d,0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x55,0x72,0x43,0x33,0x49,0x2d,0x3a,0x66,
0x35,0x31,0x31,0x3a,0x3d,0x55,0x72,0x77,0x32,0x3a,0x66,0x35,0x31,0x36,0x3a,0x7c,
0x59,0x73,0x77,0x32,0x4a,0x3a,0x66,0x31,0x31,0x6b,0x3a,0x3d,0x76,0x72,0x4b,0x31,
0x3a,0x66,0x37,0x31,0x33,0x3a,0x41,0x57,0x30,0x79,0x3a,0x66,0x31,0x72,0x31,0x3a,
0x41,0x57,0x30,0x4d,0x3a,0x66,0x31,0x72,0x31,0x3a,0x41,0x70,0x30,0x4d,0x3a,0x66,
0x31,0x72,0x31,0x3a,0x5d,0x46,0x73,0x4e,0x4a,0x3a,0x66,0x31,0x31,0x33,0x3a,0x7c,
0x46,0x74,0x4e,0x4a,0x3a,0x66,0x31,0x31,0x33,0x3a,0x5d,0x46,0x77,0x4e,0x4a,0x2d,
0x3a,0x66,0x31,0x31,0x33,0x3a,0x7c,0x46,0x76,0x4e,0x4a,0x2d,0x3a,0x66,0x31,0x31,
0x33,0x3a,0x5d,0x44,0x73,0x4b,0x4c,0x3a,0x66,0x31,0x31,0x33,0x3a,0x7c,0x44,0x74,
0x4b,0x4c,0x3a,0x66,0x31,0x31,0x33,0x3a,0x5d,0x44,0x77,0x4b,0x4c,0x2d,0x3a,0x66,
0x31,0x31,0x33,0x3a,0x7c,0x44,0x76,0x4b,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x33,0x3a,
0x2d,0x44,0x78,0x4b,0x3a,0x66,0x33,0x31,0x36,0x3a,0x3b,0x43,0x73,0x4b,0x7c,0x3a,
0x66,0x31,0x35,0x33,0x3a,0x3b,0x47,0x73,0x4e,0x3a,0x66,0x31,0x35,0x33,0x3a,0x2c,
0x43,0x78,0x4b,0x7c,0x3a,0x66,0x31,0x31,0x36,0x3a,0x2c,0x47,0x78,0x4b,0x3a,0x66,
0x31,0x31,0x36,0x3a,0x7d,0x46,0x73,0x4e,0x4a,0x3a,0x7d,0x44,0x73,0x4d,0x4c,0x3a,
0x7c,0x45,0x73,0x4d,0x32,0x3a,0x7c,0x45,0x73,0x4e,0x32,0x49,0x3a,0x49,0x43,0x73,
0x50,0x49,0x3a,0x66,0x31,0x35,0x31,0x3a,0x49,0x47,0x73,0x4b,0x3a,0x66,0x31,0x35,
0x31,0x3a,0x5e,0x54,0x72,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x54,0x71,
0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x53,0x71,0x77,0x4a,0x3a,0x66,
0x31,0x31,0x36,0x3a,0x5e,0x53,0x70,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,
0x5e,0x52,0x70,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x52,0x6f,0x77,0x4c,
0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x51,0x6f,0x77,0x4a,0x3a,0x66,0x31,0x31,
0x36,0x3a,0x5e,0x51,0x6e,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x50,
0x6e,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x50,0x6d,0x77,0x4c,0x2d,0x3a,
0x66,0x31,0x31,0x36,0x3a,0x5e,0x4f,0x6d,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,
0x5e,0x4f,0x6c,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x4e,0x6c,0x77,
0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x4e,0x6b,0x77,0x4c,0x2d,0x3a,0x66,0x31,
0x31,0x36,0x3a,0x5e,0x4d,0x6b,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x4d,
0x6a,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x4c,0x6a,0x77,0x4a,0x3a,
0x66,0x31,0x31,0x36,0x3a,0x5e,0x4c,0x69,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,
0x3a,0x5e,0x4b,0x69,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x4b,0x68,0x77,
0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x4a,0x68,0x77,0x4a,0x3a,0x66,0x31,
0x31,0x36,0x3a,0x5e,0x4a,0x67,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,
0x49,0x67,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x49,0x66,0x77,0x4c,0x2d,
0x3a,0x66,0x31,0x31,0x36,0x3a,0x3d,0x48,0x66,0x77,0x32,0x3a,0x66,0x31,0x31,0x36,
0x3a,0x5e,0x47,0x66,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x47,0x65,0x77,
0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x46,0x65,0x77,0x4a,0x3a,0x66,0x31,
0x31,0x36,0x3a,0x5e,0x46,0x64,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,
0x45,0x64,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x45,0x63,0x77,0x4c,0x2d,
0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x44,0x63,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,
0x3a,0x5e,0x44,0x62,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x43,0x62,
0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x43,0x61,0x77,0x4c,0x2d,0x3a,0x66,
0x31,0x31,0x36,0x3a,0x5e,0x42,0x61,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,
0x42,0x39,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x41,0x39,0x77,0x4a,
0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x41,0x38,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,
0x36,0x3a,0x5e,0x7a,0x38,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x7a,0x37,
0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x79,0x37,0x77,0x4a,0x3a,0x66,
0x31,0x31,0x36,0x3a,0x5e,0x79,0x36,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,
0x79,0x78,0x36,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x78,0x35,0x77,0x4c,
0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x79,0x77,0x35,0x77,0x4a,0x3a,0x66,0x31,0x31,
0x36,0x3a,0x5e,0x77,0x34,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x79,0x76,
0x34,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x76,0x33,0x77,0x4c,0x2d,0x3a,
0x66,0x31,0x31,0x36,0x3a,0x5e,0x75,0x33,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,
0x5e,0x75,0x32,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x74,0x32,0x77,
0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x74,0x31,0x77,0x4c,0x2d,0x3a,0x66,0x31,
0x31,0x36,0x3a,0x5e,0x73,0x31,0x77,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x73,
0x30,0x77,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x36,0x3a,0x6f,0x44,0x63,0x42,0x3a,0x66,
0x31,0x32,0x31,0x3a,0x3d,0x46,0x65,0x79,0x3a,0x66,0x31,0x32,0x31,0x3a,0x6f,0x4a,
0x68,0x77,0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,0x4b,0x69,0x41,0x3a,0x66,0x31,0x32,
0x31,0x3a,0x3d,0x4d,0x6b,0x78,0x3a,0x3d,0x50,0x6e,0x79,0x3a,0x66,0x31,0x32,0x31,
0x3a,0x3d,0x6e,0x30,0x77,0x3a,0x66,0x35,0x31,0x36,0x3a,0x3d,0x6f,0x30,0x79,0x31,
0x3a,0x3d,0x6f,0x30,0x7a,0x31,0x3a,0x41,0x70,0x31,0x78,0x32,0x4c,0x3a,0x66,0x31,
0x70,0x31,0x3a,0x41,0x71,0x31,0x77,0x32,0x4c,0x3a,0x66,0x31,0x34,0x31,0x3a,0x41,
0x70,0x31,0x77,0x32,0x4a,0x3a,0x66,0x31,0x34,0x31,0x3a,0x41,0x70,0x31,0x41,0x32,
0x3a,0x66,0x31,0x70,0x31,0x3a,0x41,0x71,0x31,0x42,0x32,0x3a,0x66,0x31,0x34,0x31,
0x3a,0x41,0x70,0x31,0x42,0x32,0x49,0x3a,0x66,0x31,0x34,0x31,0x3a,0x5c,0x33,0x30,
0x33,0x31,0x49,0x3a,0x5c,0x33,0x30,0x32,0x31,0x7c,0x3a,0x3d,0x34,0x30,0x32,0x31,
0x3a,0x6e,0x34,0x30,0x33,0x31,0x4c,0x3a,0x41,0x34,0x30,0x31,0x31,0x4a,0x3a,0x5c,
0x36,0x30,0x33,0x31,0x49,0x7c,0x3a,0x5c,0x36,0x30,0x32,0x31,0x3a,0x3d,0x35,0x30,
0x32,0x31,0x3a,0x6e,0x35,0x30,0x33,0x31,0x49,0x3a,0x41,0x35,0x30,0x31,0x31,0x23,
0x4d,0x4c,0x43,0x34,0x3b,0x38,0x32,0x31,0x20,0x31,0x3a,0x3d,0x30,0x30,0x30,0x3a,
0x66,0x35,0x33,0x61,0x3a,0x78,0x31,0x31,0x31,0x3a,0x66,0x34,0x32,0x34,0x3a,0x78,
0x31,0x31,0x36,0x3a,0x66,0x33,0x32,0x33,0x3a,0x29,0x31,0x31,0x35,0x4c,0x3a,0x29,
0x33,0x31,0x35,0x4a,0x3a,0x29,0x31,0x32,0x35,0x4c,0x2d,0x3a,0x29,0x33,0x32,0x35,
0x4a,0x2d,0x3a,0x78,0x32,0x31,0x35,0x3a,0x66,0x31,0x32,0x31,0x3a,0x2d,0x30,0x33,
0x31,0x3a,0x66,0x61,0x31,0x39,0x3a,0x3c,0x30,0x33,0x31,0x3a,0x66,0x61,0x31,0x31,
0x3a,0x3c,0x30,0x33,0x39,0x49,0x3a,0x66,0x61,0x31,0x31,0x3a,0x3c,0x36,0x32,0x31,
0x49,0x2d,0x3a,0x66,0x34,0x31,0x31,0x3a,0x41,0x26,0x30,0x68,0x49,0x7c,0x3a,0x66,
0x31,0x37,0x31,0x3a,0x76,0x26,0x37,0x68,0x7c,0x3a,0x5e,0x5a,0x37,0x68,0x49,0x3a,
0x66,0x32,0x31,0x31,0x3a,0x5e,0x40,0x36,0x68,0x4a,0x2d,0x3a,0x78,0x5a,0x31,0x68,
0x3a,0x6d,0x33,0x37,0x31,0x3a,0x3d,0x35,0x30,0x30,0x3a,0x66,0x7a,0x31,0x36,0x3a,
0x66,0x35,0x31,0x6b,0x3a,0x3d,0x44,0x30,0x31,0x3a,0x66,0x70,0x31,0x36,0x3a,0x5f,
0x58,0x30,0x37,0x3a,0x2d,0x58,0x30,0x32,0x3a,0x66,0x34,0x31,0x6d,0x3a,0x3d,0x58,
0x30,0x75,0x3a,0x66,0x35,0x31,0x71,0x3a,0x3d,0x30,0x30,0x79,0x3a,0x66,0x35,0x31,
0x70,0x3a,0x3d,0x33,0x30,0x57,0x3a,0x66,0x54,0x31,0x34,0x3a,0x3d,0x55,0x30,0x54,
0x3a,0x66,0x34,0x31,0x34,0x3a,0x3d,0x57,0x30,0x53,0x3a,0x66,0x34,0x31,0x34,0x3a,
0x3d,0x57,0x30,0x58,0x3a,0x3d,0x54,0x30,0x56,0x3a,0x5e,0x61,0x30,0x35,0x2d,0x3a,
0x66,0x74,0x31,0x31,0x3a,0x3c,0x58,0x31,0x6e,0x3a,0x66,0x34,0x31,0x31,0x3a,0x3c,
0x59,0x31,0x75,0x49,0x3a,0x66,0x33,0x31,0x31,0x3a,0x3d,0x58,0x30,0x6e,0x3a,0x66,
0x34,0x31,0x31,0x3a,0x27,0x58,0x30,0x6d,0x3a,0x66,0x34,0x31,0x31,0x3a,0x3d,0x61,
0x30,0x35,0x3a,0x66,0x75,0x32,0x31,0x3a,0x5e,0x61,0x32,0x35,0x49,0x3a,0x66,0x75,
0x31,0x31,0x3a,0x27,0x76,0x30,0x33,0x4a,0x3a,0x66,0x31,0x31,0x32,0x3a,0x27,0x73,
0x31,0x35,0x4c,0x3a,0x27,0x74,0x31,0x35,0x4a,0x3a,0x2c,0x71,0x32,0x34,0x32,0x2d,
0x3a,0x3b,0x71,0x32,0x33,0x33,0x3a,0x2d,0x6e,0x30,0x33,0x3a,0x66,0x38,0x31,0x32,
0x3a,0x2d,0x71,0x30,0x34,0x31,0x3a,0x66,0x33,0x31,0x31,0x3a,0x2d,0x6e,0x30,0x33,
0x31,0x3a,0x66,0x32,0x31,0x31,0x3a,0x6d,0x68,0x35,0x33,0x3a,0x3b,0x34,0x31,0x5a,
0x4a,0x3a,0x66,0x52,0x32,0x31,0x3a,0x3b,0x39,0x31,0x57,0x4c,0x3a,0x66,0x4a,0x32,
0x31,0x3a,0x78,0x72,0x31,0x56,0x3a,0x66,0x61,0x33,0x37,0x3a,0x6e,0x33,0x31,0x57,
0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,0x33,0x31,0x58,0x3a,0x66,0x31,0x32,0x33,0x3a,
0x3d,0x30,0x31,0x57,0x3a,0x66,0x33,0x32,0x31,0x3a,0x3d,0x33,0x30,0x55,0x3a,0x66,
0x34,0x31,0x33,0x3a,0x3d,0x37,0x30,0x56,0x3a,0x3d,0x35,0x30,0x54,0x3a,0x41,0x35,
0x30,0x53,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x36,0x30,0x54,0x3a,0x66,0x31,0x33,
0x31,0x3a,0x41,0x37,0x30,0x55,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x38,0x30,0x56,
0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x34,0x31,0x53,0x49,0x3a,0x66,0x31,0x32,0x31,
0x3a,0x41,0x35,0x31,0x54,0x49,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x36,0x31,0x55,
0x49,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x37,0x31,0x56,0x49,0x3a,0x66,0x31,0x32,
0x31,0x3a,0x5c,0x38,0x31,0x57,0x4a,0x7c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x77,
0x31,0x5a,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x5c,0x78,0x31,0x59,0x49,0x7c,0x3a,
0x66,0x31,0x33,0x31,0x4c,0x3a,0x6e,0x77,0x34,0x59,0x31,0x4c,0x3a,0x66,0x31,0x32,
0x31,0x3a,0x3b,0x77,0x31,0x26,0x4a,0x3a,0x66,0x34,0x32,0x31,0x3a,0x66,0x31,0x35,
0x31,0x3a,0x3d,0x77,0x33,0x5a,0x32,0x3a,0x5e,0x77,0x33,0x40,0x32,0x49,0x2d,0x3a,
0x5e,0x77,0x34,0x40,0x32,0x3a,0x3d,0x77,0x34,0x26,0x32,0x3a,0x3d,0x77,0x30,0x59,
0x3a,0x66,0x35,0x31,0x34,0x3a,0x41,0x41,0x30,0x26,0x4c,0x3a,0x66,0x31,0x33,0x31,
0x3a,0x41,0x42,0x30,0x40,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x41,0x31,0x40,
0x4a,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x42,0x31,0x5a,0x4a,0x3a,0x66,0x31,0x32,
0x31,0x3a,0x3d,0x77,0x30,0x59,0x3a,0x66,0x31,0x34,0x31,0x3a,0x6d,0x36,0x36,0x34,
0x3a,0x2d,0x4a,0x33,0x59,0x3a,0x66,0x34,0x31,0x31,0x3a,0x3c,0x4a,0x33,0x5a,0x49,
0x3a,0x66,0x34,0x31,0x31,0x3a,0x2c,0x4b,0x31,0x5a,0x32,0x4a,0x3a,0x66,0x32,0x31,
0x31,0x3a,0x5e,0x4a,0x30,0x5a,0x49,0x3a,0x66,0x33,0x31,0x31,0x3a,0x3d,0x4d,0x31,
0x5a,0x32,0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,0x4a,0x30,0x59,0x31,0x3a,0x6d,0x34,
0x34,0x33,0x3a,0x3d,0x49,0x30,0x58,0x32,0x3a,0x3d,0x4a,0x30,0x59,0x33,0x3a,0x78,
0x35,0x31,0x34,0x3a,0x66,0x34,0x32,0x36,0x3a,0x3d,0x44,0x30,0x36,0x3a,0x66,0x6b,
0x32,0x31,0x3a,0x6e,0x44,0x30,0x36,0x49,0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,0x44,
0x30,0x31,0x3a,0x66,0x70,0x32,0x31,0x3a,0x6e,0x44,0x31,0x31,0x49,0x3a,0x3d,0x57,
0x30,0x36,0x3a,0x66,0x31,0x32,0x62,0x3a,0x3d,0x26,0x30,0x31,0x3a,0x66,0x31,0x32,
0x67,0x3a,0x41,0x56,0x30,0x5a,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x57,0x30,
0x59,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x58,0x30,0x58,0x4c,0x3a,0x66,0x31,
0x33,0x31,0x3a,0x41,0x59,0x30,0x57,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x5a,
0x30,0x56,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x40,0x30,0x55,0x4c,0x3a,0x66,
0x31,0x33,0x31,0x3a,0x41,0x26,0x30,0x54,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,
0x56,0x31,0x59,0x4a,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x57,0x31,0x58,0x4a,0x3a,
0x66,0x31,0x32,0x31,0x3a,0x41,0x58,0x31,0x57,0x4a,0x3a,0x66,0x31,0x32,0x31,0x3a,
0x41,0x59,0x31,0x56,0x4a,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x5a,0x31,0x55,0x4a,
0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x40,0x31,0x54,0x4a,0x3a,0x66,0x31,0x32,0x31,
0x3a,0x41,0x53,0x30,0x56,0x4a,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x54,0x30,0x55,
0x4a,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x55,0x30,0x54,0x4a,0x3a,0x66,0x31,0x33,
0x31,0x3a,0x41,0x56,0x30,0x53,0x4a,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x57,0x30,
0x52,0x4a,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x54,0x31,0x56,0x4c,0x3a,0x66,0x31,
0x32,0x31,0x3a,0x41,0x55,0x31,0x55,0x4c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x56,
0x31,0x54,0x4c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x41,0x57,0x31,0x53,0x4c,0x3a,0x66,
0x31,0x32,0x31,0x3a,0x41,0x58,0x31,0x52,0x4c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x3d,
0x55,0x31,0x5a,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x53,0x31,0x57,0x4c,0x3a,0x66,
0x31,0x32,0x31,0x3a,0x3d,0x58,0x30,0x43,0x32,0x3a,0x66,0x31,0x33,0x66,0x3a,0x66,
0x32,0x31,0x66,0x3a,0x3d,0x26,0x30,0x43,0x32,0x3a,0x66,0x31,0x33,0x68,0x3a,0x3d,
0x40,0x30,0x43,0x32,0x3a,0x66,0x31,0x31,0x66,0x3a,0x3d,0x30,0x31,0x50,0x3a,0x66,
0x31,0x32,0x37,0x3a,0x3d,0x34,0x31,0x50,0x3a,0x66,0x31,0x32,0x33,0x3a,0x5c,0x30,
0x31,0x4f,0x7c,0x4c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x34,0x31,0x4f,0x4a,0x3a,
0x66,0x31,0x32,0x31,0x3a,0x5c,0x32,0x32,0x4c,0x7c,0x49,0x3a,0x6d,0x31,0x31,0x31,
0x3a,0x5c,0x34,0x32,0x47,0x7c,0x49,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x5c,0x32,0x32,
0x43,0x7c,0x49,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x7e,0x31,0x31,0x47,0x32,0x3a,0x3d,
0x35,0x30,0x79,0x3a,0x66,0x32,0x31,0x32,0x3a,0x41,0x30,0x30,0x79,0x4a,0x3a,0x41,
0x35,0x30,0x41,0x4c,0x3a,0x41,0x36,0x30,0x7a,0x4c,0x3a,0x41,0x37,0x30,0x79,0x4c,
0x3a,0x27,0x36,0x30,0x72,0x4a,0x3a,0x66,0x31,0x31,0x37,0x3a,0x5e,0x37,0x30,0x72,
0x4c,0x2d,0x3a,0x66,0x31,0x31,0x37,0x3a,0x5e,0x37,0x31,0x72,0x4a,0x3a,0x66,0x31,
0x31,0x37,0x3a,0x7c,0x37,0x32,0x72,0x4a,0x2d,0x3a,0x66,0x31,0x31,0x37,0x3a,0x7c,
0x38,0x32,0x72,0x4c,0x3a,0x66,0x31,0x31,0x37,0x3a,0x7c,0x38,0x31,0x72,0x4c,0x2d,
0x3a,0x66,0x31,0x31,0x37,0x3a,0x27,0x35,0x30,0x72,0x4c,0x3a,0x66,0x31,0x31,0x37,
0x3a,0x6d,0x34,0x33,0x37,0x3a,0x3b,0x79,0x30,0x33,0x31,0x4a,0x3a,0x66,0x33,0x31,
0x31,0x3a,0x3b,0x79,0x30,0x32,0x31,0x4c,0x3a,0x66,0x33,0x31,0x31,0x3a,0x5e,0x33,
0x32,0x31,0x32,0x4c,0x3a,0x5e,0x33,0x31,0x31,0x32,0x4c,0x2d,0x3a,0x2d,0x32,0x32,
0x31,0x32,0x3a,0x2d,0x32,0x31,0x31,0x32,0x2d,0x3a,0x3d,0x34,0x31,0x30,0x3a,0x66,
0x32,0x32,0x32,0x3a,0x5d,0x58,0x31,0x75,0x4c,0x3a,0x66,0x31,0x31,0x38,0x3a,0x5d,
0x26,0x31,0x75,0x4a,0x3a,0x66,0x31,0x31,0x38,0x3a,0x3e,0x46,0x30,0x31,0x3a,0x66,
0x33,0x31,0x36,0x3a,0x3e,0x30,0x30,0x44,0x3a,0x66,0x35,0x31,0x33,0x3a,0x3d,0x62,
0x30,0x57,0x33,0x3a,0x66,0x61,0x31,0x34,0x3a,0x27,0x57,0x30,0x32,0x4c,0x3a,0x66,
0x31,0x31,0x34,0x3a,0x3d,0x36,0x30,0x49,0x32,0x3a,0x66,0x32,0x33,0x36,0x3a,0x5e,
0x36,0x32,0x49,0x4a,0x3a,0x66,0x31,0x31,0x36,0x3a,0x5e,0x37,0x33,0x49,0x4a,0x3a,
0x66,0x31,0x31,0x36,0x3a,0x7c,0x30,0x31,0x7a,0x32,0x4c,0x3a,0x66,0x31,0x31,0x65,
0x62,0x6f,0x72,0x64,0x65,0x72,0x6c,0x65,0x66,0x74,0x3a,0x5c,0x51,0x31,0x32,0x31,
0x4c,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x50,0x31,0x32,0x31,0x7c,0x4a,0x3a,0x66,
0x31,0x32,0x31,0x3a,0x5c,0x51,0x31,0x35,0x31,0x7c,0x4c,0x3a,0x66,0x31,0x32,0x31,
0x3a,0x5c,0x50,0x31,0x35,0x31,0x49,0x4a,0x3a,0x66,0x31,0x32,0x31,0x3a,0x5c,0x51,
0x32,0x36,0x31,0x4c,0x3a,0x5c,0x50,0x32,0x36,0x31,0x7c,0x4a,0x3a,0x5c,0x51,0x32,
0x31,0x31,0x7c,0x4c,0x3a,0x5c,0x50,0x32,0x31,0x31,0x49,0x4a,0x3a,0x3d,0x39,0x31,
0x35,0x3a,0x66,0x31,0x32,0x66,0x3a,0x6e,0x39,0x31,0x35,0x4a,0x3a,0x66,0x31,0x32,
0x31,0x3a,0x6e,0x34,0x31,0x35,0x3a,0x66,0x31,0x32,0x31,0x3a,0x2d,0x36,0x33,0x32,
0x33,0x3a,0x66,0x32,0x31,0x32,0x3a,0x21,0x36,0x31,0x32,0x3a,0x66,0x32,0x31,0x32,
0x3a,0x2b,0x57,0x31,0x56,0x3a,0x2b,0x32,0x31,0x51,0x3a,0x2b,0x44,0x31,0x33,0x3a,
0x2a,0x32,0x31,0x37,0x49,0x23,0x4d,0x4c,0x43,0x35,0x3b,0x39,0x35,0x34,0x20,0x30,
0x3a,0x2a,0x49,0x38,0x77,0x4a,0x3a,0x76,0x71,0x31,0x74,0x31,0x4a,0x3a,0x6d,0x31,
0x31,0x31,0x3a,0x76,0x73,0x31,0x72,0x31,0x4a,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x76,
0x49,0x34,0x67,0x33,0x4a,0x3a,0x41,0x49,0x33,0x67,0x31,0x4c,0x3a,0x41,0x49,0x32,
0x67,0x31,0x4c,0x3a,0x6d,0x31,0x33,0x31,0x3a,0x76,0x4e,0x34,0x67,0x33,0x4a,0x3a,
0x41,0x4e,0x33,0x67,0x31,0x4c,0x3a,0x41,0x4e,0x32,0x67,0x31,0x4c,0x3a,0x6d,0x31,
0x33,0x31,0x3a,0x3d,0x6f,0x30,0x6b,0x3a,0x66,0x70,0x31,0x78,0x3a,0x78,0x75,0x30,
0x42,0x3a,0x66,0x38,0x31,0x62,0x3a,0x3d,0x43,0x31,0x6b,0x3a,0x66,0x62,0x32,0x35,
0x3a,0x3d,0x49,0x32,0x67,0x3a,0x66,0x35,0x31,0x34,0x3a,0x3d,0x49,0x31,0x70,0x3a,
0x66,0x35,0x32,0x6a,0x3a,0x3d,0x35,0x30,0x70,0x3a,0x66,0x67,0x33,0x35,0x3a,0x3d,
0x35,0x30,0x39,0x3a,0x66,0x72,0x31,0x67,0x3a,0x78,0x61,0x30,0x68,0x3a,0x66,0x65,
0x31,0x38,0x3a,0x3d,0x6a,0x30,0x73,0x3a,0x66,0x35,0x33,0x67,0x3a,0x3d,0x35,0x31,
0x6e,0x32,0x3a,0x66,0x35,0x31,0x32,0x3a,0x78,0x48,0x30,0x49,0x3a,0x66,0x61,0x31,
0x61,0x3a,0x5d,0x6f,0x31,0x75,0x4c,0x3a,0x66,0x31,0x31,0x39,0x3a,0x7c,0x6f,0x32,
0x75,0x4c,0x3a,0x66,0x31,0x31,0x39,0x3a,0x29,0x48,0x31,0x71,0x4a,0x3a,0x66,0x31,
0x31,0x64,0x3a,0x78,0x62,0x30,0x66,0x3a,0x66,0x62,0x31,0x32,0x3a,0x78,0x64,0x30,
0x65,0x3a,0x66,0x37,0x31,0x31,0x3a,0x78,0x35,0x30,0x39,0x3a,0x66,0x33,0x31,0x33,
0x3a,0x66,0x36,0x31,0x31,0x3a,0x66,0x31,0x31,0x36,0x3a,0x78,0x6d,0x30,0x39,0x3a,
0x66,0x63,0x31,0x31,0x3a,0x78,0x73,0x30,0x61,0x3a,0x66,0x35,0x31,0x36,0x3a,0x78,
0x75,0x30,0x67,0x3a,0x66,0x32,0x31,0x32,0x3a,0x78,0x70,0x30,0x61,0x3a,0x66,0x33,
0x31,0x32,0x3a,0x78,0x72,0x30,0x63,0x3a,0x41,0x6e,0x30,0x68,0x49,0x3a,0x41,0x6d,
0x30,0x67,0x49,0x3a,0x41,0x6c,0x30,0x66,0x49,0x3a,0x41,0x6a,0x30,0x65,0x49,0x3a,
0x41,0x61,0x30,0x68,0x4c,0x3a,0x41,0x62,0x30,0x66,0x4c,0x3a,0x41,0x64,0x30,0x65,
0x4c,0x3a,0x41,0x77,0x30,0x6a,0x3a,0x41,0x76,0x30,0x69,0x3a,0x41,0x75,0x30,0x68,
0x3a,0x41,0x74,0x30,0x67,0x3a,0x41,0x73,0x30,0x66,0x3a,0x41,0x72,0x30,0x64,0x3a,
0x41,0x71,0x30,0x63,0x3a,0x41,0x70,0x30,0x62,0x3a,0x41,0x6f,0x30,0x61,0x3a,0x41,
0x6d,0x30,0x39,0x3a,0x41,0x35,0x30,0x65,0x4a,0x3a,0x41,0x36,0x30,0x63,0x4a,0x3a,
0x41,0x37,0x30,0x62,0x4a,0x3a,0x41,0x38,0x30,0x61,0x4a,0x3a,0x41,0x61,0x30,0x39,
0x4a,0x3a,0x3d,0x69,0x30,0x75,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x68,0x30,0x75,
0x49,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x69,0x30,0x76,0x49,0x3a,0x66,0x31,0x33,
0x31,0x3a,0x78,0x6f,0x30,0x49,0x3a,0x66,0x31,0x31,0x61,0x3a,0x78,0x70,0x30,0x4e,
0x3a,0x66,0x31,0x31,0x34,0x3a,0x78,0x70,0x30,0x51,0x3a,0x66,0x34,0x31,0x31,0x3a,
0x78,0x44,0x30,0x51,0x3a,0x66,0x34,0x31,0x31,0x3a,0x78,0x47,0x30,0x4e,0x3a,0x66,
0x31,0x31,0x34,0x3a,0x41,0x75,0x30,0x4c,0x3a,0x41,0x42,0x30,0x4c,0x4a,0x3a,0x41,
0x70,0x30,0x4e,0x49,0x3a,0x41,0x71,0x30,0x50,0x49,0x3a,0x41,0x73,0x30,0x51,0x49,
0x3a,0x41,0x44,0x30,0x51,0x4c,0x3a,0x41,0x46,0x30,0x50,0x4c,0x3a,0x41,0x47,0x30,
0x4e,0x4c,0x3a,0x6e,0x4d,0x30,0x48,0x4c,0x3a,0x66,0x31,0x33,0x31,0x3a,0x3e,0x78,
0x30,0x72,0x3a,0x66,0x34,0x31,0x32,0x3a,0x3d,0x75,0x30,0x6c,0x31,0x3a,0x66,0x34,
0x31,0x34,0x3a,0x6f,0x75,0x30,0x41,0x3a,0x66,0x38,0x31,0x31,0x3a,0x78,0x6f,0x30,
0x6c,0x3a,0x66,0x32,0x31,0x38,0x3a,0x78,0x71,0x30,0x6e,0x3a,0x66,0x32,0x31,0x34,
0x3a,0x41,0x6a,0x30,0x70,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x6b,0x30,0x71,0x3a,
0x66,0x31,0x33,0x31,0x3a,0x41,0x6c,0x30,0x72,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,
0x6d,0x30,0x73,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x6e,0x30,0x74,0x3a,0x66,0x31,
0x33,0x31,0x3a,0x78,0x6b,0x30,0x70,0x3a,0x66,0x31,0x33,0x31,0x3a,0x78,0x6e,0x30,
0x73,0x3a,0x66,0x31,0x33,0x31,0x3a,0x41,0x6f,0x30,0x74,0x4a,0x3a,0x41,0x70,0x30,
0x73,0x4a,0x3a,0x41,0x71,0x30,0x72,0x4a,0x3a,0x41,0x72,0x30,0x71,0x4a,0x3a,0x41,
0x73,0x30,0x70,0x4a,0x3a,0x41,0x73,0x30,0x6f,0x49,0x3a,0x41,0x72,0x30,0x6e,0x49,
0x3a,0x41,0x71,0x30,0x6d,0x49,0x3a,0x41,0x70,0x30,0x6c,0x49,0x3a,0x41,0x6f,0x30,
0x6b,0x49,0x3a,0x3d,0x38,0x32,0x78,0x3a,0x66,0x39,0x31,0x39,0x3a,0x3d,0x35,0x32,
0x75,0x3a,0x66,0x37,0x31,0x33,0x3a,0x3d,0x67,0x32,0x43,0x3a,0x66,0x33,0x31,0x36,
0x3a,0x41,0x35,0x32,0x77,0x49,0x3a,0x41,0x36,0x32,0x78,0x49,0x3a,0x41,0x37,0x32,
0x79,0x49,0x3a,0x41,0x38,0x32,0x7a,0x49,0x3a,0x41,0x39,0x32,0x41,0x49,0x3a,0x3d,
0x37,0x32,0x78,0x3a,0x41,0x61,0x32,0x75,0x3a,0x41,0x62,0x32,0x76,0x3a,0x41,0x63,
0x32,0x77,0x3a,0x78,0x62,0x32,0x75,0x3a,0x41,0x67,0x32,0x48,0x49,0x3a,0x41,0x66,
0x32,0x47,0x49,0x3a,0x41,0x65,0x32,0x46,0x49,0x3a,0x41,0x64,0x32,0x45,0x49,0x3a,
0x41,0x65,0x32,0x79,0x3a,0x41,0x66,0x32,0x7a,0x3a,0x41,0x68,0x32,0x42,0x3a,0x41,
0x69,0x32,0x43,0x3a,0x78,0x67,0x32,0x7a,0x3a,0x78,0x65,0x32,0x78,0x3a,0x78,0x66,
0x32,0x78,0x3a,0x66,0x32,0x31,0x32,0x3a,0x41,0x39,0x32,0x42,0x4a,0x3a,0x41,0x37,
0x32,0x44,0x4a,0x3a,0x3d,0x38,0x30,0x43,0x31,0x3a,0x66,0x31,0x34,0x31,0x3a,0x78,
0x38,0x32,0x42,0x3a,0x78,0x38,0x32,0x41,0x3a,0x41,0x63,0x32,0x45,0x4c,0x3a,0x41,
0x61,0x32,0x47,0x4c,0x3a,0x3d,0x62,0x30,0x46,0x31,0x3a,0x66,0x31,0x34,0x31,0x3a,
0x78,0x63,0x32,0x46,0x3a,0x78,0x64,0x32,0x46,0x3a,0x41,0x37,0x32,0x45,0x49,0x3a,
0x41,0x38,0x32,0x46,0x49,0x3a,0x41,0x39,0x32,0x47,0x49,0x3a,0x3d,0x64,0x30,0x78,
0x31,0x3a,0x66,0x31,0x34,0x31,0x3a,0x3d,0x67,0x30,0x41,0x31,0x3a,0x66,0x31,0x34,
0x31,0x3a,0x3d,0x49,0x33,0x71,0x3a,0x66,0x35,0x36,0x64,0x3a,0x78,0x4a,0x33,0x71,
0x3a,0x66,0x33,0x35,0x64,0x3a,0x5e,0x49,0x38,0x71,0x4a,0x3a,0x66,0x31,0x31,0x64,
0x3a,0x5e,0x4d,0x38,0x71,0x4c,0x3a,0x66,0x31,0x31,0x64,0x3a,0x5d,0x4a,0x33,0x71,
0x4c,0x3a,0x66,0x31,0x31,0x64,0x3a,0x5d,0x4c,0x33,0x71,0x4a,0x3a,0x66,0x31,0x31,
0x64,0x3a,0x7c,0x4a,0x34,0x71,0x4c,0x3a,0x66,0x31,0x31,0x64,0x3a,0x7c,0x4c,0x34,
0x71,0x4a,0x3a,0x66,0x31,0x31,0x64,0x3a,0x7c,0x4a,0x36,0x71,0x4c,0x2d,0x3a,0x66,
0x31,0x31,0x64,0x3a,0x7c,0x4c,0x36,0x71,0x4a,0x2d,0x3a,0x66,0x31,0x31,0x64,0x3a,
0x5d,0x4a,0x37,0x71,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x64,0x3a,0x5d,0x4c,0x37,0x71,
0x4a,0x2d,0x3a,0x66,0x31,0x31,0x64,0x3a,0x3d,0x48,0x30,0x77,0x3a,0x66,0x31,0x38,
0x31,0x3a,0x41,0x48,0x31,0x78,0x49,0x3a,0x66,0x31,0x37,0x31,0x3a,0x41,0x48,0x31,
0x76,0x7c,0x3a,0x66,0x31,0x37,0x31,0x3a,0x78,0x49,0x38,0x76,0x3a,0x66,0x32,0x31,
0x33,0x3a,0x41,0x4a,0x38,0x78,0x7c,0x3a,0x41,0x4a,0x38,0x76,0x49,0x3a,0x3c,0x6f,
0x33,0x44,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x3d,0x6f,0x31,0x44,0x3a,0x66,0x31,
0x32,0x35,0x3a,0x27,0x70,0x33,0x44,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x3c,0x70,
0x32,0x44,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x35,0x3a,0x5e,0x71,0x34,0x44,0x4a,0x3a,
0x66,0x31,0x31,0x35,0x3a,0x5e,0x71,0x33,0x44,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x35,
0x3a,0x5e,0x72,0x35,0x44,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x5e,0x72,0x34,0x44,
0x4c,0x2d,0x3a,0x66,0x31,0x31,0x35,0x3a,0x27,0x73,0x35,0x44,0x4c,0x2d,0x3a,0x66,
0x31,0x31,0x35,0x3a,0x3c,0x73,0x36,0x44,0x4c,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,
0x3c,0x74,0x35,0x44,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x35,0x3a,0x27,0x74,0x36,0x44,
0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x3d,0x75,0x36,0x44,0x31,0x3a,0x66,0x38,0x31,
0x35,0x3a,0x3c,0x43,0x35,0x44,0x4a,0x2d,0x3a,0x66,0x31,0x31,0x35,0x3a,0x27,0x43,
0x36,0x44,0x4c,0x3a,0x66,0x31,0x31,0x35,0x3a,0x27,0x44,0x35,0x44,0x4a,0x2d,0x3a,
0x66,0x31,0x31,0x35,0x3a,0x3c,0x44,0x36,0x44,0x4c,0x3a,0x66,0x31,0x31,0x35,0x3a,
0x5e,0x45,0x35,0x44,0x4c,0x3a,0x66,0x31,0x31,0x35,0x3a,0x5e,0x45,0x34,0x44,0x4a,
0x2d,0x3a,0x66,0x31,0x31,0x35,0x3a,0x5e,0x46,0x34,0x44,0x4c,0x3a,0x66,0x31,0x31,
0x35,0x3a,0x5e,0x46,0x33,0x44,0x4a,0x2d,0x3a,0x66,0x31,0x31,0x35,0x3a,0x27,0x47,
0x33,0x44,0x4c,0x3a,0x66,0x31,0x31,0x35,0x3a,0x3c,0x47,0x32,0x44,0x4a,0x2d,0x3a,
0x66,0x31,0x31,0x35,0x3a,0x3c,0x48,0x33,0x44,0x4c,0x3a,0x66,0x31,0x31,0x35,0x3a,
0x3d,0x48,0x31,0x44,0x3a,0x66,0x31,0x32,0x35,0x3a,0x49,0x75,0x30,0x44,0x31,0x3a,
0x66,0x31,0x36,0x31,0x3a,0x49,0x75,0x30,0x48,0x31,0x4a,0x3a,0x66,0x31,0x36,0x31,
0x3a,0x49,0x42,0x30,0x44,0x31,0x7c,0x3a,0x66,0x31,0x36,0x31,0x3a,0x49,0x42,0x30,
0x48,0x31,0x4c,0x3a,0x66,0x31,0x36,0x31,0x3a,0x3c,0x41,0x31,0x6b,0x4a,0x3a,0x66,
0x31,0x31,0x35,0x3a,0x27,0x42,0x31,0x6b,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x3c,
0x43,0x32,0x6b,0x4a,0x3a,0x66,0x31,0x31,0x35,0x3a,0x27,0x44,0x32,0x6b,0x4a,0x3a,
0x66,0x31,0x31,0x35,0x3a,0x3c,0x35,0x31,0x6c,0x32,0x3a,0x66,0x35,0x31,0x31,0x3a,
0x27,0x35,0x31,0x6d,0x32,0x3a,0x66,0x35,0x31,0x31,0x3a,0x3c,0x35,0x32,0x6e,0x32,
0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x35,0x32,0x6f,0x32,0x3a,0x66,0x35,0x31,0x31,
0x3a,0x2b,0x4b,0x37,0x77,0x3a,0x2b,0x79,0x31,0x4f,0x3a,0x2b,0x6b,0x33,0x73,0x3a,
0x2b,0x67,0x31,0x62,0x3a,0x2b,0x63,0x33,0x42,0x3a,0x21,0x39,0x33,0x45,0x3a,0x21,
0x4b,0x33,0x68,0x3a,0x3b,0x6a,0x33,0x77,0x32,0x3a,0x66,0x31,0x32,0x37,0x3a,0x3b,
0x6e,0x33,0x77,0x32,0x7c,0x3a,0x66,0x31,0x32,0x37,0x3a,0x2d,0x6a,0x35,0x77,0x3a,
0x66,0x35,0x31,0x37,0x3a,0x2c,0x6a,0x34,0x78,0x2d,0x3a,0x66,0x31,0x31,0x32,0x3a,
0x2c,0x6a,0x34,0x41,0x2d,0x3a,0x66,0x31,0x31,0x32,0x3a,0x2c,0x6e,0x34,0x78,0x7c,
0x2d,0x3a,0x66,0x31,0x31,0x32,0x3a,0x2c,0x6e,0x34,0x41,0x7c,0x2d,0x3a,0x66,0x31,
0x31,0x32,0x3a,0x27,0x6a,0x35,0x77,0x32,0x4c,0x3a,0x66,0x31,0x31,0x37,0x3a,0x27,
0x6e,0x35,0x77,0x32,0x4a,0x3a,0x66,0x31,0x31,0x37,0x3a,0x29,0x70,0x31,0x4b,0x4c,
0x3a,0x66,0x31,0x31,0x33,0x3a,0x29,0x47,0x31,0x4b,0x4a,0x3a,0x66,0x31,0x31,0x33,
0x3a,0x7c,0x75,0x31,0x41,0x32,0x3a,0x66,0x38,0x31,0x31,0x3a,0x7c,0x75,0x37,0x44,
0x49,0x3a,0x66,0x38,0x31,0x31,0x3a,0x7c,0x75,0x37,0x48,0x3a,0x66,0x38,0x31,0x31,
0x3a,0x28,0x63,0x33,0x70,0x49,0x3a,0x7c,0x63,0x33,0x6f,0x2d,0x3a,0x28,0x67,0x33,
0x70,0x49,0x3a,0x7c,0x67,0x33,0x6f,0x2d,0x3a,0x3b,0x63,0x33,0x74,0x4a,0x3a,0x66,
0x35,0x31,0x31,0x3a,0x7e,0x6e,0x31,0x67,0x31,0x3a,0x7e,0x6f,0x31,0x66,0x32,0x3a,
0x7e,0x70,0x31,0x65,0x32,0x3a,0x7e,0x71,0x31,0x64,0x31,0x3a,0x6e,0x77,0x34,0x6e,
0x32,0x4c,0x3a,0x49,0x77,0x33,0x6e,0x3a,0x6d,0x31,0x32,0x31,0x3a,0x27,0x77,0x30,
0x73,0x4a,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x27,0x43,0x30,0x73,0x4a,0x3a,0x6d,0x31,
0x31,0x31,0x3a,0x7d,0x7a,0x30,0x73,0x32,0x4a,0x7c,0x3a,0x6d,0x31,0x31,0x31,0x3a,
0x3d,0x63,0x32,0x71,0x33,0x3a,0x66,0x35,0x31,0x33,0x3a,0x3b,0x67,0x30,0x38,0x32,
0x4c,0x3a,0x66,0x31,0x35,0x31,0x3a,0x3b,0x67,0x30,0x65,0x32,0x4a,0x3a,0x66,0x31,
0x35,0x31,0x3a,0x2d,0x67,0x35,0x38,0x3a,0x66,0x31,0x31,0x37,0x3a,0x7e,0x67,0x34,
0x39,0x33,0x2d,0x3a,0x7e,0x67,0x34,0x64,0x33,0x2d,
#endif // !LCR_SETTING_ONLY_SMALL_MAPS
0x23,0x4d,0x4c,0x43,0x74,0x69,0x6e,0x79,0x31,0x3b,0x33,0x33,0x30,0x20,0x30,0x3a,
0x2a,0x47,0x31,0x62,0x3a,0x2b,0x6e,0x39,0x48,0x3a,0x21,0x49,0x31,0x48,0x3a,0x3d,
0x45,0x30,0x62,0x3a,0x66,0x36,0x31,0x69,0x3a,0x5e,0x44,0x31,0x62,0x4a,0x3a,0x66,
0x31,0x31,0x69,0x3a,0x5e,0x4a,0x31,0x62,0x4c,0x3a,0x66,0x31,0x31,0x69,0x3a,0x5e,
0x44,0x30,0x62,0x4a,0x2d,0x3a,0x66,0x31,0x31,0x69,0x3a,0x5e,0x4a,0x30,0x62,0x4c,
0x2d,0x3a,0x66,0x31,0x31,0x69,0x3a,0x5e,0x45,0x31,0x73,0x3a,0x66,0x35,0x31,0x31,
0x3a,0x3e,0x45,0x30,0x6b,0x3a,0x66,0x35,0x31,0x34,0x3a,0x5e,0x43,0x32,0x76,0x49,
0x2d,0x3a,0x66,0x36,0x31,0x31,0x3a,0x5e,0x70,0x33,0x76,0x3a,0x66,0x6a,0x31,0x31,
0x3a,0x5e,0x6f,0x34,0x77,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x5e,0x6f,0x33,0x77,0x2d,
0x49,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x5e,0x6e,0x35,0x78,0x3a,0x66,0x6a,0x31,0x31,
0x3a,0x5e,0x6e,0x34,0x78,0x2d,0x49,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x5e,0x6d,0x36,
0x79,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x5e,0x6d,0x35,0x79,0x2d,0x49,0x3a,0x66,0x6a,
0x31,0x31,0x3a,0x5e,0x6c,0x37,0x7a,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x5e,0x6c,0x36,
0x7a,0x2d,0x49,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x5e,0x6b,0x38,0x41,0x3a,0x66,0x6a,
0x31,0x31,0x3a,0x5e,0x6b,0x37,0x41,0x2d,0x49,0x3a,0x66,0x6a,0x31,0x31,0x3a,0x3d,
0x6b,0x38,0x42,0x3a,0x66,0x34,0x31,0x39,0x3a,0x3d,0x6f,0x38,0x46,0x3a,0x66,0x34,
0x31,0x35,0x3a,0x5e,0x6b,0x39,0x4b,0x32,0x49,0x3a,0x66,0x36,0x31,0x31,0x3a,0x5e,
0x6b,0x38,0x4b,0x32,0x49,0x2d,0x3a,0x66,0x36,0x31,0x31,0x3a,0x3d,0x79,0x30,0x46,
0x31,0x3a,0x66,0x62,0x31,0x35,0x3a,0x5e,0x79,0x30,0x46,0x31,0x3a,0x66,0x62,0x31,
0x31,0x3a,0x5e,0x79,0x30,0x4a,0x31,0x49,0x3a,0x66,0x62,0x31,0x31,0x23,0x4d,0x4c,
0x43,0x74,0x69,0x6e,0x79,0x32,0x3b,0x31,0x37,0x35,0x20,0x31,0x3a,0x2a,0x77,0x40,
0x76,0x2d,0x3a,0x21,0x76,0x30,0x75,0x3a,0x66,0x33,0x31,0x33,0x3a,0x3d,0x77,0x63,
0x74,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x3d,0x77,0x63,0x78,0x3a,0x66,0x31,0x4b,0x31,
0x3a,0x3d,0x75,0x63,0x76,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x3d,0x79,0x63,0x76,0x3a,
0x66,0x31,0x4b,0x31,0x3a,0x41,0x76,0x63,0x77,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,
0x75,0x63,0x77,0x49,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,0x76,0x63,0x78,0x49,0x3a,
0x66,0x31,0x4b,0x31,0x3a,0x41,0x78,0x63,0x77,0x4a,0x3a,0x66,0x31,0x4b,0x31,0x3a,
0x41,0x79,0x63,0x77,0x4c,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,0x78,0x63,0x78,0x4c,
0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,0x76,0x63,0x75,0x4c,0x3a,0x66,0x31,0x4b,0x31,
0x3a,0x41,0x75,0x63,0x75,0x4a,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,0x76,0x63,0x74,
0x4a,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,0x78,0x63,0x75,0x49,0x3a,0x66,0x31,0x4b,
0x31,0x3a,0x41,0x79,0x63,0x75,0x3a,0x66,0x31,0x4b,0x31,0x3a,0x41,0x78,0x63,0x74,
0x3a,0x66,0x31,0x4b,0x31,0x3a,0x78,0x75,0x6b,0x74,0x3a,0x66,0x36,0x63,0x36,0x3a,
0x78,0x75,0x45,0x74,0x3a,0x66,0x36,0x63,0x36,0x3a,0x7e,0x77,0x55,0x76,0x33,0x2d,
0x3a,0x7e,0x77,0x56,0x76,0x33,0x3a,0x5c,0x78,0x44,0x76,0x32,0x4a,0x7c,0x3a,0x2e,
0x76,0x51,0x76,0x3a,0x2c,0x77,0x41,0x75,0x31,0x3a,0x66,0x31,0x31,0x33,0x3a,0x7e,
0x78,0x6e,0x77,0x33,0x2d,0x3a,0x7e,0x78,0x6f,0x77,0x33,0x3a,0x7e,0x77,0x65,0x75,
0x33,0x2d,0x3a,0x7e,0x77,0x66,0x75,0x33,0x3a,0x5c,0x76,0x63,0x76,0x32,0x4c,0x7c,
0x23,0x4d,0x4c,0x43,0x74,0x69,0x6e,0x79,0x33,0x3b,0x33,0x38,0x31,0x20,0x32,0x3a,
0x2a,0x77,0x31,0x64,0x3a,0x21,0x77,0x31,0x64,0x3a,0x2b,0x77,0x31,0x41,0x3a,0x76,
0x77,0x31,0x68,0x4a,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x76,0x77,0x31,0x71,0x4a,0x3a,
0x6d,0x31,0x31,0x31,0x3a,0x76,0x77,0x31,0x79,0x4a,0x3a,0x6d,0x31,0x31,0x31,0x3a,
0x76,0x79,0x31,0x72,0x31,0x4a,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x76,0x79,0x31,0x75,
0x32,0x4a,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x76,0x76,0x31,0x74,0x33,0x4a,0x3a,0x6d,
0x31,0x31,0x31,0x3a,0x76,0x75,0x31,0x6b,0x31,0x49,0x3a,0x76,0x75,0x31,0x6c,0x31,
0x7c,0x3a,0x76,0x75,0x31,0x76,0x32,0x49,0x3a,0x76,0x75,0x31,0x77,0x32,0x7c,0x3a,
0x76,0x79,0x31,0x6c,0x4c,0x3a,0x76,0x79,0x31,0x6d,0x3a,0x76,0x79,0x31,0x76,0x31,
0x4c,0x3a,0x76,0x79,0x31,0x77,0x31,0x3a,0x76,0x79,0x31,0x79,0x4c,0x3a,0x76,0x79,
0x31,0x7a,0x3a,0x41,0x77,0x31,0x6d,0x4c,0x3a,0x6d,0x31,0x31,0x31,0x3a,0x3d,0x74,
0x30,0x64,0x32,0x3a,0x66,0x37,0x31,0x70,0x3a,0x3d,0x74,0x31,0x42,0x31,0x3a,0x66,
0x37,0x31,0x31,0x3a,0x5e,0x74,0x31,0x64,0x31,0x4a,0x3a,0x66,0x31,0x31,0x70,0x3a,
0x5e,0x7a,0x31,0x64,0x31,0x4c,0x3a,0x66,0x31,0x31,0x70,0x3a,0x27,0x75,0x30,0x6e,
0x31,0x49,0x3a,0x66,0x35,0x31,0x31,0x3a,0x27,0x75,0x30,0x6f,0x31,0x3a,0x66,0x35,
0x31,0x31,0x3a,0x7e,0x79,0x31,0x69,0x31,0x3a,0x7e,0x77,0x31,0x6a,0x33,0x3a,0x7e,
0x78,0x31,0x6c,0x3a,0x7e,0x78,0x31,0x70,0x33,0x3a,0x7e,0x75,0x31,0x71,0x32,0x3a,
0x7e,0x77,0x31,0x73,0x32,0x3a,0x7e,0x78,0x31,0x76,0x3a,0x7e,0x76,0x31,0x77,0x31,
0x3a,0x7e,0x78,0x31,0x78,0x33,0x23,0x4d,0x4c,0x43,0x74,0x69,0x6e,0x79,0x34,0x3b,
0x31,0x37,0x32,0x20,0x32,0x3a,0x2a,0x4d,0x31,0x41,0x4a,0x3a,0x3d,0x43,0x32,0x77,
0x3a,0x66,0x61,0x63,0x61,0x3a,0x3d,0x43,0x36,0x77,0x32,0x3a,0x66,0x61,0x31,0x61,
0x3a,0x3d,0x48,0x32,0x77,0x32,0x3a,0x66,0x31,0x63,0x61,0x3a,0x3d,0x43,0x30,0x7a,
0x3a,0x66,0x62,0x31,0x34,0x3a,0x78,0x41,0x32,0x78,0x3a,0x66,0x67,0x6f,0x38,0x3a,
0x5e,0x43,0x32,0x77,0x2d,0x3a,0x66,0x62,0x31,0x31,0x3a,0x5e,0x43,0x31,0x78,0x2d,
0x3a,0x66,0x62,0x31,0x31,0x3a,0x5e,0x43,0x30,0x7a,0x2d,0x3a,0x66,0x62,0x31,0x31,
0x3a,0x5e,0x43,0x30,0x43,0x2d,0x49,0x3a,0x66,0x62,0x31,0x31,0x3a,0x5e,0x43,0x31,
0x45,0x2d,0x49,0x3a,0x66,0x62,0x31,0x31,0x3a,0x5e,0x43,0x32,0x46,0x2d,0x49,0x3a,
0x66,0x62,0x31,0x31,0x3a,0x76,0x43,0x32,0x77,0x4c,0x2d,0x3a,0x76,0x43,0x32,0x46,
0x2d,0x3a,0x76,0x4d,0x32,0x77,0x49,0x2d,0x3a,0x76,0x4d,0x32,0x46,0x4a,0x2d,0x3a,
0x2d,0x46,0x65,0x77,0x32,0x3a,0x66,0x34,0x31,0x31,0x3a,0x2d,0x46,0x65,0x46,0x32,
0x3a,0x66,0x34,0x31,0x31,0x3a,0x41,0x43,0x33,0x77,0x7c,0x3a,0x66,0x31,0x62,0x31,
0x3a,0x41,0x4d,0x33,0x77,0x3a,0x66,0x31,0x62,0x31,0x3a,0x41,0x43,0x33,0x46,0x49,
0x3a,0x66,0x31,0x62,0x31,0x3a,0x41,0x4d,0x33,0x46,0x4c,0x3a,0x66,0x31,0x62,0x31,
0x3a,0x5d,0x43,0x32,0x78,0x49,0x3a,0x66,0x62,0x31,0x31,0x3a,0x7c,0x43,0x33,0x78,
0x49,0x3a,0x66,0x62,0x31,0x31,0x3a,0x27,0x43,0x31,0x79,0x49,0x3a,0x66,0x62,0x31,
0x31,0x3a,0x3c,0x43,0x31,0x7a,0x49,0x3a,0x66,0x62,0x31,0x31,0x3a,0x5d,0x43,0x32,
0x45,0x3a,0x66,0x62,0x31,0x31,0x3a,0x7c,0x43,0x33,0x45,0x3a,0x66,0x62,0x31,0x31,
0x3a,0x27,0x43,0x31,0x44,0x3a,0x66,0x62,0x31,0x31,0x3a,0x3c,0x43,0x31,0x43,0x3a,
0x66,0x62,0x31,0x31,0x3a,0x21,0x48,0x67,0x78,0x3a,0x21,0x48,0x67,0x77,0x3a,0x21,
0x48,0x67,0x45,0x3a,0x21,0x48,0x67,0x46,0x23,0x4d,0x4c,0x43,0x74,0x69,0x6e,0x79,
0x35,0x3b,0x32,0x39,0x36,0x20,0x31,0x3a,0x2a,0x78,0x31,0x71,0x3a,0x2b,0x75,0x31,
0x46,0x3a,0x21,0x78,0x31,0x6f,0x3a,0x3d,0x6a,0x30,0x6f,0x3a,0x66,0x67,0x31,0x6a,
0x3a,0x78,0x6a,0x30,0x6f,0x3a,0x66,0x63,0x31,0x38,0x3a,0x78,0x6a,0x30,0x45,0x3a,
0x66,0x36,0x31,0x33,0x3a,0x3d,0x70,0x30,0x77,0x32,0x3a,0x66,0x37,0x31,0x62,0x3a,
0x41,0x6a,0x30,0x44,0x49,0x3a,0x41,0x70,0x30,0x47,0x32,0x49,0x3a,0x41,0x6a,0x30,
0x77,0x4a,0x3a,0x5e,0x6d,0x31,0x41,0x31,0x49,0x3a,0x66,0x64,0x31,0x31,0x3a,0x3d,
0x6d,0x31,0x77,0x3a,0x66,0x39,0x31,0x31,0x3a,0x5e,0x76,0x31,0x6f,0x4a,0x3a,0x66,
0x31,0x31,0x38,0x3a,0x41,0x76,0x31,0x77,0x4c,0x3a,0x3d,0x74,0x30,0x41,0x3a,0x66,
0x32,0x33,0x34,0x3a,0x5e,0x7a,0x31,0x6f,0x4c,0x3a,0x66,0x31,0x31,0x6a,0x3a,0x5e,
0x7a,0x30,0x6f,0x4c,0x2d,0x3a,0x66,0x31,0x31,0x6a,0x3a,0x56,0x6d,0x31,0x78,0x49,
0x3a,0x56,0x6d,0x31,0x7a,0x3a,0x2d,0x77,0x30,0x71,0x31,0x3a,0x2d,0x79,0x30,0x71,
0x31,0x23
};

#define LCR_IMAGE_SIZE 64 ///< one-dimension resolution of bitmap image
#define LCR_IMAGE_STORE_SIZE (LCR_IMAGE_SIZE * LCR_IMAGE_SIZE + 256 * 2)

struct
{
  uint16_t palette[256];
  const uint8_t *pixel;
  const uint8_t *image;
} LCR_currentImage;

uint16_t LCR_getFontChar(char c)
{
  /*
    Font consists of kind of "segmented display" characters, encoded with 16
    bits (each for one segment) like this (numbered from LSB):

     __0__ __2__
    |\12  |\13  |
   1| \  5| \  9|
    |  \  |  \  |
    |__4\_|__6\_|
    |\14  |\15  |
   3| \  7| \ 11|
    |  \  |  \  |
    |__8\_|_10\_|
  */

#define _F(a,b,c,d,e,f,g,h,i,j) return (1 << a) | (1 << b) | (1 << c) | (1 <<\
  d) | (1 << e) | (1 << f) | (1 << g) | (1 << h) | (1 << i) | (1 << j); break;

  switch (c)
  {
    case 'O': case 'o':
    case '0':           _F( 0, 2, 1, 9, 3,11, 8,10,10,10)
    case '1':           _F( 0, 5, 7, 8,10,10,10,10,10,10)
    case 'Z': case 'z':
    case '2':           _F( 0, 2, 9, 4, 6, 3, 8,10,10,10)
    case '3':           _F( 0, 2, 9, 6,11, 8,10,10,10,10)
    case '4':           _F( 1, 4, 6, 9,11,11,11,11,11,11)
    case 'S': case 's':
    case '5':           _F( 0, 2, 1, 4, 6,11, 8,10,10,10)
    case '6':           _F( 0, 2, 1, 3, 8,10,11, 6, 4, 4)
    case '7':           _F( 0, 2, 9,11,11,11,11,11,11,11)
    case '8':           _F( 0, 2, 1, 9, 4, 6, 3,11, 8,10)
    case '9':           _F( 0, 2, 1, 9, 4, 6,11, 8,10,10)
    case 'A': case 'a': _F( 0, 1,13, 4, 6, 3,11,11,11,11)
    case 'B': case 'b': _F( 0, 1,13, 4, 6, 3,11, 8,10,10)
    case 'C': case 'c': _F( 0, 2, 1, 3, 8,10,10,10,10,10)
    case 'D': case 'd': _F( 0, 1, 3, 8,10,13,11,11,11,11)
    case 'E': case 'e': _F( 0, 2, 1, 3, 4, 8,10,10,10,10)
    case 'F': case 'f': _F( 0, 2, 1, 3, 4, 4, 4, 4, 4, 4)
    case 'G': case 'g': _F( 0, 2, 1, 3, 8,10,11, 6, 6, 6)
    case 'H': case 'h': _F( 1, 3, 4, 6, 9,11,11,11,11,11)
    case 'I': case 'i': _F( 0, 2, 5, 7, 8,10,10,10,10,10)
    case 'J': case 'j': _F( 9,11,10,14,14,14,14,14,14,14)
    case 'K': case 'k': _F( 1, 3, 4,15, 5, 2, 2, 2, 2, 2)
    case 'L': case 'l': _F( 1, 3, 8,10,10,10,10,10,10,10)
    case 'M': case 'm': _F( 1, 3,12, 5, 2, 9,11,11,11,11)
    case 'N': case 'n': _F( 1, 3,12,15, 9,11,11,11,11,11)
    case 'P': case 'p': _F( 0, 2, 1, 9, 4, 6, 3, 3, 3, 3)
    case 'Q': case 'q': _F( 0, 2, 1, 9, 3,11, 8,10,15,15)
    case 'R': case 'r': _F( 0, 2, 1, 9, 4, 6, 3,15,15,15)
    case 'T': case 't': _F( 0, 2, 5, 7, 7, 7, 7, 7, 7, 7)
    case 'U': case 'u': _F( 1, 3, 8,10,11, 9, 9, 9, 9, 9)
    case 'V': case 'v': _F( 1,14,10,11, 9, 9, 9, 9, 9, 9)
    case 'W': case 'w': _F( 1,14,10,11, 7, 9, 9, 9, 9, 9)
    case 'X': case 'x': _F(12,15, 3, 4, 6, 9, 9, 9, 9, 9)
    case 'Y': case 'y': _F(12, 6, 9, 7, 8, 8, 8, 8, 8, 8)
    case ':':           _F( 4, 8, 8, 8, 8, 8, 8, 8, 8, 8)
    case '!':           _F( 1, 8, 8, 8, 8, 8, 8, 8, 8, 8)
    case '?':           _F( 0, 2, 9, 6, 7, 7, 7, 7, 7, 7)
    case ';':           _F( 0, 7, 7, 7, 7, 7, 7, 7, 7, 7)
    case '.':           _F( 8, 8, 8, 8, 8, 8, 8, 8, 8, 8)
    case ',':           _F( 7, 7, 7, 7, 7, 7, 7, 7, 7, 7)
    case '-':           _F( 4, 6, 6, 6, 6, 6, 6, 6, 6, 6)
    case '\'':          _F( 5, 5, 5, 5, 5, 5, 5, 5, 5, 5)
    case '"':           _F( 1, 5, 5, 5, 5, 5, 5, 5, 5, 5)
    case '+':           _F( 4, 6, 5, 7, 7, 7, 7, 7, 7, 7)
    case '_':           _F( 8,10,10,10,10,10,10,10,10,10)
    case '|':           _F( 5, 7, 7, 7, 7, 7, 7, 7, 7, 7)
    case '=':           _F( 4, 6, 0, 2, 2, 2, 2, 2, 2, 2)
    case '#':           _F( 5, 7,14,14,14,14,14,14,14,14) // used for "check"
    default: return 0; break;
  }

#undef _F
}

#define LCR_IMAGE_WALL_CONCRETE   0
#define LCR_IMAGE_WALL_WOOD       1
#define LCR_IMAGE_GROUND_CONCRETE 2
#define LCR_IMAGE_GROUND_GRASS    3
#define LCR_IMAGE_GROUND_DIRT     4
#define LCR_IMAGE_GROUND_ICE      5
#define LCR_IMAGE_GROUND_ACCEL    6
#define LCR_IMAGE_GROUND_FAN      7
#define LCR_IMAGE_CAR             20

#if !LCR_SETTING_POTATO_GRAPHICS
static const uint8_t LCR_images[] =
{
  // 0: wall concrete
  0x10,0x84,0x92,0x94,0xf3,0x9c,0x14,0xa5,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,
  0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x6e,0x6b,
  0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,
  0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x75,0xad,
  0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,
  0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,
  0x10,0x22,0x18,0x22,0xf0,0x7b,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,
  0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x72,0x8c,0x10,0x25,0x18,0x25,
  0x51,0x8c,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,
  0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,
  0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,
  0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0xf4,0x9c,
  0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,
  0x10,0x47,0x18,0x47,0x00,0x60,0x10,0x7c,0x08,0x60,0x10,0x60,0x18,0x60,0x00,0x61,
  0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,
  0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,
  0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,
  0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,
  0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,
  0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x18,0x84,0x00,0x85,0x08,0x85,
  0x10,0x85,0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,
  0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,
  0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,
  0x10,0xa3,0xae,0x73,0x18,0xa3,0x00,0xa4,0x08,0xa4,0x10,0xa4,0xef,0x7b,0x18,0xa4,
  0x00,0xa5,0x08,0xa5,0x92,0x8c,0xb6,0xb5,0x10,0xa5,0x18,0xa5,0x14,0x9d,0x00,0xa6,
  0x08,0xa6,0x10,0xa6,0x18,0xa6,0xb4,0x8c,0x00,0xa7,0x08,0xa7,0xcd,0x4a,0x12,0x7c,
  0x10,0xa7,0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x00,0xc1,0x08,0xc1,
  0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0xac,0x42,0x00,0xc3,
  0x08,0xc3,0x10,0xc3,0x18,0xc3,0x96,0xad,0x00,0xc4,0x8b,0x42,0xe8,0x31,0x32,0x8c,
  0x08,0xc4,0xed,0x52,0x10,0xc4,0x6b,0x42,0x13,0x9d,0x18,0xc4,0x00,0xc5,0x09,0x32,
  0x6a,0x3a,0xc7,0x41,0xe8,0x29,0x08,0xc5,0xa7,0x29,0xc7,0x29,0x86,0x49,0xcb,0x72,
  0x76,0xad,0x0b,0x63,0x2a,0x32,0x87,0x39,0xf0,0x83,0x8b,0x72,0xa8,0x49,0x71,0x94,
  0xc2,0xc2,0xc2,0xcb,0xc2,0x3d,0x3d,0xcb,0x3d,0xcb,0x6b,0x32,0x32,0x32,0x32,0x32,
  0x32,0x32,0x32,0x32,0x6b,0x32,0x32,0x6b,0x3d,0x3d,0x3d,0x3d,0x3d,0xc2,0x3d,0x3d,
  0xcb,0xcb,0xcb,0xcb,0x3d,0xcf,0xcf,0x6b,0x7d,0x32,0x32,0x0f,0x0f,0x32,0x7d,0x32,
  0x32,0x7d,0x32,0x0f,0x32,0x32,0x32,0x32,0x6b,0xcf,0x3d,0x3d,0x3d,0x3d,0xcb,0x3d,
  0x32,0x32,0x32,0x00,0x00,0x00,0x00,0xfc,0x32,0x32,0xb9,0xb9,0x32,0x0f,0xb9,0x32,
  0x32,0x32,0x32,0x32,0xb9,0xb9,0x0f,0x32,0xb9,0x0f,0x0f,0x0f,0x0f,0xf9,0x0f,0x3d,
  0xcb,0x32,0x32,0x40,0x00,0x00,0x32,0x32,0x32,0x32,0x32,0xb9,0x32,0xb9,0x32,0x32,
  0xb9,0x32,0x32,0x32,0x0f,0x0f,0x32,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xf9,0x0f,0x32,
  0xc2,0x3d,0x5f,0xc6,0xc6,0xcb,0x5f,0xc2,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xc2,0xcb,
  0xcb,0xcb,0xc2,0xc2,0x3d,0x3d,0x3d,0x3d,0x3d,0xcf,0x3d,0x3d,0x3d,0x5f,0x6b,0xc2,
  0xcb,0x6b,0xcb,0xcb,0xc2,0xcf,0xcf,0xc2,0x3d,0xc2,0xc2,0x3d,0x3d,0x3d,0xcf,0x3d,
  0x3d,0xcb,0xcb,0xcb,0xcf,0xcf,0xcf,0x6b,0xcf,0x32,0xcf,0x3d,0x3d,0xcb,0x3d,0x5f,
  0x5f,0x32,0x00,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0x3d,0xe7,0xe7,0x5f,0x01,0xc2,0xcb,
  0xc2,0x01,0x01,0x3d,0x3d,0x3d,0x3d,0xe7,0x00,0xcf,0xcf,0x00,0x32,0x00,0x32,0x3d,
  0x3d,0x32,0x3d,0x01,0x01,0x3d,0x3d,0x5f,0x3d,0x01,0x3d,0x3d,0x3d,0x3d,0xe7,0x3d,
  0xe7,0xe7,0x3d,0x3d,0xe7,0x3d,0xe7,0x32,0xe7,0xcf,0x32,0x6b,0x32,0x32,0x32,0x5f,
  0x6b,0x32,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x32,
  0x32,0xb9,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x32,
  0x0f,0xb9,0xe7,0x3d,0x3d,0x5f,0xcb,0xcb,0xcb,0x5f,0xcb,0xcb,0x5f,0x5f,0x5f,0x5f,
  0x5f,0x5f,0xcb,0xcb,0xcb,0xcb,0x01,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0x3d,0x0f,0x32,
  0x32,0x32,0x3d,0x3d,0x5f,0xcb,0xcb,0xcb,0xcb,0x5f,0xcb,0xcb,0x5f,0x5f,0xc6,0xc6,
  0x5f,0xc6,0xcb,0xcb,0x5f,0xcb,0xc2,0xcb,0xc6,0x5f,0x01,0xcb,0xc2,0x6b,0x0f,0x32,
  0x3d,0x32,0xcf,0x6b,0xcf,0x6b,0xcf,0x6b,0x6b,0x6b,0x32,0x6b,0x32,0x6b,0x6b,0x6b,
  0x32,0x32,0x32,0x32,0x32,0x32,0x32,0xb9,0x32,0x32,0x32,0xcf,0x32,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x3d,0x00,0x6b,0xe7,0xcf,0x6b,0x6b,0x6b,0x32,0x32,0x6b,0x32,0x32,0x6b,
  0x6b,0x32,0x32,0x3d,0x3d,0x6b,0x32,0x32,0x32,0x32,0x6b,0x6b,0x7d,0x0f,0x0f,0x5f,
  0x3d,0x0f,0xcf,0x32,0xcf,0x3d,0xcf,0xcf,0xcf,0x6b,0x6b,0x6b,0x6b,0xcf,0xcf,0x6b,
  0x6b,0x6b,0xcf,0x6b,0x6b,0x3d,0x00,0xe7,0x3d,0x32,0x32,0x32,0xcf,0x32,0x0f,0x0f,
  0x0f,0x0f,0x3d,0x6b,0xcf,0xe7,0x00,0x6b,0xe7,0x6b,0x6b,0x32,0x6b,0x32,0x6b,0xc2,
  0x3d,0x6b,0x6b,0x32,0x00,0x40,0x6b,0x6b,0x6b,0x32,0x32,0x00,0x32,0x7d,0x0f,0x00,
  0xe9,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xf9,0xf9,0x0f,0x0f,0x0f,0xf9,0xf9,
  0xe9,0xf9,0xe9,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xf9,0x0f,0x0f,0x0f,0xe9,0xf9,0x0f,
  0x0f,0xe9,0xe9,0x0f,0xe9,0xe9,0x0f,0xf9,0xe9,0xf9,0x0f,0x0f,0x0f,0x0f,0xf9,0x0f,
  0x40,0x00,0xfc,0x00,0xfc,0x40,0x40,0x00,0x00,0x00,0x00,0x00,0xb9,0x00,0x3d,0x32,
  0x0f,0xbe,0xb9,0x40,0xfc,0xfc,0x32,0x32,0x32,0x00,0x32,0x32,0xb9,0xb9,0x32,0x00,
  0x00,0x00,0x00,0x00,0x40,0x00,0x40,0x40,0x40,0x00,0x00,0x40,0xbe,0x00,0x3d,0x32,
  0x0f,0x32,0xb9,0x40,0x3d,0x00,0x40,0x01,0xff,0xff,0x40,0x00,0x01,0x40,0x00,0x00,
  0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x01,0x02,0x02,0x01,0xb9,0x3d,0x40,0x00,
  0x32,0xbe,0xe7,0x3d,0x40,0x02,0x01,0x01,0x02,0x01,0x01,0x01,0x3d,0x3d,0x01,0x01,
  0x02,0x02,0x01,0x5f,0x02,0x02,0x5f,0x02,0x03,0x01,0x01,0x01,0xb9,0x32,0x01,0x00,
  0xb9,0x32,0x32,0x3d,0x40,0x5f,0x03,0x1f,0x5f,0x03,0x03,0x02,0x5f,0x5f,0x01,0x01,
  0x5f,0x3d,0x3d,0x3d,0x3d,0x01,0x01,0x01,0x01,0x40,0x40,0x3d,0x01,0x01,0xbe,0x40,
  0x00,0x40,0x5f,0x00,0xfc,0x3d,0x40,0x3d,0x3d,0x01,0x01,0x02,0x01,0x01,0xff,0x01,
  0x02,0x5f,0x03,0x5f,0x01,0x5f,0x01,0x5f,0x5f,0x5f,0x5f,0x01,0x40,0x5f,0xbe,0x00,
  0xbe,0x00,0x01,0x01,0x32,0x5f,0x5f,0x02,0x03,0x1f,0x03,0x5f,0x5f,0x03,0x5f,0x5f,
  0x32,0xb9,0xb9,0xb9,0xf9,0xf9,0xe5,0xf9,0xb9,0xb9,0xb9,0xb9,0xff,0xbe,0xb9,0x0f,
  0xf7,0x0f,0x40,0xf9,0x0f,0x0f,0x0f,0xbe,0xb9,0xbe,0xbe,0xbe,0xbe,0xbe,0xb9,0xb9,
  0xbe,0xb9,0x32,0x32,0x32,0x32,0x7d,0x7d,0x0f,0x32,0x32,0xbe,0x40,0x40,0xb9,0xb9,
  0x0f,0xbe,0x01,0xbe,0x0f,0x0f,0x0f,0xbe,0xb9,0x0f,0xb9,0xb9,0xbe,0x32,0x32,0x0f,
  0x01,0x00,0xbe,0x01,0x01,0x00,0x01,0x00,0xe9,0x0f,0x6b,0x3d,0x40,0x6b,0x00,0xbe,
  0xb9,0xb9,0xb9,0xb9,0x00,0xbe,0x00,0xb9,0xb9,0xbe,0x40,0x02,0xb9,0xbe,0xb9,0xb9,
  0xb9,0xb9,0xbe,0x00,0x01,0x01,0xbe,0x00,0x40,0x01,0x01,0x40,0x32,0x32,0x6b,0x40,
  0x40,0x40,0x00,0x32,0x03,0xcb,0xb9,0xe9,0xb9,0x5f,0x00,0xff,0x00,0xb9,0x0f,0x01,
  0x02,0x01,0x01,0x1f,0x03,0x01,0xbe,0x0f,0x6b,0x01,0x5f,0xc3,0x03,0x1f,0xc3,0xc3,
  0x03,0x03,0x1f,0x5f,0xc3,0x03,0x02,0x40,0x01,0x02,0x02,0x03,0x02,0x1f,0x03,0xc3,
  0x1f,0x1f,0x1f,0x02,0xc3,0x1f,0x03,0x03,0x1f,0x1f,0x1f,0x03,0x02,0x03,0xc3,0xc3,
  0xe3,0xe3,0xc3,0x02,0xc3,0x03,0x5f,0x3d,0xe9,0xbe,0x01,0x03,0x1f,0x40,0x6b,0x5f,
  0x5f,0x3d,0xc2,0x3d,0x01,0xb9,0x0f,0xe7,0x40,0x02,0x1f,0x1f,0x03,0x1f,0x1f,0x1f,
  0x03,0x03,0x03,0x02,0x01,0x02,0x02,0xff,0xff,0x03,0x5f,0x1f,0xc6,0x03,0x1f,0xf8,
  0x1f,0x1f,0x1f,0x01,0x1f,0x1f,0x1f,0x03,0x03,0xe3,0xe3,0x1f,0x02,0xc6,0xc3,0xc3,
  0xc3,0xe3,0xf8,0x5f,0xc3,0x1f,0x01,0xff,0x3d,0xb9,0xb9,0x01,0x01,0x40,0x6b,0x5f,
  0x5f,0x01,0x3d,0x3d,0x0f,0xe9,0x00,0x40,0x02,0x01,0x02,0x03,0x03,0x03,0x1f,0xf8,
  0x1f,0x1f,0x03,0x02,0x02,0x5f,0x40,0x40,0x40,0x02,0x01,0x03,0x5f,0x03,0x1f,0xe3,
  0x1f,0x03,0x03,0xc2,0xc3,0x03,0x03,0x02,0x03,0x1f,0x1f,0x1f,0x5f,0xc6,0xf8,0xe3,
  0xc3,0xc3,0xc3,0xc2,0xc3,0x1f,0xc6,0x5f,0x03,0xc2,0xb9,0xf9,0x3d,0x32,0x00,0x02,
  0x5f,0xc2,0xcb,0x40,0xe9,0x00,0x02,0x02,0x02,0x02,0x02,0x03,0x5f,0x03,0xc3,0x1f,
  0x03,0x03,0x03,0x5f,0xc3,0x03,0x02,0x01,0x01,0x02,0x02,0x1f,0x03,0xf8,0x5f,0x03,
  0x1f,0x1f,0x03,0x01,0xe3,0x1f,0x1f,0x03,0x03,0x1f,0x1f,0xe3,0x5f,0xc6,0xf8,0xc3,
  0xe3,0xc3,0xc3,0x03,0xc3,0x03,0x5f,0x03,0x02,0x01,0x3d,0x0f,0x32,0xcb,0x6b,0x5f,
  0x5f,0x3d,0x01,0x0f,0x32,0x03,0x1f,0x03,0x02,0x1f,0x1f,0x03,0x03,0x03,0xc3,0x1f,
  0x03,0xe3,0x1f,0x02,0xc3,0x02,0x02,0x01,0x02,0x02,0x01,0x5f,0x5f,0x03,0x03,0xec,
  0x1f,0x03,0x03,0x02,0xe3,0x1f,0x1f,0x03,0x1f,0x03,0x03,0x1f,0x5f,0xc6,0xc3,0xc3,
  0xe3,0x1f,0x1f,0x01,0x1f,0x02,0x01,0x01,0x02,0x02,0x02,0x32,0xf9,0xe7,0x40,0x03,
  0x5f,0x01,0x32,0x0f,0x03,0x1f,0x02,0x01,0x01,0x03,0x02,0x1f,0x03,0x02,0x02,0x02,
  0x02,0x02,0x01,0x02,0x03,0x02,0x01,0x40,0x01,0x01,0x01,0x03,0x02,0x02,0x02,0x02,
  0x01,0x01,0x02,0xc2,0x03,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0xe3,0x02,0x02,0x5f,0xec,
  0xc6,0x02,0xc6,0x02,0x03,0x1f,0x03,0x03,0x02,0xc2,0xc6,0xc2,0x32,0x0f,0x3d,0x5f,
  0x5f,0xc2,0x32,0x3d,0x03,0x1f,0x03,0x02,0x01,0x40,0x01,0xc6,0x03,0x1f,0x1f,0x03,
  0x02,0x03,0x03,0x03,0x00,0x40,0x02,0x01,0x02,0x02,0x40,0x02,0x01,0x03,0x03,0x03,
  0x02,0x03,0x03,0x03,0x02,0x02,0xc6,0xc2,0x02,0xc6,0x03,0x02,0x03,0x03,0x1f,0x1f,
  0x03,0x03,0x1f,0x03,0x01,0x02,0xc6,0x03,0x02,0xc2,0xc6,0x02,0x00,0x32,0x3d,0x5f,
  0x5f,0x3d,0x32,0x01,0x01,0x1f,0xff,0x01,0x01,0x01,0x02,0x02,0x1f,0x1f,0x03,0x03,
  0x02,0x02,0x1f,0x03,0x40,0x01,0x02,0xff,0x40,0x40,0x01,0x3d,0xf8,0x03,0x02,0x02,
  0x02,0x01,0x02,0xc6,0x02,0x02,0x02,0x01,0x02,0x02,0x1f,0xc2,0x1f,0x03,0x03,0x02,
  0x03,0x03,0x03,0x03,0x01,0x01,0xff,0x40,0x00,0x40,0x02,0x01,0x01,0x32,0x40,0x5f,
  0x5f,0x3d,0x32,0x5f,0x01,0x02,0x02,0x01,0x01,0x03,0x02,0x02,0x1f,0xec,0x02,0x1f,
  0x03,0x03,0x03,0x1f,0xc2,0x01,0x01,0x40,0x40,0x02,0x01,0xc2,0x03,0x02,0x02,0x03,
  0x03,0xec,0x02,0x03,0x02,0xec,0x03,0xc6,0x02,0x1f,0x1f,0x00,0x1f,0x02,0x03,0x1f,
  0x03,0x03,0x03,0x03,0x02,0x01,0x02,0xc2,0x40,0x02,0x03,0x01,0x03,0x32,0xc2,0x5f,
  0x03,0x40,0xbe,0x02,0x01,0x02,0x03,0x02,0x03,0x02,0x03,0x5f,0xc3,0x03,0x5f,0x03,
  0x03,0x03,0x02,0xc3,0xc2,0x02,0x01,0xff,0x40,0x01,0x02,0x40,0x1f,0xec,0x02,0x03,
  0xec,0x03,0x03,0xf8,0x5f,0x1f,0x1f,0xf8,0x1f,0x1f,0xc3,0x01,0x1f,0x03,0x03,0xc3,
  0xc3,0x03,0x1f,0x03,0x5f,0x02,0x01,0x40,0x01,0x03,0x03,0x40,0x03,0xbe,0xc2,0x5f,
  0x5f,0x40,0x00,0x01,0x40,0x40,0x02,0x01,0x01,0x03,0x02,0x02,0x1f,0x03,0x02,0x03,
  0x03,0x03,0x01,0x1f,0x40,0x01,0x01,0x40,0xff,0x01,0x40,0x40,0x03,0x03,0x02,0xc6,
  0x02,0x02,0x02,0x1f,0x02,0x03,0x03,0xc6,0x03,0x03,0x03,0x3d,0x1f,0x03,0x02,0x03,
  0x03,0x03,0x02,0x03,0x02,0x02,0x03,0x02,0x02,0x03,0x1f,0xc2,0x1f,0x6b,0x3d,0x5f,
  0x5f,0x3d,0x6b,0x02,0xff,0x01,0x01,0x40,0x01,0x01,0x02,0x5f,0x03,0x02,0x02,0x02,
  0x02,0x02,0x03,0x03,0x40,0x02,0x01,0x40,0xff,0x40,0x40,0x40,0x1f,0x03,0x5f,0x02,
  0x02,0x02,0x03,0x03,0x01,0x03,0x1f,0x02,0x03,0x03,0x03,0x00,0x03,0xec,0x02,0x02,
  0x02,0x02,0x02,0x1f,0x02,0x03,0x1f,0x03,0x03,0x03,0x1f,0x40,0x03,0x00,0x3d,0x5f,
  0x01,0x00,0x6b,0x02,0x01,0x02,0x02,0x02,0x3d,0x02,0x02,0x5f,0x1f,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x01,0x3d,0x00,0x00,0x00,0x40,0x40,0x40,0x03,0xc6,0x01,0x01,
  0xec,0x03,0x03,0x02,0x01,0x01,0xec,0x01,0x02,0x02,0x02,0xb9,0x02,0xc2,0x02,0x03,
  0x03,0x03,0x1f,0xc6,0x01,0x01,0x01,0xc2,0xc2,0x02,0x03,0x01,0x03,0x00,0x3d,0x02,
  0x5f,0x40,0x32,0x6b,0x6b,0x40,0x40,0x40,0x40,0xc2,0x01,0x5f,0x5f,0xc6,0x03,0x02,
  0x02,0x02,0x02,0x01,0x40,0x01,0xff,0x40,0x40,0x01,0x02,0x02,0x02,0x02,0x02,0x01,
  0x02,0x01,0x02,0x03,0x01,0x40,0xc6,0x02,0x02,0x03,0x02,0x00,0x40,0x03,0xc3,0xc3,
  0xe3,0x1f,0x03,0x5f,0x01,0x02,0x01,0x01,0x01,0x02,0x1f,0x02,0x02,0x40,0xcb,0x5f,
  0x5f,0x3d,0x32,0x3d,0x01,0x03,0x03,0x02,0x02,0x02,0x40,0x02,0x03,0x40,0x01,0x40,
  0x00,0x40,0x40,0xb9,0x00,0x40,0x40,0xff,0x40,0x01,0x01,0x02,0x40,0x6b,0x40,0x00,
  0x6b,0x00,0x00,0x6b,0x01,0x02,0x02,0x40,0x02,0xc6,0x03,0x01,0x00,0x40,0x03,0x1f,
  0x03,0x03,0x5f,0x6b,0x01,0xc2,0x01,0x01,0x03,0x01,0x02,0x02,0x02,0x00,0x3d,0xc6,
  0x5f,0x40,0x32,0x01,0x02,0x1f,0x03,0x01,0x40,0x01,0x01,0x03,0x01,0x02,0x01,0x01,
  0xbe,0x40,0x00,0xb9,0xff,0x01,0x40,0x01,0x40,0x40,0x01,0x01,0x40,0x01,0x00,0x00,
  0x00,0x40,0x01,0x6b,0x03,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x00,0x3d,0x02,0x02,
  0x03,0x1f,0xc6,0x6b,0x03,0x02,0x02,0x02,0x02,0x01,0x01,0x03,0x02,0x6b,0x01,0x03,
  0x5f,0x00,0xbe,0x40,0xec,0x03,0x03,0x02,0x02,0x02,0x02,0x03,0x02,0xc6,0x02,0x02,
  0x40,0x40,0x40,0x00,0x02,0x02,0x40,0x40,0x00,0x40,0x01,0x01,0x00,0x01,0x40,0x01,
  0x40,0x40,0xff,0x6b,0x02,0x00,0x00,0x40,0x40,0x40,0x02,0x01,0x00,0x01,0x03,0x03,
  0x03,0x03,0x03,0x00,0x1f,0x03,0x02,0x01,0x01,0x40,0xc6,0x02,0x02,0xbe,0x02,0x03,
  0x5f,0x40,0xb9,0x00,0x02,0x01,0x01,0x00,0x40,0x00,0x40,0x02,0x3d,0xc2,0x02,0x02,
  0x01,0x01,0x3d,0x00,0x01,0x01,0x40,0x00,0x00,0x40,0x00,0x01,0x3d,0x02,0x40,0x40,
  0x6b,0x6b,0x40,0xbe,0x02,0x40,0x40,0x40,0x40,0x01,0x02,0x02,0x40,0x01,0x03,0x02,
  0x5f,0x5f,0x02,0x00,0x03,0x01,0x40,0x40,0x01,0x40,0x01,0x00,0x00,0x0f,0xc2,0x03,
  0xcb,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0xcb,0xc2,0x3d,0x3d,0x3d,0xcf,0xcf,0xcf,0xcf,
  0xcf,0xcf,0x3d,0xcf,0xcf,0x3d,0xc2,0xcb,0xcb,0xcb,0xcb,0x5f,0xc2,0xcb,0xcb,0x5f,
  0x5f,0x5f,0xcb,0xcb,0xcb,0xcb,0xcb,0xc2,0x3d,0x3d,0x3d,0xcf,0xcf,0xcf,0x6b,0x6b,
  0xcf,0xcf,0xcf,0x6b,0xcf,0xcf,0xcf,0xcf,0xc2,0x3d,0xcb,0xcb,0x5f,0x5f,0x5f,0x5f,
  0xcf,0x7d,0xe9,0x79,0xf9,0xe9,0xf9,0x0f,0x0f,0xb9,0x0f,0x79,0x0f,0x0f,0x0f,0x32,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x32,0x0f,0x0f,0x0f,0xb9,0xb9,0x0f,0x32,0xe7,
  0xe7,0x32,0xe9,0x79,0xf9,0xf9,0x0f,0x0f,0xb9,0x0f,0x0f,0x0f,0xb9,0x0f,0x0f,0x32,
  0x6b,0x32,0x7d,0x0f,0x7d,0x7d,0x32,0x0f,0x32,0x0f,0x0f,0x32,0x32,0x0f,0x32,0xcf,
  0x3d,0xcf,0x5f,0xcb,0xcb,0x3d,0xcf,0xcf,0x3d,0x3d,0xcf,0xcb,0xcb,0xcb,0xcb,0xcb,
  0xcb,0xcf,0xcb,0xcb,0xcf,0xcf,0xcf,0xcf,0xcf,0x3d,0xcf,0xcb,0x5f,0xcb,0xe7,0xcb,
  0xcb,0x6b,0x5f,0xcb,0xcb,0xcb,0xc2,0x3d,0xc2,0xcb,0xc2,0xcb,0xc6,0x5f,0xc6,0xc6,
  0xc6,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xc2,0xc2,0xcf,0xcf,0xcb,0xcb,0x32,0xcb,
  0xe7,0x7d,0x32,0x32,0x32,0x32,0x32,0xcf,0x32,0x6b,0xcf,0xcf,0x3d,0xe7,0x00,0x3d,
  0xcf,0xcf,0xcf,0xe7,0xcf,0x32,0x32,0xcf,0x6b,0x32,0x32,0xe7,0xe7,0x32,0x32,0x3d,
  0xc2,0x32,0x32,0x32,0xcf,0xcf,0xcf,0x32,0xcf,0xcf,0xcf,0xe7,0xe7,0xcf,0x00,0x3d,
  0x3d,0xe7,0x3d,0x3d,0x3d,0x3d,0x3d,0xe7,0xcf,0x32,0x32,0xe7,0xcf,0x32,0x0f,0xcf,
  0x7d,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x7d,0x32,
  0x32,0x7d,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x32,
  0x32,0x0f,0xcf,0x3d,0x3d,0x3d,0xc2,0xc2,0xcb,0x3d,0x01,0xcb,0xc2,0x5f,0x5f,0x5f,
  0xcb,0xcb,0xcb,0xcb,0x3d,0xcb,0xcb,0xcb,0xcb,0x3d,0x3d,0x3d,0xe7,0x32,0x0f,0x32,
  0x32,0xb9,0xe7,0x3d,0x3d,0xcb,0x5f,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0x5f,0xcb,
  0xcb,0x5f,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xc2,0x3d,0xcb,0x3d,0x0f,0x32,
  0x0f,0x0f,0x6b,0x6b,0x32,0x6b,0x32,0x32,0x32,0x32,0x32,0x32,0x6b,0x6b,0x6b,0x6b,
  0xcf,0x6b,0x32,0xcf,0x6b,0x32,0x32,0x32,0x6b,0x6b,0x6b,0x6b,0x32,0x7d,0x0f,0x6b,
  0x32,0x0f,0x6b,0xcf,0xcf,0x6b,0xcf,0x6b,0x32,0x6b,0x32,0x32,0x6b,0x32,0x6b,0x6b,
  0x6b,0x6b,0x32,0x32,0x32,0x32,0x32,0x32,0x32,0x32,0x32,0xcf,0x32,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x3d,0x40,0xcf,0xe7,0xcf,0x6b,0xcf,0x6b,0x6b,0x32,0x6b,0x32,0x6b,0x3d,
  0x3d,0x6b,0x6b,0x32,0x6b,0x00,0x6b,0x6b,0x6b,0x32,0x32,0x00,0x32,0x32,0x0f,0x32,
  0x32,0x0f,0xcf,0x6b,0xcf,0x3d,0xe7,0x6b,0xcf,0x6b,0x6b,0x6b,0x6b,0xcf,0xcf,0xcf,
  0x6b,0x6b,0xcf,0xcf,0x6b,0x3d,0xe7,0x6b,0xe7,0xcf,0x32,0x32,0x6b,0x32,0x0f,0x0f,
  0xf9,0xf9,0xf9,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xe9,0xf9,0x0f,
  0x0f,0xe9,0xe9,0x0f,0x0f,0xf9,0x0f,0x0f,0xf9,0xf9,0x0f,0x0f,0x0f,0x0f,0x0f,0xf9,
  0xe9,0xf9,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x79,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xf9,0xf9,
  0x00,0x40,0x01,0x01,0x40,0x40,0x01,0x01,0x01,0x3d,0x3d,0x01,0x0f,0x40,0x3d,0x32,
  0xb9,0xb9,0xbe,0x01,0x40,0x00,0x00,0x3d,0x3d,0x3d,0x00,0x3d,0x40,0x00,0x00,0x00,
  0x00,0x32,0xbe,0x32,0x32,0x32,0x00,0x00,0x32,0x00,0x00,0x00,0xb9,0x00,0x01,0x00,
  0x0f,0x32,0xb9,0x01,0x40,0x40,0x40,0x00,0x00,0x40,0x01,0x00,0x40,0x00,0x00,0x40,
  0x01,0x02,0x5f,0x5f,0x03,0x03,0x03,0x5f,0x1f,0x03,0x5f,0x01,0x6b,0xc2,0x00,0x00,
  0xfc,0xbe,0x3d,0x3d,0x40,0xcb,0x01,0x03,0x02,0x5f,0x02,0x5f,0x5f,0x01,0x5f,0x02,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x01,0x02,0x01,0xb9,0x3d,0x3d,0x00,
  0xbe,0x32,0x00,0x3d,0x40,0x02,0x02,0x01,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,
  0x5f,0xcb,0x5f,0x5f,0x5f,0x03,0x03,0x5f,0x02,0x5f,0x02,0x01,0x5f,0x3d,0x00,0x00,
  0x32,0x01,0x5f,0x32,0x3d,0x5f,0x5f,0xcb,0x5f,0x01,0xcb,0x01,0xcb,0x03,0x5f,0x02,
  0x01,0xff,0x01,0x01,0x01,0x01,0x01,0x00,0xe7,0x00,0x40,0x40,0x01,0x5f,0xbe,0x40,
  0xbe,0x40,0x5f,0x3d,0xfc,0xff,0x40,0xff,0x01,0x3d,0x01,0x40,0x3d,0x3d,0x3d,0x5f,
  0xf9,0xb9,0x0f,0xb9,0x0f,0x0f,0xf9,0x0f,0xb9,0xf9,0x0f,0x0f,0xb9,0xb9,0xbe,0x0f,
  0x0f,0xb9,0x40,0xb9,0xbe,0xb9,0x32,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0xb9,
  0xf7,0x0f,0xb9,0xb9,0xb9,0xb9,0xb9,0x0f,0x0f,0x0f,0xf9,0x0f,0xbe,0x00,0xb9,0xf7,
  0xf9,0xb9,0x40,0x0f,0x0f,0x0f,0xf9,0xf9,0xe5,0xe5,0xf9,0x0f,0xf9,0x0f,0x0f,0x0f,
  0x40,0x00,0x00,0x40,0xc2,0x40,0x40,0x40,0x40,0x40,0x40,0x00,0xb9,0x40,0x02,0x02,
  0x40,0x01,0x40,0x40,0x01,0x01,0x03,0xbe,0x01,0x3d,0xc3,0xe3,0xca,0x1f,0xc3,0xc3,
  0xca,0xc3,0xc3,0xc3,0x01,0x03,0x03,0x02,0xf8,0x1f,0xb9,0x01,0x00,0x40,0x01,0x40,
  0x32,0x00,0x00,0xb9,0x01,0xbe,0xbe,0x0f,0xb9,0xbe,0x00,0x40,0xb9,0x00,0x3d,0x40,
  0x1f,0x1f,0x03,0xc6,0xec,0x1f,0x1f,0x03,0x1f,0xf8,0xe3,0x03,0x01,0x1f,0xe3,0xc3,
  0xe3,0x1f,0x1f,0xe3,0xc3,0x1f,0x03,0x6b,0x5f,0x01,0x1f,0xf8,0xca,0x5f,0x1f,0xc3,
  0x1f,0xf8,0xf8,0xc3,0x5f,0xc3,0x1f,0x5f,0xc3,0xc3,0x40,0x1f,0x02,0x1f,0xc3,0x1f,
  0x03,0x03,0xc6,0x40,0x03,0x02,0x02,0x40,0x01,0x03,0x03,0xf8,0x5f,0x5f,0xc3,0x1f,
  0x1f,0x1f,0x5f,0x03,0x1f,0x03,0x03,0x02,0x03,0x1f,0x1f,0x1f,0x01,0x1f,0xe3,0xe3,
  0xc3,0xc3,0x1f,0x03,0x1f,0x1f,0x1f,0x6b,0x5f,0x03,0xe3,0x5f,0xca,0x03,0x03,0xc3,
  0xc3,0x1f,0xf8,0xc3,0x03,0xc3,0xc3,0x5f,0xf8,0x01,0xbe,0x1f,0x03,0x1f,0xc3,0x1f,
  0x1f,0x1f,0x03,0x40,0x1f,0x01,0x40,0x40,0x40,0x02,0x02,0x1f,0x5f,0x02,0xc3,0xf8,
  0x1f,0x1f,0x02,0x02,0x1f,0x1f,0x1f,0x03,0x1f,0x1f,0xe3,0xf8,0x02,0x1f,0xc3,0xc3,
  0xc3,0xc3,0xc3,0xc3,0xc3,0x1f,0x1f,0x32,0x5f,0xc6,0x1f,0x1f,0xca,0x1f,0x03,0x03,
  0x03,0xf8,0xf8,0xe3,0x03,0xe3,0xf8,0x03,0xf8,0xf8,0xb9,0x1f,0x02,0x1f,0xc3,0x1f,
  0x03,0x03,0xc6,0xc2,0xc3,0x03,0x01,0x01,0x01,0x02,0x5f,0xf8,0x03,0x03,0x1f,0x1f,
  0x1f,0x03,0xec,0x02,0x1f,0x1f,0x1f,0xc6,0x03,0x1f,0x03,0xe3,0x02,0x03,0xc3,0xf8,
  0x1f,0x1f,0xc6,0x03,0x03,0xc6,0x03,0x6b,0x02,0xc6,0x03,0x03,0xc3,0xc6,0x1f,0x1f,
  0x03,0x1f,0xe3,0xf8,0x5f,0xe3,0xf8,0x03,0x03,0xf8,0xbe,0x1f,0x02,0x03,0xc3,0xc6,
  0x03,0xf8,0x1f,0x01,0xc3,0x02,0x02,0x01,0x01,0x01,0x02,0x03,0x01,0x03,0x03,0x02,
  0x02,0x02,0xc6,0xc2,0xc6,0x1f,0xe3,0xe3,0x1f,0x1f,0x1f,0xe3,0x02,0x03,0xc6,0x03,
  0x1f,0xc6,0x1f,0x5f,0x1f,0x03,0x1f,0x32,0x01,0x02,0xc6,0xc6,0xe3,0x1f,0xc6,0x03,
  0x1f,0x5f,0x03,0xf8,0x02,0x5f,0xf8,0x5f,0x5f,0xc6,0xbe,0x03,0x03,0x5f,0x1f,0x03,
  0x03,0x03,0xc6,0x5f,0x40,0x02,0x02,0x01,0x01,0x02,0x01,0x5f,0x03,0x02,0x03,0xc6,
  0x02,0x02,0x03,0xc6,0x02,0xc6,0xec,0xec,0x03,0x03,0x1f,0x03,0x02,0x03,0x1f,0x03,
  0x03,0x03,0x1f,0x02,0x02,0x1f,0x1f,0xb9,0xe3,0x02,0xc6,0x02,0x1f,0x02,0x5f,0x5f,
  0x03,0x01,0x01,0xf8,0xc2,0x5f,0x5f,0xc2,0x01,0x5f,0xbe,0xf8,0x1f,0x03,0x1f,0xc6,
  0x02,0x02,0x02,0x02,0x6b,0x02,0x01,0xc2,0x01,0x02,0x02,0xc6,0x02,0x02,0x02,0x03,
  0xc6,0x01,0x01,0x02,0x02,0x02,0x01,0xff,0x01,0x5f,0xc6,0x03,0x03,0x03,0x03,0xec,
  0x03,0x03,0x1f,0xc6,0xc2,0x02,0x02,0xb9,0xe3,0xc2,0xc6,0xc2,0xc6,0xc2,0x3d,0xc2,
  0xc6,0x40,0x3d,0x5f,0x3d,0x5f,0xc2,0x00,0xc2,0x5f,0xbe,0x02,0x1f,0x1f,0x1f,0x02,
  0x01,0x03,0x1f,0x03,0x40,0x01,0x01,0x40,0x40,0x40,0x01,0xc2,0x1f,0x03,0x5f,0xc6,
  0x03,0x01,0x02,0x01,0x01,0x03,0xc6,0x03,0x5f,0x1f,0x03,0x02,0x03,0x03,0x03,0x03,
  0x03,0x03,0x1f,0x03,0x3d,0x03,0x03,0x0f,0x02,0x6b,0x02,0x6b,0xcb,0x3d,0x3d,0x40,
  0x3d,0x40,0x01,0x01,0x3d,0x02,0x3d,0xbe,0x3d,0x01,0xb9,0xcb,0xc3,0x03,0x02,0x03,
  0x03,0x03,0x03,0x03,0x40,0x01,0x02,0x40,0x40,0x01,0x01,0x40,0x1f,0x03,0x03,0x5f,
  0xff,0x03,0x1f,0x1f,0x02,0x1f,0x1f,0x1f,0x1f,0x1f,0x1f,0x02,0x03,0x03,0x1f,0xc3,
  0x1f,0x1f,0x03,0x1f,0xc2,0x1f,0xc6,0x0f,0xc2,0x6b,0xcb,0x6b,0xc2,0x3d,0xc2,0x3d,
  0x3d,0x40,0xc2,0x01,0x6b,0x01,0xc2,0x6b,0x40,0x00,0x00,0x5f,0xc3,0x03,0x01,0xf8,
  0x03,0x03,0x02,0x1f,0x01,0x02,0x02,0xff,0x40,0x01,0x02,0x3d,0x1f,0xc6,0x03,0xec,
  0xff,0x01,0xc6,0x03,0x01,0x1f,0x02,0x03,0x03,0x03,0x03,0x5f,0x03,0xec,0x03,0x1f,
  0x03,0x03,0x02,0x1f,0xc2,0x1f,0x1f,0x0f,0x3d,0x6b,0x02,0x6b,0xc2,0x3d,0x40,0x3d,
  0x6b,0xb9,0x40,0x3d,0x6b,0xc2,0x40,0x32,0x00,0x01,0xbe,0x01,0xc3,0x03,0x02,0xe3,
  0x1f,0x03,0x02,0x1f,0xc2,0x01,0x02,0x01,0x01,0x02,0xff,0x40,0x1f,0x03,0x02,0x02,
  0x02,0xec,0x03,0x02,0x01,0xf8,0x03,0xc6,0x03,0x03,0x02,0x01,0x02,0x02,0x02,0xec,
  0xec,0x02,0x03,0x1f,0xc2,0xc3,0x1f,0xe9,0x6b,0x6b,0x40,0xb9,0x40,0x40,0x6b,0x32,
  0x40,0xb9,0x6b,0x6b,0x32,0xc2,0x6b,0xb9,0x6b,0x00,0xb9,0x02,0x1f,0x02,0x02,0x02,
  0x03,0xec,0x02,0x1f,0x40,0x01,0x01,0x40,0x00,0x40,0x01,0x40,0x1f,0x03,0x5f,0x02,
  0x02,0x02,0x02,0x02,0xc2,0x03,0x02,0x01,0x02,0x02,0xc2,0x00,0x01,0x01,0x03,0x03,
  0x03,0x03,0x1f,0xc6,0x40,0x03,0xc6,0xe9,0x6b,0x6b,0x40,0x6b,0x3d,0x6b,0x6b,0xb9,
  0xb9,0x6b,0x40,0x40,0x0f,0xb9,0x0f,0x0f,0xb9,0x0f,0x6b,0xc6,0xc3,0x03,0x03,0x03,
  0x03,0x02,0x03,0x03,0x40,0x40,0x40,0x00,0x40,0x40,0xc2,0x40,0x1f,0x03,0xff,0xc2,
  0x01,0xff,0xec,0x03,0x40,0xc2,0x02,0x02,0x01,0x02,0x40,0x00,0x01,0x03,0x1f,0x1f,
  0x1f,0x03,0x03,0x02,0x01,0x03,0xc6,0x0f,0xbe,0x6b,0xc2,0x6b,0x02,0xc2,0x00,0x0f,
  0x6b,0x32,0x6b,0x32,0x0f,0x0f,0x0f,0xf9,0x0f,0xb9,0xb9,0xc6,0xc6,0x03,0x02,0x02,
  0x02,0x02,0x02,0xc2,0x02,0x01,0x00,0x40,0x40,0x01,0x02,0x01,0x02,0x02,0x01,0x01,
  0x0f,0xb9,0xbe,0x0f,0xbe,0x6b,0x00,0xbe,0x00,0x00,0x00,0xb9,0x0f,0x32,0xc2,0x3d,
  0x40,0x40,0x00,0x0f,0xb9,0xbe,0x40,0x0f,0x0f,0xf9,0xf9,0xf9,0xf9,0xf9,0x0f,0xf9,
  0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xb9,0x40,0x40,0xf9,0x00,0xbe,0xb9,0x6b,0xbe,
  0xbe,0xb9,0xbe,0x0f,0x0f,0xb9,0xb9,0xb9,0xb9,0xbe,0xbe,0xfc,0x0f,0x0f,0xb9,0x0f,
  0xe5,0xe5,0xde,0xde,0xe5,0xe5,0xe5,0xe5,0xe5,0xde,0xfd,0xfd,0xfd,0xde,0xe5,0xe5,
  0xe5,0xe5,0xde,0xe5,0xe5,0xde,0xde,0xde,0xe5,0xde,0xe9,0xde,0xe5,0xde,0xce,0xf9,
  0xf9,0xf9,0xde,0xde,0xe5,0xe5,0xce,0xde,0xde,0xde,0xde,0xde,0xde,0xde,0xce,0xde,
  0xce,0xde,0xde,0xde,0xe5,0xfd,0xfd,0xde,0xde,0xde,0xde,0xeb,0xe5,0xde,0xe5,0xde,
  0xe9,0xe9,0xe9,0xce,0xe9,0xce,0xce,0xe9,0xce,0xce,0xf7,0xf7,0xf7,0xce,0xe9,0xe9,
  0xe9,0xce,0xce,0xce,0xe9,0xce,0xe9,0xce,0xce,0xce,0xe9,0xe9,0xe9,0xe9,0xce,0xf7,
  0xf7,0xf7,0xce,0xce,0xce,0xde,0xce,0xce,0xde,0xde,0xce,0xde,0xce,0xce,0xce,0xce,
  0xe9,0xce,0xce,0xce,0xe5,0xfd,0xfd,0xde,0xce,0xce,0xe9,0xce,0xce,0xce,0xce,0xe9,
  0xf0,0xf0,0xf0,0xf0,0xeb,0xeb,0xeb,0xf0,0xeb,0xe5,0xfd,0xfd,0xfd,0xeb,0xeb,0xe5,
  0xe5,0xe5,0xe5,0xe5,0xeb,0xe5,0xeb,0xe5,0xe5,0xeb,0xe5,0xeb,0xeb,0xeb,0xf0,0x74,
  0x74,0xeb,0xf0,0xeb,0xeb,0xe5,0xe5,0xe5,0xeb,0xe5,0xeb,0xeb,0xf0,0xf0,0xeb,0xf0,
  0xe5,0xf0,0xf0,0xf0,0x74,0x74,0x74,0xfa,0xf0,0xf0,0xf0,0xf0,0xfa,0xf0,0xf0,0xf0,
  0xf4,0xf4,0xf4,0xf4,0xf4,0xe6,0xf5,0xf5,0xf5,0xf4,0xf1,0xfe,0xfe,0xef,0xe6,0xef,
  0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xef,0xe6,0xe6,0xe6,0xf2,0xf5,0xf5,0xf1,
  0xfe,0xfe,0xe6,0xef,0xe6,0xe6,0xe6,0xef,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xf2,0xf5,
  0xe6,0xe6,0xf5,0xf4,0xfb,0xf6,0xfb,0xf4,0xf4,0xf4,0xf4,0xe6,0xf4,0xf4,0xf4,0xf5,
  // 1: wall wood
  0x53,0xb5,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,
  0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x94,0xc5,0x00,0x03,0x08,0x03,
  0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,
  0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,
  0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,
  0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,
  0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,
  0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,
  0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,
  0xf5,0xcd,0x12,0xb5,0x10,0x41,0x08,0x4a,0x18,0x41,0x00,0x42,0x10,0x42,0x18,0x42,
  0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,
  0x00,0x45,0x08,0x45,0xd0,0xac,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,
  0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,
  0xb0,0xac,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,
  0x10,0x62,0x18,0x62,0x09,0x6b,0x00,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,
  0x10,0x64,0x18,0x64,0xac,0x83,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,
  0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x31,0x84,
  0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,
  0x00,0x82,0x29,0x8b,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x10,0x83,0x18,0x83,
  0x00,0x84,0x08,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,
  0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,
  0x2f,0x94,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,
  0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x11,0xad,0x00,0xa3,0x74,0xbd,0x08,0xa3,
  0x10,0xa3,0x95,0xbd,0x18,0xa3,0x4e,0x9c,0xb2,0xc5,0x00,0xa4,0x08,0xa4,0x10,0xa4,
  0x18,0xa4,0x00,0xa5,0x08,0xa5,0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x51,0x94,
  0x18,0xa6,0xd6,0xcd,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0xcc,0x93,0x00,0xc0,
  0x08,0xc0,0x10,0xc0,0x8f,0xac,0xf1,0xb4,0x18,0xc0,0x00,0xc1,0xec,0x8b,0x08,0xc1,
  0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x4f,0xa4,0x12,0xad,0x10,0xc2,0x18,0xc2,
  0x73,0xc5,0x00,0xc3,0x08,0xc3,0x0a,0x6b,0x10,0xc3,0x2e,0x84,0x16,0xce,0x18,0xc3,
  0x00,0xc4,0xe6,0x41,0x08,0xc4,0x8f,0x9c,0x10,0xc4,0x2d,0xa4,0x11,0xb5,0xae,0x83,
  0x4a,0x8b,0xd4,0x94,0xab,0x93,0x4c,0xa4,0xad,0xac,0xee,0xc4,0xcd,0xb4,0x4a,0x6b,
  0x4c,0x83,0x6b,0x73,0xcf,0xac,0x8e,0xb4,0xca,0x6a,0x15,0xa5,0x8b,0x83,0xef,0x8b,
  0xef,0xef,0xef,0xef,0xef,0xef,0xef,0x74,0xef,0xff,0xa8,0xef,0xef,0xef,0xa8,0xa8,
  0xef,0xef,0xef,0x74,0x74,0xef,0xef,0x74,0x74,0xf7,0x74,0xf7,0x74,0xf7,0x74,0xe3,
  0x6f,0xf9,0xe3,0xf9,0xf9,0xf9,0xe3,0xe3,0xe3,0xe3,0xe3,0xf9,0xe3,0xe3,0xe3,0xf9,
  0xe3,0xe3,0xe3,0xe3,0xe3,0xfc,0xe3,0xe3,0xe3,0x74,0x74,0x74,0xf9,0xef,0xef,0xef,
  0x91,0x91,0x91,0x91,0x91,0x91,0x7a,0xce,0x7a,0x7a,0x7a,0x7a,0x7a,0x7a,0x7a,0xf9,
  0x7a,0x7a,0xf9,0xf9,0xf7,0xe3,0xe3,0xe3,0xf9,0xf7,0xf7,0xe3,0xe3,0xe3,0xe3,0xe3,
  0x4b,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xf9,0xe3,0xf9,0xf9,
  0xe3,0xe3,0xe3,0xe3,0xf9,0xf9,0xf9,0xf7,0x72,0x72,0x91,0x91,0x91,0x91,0x91,0x91,
  0xf6,0xfb,0xf6,0xf6,0xf6,0xf3,0xd2,0x5a,0xdc,0xa8,0xa8,0xbb,0xbb,0xa8,0xa8,0xbb,
  0xd2,0xd2,0xff,0xff,0xa8,0xed,0xed,0xbb,0xbb,0xa8,0xa8,0xa8,0xbb,0xa8,0xa8,0xed,
  0x6f,0x7a,0x7a,0x7a,0xce,0xd6,0x7a,0xd6,0xce,0xce,0xce,0xed,0xce,0x7a,0xed,0xed,
  0xed,0xed,0x7a,0xd6,0xed,0xed,0xed,0x7a,0xf3,0xf3,0x91,0xf0,0xf2,0xce,0xf3,0xf6,
  0xfb,0xfb,0xfb,0xfb,0xf6,0xf6,0xd2,0xd3,0xd2,0xfa,0xfa,0xdc,0xff,0xa8,0xa8,0xa8,
  0xa8,0xdc,0xef,0xef,0xff,0xbb,0xce,0xe3,0x7a,0xef,0xef,0xef,0xce,0x7a,0xce,0xce,
  0x6f,0xed,0x7a,0x7a,0x7a,0x7a,0x7a,0x7a,0x7a,0x7a,0xd6,0xce,0xd6,0x7a,0x7a,0x7a,
  0x7a,0xd6,0xf9,0xf9,0x7a,0xd6,0x7a,0xf9,0xed,0xfb,0xf3,0xf6,0xf6,0xf6,0xfb,0xfb,
  0xf3,0xf3,0xf3,0xf3,0xf3,0xf2,0xfb,0xd3,0xff,0xd2,0xa8,0xff,0xff,0xff,0xa8,0xff,
  0xa8,0xd2,0xa8,0xa8,0xd2,0xfb,0xdd,0x00,0xdd,0xf6,0xed,0xd2,0xfb,0xf6,0xed,0xed,
  0x6f,0xce,0xed,0xed,0xf3,0xce,0xce,0xce,0xc7,0xfd,0xc7,0xce,0xed,0xed,0xce,0xce,
  0xce,0xce,0x7a,0x7a,0xce,0xf3,0xf3,0xd6,0xd2,0xfb,0x91,0xf2,0xce,0xed,0xed,0xf3,
  0xed,0xed,0xed,0xce,0xf2,0xf2,0xbb,0xee,0xbb,0xd2,0xa8,0xef,0xef,0xef,0xff,0xbb,
  0xd2,0xa8,0xf5,0xd2,0xef,0x87,0xb9,0x87,0x00,0x87,0xed,0xed,0xce,0xed,0xce,0x7a,
  0x6f,0x7a,0xf9,0x7a,0x7a,0x7a,0xd6,0xef,0xc8,0xc7,0xb9,0xff,0x7a,0xed,0xed,0x7a,
  0xce,0xd6,0xce,0xf3,0xf3,0xed,0xed,0xf3,0x5a,0xd2,0x91,0xf2,0xf2,0xf2,0xf3,0xce,
  0xd2,0xfb,0xfb,0xf6,0xf3,0xf3,0xfb,0xee,0x7a,0xf9,0xd2,0xef,0xef,0xef,0x87,0xc7,
  0xdc,0xa8,0xa8,0xbb,0xbb,0x87,0xc7,0xef,0xe3,0xef,0xfb,0xed,0xfb,0xed,0xed,0xf5,
  0x6f,0xed,0xed,0xf3,0xf3,0xf3,0xf3,0x87,0xc7,0xef,0xe3,0xff,0xce,0xed,0xf6,0xf3,
  0xc7,0xce,0xf8,0x7a,0x7a,0x7a,0xf9,0x7a,0xd3,0x5a,0xf3,0xf3,0xed,0xfb,0xfb,0xfb,
  0xed,0xf3,0xf2,0xf2,0xf2,0xf2,0xed,0xbc,0x74,0x7a,0xff,0xef,0xef,0xef,0xeb,0x87,
  0xff,0xef,0xef,0xef,0xef,0xef,0x87,0xfc,0xe3,0xa8,0xa8,0xa8,0xbb,0xa8,0xce,0xce,
  0x4b,0xd6,0xce,0x7a,0x7a,0x7a,0x7a,0x7a,0xc7,0x6f,0x74,0x7a,0xd6,0x7a,0xce,0x7a,
  0xeb,0xce,0xce,0xf3,0xf6,0xf5,0xf3,0xf5,0xbc,0xd2,0x91,0xf2,0xf2,0xf2,0xf3,0xed,
  0xef,0xc7,0xff,0x7a,0x7a,0xf9,0xf7,0x7a,0xbb,0xd2,0xef,0xef,0xef,0xef,0x5a,0xff,
  0xef,0xef,0xef,0xef,0xef,0xff,0xef,0xf9,0xf8,0xef,0xef,0xef,0xef,0xce,0xd6,0xce,
  0xfc,0xce,0xce,0xce,0xed,0xed,0xce,0xce,0x7a,0xe3,0xf9,0xce,0xce,0x7a,0x7a,0xfe,
  0x00,0xd6,0xce,0x7a,0xce,0xce,0xed,0xed,0x7a,0xf9,0xf9,0xf8,0x7a,0xe5,0xa8,0x7a,
  0xe3,0xbb,0xbb,0xa8,0xa8,0xd6,0x7a,0x7a,0xe5,0xa8,0xef,0xef,0xef,0xef,0xdd,0xa8,
  0xef,0xef,0xef,0xef,0xef,0xa8,0xef,0xef,0xef,0xef,0xa8,0xef,0xce,0xed,0xed,0x7a,
  0x72,0x7a,0x7a,0x7a,0xf9,0x7a,0x7a,0xd6,0x7a,0xf9,0xd6,0xd6,0xd6,0xd6,0x7a,0xfe,
  0x5a,0xce,0xce,0xce,0xd6,0xce,0xed,0xed,0x7a,0x7a,0x7a,0xe5,0xa8,0xbb,0xa8,0xe3,
  0x7a,0xf9,0xf9,0xf9,0xf7,0xf9,0xe3,0xf9,0xf7,0xef,0xef,0xef,0x74,0xe3,0xb9,0xd2,
  0x74,0xef,0x7a,0x7a,0x7a,0xce,0x7a,0xf8,0xf9,0xf9,0xf7,0xe3,0xf7,0x74,0x74,0xf9,
  0x6f,0x7a,0xf9,0xf7,0x7a,0x7a,0x7a,0xf8,0xf9,0x7a,0xce,0xfe,0xf8,0xfe,0xf8,0xd6,
  0xb9,0xdd,0xfe,0xf0,0xf2,0xf2,0xce,0xf3,0xfe,0x7a,0xce,0x7a,0x7a,0x7a,0x7a,0x7a,
  0xe3,0xe3,0xe3,0x7a,0xf8,0xf8,0xf8,0xf8,0x7a,0xf9,0xfc,0xe3,0xe3,0xe3,0xdd,0xa8,
  0x4b,0xfc,0xf9,0xf8,0x7a,0xd6,0x7a,0xa8,0xff,0x7a,0xf8,0xf8,0xf9,0xe3,0xe3,0xe3,
  0x4b,0xe3,0xe3,0xe3,0xe3,0xe3,0xfc,0xe3,0xfc,0xe3,0xe3,0xfc,0xe3,0xf9,0xe3,0xe3,
  0x5a,0xd6,0xf0,0xf0,0xfe,0xe3,0xfe,0xce,0xfe,0xce,0xf8,0xf8,0x7a,0xfe,0xf8,0xe3,
  0xe3,0xf9,0xf9,0xe3,0xf9,0xf9,0xf9,0xe3,0xf9,0xf9,0xf8,0xe3,0xf9,0xe3,0xc7,0xef,
  0xe3,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xe3,0xf9,0xe3,0xf8,0x6f,
  0x72,0xe3,0xe3,0x72,0xf9,0xe3,0xe3,0xe3,0xf7,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0x72,
  0xe3,0xf7,0xf9,0xf9,0xf7,0xe3,0xe3,0xe3,0xe3,0xe3,0xe3,0xf9,0xf9,0xf9,0xf7,0xe3,
  0xa8,0xeb,0xeb,0xeb,0xeb,0xeb,0xa8,0xa8,0xeb,0x5a,0xeb,0xeb,0x5a,0xd6,0xf9,0xd6,
  0xbb,0xeb,0xeb,0xeb,0xeb,0xeb,0xeb,0xeb,0xbb,0xeb,0xeb,0xa8,0xa8,0xd6,0xeb,0x7a,
  0xf9,0xbb,0xbb,0xbb,0xeb,0xbb,0xbb,0xed,0xbb,0xbb,0xbb,0xeb,0xeb,0xeb,0xbb,0x7a,
  0xef,0xe5,0xeb,0xbb,0xeb,0xa8,0xa8,0xa8,0xa8,0xa8,0xeb,0xc7,0xeb,0xc7,0xc7,0xa8,
  0xee,0xb6,0x00,0x00,0x49,0xb6,0x5a,0xb4,0x5a,0xb6,0x00,0x00,0x00,0x00,0xe5,0x00,
  0x00,0xb6,0x49,0x49,0xee,0x00,0xb6,0x00,0xee,0x49,0x00,0xb4,0xdd,0xee,0xee,0xed,
  0xa8,0xd3,0xee,0xee,0xb6,0xee,0xee,0xd3,0xd3,0x49,0x49,0x49,0xd3,0x49,0xd3,0xbb,
  0xe5,0xeb,0x49,0xdd,0x00,0xdd,0xb4,0x5a,0x49,0xb4,0x00,0x00,0xdd,0x00,0xdd,0xdd,
  0xb6,0x0d,0x0d,0xe0,0x0d,0xb6,0x49,0x49,0x00,0x48,0x48,0xc9,0x0d,0xb6,0xee,0x5a,
  0xb6,0x0d,0x0d,0x0d,0xe0,0xe0,0x0d,0x00,0x49,0x49,0x48,0x00,0xb6,0xb6,0xd3,0xfa,
  0xbb,0xee,0xbc,0x00,0x0d,0xe0,0xee,0xee,0x00,0xe0,0xb6,0x00,0x00,0xe0,0xe0,0xd3,
  0x5a,0x68,0x49,0x00,0x49,0x49,0xb4,0xdd,0x49,0x49,0xb6,0xb6,0xb6,0xb9,0x00,0x49,
  0xb6,0x48,0x48,0x0d,0x0d,0xe0,0xee,0x00,0x48,0xc9,0xc9,0x0d,0x48,0x48,0xb6,0xb6,
  0xee,0xc9,0x48,0xbc,0x48,0x48,0x0d,0x00,0x00,0xe0,0x0d,0xb6,0x00,0x00,0x00,0xee,
  0xbb,0xee,0xbc,0xe0,0x0d,0xe0,0x00,0xee,0xe0,0xe0,0xe0,0xbc,0xb6,0x0d,0x0d,0xee,
  0x5a,0x5a,0xee,0x00,0xb6,0x00,0x00,0x49,0x00,0x00,0x0d,0x0d,0xb6,0x0d,0x49,0x49,
  0xe0,0x48,0x48,0x48,0x0d,0xe0,0x00,0x49,0x0d,0xe6,0x48,0x0d,0x48,0xb6,0xb6,0xb6,
  0xb6,0xe6,0x48,0xe0,0x48,0x48,0xe0,0x00,0x49,0x00,0xe0,0x00,0xee,0xee,0x00,0xee,
  0xa8,0xee,0xbc,0xe0,0x48,0x0d,0xb6,0x00,0x00,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0x00,
  0x00,0x5a,0x49,0x00,0xd3,0x49,0x00,0xb4,0x00,0x00,0x0d,0x0d,0x00,0xb6,0x49,0x5a,
  0x0d,0xe6,0xe6,0xe6,0x48,0xe0,0x49,0xee,0x0d,0xe6,0x00,0xb6,0xc9,0x48,0xb6,0xc9,
  0x49,0x48,0x48,0x0d,0x48,0x0d,0xe0,0xb6,0xb4,0xb6,0x0d,0xb6,0x00,0x00,0xee,0xd2,
  0xce,0xee,0xbc,0xe0,0x0d,0x0d,0xe0,0x00,0xb6,0xe0,0xe0,0xe0,0x00,0xe0,0xe0,0xd3,
  0x49,0xd3,0x49,0x00,0x49,0x49,0x49,0xb4,0x00,0xb6,0x0d,0x0d,0x49,0x00,0x00,0x5a,
  0xb6,0x48,0xe6,0xc9,0x0d,0x0d,0xb6,0xb6,0x0d,0x48,0xb6,0x48,0x48,0x48,0xe0,0x48,
  0x49,0xc9,0x0d,0x0d,0x48,0x0d,0x0d,0x00,0x49,0x0d,0x0d,0xb6,0xb6,0xb6,0x00,0xd2,
  0xbb,0xee,0xbc,0x0d,0x0d,0x0d,0xe0,0xe0,0x0d,0x0d,0x0d,0x0d,0xe0,0x0d,0xe0,0x68,
  0x49,0x49,0x49,0xb6,0x00,0x49,0xb6,0x5a,0x49,0x0d,0x0d,0x0d,0xb4,0x00,0x00,0xb4,
  0x0d,0xe6,0xe6,0xe6,0xe0,0x00,0xb6,0x0d,0xc9,0x48,0xc9,0x48,0x0d,0x48,0x48,0x0d,
  0x68,0xc9,0x0d,0xe0,0xe0,0xbc,0x0d,0xe0,0xee,0xe0,0x0d,0x0d,0xe0,0xe0,0xe0,0xd3,
  0xbb,0xee,0xee,0xe0,0xe0,0x0d,0xe0,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0xe0,0x49,
  0xd3,0x49,0x00,0x00,0x49,0xb6,0xb6,0x00,0xb6,0xb6,0x0d,0xb6,0x00,0x0d,0x00,0x00,
  0xc9,0xc9,0xe6,0x48,0xe0,0x49,0x0d,0x48,0x48,0xe6,0x48,0xc9,0x0d,0x0d,0x48,0xe0,
  0xd3,0x0d,0x48,0x0d,0x0d,0x48,0x0d,0x48,0xbc,0x0d,0x0d,0x0d,0xb6,0x48,0xbc,0xee,
  0xbb,0xf5,0xbc,0xe0,0xb6,0xbc,0xe0,0xe0,0x0d,0x0d,0x0d,0xe0,0x0d,0x0d,0xee,0xd3,
  0x49,0x49,0x49,0x49,0xd3,0x00,0x00,0x49,0xb6,0xb6,0xb6,0x0d,0x48,0x00,0xb6,0x00,
  0xc9,0xe6,0xe6,0x0d,0x00,0xd3,0x0d,0x0d,0x0d,0xe6,0x48,0xc9,0x0d,0x0d,0x0d,0x48,
  0x49,0x0d,0x48,0x0d,0x0d,0x48,0xe0,0x48,0xee,0xe0,0x48,0x48,0xe0,0xc9,0xbc,0xd2,
  0xbb,0xee,0xbc,0xe0,0xbc,0xe0,0x0d,0x48,0x0d,0x48,0x0d,0x0d,0x0d,0x00,0xb6,0xee,
  0xee,0xee,0x49,0xb6,0x00,0x00,0xb6,0x00,0xb6,0xb6,0x48,0x48,0xb6,0xb6,0xb6,0x00,
  0x48,0xe6,0x0d,0x0d,0x00,0x5a,0x0d,0x0d,0x48,0xe6,0x48,0xe6,0xc9,0x0d,0x48,0xc9,
  0xb6,0x00,0x48,0x0d,0x48,0x0d,0x00,0x0d,0xb6,0xee,0x48,0x0d,0x0d,0x48,0xbc,0xfa,
  0xbb,0xbc,0xe0,0xe0,0xe0,0xe0,0xe0,0x48,0x0d,0x0d,0x0d,0x0d,0x0d,0xb6,0xb6,0xee,
  0xdd,0x00,0x00,0xb6,0x49,0xb6,0xb6,0xb6,0xb6,0x00,0x48,0x48,0xb6,0x48,0x0d,0xb6,
  0xc9,0xe6,0x00,0x49,0x5a,0x5a,0x48,0xb6,0xc9,0xe6,0xe6,0xc9,0xc9,0x0d,0x48,0x48,
  0x00,0xb6,0x48,0xb6,0x0d,0x48,0x49,0xe0,0x00,0xb6,0x48,0x48,0x0d,0x0d,0xbc,0xfa,
  0xeb,0xbc,0xbc,0xe0,0xbc,0x0d,0xe0,0xbc,0xe0,0x0d,0x0d,0xb6,0x48,0x0d,0x5a,0xd3,
  0x5a,0xb6,0x49,0x00,0x00,0x00,0x0d,0xb6,0xb6,0xb6,0x48,0xe6,0xb6,0xb9,0xb6,0xb6,
  0xe6,0xe6,0x48,0xe0,0xdd,0xb4,0x00,0x0d,0x0d,0xc9,0x48,0x48,0x0d,0xb6,0xc9,0x48,
  0xb6,0x0d,0x48,0x0d,0xbc,0x48,0xe0,0x00,0x00,0x0d,0xb6,0x00,0x00,0xe0,0xd3,0xee,
  0xbb,0xee,0xe0,0xe0,0xe0,0x48,0xe0,0xe0,0xe0,0x0d,0xe0,0x0d,0x48,0x0d,0x00,0xd3,
  0xee,0x00,0x49,0xb6,0x00,0xb6,0x00,0xb6,0x48,0x0d,0x48,0xc9,0x0d,0x48,0x0d,0xb6,
  0xe6,0xe6,0xe6,0xc9,0x0d,0x00,0x00,0x0d,0x48,0xc9,0x0d,0xc9,0x48,0x0d,0x0d,0x0d,
  0xb6,0x48,0x48,0x0d,0x48,0x0d,0x0d,0xb6,0xe0,0x0d,0x0d,0x00,0xe0,0x0d,0xbc,0xd3,
  0xbb,0xbc,0xbc,0xbc,0xbc,0x0d,0xe0,0xe0,0xe0,0x48,0x0d,0x0d,0x0d,0xe0,0xe0,0xee,
  0xee,0x49,0x49,0x0d,0x00,0x00,0x00,0x49,0xc9,0x0d,0x0d,0xc9,0x0d,0xb6,0xb6,0x00,
  0x0d,0xe6,0xe6,0xe6,0x48,0x0d,0x0d,0x0d,0x48,0xe6,0x0d,0xe6,0xc9,0x48,0x48,0x48,
  0xb6,0xb6,0x48,0x0d,0x00,0x48,0x48,0x0d,0xb6,0xb6,0x0d,0x00,0x0d,0xb6,0xbc,0xfa,
  0xbb,0xbc,0x0d,0x48,0x0d,0xe0,0xe0,0xe0,0x0d,0x48,0x0d,0x0d,0xe0,0x00,0xe0,0x00,
  0xee,0x49,0x49,0x00,0x00,0x49,0x00,0x00,0x0d,0x0d,0x0d,0x48,0x0d,0xb6,0xb6,0xb6,
  0x48,0xe6,0xb9,0xb6,0xb9,0xb6,0x0d,0x0d,0xe6,0xe6,0xb6,0x0d,0x48,0x48,0x48,0x0d,
  0xb6,0x0d,0xb6,0xb6,0xbc,0xc9,0x0d,0x48,0x48,0x48,0x0d,0x48,0x48,0x00,0xb9,0xbb,
  0xce,0xee,0x00,0x00,0x48,0xe0,0xe0,0xe0,0x48,0xe0,0x0d,0x0d,0x0d,0x49,0xb9,0x49,
  0x00,0xdd,0xdd,0xdd,0x00,0x00,0x00,0x00,0xb6,0xc9,0xe6,0xc9,0x48,0xdd,0xfd,0xdd,
  0xe6,0xf1,0xc7,0x00,0xc9,0x0d,0x49,0xe0,0x48,0xc9,0x48,0xb6,0x49,0xf1,0x87,0x00,
  0x49,0x00,0xc7,0xf1,0x00,0x0d,0x0d,0x00,0x0d,0x48,0x48,0x48,0xb6,0xc7,0xc7,0xa8,
  0x68,0x68,0xf1,0xf1,0x48,0xe0,0xbc,0xbc,0xe0,0x48,0xbc,0x00,0x48,0xc7,0xf1,0xc7,
  0xb6,0xc7,0xf1,0xc7,0xb6,0x49,0xb6,0x00,0xb6,0xc9,0xb6,0xc9,0xb6,0xc7,0xf1,0xc7,
  0xe6,0xdd,0x87,0x00,0x48,0x0d,0x0d,0x00,0xe6,0x0d,0xb6,0xe6,0xd3,0xff,0xc7,0x48,
  0xb6,0x00,0xef,0xc7,0xe0,0x48,0x48,0xbc,0x48,0xe6,0x48,0xe6,0x48,0xeb,0xe5,0xbb,
  0xbb,0xfa,0xef,0x5a,0xbc,0xe0,0xbc,0xe0,0x0d,0x48,0x48,0xe0,0x48,0xb4,0xe5,0xb4,
  0x49,0x49,0xe5,0xdd,0x0d,0x00,0x00,0xb6,0xb6,0xc9,0xc9,0x0d,0x48,0xb4,0xe5,0xb4,
  0xe6,0xe6,0xe6,0xb9,0x48,0xe6,0x00,0xb6,0xb6,0xc9,0x00,0xb6,0x49,0x48,0xc9,0xb9,
  0xb6,0x0d,0xb6,0x0d,0xb6,0x0d,0x49,0x00,0x00,0x00,0x00,0x48,0xb6,0xb6,0xb6,0x68,
  0xed,0xee,0x00,0x00,0xe0,0xb6,0x00,0xb6,0xe0,0x48,0x0d,0x0d,0x0d,0xb6,0xb6,0xee,
  0x49,0xb6,0x49,0x0d,0xc9,0x49,0x00,0xb6,0xb6,0xb6,0xb6,0xee,0x48,0xc9,0xc9,0xb6,
  0xff,0xe5,0xc7,0xc7,0xf9,0xf9,0xf9,0x7a,0x7a,0x7a,0xf9,0xa8,0xd6,0x7a,0xe5,0x72,
  0xa8,0xd6,0xa8,0xeb,0xbb,0xa8,0xa8,0xd6,0xd6,0xa8,0xd6,0xff,0xff,0xd6,0xd6,0xf9,
  0xce,0xd6,0x7a,0xd6,0xf9,0x7a,0x7a,0x7a,0x7a,0xf8,0xe3,0xd6,0xfe,0xfe,0xfe,0xe3,
  0x7a,0x7a,0xd6,0x7a,0xf8,0x7a,0x7a,0x7a,0x7a,0x7a,0xf8,0xd6,0xd6,0xd6,0x7a,0xf8,
  0xef,0xdd,0xb9,0x00,0x7a,0x7a,0xbf,0xdd,0xdd,0xbb,0x7a,0xdd,0x5a,0x68,0x0d,0xfc,
  0x00,0x68,0xb4,0xdd,0xb6,0x00,0x49,0x49,0xb4,0x00,0x49,0xd2,0xb4,0x5a,0x49,0xf8,
  0x68,0xbb,0xbb,0xdc,0x7a,0xbb,0xbb,0xbb,0xa8,0xa8,0xf9,0xd2,0xce,0xbb,0xbb,0xe3,
  0x68,0x68,0xd3,0xa8,0xa8,0x68,0x68,0xdc,0xbb,0x5a,0x68,0xeb,0x68,0x68,0x5a,0xeb,
  0xdd,0xdd,0xc7,0xc7,0x7a,0xf8,0x68,0x68,0xb4,0x68,0x7a,0x5a,0x68,0x5a,0xb6,0xfc,
  0x68,0xbb,0x49,0x5a,0x68,0xdd,0x00,0xdd,0x5a,0xb4,0xd3,0x49,0x68,0xdc,0xbb,0xf8,
  0xd2,0xbb,0xbb,0xbb,0x7a,0xa8,0xbb,0xdc,0xbb,0xbb,0xe3,0x68,0xce,0xbb,0xbb,0xe3,
  0xeb,0xeb,0xa8,0xa8,0xbb,0xbb,0xbb,0xbb,0xbb,0xeb,0xbb,0xbb,0xa8,0xdd,0x68,0xeb,
  0xdd,0xdd,0x00,0xc7,0x7a,0xef,0x5a,0xeb,0x5a,0xa8,0x7a,0xb4,0xeb,0x68,0xb6,0xfc,
  0x0d,0xb6,0x0d,0x00,0x49,0xb6,0xdd,0xdd,0x5a,0xeb,0x68,0x5a,0xdd,0xb4,0x68,0xf8,
  0xd2,0xce,0xbb,0xd2,0xf9,0xa8,0x68,0xd2,0xbb,0xdc,0xfc,0xd3,0xd6,0xbb,0xbb,0xf7,
  0x0d,0x49,0x5a,0xb4,0xd3,0x49,0x68,0x68,0xdd,0xb4,0x68,0xb4,0x49,0xb4,0xb4,0xbb,
  0xdd,0xdd,0xdd,0x87,0xf9,0xf8,0xbb,0xeb,0x0d,0xdd,0xef,0x00,0xdd,0x5a,0xb6,0x4b,
  0xe5,0x7a,0x7a,0x7a,0x7a,0xe5,0xef,0x7a,0x7a,0x7a,0xf9,0xf9,0xf9,0x7a,0xe5,0xe3,
  0xbb,0xa8,0xce,0xdc,0xf9,0xa8,0x68,0x5a,0xdc,0xbb,0xfc,0x68,0xbb,0xdc,0xd2,0xf9,
  0x7a,0x7a,0x7a,0x7a,0xf8,0x7a,0x7a,0xd6,0xd6,0x7a,0xd6,0xd6,0x7a,0x7a,0x7a,0xf8,
  0xc7,0xdd,0xdd,0xc7,0x7a,0xef,0xa8,0x5a,0x00,0xb4,0x7a,0x00,0xbb,0xbb,0x00,0xe3,
  0xc7,0xeb,0xa8,0xc7,0xef,0xc7,0xeb,0xeb,0xc7,0xa8,0xff,0xa8,0xa8,0xeb,0xdd,0xf9,
  0xdc,0xbb,0xbb,0xeb,0x7a,0xa8,0xd2,0xb4,0xbb,0xbb,0xfc,0xd3,0xdc,0xdc,0xdc,0x72,
  0x68,0xd2,0x68,0xd3,0x49,0x00,0xd3,0xb4,0x49,0x49,0xd3,0x49,0xd3,0xbf,0xb4,0xbf,
  0x68,0x5a,0xb9,0xdd,0x7a,0x7a,0xa8,0x68,0xb4,0xeb,0x7a,0xdd,0xbb,0xa8,0xb6,0xfc,
  0xef,0x7a,0xbb,0xce,0xeb,0xff,0xff,0xff,0xa8,0xa8,0xce,0x7a,0xf8,0xef,0xa8,0xfc,
  0xbb,0xbb,0xd2,0x68,0x7a,0xa8,0x68,0xbf,0xeb,0xa8,0x6f,0x5a,0xbb,0xd2,0xbb,0x72,
  0xeb,0xdc,0xdc,0x68,0x68,0xb4,0x5a,0xee,0x68,0x5a,0x5a,0xee,0xb4,0xd3,0xd3,0xd2,
  0x5a,0xdd,0x00,0x00,0x7a,0xbb,0xeb,0xdd,0xb4,0xeb,0xef,0x00,0xa8,0xa8,0xb6,0xe3,
  0x68,0x5a,0xb4,0x5a,0xdd,0xdd,0x68,0xb4,0x5a,0x68,0x5a,0xbb,0xce,0xa8,0xbb,0xf9,
  0xdc,0xa8,0xeb,0x68,0x7a,0xa8,0xb4,0x68,0x00,0xbb,0xfc,0x68,0xbb,0x68,0xd2,0xe3,
  0xeb,0xbb,0xa8,0xa8,0x68,0x68,0x68,0x68,0xb4,0x5a,0x68,0x49,0xd3,0x00,0xd3,0x68,
  0xdd,0x5a,0xdd,0xb6,0xe5,0xa8,0xdd,0x49,0xa8,0xa8,0xef,0xb4,0xa8,0xa8,0x00,0xe9,
  0xf9,0xf7,0xf7,0xf9,0xf9,0xe3,0xe3,0xe3,0xe3,0xe3,0xf9,0xe3,0x7a,0xf8,0xe5,0xfc,
  0xdc,0xbb,0xbb,0x68,0x7a,0xeb,0x49,0xeb,0x68,0x68,0xe3,0x5a,0xd2,0xd2,0x68,0xe3,
  0xf8,0x7a,0x7a,0x7a,0xf8,0xf8,0xf8,0x7a,0x7a,0x7a,0xf8,0x7a,0x7a,0xd6,0xd6,0x7a,
  0x00,0xdd,0xdd,0xb9,0xef,0xc7,0x68,0x00,0x68,0xe5,0x7a,0x49,0x68,0xbf,0x49,0xe3,
  0x68,0x68,0x49,0x00,0xb6,0xb6,0x00,0x49,0x49,0x00,0x00,0x00,0x68,0xdd,0xb4,0xf8,
  0x68,0x68,0x68,0x5a,0x7a,0x68,0x5a,0xa8,0x68,0xb4,0xfc,0x68,0xdc,0x5a,0x5a,0x72,
  0xb4,0xee,0x00,0xe0,0xe0,0xe0,0x00,0xbc,0x49,0x49,0xd3,0x68,0xa8,0xa8,0xd3,0xdc,
  0xc7,0x5a,0x68,0xb6,0x7a,0xc7,0x5a,0xb6,0x49,0xff,0xff,0x00,0xd2,0x68,0x5a,0xf9,
  0xb4,0x49,0x00,0x00,0x00,0x0d,0x00,0x00,0x00,0xeb,0xbb,0xa8,0xf8,0xf8,0xbf,0x7a,
  0xdc,0xbb,0x68,0xd2,0x7a,0xa8,0xbb,0xa8,0x68,0xb4,0xfc,0xdc,0xbb,0xd3,0x68,0xe3,
  0x5a,0xb4,0xd3,0x49,0x0d,0xd3,0x49,0x00,0x5a,0x00,0xdc,0x68,0x5a,0x68,0xdc,0xdc,
  0xc7,0xc7,0xeb,0x00,0x7a,0xc7,0x68,0xb6,0x49,0xeb,0xef,0xdd,0xb4,0xbf,0xb4,0xf9,
  0xb4,0x00,0xb6,0x00,0xbf,0xdd,0x00,0x49,0xdd,0xdd,0xeb,0xbf,0xbb,0xef,0x68,0xf8,
  0xdc,0xbb,0xbb,0xa8,0x7a,0x68,0x5a,0x68,0x5a,0x68,0xe3,0xdc,0xdc,0xd2,0xbb,0xe3,
  0xeb,0xdc,0x68,0x5a,0xd3,0xd3,0xb4,0x0d,0x5a,0x49,0xbf,0xbf,0x68,0x68,0x68,0x68,
  0xdd,0x87,0xc7,0xc7,0xf9,0xff,0xc7,0xeb,0x68,0xa8,0x7a,0xa8,0xc7,0xdd,0x5a,0xef,
  0x5a,0xc7,0x00,0xdd,0xef,0xe5,0xeb,0x68,0x68,0xc7,0x68,0x68,0xc7,0xef,0xc7,0x7a,
  0xe5,0x7a,0xa8,0xa8,0x7a,0x87,0xc7,0xc7,0xa8,0xef,0xf9,0xbb,0xbb,0xeb,0xbb,0xf9,
  0xc7,0xa8,0xeb,0xa8,0xeb,0x68,0xbb,0xeb,0x68,0x68,0xbb,0xa8,0xbb,0xe5,0xbb,0xa8,
  0xc7,0xb9,0xb9,0xdd,0xb9,0xc9,0xb9,0x00,0xb6,0xe6,0xb6,0xdd,0xc7,0xb6,0xe6,0xe6,
  0xe6,0xc7,0xb9,0xc9,0xb9,0xdd,0xef,0xc7,0xc7,0xc7,0xc7,0xdd,0xdd,0xb9,0xe5,0xb9,
  0x00,0xe6,0xb9,0xb6,0xe6,0xb9,0xb9,0xc9,0xb6,0xdd,0x00,0xc7,0xdd,0x00,0x00,0x87,
  0x00,0xc7,0x00,0xb6,0xdd,0xc7,0xdd,0xb9,0xf1,0xe5,0xc7,0xb6,0xe6,0x00,0xc7,0x00,
  0xff,0xe6,0xe6,0xb9,0x00,0xc9,0xe6,0xe6,0xe6,0xe6,0xb9,0xb6,0xb9,0xe6,0xb9,0xb9,
  0xb9,0xb9,0xe6,0xb9,0xb6,0xc9,0xc7,0xc7,0x00,0xc7,0xdd,0xc7,0xdd,0xb6,0xb9,0xdd,
  0xb6,0xb6,0xc9,0xb9,0x00,0xb6,0xb6,0xb6,0xb6,0xe6,0x00,0xdd,0x00,0xb6,0xb9,0x00,
  0xb6,0x00,0xdd,0xb6,0xb9,0xc9,0xb9,0xb9,0xc9,0xf1,0x00,0xb9,0xc7,0xdd,0xb6,0xdd,
  0xb9,0xc9,0xe6,0xe6,0xc9,0xe6,0xf1,0xb9,0xb9,0xe6,0xc7,0xb4,0xc7,0xb6,0xe6,0xb9,
  0xb9,0xe6,0xe6,0xb9,0xb9,0x00,0xf1,0xb9,0x87,0xdd,0xdd,0x00,0xb6,0xb6,0x00,0x5a,
  0xe6,0xe6,0xb6,0xb9,0xe6,0xb9,0xb9,0xe6,0xb6,0xb9,0xb9,0xb9,0xb6,0xb6,0x00,0xdd,
  0xb6,0x00,0x00,0x00,0x00,0xb9,0xb6,0xe6,0xb6,0xc7,0xf1,0xb9,0x87,0xb6,0x00,0xfd,
  0xe9,0xe9,0xe9,0x4b,0x4b,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,
  0xe9,0xe9,0xe9,0x72,0xfe,0xfe,0xfe,0xfe,0x72,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,
  0xe9,0xe9,0xe9,0xe9,0xe9,0x4b,0xe9,0xe9,0x4b,0x72,0x72,0x72,0x72,0x72,0xe9,0xe9,
  0xe9,0xe9,0xe9,0xe9,0x4b,0xe9,0x4b,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,
  0xff,0xef,0xc7,0xa8,0xc7,0xa8,0xc7,0xa8,0xa8,0xa8,0xff,0xff,0xff,0xc7,0xe5,0xff,
  0x7a,0x7a,0xce,0xfa,0xbc,0xbc,0xbc,0xfc,0xd6,0xa8,0xa8,0xa8,0x7a,0xa8,0x7a,0x7a,
  0xe3,0xa8,0xa8,0xff,0xa8,0xa8,0xa8,0xc7,0xa8,0x7a,0xce,0x7a,0x7a,0xfa,0xd6,0xff,
  0xff,0x7a,0xa8,0xe5,0xa8,0xc7,0xc7,0x7a,0x7a,0x7a,0xe5,0xe5,0xe5,0xe5,0xa8,0xe5,
  0x68,0xdd,0x00,0x00,0x00,0x00,0xdd,0xb6,0xdd,0xb6,0xdd,0xdd,0x00,0x00,0xdd,0xb4,
  0xff,0xa8,0xfa,0xbc,0xe0,0xee,0x7a,0xdc,0x68,0xb4,0xb4,0x68,0xa8,0x5a,0xd2,0x7a,
  0xf9,0xb4,0xeb,0xb4,0xdd,0xb4,0x00,0x00,0x00,0x68,0xd6,0xed,0xd6,0xd6,0xf4,0xeb,
  0xbb,0x68,0xb4,0xdd,0x00,0x00,0xb9,0x68,0x68,0x68,0xdd,0xdd,0x5a,0xdd,0x5a,0x68,
  0x49,0xb6,0xb9,0x0d,0xc9,0xb6,0x0d,0xb6,0x0d,0xc9,0x00,0x00,0x0d,0xb9,0xb9,0xdd,
  0xa8,0xfa,0xbc,0xbc,0xfa,0xeb,0x49,0x49,0xd3,0x00,0x49,0x49,0x68,0x00,0xee,0x68,
  0xff,0x0d,0xb6,0x49,0xb6,0xb6,0x0d,0x48,0x48,0x0d,0xb6,0xbb,0xf3,0x7a,0xe5,0xf6,
  0xbb,0x5a,0x00,0xb6,0x00,0x0d,0xc9,0x5a,0x49,0x5a,0xd3,0x00,0xdd,0xb6,0x49,0xdd,
  0x00,0xb6,0xb6,0xc9,0x0d,0x00,0x0d,0x48,0x0d,0xe6,0x00,0x0d,0x0d,0xc9,0xb6,0xd2,
  0xfa,0xe0,0xd3,0xee,0xeb,0x00,0x49,0x68,0xd3,0x49,0xb6,0x49,0xd3,0x68,0xd3,0xd2,
  0xa8,0xe0,0x0d,0x00,0x0d,0x48,0x48,0x48,0x0d,0x48,0x48,0x49,0xe5,0xd3,0xce,0x7a,
  0xee,0xdc,0x49,0x0d,0xb6,0x48,0xe6,0x68,0xd3,0x49,0x00,0x00,0x68,0x00,0xb4,0xdd,
  0x00,0x00,0xb6,0x0d,0x0d,0xb9,0x0d,0xb6,0x0d,0xc9,0x00,0x0d,0x48,0xdd,0xa8,0xfa,
  0xee,0xbb,0xbc,0x68,0x49,0x49,0xb4,0xee,0x49,0xee,0x00,0xd3,0xee,0x5a,0x49,0x5a,
  0xeb,0x0d,0x48,0x0d,0x0d,0x48,0x48,0x0d,0xc9,0x0d,0x48,0x48,0xb6,0xd6,0xd3,0xce,
  0xbb,0xbc,0x49,0xb6,0xb6,0x48,0x48,0xb4,0xb4,0x49,0x49,0x49,0x49,0xdd,0x00,0xdd,
  0x00,0x49,0x00,0xc9,0xb6,0xb6,0xb6,0xb6,0x00,0xb6,0x00,0xb6,0xdd,0xce,0xd2,0x7a,
  0xa8,0xee,0xc7,0x49,0xb4,0xb4,0x5a,0x68,0xd3,0xd3,0x49,0xd3,0x5a,0x68,0x49,0x5a,
  0x68,0x0d,0x48,0x0d,0x48,0x48,0x48,0x0d,0xc9,0x0d,0x0d,0x48,0x0d,0xd3,0xbb,0xfa,
  0x5a,0xf3,0xf5,0x5a,0xb6,0xb6,0xb6,0x5a,0xb4,0xb4,0x49,0x00,0xb4,0xb4,0x49,0x68,
  0x68,0x49,0xdd,0xb9,0x00,0xb6,0xb9,0xb9,0xb6,0x0d,0xb6,0xdd,0xce,0xbb,0x7a,0x5a,
  0xbc,0xa8,0x49,0x00,0xb4,0xd3,0x68,0x68,0x5a,0xd3,0xd3,0xbb,0xd2,0x5a,0xee,0x5a,
  0x68,0x0d,0x0d,0xb6,0x48,0x48,0x48,0x0d,0xc9,0x48,0x49,0x0d,0xb6,0x00,0xb4,0x7a,
  0xfa,0xf5,0xf5,0xbc,0xb4,0x0d,0xb6,0x68,0x49,0x49,0x00,0xb6,0x49,0x49,0x49,0xd2,
  0x68,0x68,0x49,0x00,0x00,0x00,0xb9,0x0d,0x0d,0xc9,0xdd,0xd2,0xfa,0x7a,0xbb,0xee,
  0xef,0xdd,0x49,0x68,0xb4,0xbb,0x68,0xd2,0xfa,0x49,0xd3,0xbb,0xd3,0xd3,0x5a,0x5a,
  0x49,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x0d,0x48,0x48,0x00,0xe0,0x49,0x00,0x49,0x5a,
  0xd6,0xd2,0xf5,0xf5,0xbc,0x5a,0x00,0xb4,0xb4,0xb4,0x00,0x00,0x5a,0x49,0x49,0x5a,
  0x68,0xb4,0x00,0xb6,0x00,0x00,0xb9,0xb6,0x0d,0x00,0xbb,0xfa,0xbb,0xd2,0xee,0xa8,
  0x5a,0x5a,0xb4,0x5a,0x68,0x68,0x5a,0xee,0xd3,0xd3,0xee,0xb4,0x5a,0xd3,0xd3,0xd3,
  0xd3,0x00,0xe0,0xe0,0x0d,0x0d,0xb6,0x0d,0x0d,0x0d,0x00,0xb6,0x0d,0x00,0xee,0x00,
  0xa8,0x7a,0xf4,0xf5,0xf5,0xbc,0xb4,0x49,0xb4,0xb4,0xb4,0xb4,0xd2,0x49,0xdd,0x5a,
  0x68,0xb4,0xdd,0x00,0x49,0xb4,0xb6,0x49,0x00,0xbb,0xd2,0xed,0xfb,0xbc,0xeb,0xeb,
  0x5a,0xeb,0xd3,0x68,0x49,0xbf,0x68,0xee,0x5a,0x68,0x5a,0xd2,0xd3,0xd3,0xee,0x5a,
  0xd2,0x49,0xb6,0x00,0x0d,0xb6,0xb6,0xb6,0xb6,0x0d,0xb6,0xb6,0xb6,0x68,0xee,0x00,
  0xf8,0xeb,0xe5,0xed,0xf5,0xf4,0xbb,0xa8,0x68,0x5a,0x5a,0xdd,0xdd,0xdd,0xd3,0xbb,
  0x68,0x5a,0xdd,0x00,0x00,0xb4,0xb6,0xdd,0xd6,0xdc,0x5a,0xbb,0xfa,0xc7,0x00,0xbf,
  0xeb,0xbb,0x68,0x5a,0xb4,0xbb,0x5a,0xd2,0xd2,0xd2,0xd2,0xd2,0x5a,0xd3,0x5a,0x68,
  0x68,0x49,0x49,0x00,0x00,0x00,0xb6,0xb6,0xb6,0x00,0xb6,0xb9,0x00,0x49,0x00,0xb4,
  0xf8,0xbb,0x5a,0x7a,0xce,0xf2,0xd6,0xf7,0xe5,0x00,0xb4,0x00,0x00,0x49,0x49,0xbb,
  0xbf,0x68,0xb4,0x49,0xdd,0x5a,0xdd,0x7a,0xd6,0xe5,0xeb,0xbb,0xff,0x00,0x68,0xbf,
  0xed,0xbb,0xbb,0xd2,0x68,0xdc,0xeb,0xfa,0xbb,0xbb,0xd2,0xbb,0xed,0xd2,0x5a,0xbb,
  0xeb,0xb6,0xb4,0xee,0xee,0xd3,0x00,0xb6,0x49,0xd3,0xb6,0xb6,0x00,0xb4,0x00,0xb4,
  0xe5,0x7a,0x00,0xdd,0xf9,0xf3,0xfe,0xed,0xce,0x68,0xeb,0x00,0x49,0x49,0x49,0xeb,
  0xa8,0xc7,0xf1,0x5a,0x68,0xbf,0x7a,0xa8,0xff,0xe5,0x7a,0xc7,0xa8,0xfd,0xf1,0xbb,
  0xbb,0xd2,0xc7,0xdd,0xd2,0x68,0xed,0xed,0xbb,0xbb,0xed,0xed,0xed,0x68,0xfd,0xa8,
  0xbb,0x68,0xc7,0xdd,0xd3,0x68,0x49,0x49,0xee,0xd3,0x00,0x49,0x00,0xdd,0xf1,0xc7,
  0xbb,0xc7,0x87,0x00,0x5a,0x7a,0xd6,0xce,0xf9,0xf6,0xe5,0xdd,0xb4,0xf1,0xc7,0xef,
  0x5a,0x87,0xf1,0xc7,0xa8,0xa8,0x7a,0xe5,0xf9,0xe5,0x87,0x5a,0xf8,0xf1,0xc7,0xa8,
  0xf2,0xef,0xf1,0xf1,0xce,0xbb,0xbb,0xce,0xed,0xbb,0xed,0xce,0xed,0xff,0xfd,0xef,
  0xbb,0xa8,0xf1,0xc7,0x5a,0xd2,0xd3,0xb4,0x5a,0x68,0x49,0x49,0x49,0x87,0xfd,0xa8,
  0xd2,0xff,0xfd,0xc7,0xdd,0xeb,0xfe,0xfe,0xfe,0xce,0xf5,0xa8,0xb4,0x87,0xf1,0xfe,
  0xeb,0xd6,0xef,0xeb,0xd2,0xfb,0xf9,0x7a,0xe5,0x7a,0x68,0xeb,0xd2,0x7a,0x7a,0xce,
  0xd6,0xa8,0xfc,0xf9,0xce,0xce,0xce,0xf8,0xce,0xce,0xf3,0xf2,0xce,0xf2,0xfc,0xf0,
  0xa8,0xdc,0xf8,0xbb,0x68,0xfa,0xd2,0xd2,0xd2,0x68,0x5a,0x68,0xd3,0xbb,0x7a,0x5a,
  0xce,0xbb,0xd6,0xeb,0xbb,0xeb,0xeb,0xf7,0xf7,0xf3,0xf5,0xf4,0xa8,0xbb,0xd6,0xce,
  // 2: ground concrete
  0x2d,0x63,0x4d,0x6b,0x6e,0x6b,0x8e,0x73,0xcf,0x73,0xf0,0x7b,0x00,0x00,0x08,0x00,
  0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,
  0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,
  0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x8e,0x6b,0x18,0x07,0x00,0x20,
  0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,
  0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,
  0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,
  0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,
  0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,
  0x08,0x42,0x10,0x42,0x18,0x42,0x10,0x84,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,
  0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,
  0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,
  0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,
  0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,
  0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,
  0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,
  0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,
  0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,
  0x00,0x84,0x08,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,
  0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,
  0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,
  0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x0c,0x5b,
  0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,0x18,0xa5,
  0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,
  0x00,0xc0,0x08,0xc0,0x10,0xc0,0x10,0x7c,0x18,0xc0,0x00,0xc1,0x08,0xc1,0x10,0xc1,
  0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,
  0x18,0xc3,0x00,0xc4,0xcb,0x5a,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,
  0x10,0xc5,0x18,0xc5,0x00,0xc6,0x08,0xc6,0x10,0xc6,0xec,0x5a,0x18,0xc6,0x00,0xc7,
  0x08,0xc7,0x10,0xc7,0x18,0xc7,0x00,0xe0,0x08,0xe0,0x10,0xe0,0x18,0xe0,0x00,0xe1,
  0x52,0x84,0x08,0xe1,0x18,0xe1,0x6e,0x73,0x08,0xe2,0x10,0xe2,0x18,0xe2,0x10,0xe3,
  0xb0,0x73,0x08,0xe4,0x0b,0x5b,0x18,0xe4,0x08,0xe5,0x00,0xe6,0x10,0xe6,0xaa,0x52,
  0xff,0xda,0xe5,0x00,0x00,0x00,0xb7,0xda,0xb7,0x00,0xb7,0xb7,0xb7,0xb7,0x00,0x00,
  0x00,0x00,0x00,0x00,0xb7,0xda,0xe5,0xb7,0xb7,0xda,0xda,0xb7,0xb7,0xb7,0xb7,0xda,
  0xda,0xe5,0xda,0xb7,0x00,0xb7,0x00,0x00,0x00,0xfa,0xb7,0xb7,0xb7,0xb7,0x00,0x00,
  0x00,0x00,0xb7,0xb7,0x00,0xb7,0x00,0xb7,0xb7,0xb7,0x00,0x00,0x00,0xb7,0xb7,0xda,
  0xda,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x02,0x00,0x01,
  0x02,0x25,0x01,0x25,0x25,0x01,0x01,0x02,0x00,0x01,0x00,0x00,0xb7,0x00,0x00,0xda,
  0x00,0x00,0x00,0x00,0x01,0x01,0x02,0x03,0x25,0x01,0x01,0x01,0x01,0x02,0x02,0x02,
  0x02,0x01,0x01,0x02,0x00,0x02,0x25,0x01,0x02,0x02,0x01,0x01,0x02,0x02,0x01,0xb7,
  0x00,0x00,0x00,0x01,0x01,0x01,0x02,0x02,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x01,
  0x01,0x02,0x03,0x02,0x02,0x01,0x25,0x25,0x02,0x00,0x02,0x01,0x00,0x00,0x01,0xe5,
  0xb7,0x01,0x01,0x00,0x01,0x01,0x02,0x02,0x02,0x01,0x25,0x03,0x25,0x04,0x03,0x01,
  0x01,0x25,0x25,0x02,0x02,0x02,0x02,0x25,0x02,0x01,0x25,0x02,0x01,0x02,0x01,0xb7,
  0x00,0x00,0x02,0x00,0x01,0x03,0x02,0x04,0x25,0x01,0x02,0x02,0x03,0x04,0x04,0x25,
  0x03,0x03,0x04,0x04,0x02,0x02,0x03,0x03,0x03,0x03,0x04,0x02,0x00,0x00,0x01,0xb7,
  0x00,0x01,0x01,0x00,0x01,0x01,0x03,0x03,0x01,0x03,0x03,0x03,0x01,0x02,0x25,0x25,
  0x03,0x25,0x03,0x03,0x25,0x25,0x02,0x25,0x25,0x25,0x01,0x02,0x02,0x00,0x01,0xb7,
  0x00,0x00,0x01,0x00,0x03,0x04,0x04,0x04,0x01,0x00,0x01,0x01,0x25,0x04,0x04,0x03,
  0x04,0x04,0x04,0x05,0x03,0x03,0x05,0x04,0x05,0x03,0x03,0x02,0x00,0x01,0x03,0xb7,
  0xb7,0x00,0x25,0x02,0x04,0x04,0x05,0x03,0x04,0x04,0x04,0x04,0x25,0x03,0x04,0x04,
  0x04,0x04,0x04,0x04,0x01,0x03,0x03,0x04,0x03,0x03,0x25,0x03,0x02,0x02,0x02,0xb7,
  0x00,0x00,0x00,0xb7,0x25,0x25,0x01,0x01,0x01,0x00,0x02,0x02,0x03,0x03,0x03,0x04,
  0x04,0x04,0x03,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x02,0x00,0x00,0x00,0xb7,
  0xb7,0x01,0x25,0x02,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x03,0x02,0x25,0x03,0x03,
  0x04,0x05,0x03,0x03,0x04,0x01,0x01,0x25,0x02,0x01,0x04,0x04,0x01,0x02,0x02,0xb7,
  0x00,0x00,0x00,0x00,0x02,0x01,0x00,0x03,0x00,0x00,0x02,0x02,0x02,0x25,0x03,0x03,
  0x03,0x05,0x25,0x03,0x04,0x04,0x04,0x03,0x03,0x02,0x25,0x25,0x00,0x00,0x00,0xda,
  0x01,0x02,0x00,0x02,0x04,0x03,0x04,0x04,0x04,0x03,0x04,0x03,0x25,0x03,0x03,0x03,
  0x03,0x03,0x03,0x05,0x05,0x03,0x02,0x01,0x03,0x02,0x02,0x25,0x02,0x01,0x02,0x00,
  0x00,0x00,0xb7,0x00,0x01,0x01,0x00,0x00,0x00,0x01,0x03,0x01,0x25,0x04,0x04,0x02,
  0x01,0x03,0x03,0x04,0x05,0x04,0x03,0x02,0x02,0x04,0x03,0x02,0x00,0x00,0x01,0xe5,
  0x00,0x02,0x02,0x01,0x03,0x04,0x05,0x04,0x05,0x04,0x03,0x25,0x25,0x02,0x04,0x04,
  0x04,0x03,0x04,0x03,0x02,0x02,0x01,0x02,0x03,0x25,0x25,0x03,0x02,0x02,0x02,0xe5,
  0x00,0x00,0xb7,0xb7,0x25,0x00,0x00,0x02,0x02,0x01,0x01,0x02,0x04,0x04,0x02,0x02,
  0x01,0x03,0x04,0x04,0x05,0x04,0x25,0x03,0x03,0x02,0x02,0x04,0x01,0x01,0x00,0xb7,
  0x00,0x02,0x02,0x01,0x04,0x04,0x05,0xf8,0x05,0x04,0x04,0x02,0x02,0x04,0x04,0x03,
  0x04,0x04,0x04,0x03,0x03,0x01,0x02,0x25,0x03,0x01,0x02,0x01,0x02,0x02,0x01,0xda,
  0x00,0x02,0x00,0x00,0x01,0x01,0x02,0x02,0x01,0x00,0x01,0x02,0x03,0x04,0x04,0x01,
  0x01,0x03,0x04,0x04,0x05,0x03,0x05,0x04,0x05,0x04,0x04,0x03,0x01,0x02,0x00,0xb7,
  0x00,0x02,0x01,0x02,0x04,0x05,0x04,0x04,0x03,0x04,0x04,0x03,0x25,0x03,0x03,0x03,
  0x04,0x03,0x03,0x03,0x01,0x03,0x03,0x02,0x03,0x01,0x01,0x02,0x02,0x00,0x00,0xda,
  0xb7,0x01,0x00,0x02,0x01,0x01,0x02,0x00,0x01,0x02,0x02,0x01,0x00,0x04,0x03,0x25,
  0x03,0x04,0x04,0x04,0x03,0x03,0x05,0x05,0x04,0x03,0x04,0x03,0x02,0x25,0x02,0xb7,
  0x00,0x25,0x25,0x04,0x05,0x04,0x04,0x03,0x04,0x04,0x05,0x03,0x04,0x03,0x03,0x04,
  0x04,0x03,0x04,0x03,0x03,0x03,0x02,0x25,0x02,0x03,0x04,0x03,0x01,0x01,0x01,0xb7,
  0xb7,0x01,0x00,0x00,0x02,0x02,0x01,0x00,0x02,0x03,0x02,0x02,0x03,0x01,0x04,0x04,
  0x03,0x05,0x05,0x03,0x03,0x04,0x05,0x05,0x05,0x04,0x25,0x03,0x02,0x04,0x25,0xe5,
  0x00,0x02,0x01,0x03,0x04,0x03,0x03,0x04,0x04,0x04,0x05,0x03,0x03,0x04,0x03,0x03,
  0x04,0x03,0x03,0x03,0x02,0x02,0x03,0x03,0x02,0x03,0x03,0x03,0x25,0x01,0x01,0xb7,
  0x00,0x01,0x02,0x02,0x02,0x02,0x25,0x01,0x02,0x03,0xf3,0xf3,0x03,0x03,0x04,0x04,
  0x05,0x04,0x03,0x03,0x04,0x03,0x05,0x04,0x04,0x04,0x03,0x02,0x01,0x02,0x00,0xda,
  0x00,0x04,0x02,0x01,0x04,0x03,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x05,0x05,
  0x04,0x03,0x04,0x04,0x03,0x25,0x04,0x03,0x03,0x03,0x03,0x04,0x04,0x01,0x01,0xb7,
  0xb7,0x00,0x00,0x00,0x01,0x02,0x04,0x04,0x03,0xf3,0x01,0x01,0x03,0x04,0x05,0x03,
  0x04,0x03,0x02,0x04,0x04,0x05,0x03,0x03,0x04,0x04,0x03,0x05,0x25,0x01,0x02,0xe5,
  0x00,0x02,0x25,0x25,0x03,0x05,0x04,0x04,0x04,0x03,0x03,0x04,0x03,0x03,0x05,0x53,
  0x04,0x04,0x04,0x03,0x25,0x25,0x01,0x03,0x25,0x03,0x03,0x03,0x02,0x02,0x01,0xb7,
  0x01,0x02,0x00,0x00,0x02,0x03,0x03,0x04,0x04,0x03,0x01,0x04,0x03,0x04,0x04,0x04,
  0x03,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x25,0x02,0x01,0xb7,
  0x02,0x02,0x25,0x01,0x04,0x03,0x04,0x04,0x04,0x25,0x03,0x03,0x03,0x04,0x04,0x05,
  0x05,0x04,0x04,0x04,0x03,0x03,0x04,0x04,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0xb7,
  0x00,0x00,0x00,0x00,0x25,0x02,0x02,0x03,0x03,0x25,0x02,0x03,0x03,0x04,0x03,0x03,
  0x04,0x04,0x03,0x04,0x04,0x03,0x04,0x04,0x03,0x03,0x04,0x03,0x25,0x25,0x02,0xb7,
  0x00,0x02,0x01,0x04,0x05,0x04,0x04,0x25,0x02,0x02,0x03,0x04,0x03,0x03,0x04,0x05,
  0x05,0x05,0x53,0x04,0x03,0x04,0x03,0x03,0x03,0x04,0x03,0x03,0x02,0x01,0x01,0xb7,
  0x00,0x00,0x00,0x02,0x04,0x02,0x03,0x02,0x03,0x03,0x03,0x03,0x25,0x05,0x04,0x04,
  0x05,0x05,0x04,0x04,0x04,0x04,0x05,0x05,0x04,0x05,0x05,0x05,0x02,0x02,0x25,0xb7,
  0x01,0x01,0x03,0x04,0x04,0x05,0x04,0x02,0x04,0x03,0x04,0x04,0x25,0x04,0x05,0x05,
  0x04,0x05,0x05,0x04,0x02,0x03,0x03,0x03,0x25,0x01,0x25,0x25,0x01,0x01,0x00,0xda,
  0x00,0x01,0x02,0x02,0x03,0x03,0x02,0x03,0x03,0x02,0x02,0x02,0x04,0x04,0x04,0x05,
  0x05,0x05,0x04,0x04,0x04,0x05,0xcb,0x05,0x04,0x04,0x05,0x04,0x02,0x03,0x03,0x00,
  0x01,0x25,0x01,0x04,0x05,0x03,0x03,0x04,0x04,0x04,0x04,0x03,0x01,0x04,0x04,0x05,
  0x05,0x04,0x04,0x05,0x04,0x03,0x03,0x04,0x03,0x04,0x03,0x02,0x25,0x01,0x00,0xb7,
  0xb7,0x00,0x01,0x02,0x02,0x02,0x25,0x01,0x01,0x01,0x25,0x02,0x02,0x04,0x04,0x04,
  0x05,0x04,0x04,0x04,0xcb,0x04,0x04,0x05,0x04,0x05,0xcb,0x05,0x25,0x03,0x25,0x00,
  0x02,0x04,0x04,0x03,0x04,0x03,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x04,0x05,0x53,
  0x04,0x04,0x04,0x05,0x04,0x03,0x04,0x04,0x04,0x04,0x03,0x02,0x02,0x02,0x01,0xb7,
  0xb7,0x01,0x00,0x00,0x02,0x01,0x02,0x02,0x01,0x01,0x01,0x03,0x04,0x03,0x04,0x04,
  0x04,0x04,0x04,0x04,0x05,0x05,0x04,0x04,0xcb,0x53,0x05,0x05,0x03,0x25,0x04,0x00,
  0x00,0x03,0x04,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x03,0x04,0x03,0x04,0x04,0x05,
  0x03,0x04,0x03,0x04,0x05,0x04,0x03,0x04,0x04,0x03,0x03,0x03,0x01,0x02,0x02,0xb7,
  0x00,0x02,0x01,0x00,0x01,0x02,0x02,0x01,0x02,0x01,0x00,0x01,0x02,0x02,0x03,0x03,
  0x03,0x04,0x04,0x04,0x05,0x05,0x04,0x05,0x04,0x05,0xcb,0x05,0x03,0x03,0x03,0xb7,
  0x01,0x04,0x03,0x03,0x04,0x25,0x03,0x04,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x05,
  0x05,0x04,0x04,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x02,0x02,0x02,0xb7,
  0x00,0x02,0x00,0x00,0x01,0x01,0x02,0x02,0x01,0x03,0x04,0x25,0x02,0x03,0x01,0x03,
  0x02,0x04,0x03,0x04,0x04,0x04,0x05,0x05,0x53,0x05,0x05,0x53,0x04,0x04,0x04,0x00,
  0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,
  0x05,0x04,0x05,0x03,0x03,0x04,0x04,0x05,0x04,0x04,0x03,0x03,0x03,0x02,0x01,0xb7,
  0x00,0x00,0x00,0x02,0x03,0x02,0x01,0x03,0x03,0x04,0x04,0x03,0x03,0x01,0x02,0x25,
  0x25,0x04,0x04,0x03,0x25,0x05,0x53,0x05,0x05,0x05,0x05,0xcb,0x05,0x04,0x04,0x00,
  0x01,0x25,0x02,0x02,0x04,0x04,0x03,0x04,0x03,0x04,0x04,0x05,0x03,0x04,0x04,0x03,
  0x04,0x05,0x04,0x05,0x04,0x03,0x04,0x04,0x04,0x03,0x03,0x04,0x25,0x25,0x02,0xb7,
  0xb7,0x00,0x01,0x02,0x02,0x03,0x03,0x04,0x04,0x04,0x04,0x05,0x03,0x25,0x25,0x25,
  0x04,0x03,0x03,0x03,0x04,0x04,0x05,0x53,0x05,0x05,0x05,0x05,0x04,0x04,0x03,0xb7,
  0x02,0x04,0x01,0x02,0x05,0x03,0x04,0x05,0x05,0x04,0x04,0x04,0x04,0x05,0x05,0x04,
  0x04,0x04,0x05,0x05,0x04,0x04,0x03,0x25,0x25,0x03,0x04,0x04,0x25,0x01,0x01,0x00,
  0xb7,0x01,0x01,0x02,0x02,0x01,0x03,0x04,0x01,0x03,0x04,0x03,0x04,0x04,0x03,0x03,
  0x04,0x04,0x03,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x03,0x00,
  0x02,0x04,0x04,0x04,0x05,0x05,0x05,0x04,0x05,0x05,0x05,0x04,0x05,0x05,0x05,0x03,
  0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x04,0x04,0x25,0x01,0x01,0xb7,
  0x01,0x02,0x00,0x01,0x01,0x04,0x02,0x04,0x04,0x03,0x03,0x04,0x04,0x04,0x05,0x05,
  0x03,0x03,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x03,0x03,0x00,
  0x04,0x04,0x04,0x03,0x53,0x53,0x05,0x05,0x53,0x04,0x53,0xcb,0x05,0x05,0x05,0x04,
  0x05,0x05,0x04,0x03,0x04,0x04,0x04,0x25,0x04,0x03,0x04,0x03,0x01,0x02,0x02,0xb7,
  0x00,0x00,0xb7,0x00,0x01,0x04,0x25,0x03,0x02,0x01,0x01,0x25,0x03,0x04,0x04,0x03,
  0x05,0x05,0x03,0x04,0x04,0x05,0x04,0x04,0x04,0x05,0x05,0xcb,0x04,0x05,0x04,0xb7,
  0x02,0x03,0x03,0x04,0x53,0x53,0x05,0x04,0x04,0x05,0x05,0x05,0x05,0x03,0x04,0x05,
  0x05,0x05,0x53,0x04,0x03,0x04,0x05,0x04,0x03,0x04,0x03,0x03,0x02,0x01,0x01,0x00,
  0xb7,0x00,0x00,0x00,0x01,0x01,0x03,0x03,0x25,0x25,0x25,0x01,0x03,0x03,0x02,0x03,
  0x04,0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x53,0xcb,0x05,0xf8,0x05,0x02,0x00,
  0x01,0x03,0x04,0x04,0x05,0x05,0x05,0x04,0x53,0x53,0x05,0x04,0x05,0x04,0x04,0x05,
  0x05,0x05,0x05,0x04,0x03,0x03,0x04,0x04,0x04,0x03,0x03,0x03,0x01,0x01,0x00,0xb7,
  0xda,0x00,0xb7,0x00,0x00,0x01,0x00,0x00,0x02,0xb7,0x02,0x02,0x01,0x01,0x01,0x03,
  0x04,0x03,0x03,0x25,0x25,0x25,0x04,0x02,0x04,0x03,0x04,0x02,0x25,0x03,0x25,0xb7,
  0x02,0x03,0x25,0x25,0x04,0x03,0x03,0x04,0x04,0x04,0x04,0x03,0x01,0x03,0x25,0x02,
  0x03,0x04,0x03,0x03,0x02,0x02,0x01,0x02,0x01,0x02,0x03,0x25,0x01,0x01,0x00,0xb7,
  0xb7,0x00,0x00,0x00,0x00,0x00,0xb7,0x00,0x01,0x01,0x02,0x02,0x00,0x25,0x25,0x25,
  0x02,0x25,0x01,0x25,0x04,0x03,0x02,0x02,0x03,0x04,0x04,0x04,0x25,0x02,0x03,0xb7,
  0x01,0x25,0x02,0x25,0x05,0x03,0x03,0x04,0x04,0x04,0x25,0x02,0x03,0x04,0x02,0x02,
  0x01,0x01,0x02,0x02,0x01,0x02,0x02,0x00,0x01,0x02,0x02,0x02,0x02,0x00,0x00,0xb7,
  0xb7,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x01,0x02,0x00,0x02,0x00,0x02,0x01,
  0x25,0x01,0x03,0x03,0x04,0x03,0x03,0x03,0x04,0x05,0x04,0x04,0x03,0x01,0x01,0xb7,
  0x00,0x03,0x04,0x01,0x00,0x03,0x04,0x04,0x25,0x04,0x04,0x03,0x04,0x04,0x03,0x02,
  0x01,0x25,0x01,0x25,0x01,0x02,0x01,0x01,0x00,0x01,0x01,0x00,0x01,0x02,0x00,0xda,
  0xda,0xe5,0xe5,0xfa,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0xb7,0x00,0x00,
  0x01,0x00,0x00,0x01,0x00,0x01,0x00,0x01,0x00,0x25,0x25,0x02,0x01,0x00,0xb7,0xb7,
  0x00,0x02,0x01,0x00,0x02,0x01,0x02,0x02,0x02,0x02,0x03,0x25,0x02,0x01,0x02,0x01,
  0x01,0x01,0x00,0xb7,0x00,0xb7,0x00,0x00,0xb7,0x00,0xb7,0x00,0x00,0x00,0xe5,0xda,
  0xda,0xe5,0xda,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xda,0xda,0xb7,0xda,0xb7,0xb7,0xb7,
  0xb7,0x00,0x00,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0x00,0xb7,0xda,
  0xda,0xb7,0xb7,0x00,0xb7,0x00,0xb7,0xb7,0x00,0x00,0x00,0xb7,0xb7,0x00,0x00,0x00,
  0xb7,0xb7,0xb7,0xb7,0xda,0xb7,0xb7,0xb7,0xb7,0xb7,0xda,0xb7,0xb7,0xb7,0xe5,0xda,
  0xb7,0x00,0x00,0x00,0x25,0x00,0x00,0x01,0x01,0x00,0x01,0x01,0x00,0x25,0x01,0x02,
  0x03,0x03,0x04,0x04,0x01,0x03,0x04,0x04,0x02,0x25,0x01,0x01,0x02,0x03,0x25,0xb7,
  0x01,0x25,0x03,0x03,0x04,0x04,0x04,0x25,0x03,0x04,0x03,0x04,0x03,0x03,0x25,0x25,
  0x03,0x04,0x02,0x03,0x02,0x01,0x02,0x25,0x02,0x01,0x25,0x02,0x00,0x00,0x00,0xda,
  0xda,0x00,0x25,0x01,0x00,0x00,0x00,0x00,0x25,0x02,0x02,0x25,0x00,0x01,0x03,0x03,
  0x04,0x04,0x04,0x25,0x02,0x03,0x05,0xf8,0x04,0x03,0x05,0x04,0x25,0x04,0x02,0xb7,
  0x03,0x04,0x02,0x03,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x03,0x03,0x03,0x04,0x25,0x02,0x01,0x03,0x01,0x02,0x02,0x01,0x00,0x00,0x00,0xb7,
  0xb7,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x25,0x03,0x01,0x03,0x04,0x25,0x02,
  0x02,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x25,0x25,0x02,0x00,
  0x25,0x03,0x04,0x03,0x03,0x04,0x04,0x05,0x04,0x04,0x03,0x03,0x04,0x04,0x04,0x04,
  0x04,0x25,0x03,0x04,0x03,0x03,0x01,0x03,0x03,0x02,0x03,0x02,0x02,0x00,0x00,0xe5,
  0xb7,0x00,0x00,0x02,0x01,0x02,0x02,0x02,0x03,0x01,0x02,0x02,0x04,0x04,0x03,0x03,
  0x03,0x03,0x04,0x05,0x05,0x53,0x05,0x04,0x04,0x04,0x05,0x05,0x04,0x25,0x03,0x00,
  0x25,0x03,0x05,0x03,0x04,0x05,0x05,0x53,0x05,0x04,0x05,0x05,0x05,0x53,0x53,0x53,
  0xcb,0x05,0x04,0x04,0x05,0x04,0x03,0x04,0x04,0x04,0x04,0x03,0x02,0x02,0x02,0xb7,
  0xe5,0x00,0x02,0x02,0x02,0x03,0x02,0x03,0x02,0x02,0x02,0x03,0x04,0x25,0x03,0x03,
  0x04,0x04,0x05,0x05,0xcb,0x05,0x05,0x05,0x03,0x04,0x04,0x05,0x04,0x04,0x04,0x00,
  0x03,0x04,0x03,0x04,0x05,0x05,0x04,0x05,0x04,0x04,0x05,0x05,0x04,0x05,0xcb,0x05,
  0xcb,0x05,0x05,0x03,0x03,0x04,0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x00,0xb7,
  0x00,0x01,0x00,0x01,0x04,0x03,0x03,0x03,0x01,0x03,0x04,0x04,0x03,0x04,0x03,0x03,
  0x05,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x04,0x05,0x00,
  0x25,0x03,0x04,0x04,0x05,0x05,0x05,0x04,0x04,0x53,0x53,0x05,0x04,0x05,0x53,0x05,
  0x05,0x05,0x53,0x04,0x03,0x04,0x05,0xf8,0xf8,0x03,0x03,0x04,0x04,0x02,0x02,0xb7,
  0xb7,0x00,0x02,0x02,0x04,0x03,0x01,0x03,0x03,0x04,0x05,0x05,0x04,0x03,0x04,0x04,
  0x05,0x05,0x05,0x04,0x05,0x53,0x05,0x05,0x05,0x05,0x03,0x04,0x04,0x04,0x04,0xb7,
  0x01,0x03,0x04,0x05,0x03,0x05,0x05,0x05,0x05,0x05,0xcb,0x05,0x04,0x04,0x05,0x03,
  0x05,0x05,0x05,0x05,0x04,0x03,0x03,0x04,0x04,0x05,0x05,0x05,0x04,0x01,0x25,0xb7,
  0xb7,0x00,0x04,0x02,0x04,0x04,0x03,0x25,0x04,0x04,0x04,0x05,0x03,0x03,0x04,0x04,
  0x05,0x05,0x04,0x04,0x04,0x05,0x05,0x53,0x05,0x04,0x04,0x05,0x03,0x04,0x04,0xb7,
  0x02,0x04,0x04,0x05,0x05,0x03,0x05,0x53,0x53,0x05,0x05,0x04,0x04,0x05,0x53,0x05,
  0x04,0x04,0x53,0xcb,0xcb,0x05,0x04,0x03,0x04,0x05,0x05,0x04,0x04,0x03,0x01,0xb7,
  0xb7,0x01,0x02,0x03,0x05,0x02,0x04,0x03,0x03,0x05,0x53,0x04,0x04,0x04,0x05,0x03,
  0x04,0x04,0x04,0x05,0x05,0x05,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x03,0x02,0xb7,
  0x03,0x04,0x05,0x04,0x04,0x05,0x05,0x53,0xcb,0x05,0x05,0x03,0x05,0x53,0x05,0x05,
  0x05,0x04,0x03,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x05,0x05,0x04,0x02,0x02,0xb7,
  0xb7,0x01,0x01,0x02,0x25,0x04,0x04,0x03,0x03,0x04,0x03,0x04,0x04,0xcb,0xf0,0x05,
  0x04,0x03,0x05,0x05,0xf0,0xcb,0x05,0x04,0x05,0x05,0x05,0x05,0x04,0x03,0x25,0xb7,
  0x25,0x05,0x04,0x03,0xcb,0x05,0x53,0x53,0x53,0x04,0x05,0x05,0x05,0x05,0x05,0x04,
  0x05,0x05,0x03,0x04,0x05,0x04,0x04,0x05,0x04,0x04,0x04,0x04,0x03,0x02,0x02,0xb7,
  0x00,0x01,0x01,0x01,0x04,0x05,0x04,0x04,0x04,0x04,0x03,0x03,0x04,0xcb,0x05,0x04,
  0x05,0x04,0x04,0x05,0x05,0x04,0x04,0x04,0x04,0x03,0x05,0x04,0x03,0x04,0x03,0xb7,
  0x01,0x04,0x02,0x04,0xcb,0x53,0x05,0x05,0x05,0x04,0x05,0x05,0x53,0x05,0x05,0x04,
  0xcb,0x04,0x04,0x05,0x03,0x04,0x25,0x03,0x03,0x04,0x03,0x03,0x03,0x03,0x02,0xe5,
  0x00,0x02,0x01,0x04,0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x02,0x04,0x04,0x04,
  0x05,0x53,0x04,0x04,0x04,0x05,0x05,0xf8,0x05,0x04,0x04,0x04,0x05,0x03,0x03,0xb7,
  0x01,0x03,0x04,0x04,0xcb,0xcb,0x53,0x04,0x05,0x05,0x05,0x05,0x05,0xcb,0x05,0x05,
  0x05,0x05,0x04,0x04,0x03,0x03,0x03,0x04,0x02,0x03,0x25,0x03,0x03,0x25,0x02,0xe5,
  0x00,0x01,0x03,0x04,0x05,0x04,0x03,0x03,0x04,0x05,0x04,0x04,0x04,0x03,0x04,0x05,
  0x05,0x05,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x53,0x04,0x03,0x03,0x00,
  0x25,0x04,0x03,0x04,0xf8,0x05,0x05,0x53,0x05,0x05,0x05,0x05,0x04,0x53,0xcb,0x05,
  0x53,0x53,0x04,0x04,0x04,0x05,0x03,0x05,0x03,0x04,0x03,0x03,0x04,0x03,0x02,0xb7,
  0x01,0x25,0x03,0x02,0xcb,0x05,0x05,0x04,0x04,0x04,0x05,0x04,0x04,0x05,0x05,0x05,
  0x53,0x05,0x04,0x05,0x05,0x04,0x05,0x05,0x04,0x04,0xcb,0x05,0x04,0x04,0x25,0x00,
  0x04,0x04,0x03,0x03,0x03,0x03,0x05,0x05,0x05,0x53,0x04,0x05,0x05,0xcb,0x53,0xf0,
  0x53,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x04,0x03,0x04,0x02,0x01,0xb7,
  0x00,0x02,0x01,0x02,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x53,0x53,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x03,0x25,0x00,
  0x02,0x25,0x04,0x03,0x03,0x05,0x05,0xf8,0xcb,0x04,0x05,0xcb,0x04,0xcb,0x53,0x53,
  0x05,0x53,0x04,0x04,0x04,0x04,0x03,0x03,0x04,0x01,0x04,0x03,0x03,0x01,0x01,0xb7,
  0x00,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x53,0xcb,0x05,0x05,
  0x53,0x53,0x05,0x05,0x05,0x05,0x04,0x04,0x03,0x05,0x05,0x05,0x04,0x03,0x02,0xb7,
  0x25,0x04,0x03,0x03,0x04,0x04,0x04,0x05,0x04,0x04,0x05,0xcb,0x04,0x05,0x53,0x05,
  0x05,0x05,0x05,0x04,0x04,0x04,0x03,0x02,0x04,0x03,0x03,0x03,0x03,0x02,0x02,0x00,
  0x00,0x01,0x01,0x01,0x03,0x04,0x04,0x05,0x03,0x04,0x05,0x04,0x53,0x53,0x04,0x04,
  0x53,0x05,0x04,0x04,0x05,0x03,0x04,0x05,0x04,0x04,0x04,0x04,0x03,0x04,0x03,0xb7,
  0x02,0x03,0x03,0x03,0x04,0x04,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x04,0x05,0x53,
  0xcb,0x05,0x05,0x04,0x03,0x03,0x25,0x04,0x04,0x02,0x03,0x03,0x03,0x03,0x02,0x00,
  0x00,0x02,0x01,0x02,0x04,0x03,0x03,0x03,0x03,0x05,0x05,0x05,0x04,0x04,0x05,0x05,
  0x05,0x04,0xcb,0x03,0x03,0x04,0x04,0x05,0x53,0xcb,0x03,0x03,0x03,0x04,0x25,0x00,
  0x01,0x01,0x03,0x25,0x05,0x05,0xcb,0x05,0x05,0x04,0xcb,0x53,0x04,0x04,0x05,0x53,
  0x53,0x05,0x04,0x03,0x04,0x03,0x03,0x04,0x03,0x02,0x03,0x04,0x03,0x25,0x01,0x00,
  0xb7,0x01,0x02,0x02,0x03,0x03,0x01,0x01,0x04,0x03,0x03,0x05,0x04,0x03,0x04,0x04,
  0x03,0x04,0x04,0x03,0x03,0x03,0x04,0x05,0x05,0x04,0x03,0x04,0x03,0x03,0x04,0xb7,
  0x01,0x03,0x01,0x25,0x04,0x04,0x05,0x05,0x05,0x04,0x04,0x05,0x03,0x05,0x05,0xcb,
  0x05,0x04,0x03,0x04,0x03,0x04,0x04,0x03,0x02,0x04,0x04,0x04,0x04,0x25,0x01,0x00,
  0x00,0x00,0x01,0x00,0x02,0x01,0x25,0x02,0x01,0x02,0x02,0x02,0x03,0x03,0x05,0x04,
  0x04,0x04,0x03,0x04,0x04,0x03,0x03,0x05,0x04,0x04,0x05,0x04,0x04,0x01,0x01,0xb7,
  0x25,0x25,0x25,0x02,0x03,0x04,0x05,0x05,0xcb,0x53,0x05,0x04,0x05,0x04,0x05,0x05,
  0x05,0x03,0x04,0x04,0x02,0x03,0x25,0x03,0x02,0x03,0x04,0x04,0x25,0x01,0x02,0xb7,
  0x00,0x02,0x00,0x03,0x25,0x01,0x03,0x03,0x03,0x03,0x02,0x25,0x04,0x03,0x03,0x03,
  0x03,0x02,0x04,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x04,0x01,0x03,0xb7,
  0x02,0x04,0x02,0x02,0x03,0x03,0x05,0xcb,0xcb,0x05,0x05,0x05,0x04,0x05,0x05,0x04,
  0x04,0x04,0x04,0x04,0x03,0x04,0x03,0x03,0x02,0x01,0x25,0x01,0x03,0x25,0x25,0x00,
  0x00,0x02,0x01,0x00,0x02,0x02,0x03,0xf3,0x03,0x02,0x03,0x03,0x04,0x04,0x04,0x04,
  0x03,0x03,0x03,0x04,0x03,0x04,0x03,0x04,0x04,0x04,0x04,0x05,0x02,0x25,0x04,0x00,
  0x00,0x02,0x03,0x02,0x04,0x04,0x03,0x05,0x04,0xcb,0x05,0x05,0xcb,0x04,0x04,0x04,
  0x04,0x04,0x05,0x04,0x03,0x03,0x03,0x03,0x02,0x00,0x01,0x03,0x04,0x02,0x02,0xb7,
  0x00,0x02,0x00,0x01,0x03,0x02,0x03,0x03,0x01,0x02,0x04,0x04,0x03,0x04,0x04,0x03,
  0x04,0x03,0x03,0x03,0x04,0x03,0x04,0x04,0x04,0x04,0x03,0x04,0x05,0x04,0x25,0xb7,
  0x01,0x01,0x03,0x25,0x05,0x04,0x04,0x04,0x05,0x05,0x05,0x04,0x05,0x53,0x05,0x05,
  0x04,0x05,0x04,0x04,0x03,0x01,0x03,0x04,0x03,0x02,0x01,0x04,0x04,0x02,0x01,0xb7,
  0x00,0x00,0x01,0x01,0x03,0x02,0x04,0x00,0x01,0x02,0x02,0x03,0x03,0x03,0x02,0x03,
  0x03,0x03,0x04,0x02,0x03,0x03,0x04,0x04,0x04,0x04,0x03,0x03,0x04,0x25,0x25,0xb7,
  0x01,0x02,0x01,0x25,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x04,0x04,0x05,0x04,
  0x04,0x05,0x04,0x04,0x02,0x02,0x01,0x03,0x03,0x01,0x03,0x04,0x04,0x02,0x01,0xb7,
  0x00,0x00,0x01,0x00,0x03,0x03,0x01,0x01,0x01,0x02,0x02,0x02,0x03,0x04,0x02,0x03,
  0x04,0x04,0x04,0x01,0x03,0x03,0x03,0x05,0x05,0x03,0x04,0x04,0x25,0x25,0x02,0xb7,
  0x04,0x04,0x25,0x25,0x03,0x04,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x05,0x05,0x05,
  0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x03,0x25,0x03,0x04,0x04,0x02,0x01,0x00,
  0x00,0x00,0x00,0x00,0x25,0x03,0x25,0x02,0x01,0x25,0x02,0x01,0x25,0x04,0x03,0x03,
  0x03,0x25,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x04,0x05,0x04,0x04,0x01,0x01,0x00,
  0x02,0x25,0x02,0x02,0x02,0x04,0xcb,0x05,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,
  0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x01,0x03,0x03,0x04,0x04,0x04,0x02,0x02,0xb7,
  0x00,0x01,0x01,0x00,0x01,0x04,0x03,0x03,0x04,0x03,0x01,0x03,0x04,0x04,0x04,0x04,
  0x04,0x03,0x04,0x04,0x04,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x02,0x02,0x01,0x00,
  0x02,0x01,0x03,0x02,0x03,0x04,0x04,0x05,0x04,0x04,0x05,0x04,0x04,0x04,0x05,0xcb,
  0x04,0x04,0x04,0x03,0x03,0x03,0x25,0x03,0x03,0x02,0x03,0x03,0x25,0x25,0x00,0xb7,
  0x00,0x02,0x01,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x02,0x01,0x02,0x01,
  0x01,0x01,0x01,0x02,0x01,0x01,0x01,0x02,0x25,0x01,0x25,0x01,0x01,0x02,0x02,0xb7,
  0x00,0x02,0x01,0x02,0x02,0x04,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x03,0x04,0x04,0x01,0x03,0x02,0x25,0x01,0x00,0x01,0x00,0x01,0x01,0x02,0xe5,
  0xb7,0x00,0x00,0x02,0x01,0x00,0x00,0x00,0x00,0x02,0x00,0x01,0x01,0x01,0x00,0x02,
  0x02,0x25,0x02,0x25,0x25,0x01,0x02,0x02,0x02,0x25,0x01,0x01,0x01,0x01,0x02,0xb7,
  0x01,0x25,0x02,0x02,0x01,0x25,0x03,0x03,0x02,0x04,0x04,0x25,0x25,0x03,0x03,0x03,
  0x25,0x25,0x02,0x01,0x25,0x01,0x01,0x00,0x00,0x01,0x01,0x02,0x01,0x01,0x00,0xda,
  0xe5,0x00,0x01,0x00,0x00,0x01,0x02,0x00,0x00,0x02,0x02,0x02,0x02,0x00,0x02,0x01,
  0x01,0x02,0x02,0x01,0x01,0x01,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x02,0x01,0xda,
  0x00,0x01,0x00,0x01,0x00,0x01,0x01,0x25,0x03,0x01,0x02,0x25,0x25,0x25,0x25,0x01,
  0x01,0x02,0x25,0x02,0x02,0x02,0x01,0x01,0x01,0x25,0x02,0x25,0x02,0x00,0x00,0xb7,
  0xda,0xb7,0xb7,0xe5,0x00,0xb7,0x00,0x00,0xb7,0x00,0x00,0x00,0x00,0xb7,0x00,0xb7,
  0x00,0x01,0x00,0xda,0xb7,0xb7,0x00,0xb7,0xb7,0x00,0xb7,0xb7,0xb7,0x00,0xe5,0xda,
  0xda,0xb7,0xb7,0xb7,0xb7,0x00,0x00,0x00,0x00,0x00,0xb7,0x00,0x00,0x00,0x00,0x00,
  0x00,0xb7,0x00,0x00,0x00,0xb7,0xb7,0xb7,0xb7,0x01,0x00,0x01,0x00,0xb7,0xe5,0xda,
  // 3: ground grass
  0xe5,0x6c,0xa5,0x6c,0x64,0x6c,0x24,0x64,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,
  0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,
  0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,
  0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,
  0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x05,0x6d,0x00,0x20,0x08,0x20,0x10,0x20,
  0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,
  0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,
  0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,
  0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,
  0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,
  0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,
  0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,
  0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,
  0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,
  0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,
  0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,
  0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,
  0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,
  0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,
  0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,0x08,0x86,0x24,0x5c,
  0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x23,0x5c,0x10,0x87,0x18,0x87,0x00,0xa0,
  0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,
  0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x23,0x64,0x10,0xa3,0x63,0x64,
  0x25,0x75,0x18,0xa3,0x00,0xa4,0x44,0x5c,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,
  0x08,0xa5,0x10,0xa5,0x18,0xa5,0xe5,0x74,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,
  0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x26,0x75,
  0x18,0xc0,0xc6,0x6c,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x63,0x54,0x18,0xc1,0x00,0xc2,
  0x08,0xc2,0x10,0xc2,0x18,0xc2,0x83,0x64,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x18,0xc3,
  0x00,0xc4,0x45,0x75,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x45,0x6d,
  0xc7,0x74,0x10,0xc5,0x18,0xc5,0x00,0xc6,0x08,0xc6,0x10,0xc6,0x18,0xc6,0xe3,0x6b,
  0x00,0xc7,0xa4,0x74,0x44,0x74,0xc4,0x5b,0xc3,0x5c,0x08,0xc7,0x26,0x6d,0x86,0x74,
  0x66,0x6d,0x10,0xc7,0x85,0x75,0xe4,0x63,0x66,0x6c,0x86,0x6d,0xa6,0x6c,0x84,0x5c,
  0xbb,0x02,0x24,0xfa,0x00,0xc3,0x24,0xdb,0x00,0x24,0xe1,0x24,0x02,0x01,0xbb,0x9f,
  0x9f,0x03,0x01,0xdb,0x00,0x00,0x00,0xcf,0x01,0x01,0xdb,0x24,0xb8,0x00,0xe7,0xb8,
  0x00,0x02,0x01,0x00,0xe8,0x00,0x01,0x24,0x01,0xd1,0x00,0x00,0x01,0x01,0x24,0x01,
  0x03,0xcf,0x24,0x24,0x00,0x24,0xcf,0x24,0xb8,0x01,0x01,0xf7,0xfe,0x01,0x00,0xdb,
  0x24,0xdb,0x00,0xcf,0x24,0x24,0xdb,0x24,0x00,0x00,0x24,0x24,0xd1,0x00,0x9f,0xa4,
  0x9f,0x9f,0xcf,0x00,0xcf,0xcf,0x24,0x00,0x24,0x01,0x24,0x00,0xb8,0xb8,0xe1,0xdb,
  0x01,0x01,0x01,0x00,0xcf,0x24,0x00,0x24,0x00,0x24,0x01,0x02,0x00,0xe1,0x02,0x03,
  0xf1,0x02,0x24,0x24,0xe1,0x24,0xcf,0x01,0x00,0x01,0x01,0xd1,0xf6,0xe7,0x00,0x01,
  0x00,0x00,0x00,0x00,0x24,0x24,0xdb,0x24,0x00,0xdb,0x03,0x01,0xdb,0xbb,0x9f,0x9f,
  0xf3,0x9f,0x9f,0x00,0x00,0x24,0x24,0xb8,0x00,0xdb,0xdb,0x24,0x24,0xf6,0x01,0x00,
  0x03,0x01,0x24,0x00,0x24,0xdb,0x00,0x24,0xe7,0x24,0x00,0x00,0x00,0xbb,0x01,0xbb,
  0x03,0x00,0x01,0x24,0x24,0xf6,0xe1,0x00,0x01,0xdb,0xc3,0xd1,0xcf,0xe7,0x00,0x00,
  0xf6,0xdb,0xb8,0x24,0x24,0x00,0xdb,0x01,0x00,0x02,0x00,0xff,0x9f,0xf3,0x03,0xbb,
  0x01,0x03,0x9f,0x03,0x24,0x24,0x00,0xdb,0x01,0x24,0x24,0x24,0xe7,0xdb,0x00,0x01,
  0xdb,0xcf,0xd1,0x00,0xf6,0x24,0xdb,0x00,0x24,0x00,0x02,0x01,0x03,0xbb,0x03,0x02,
  0x01,0xf1,0x02,0x01,0x00,0xb8,0x01,0xcf,0xf7,0xdb,0x24,0xf6,0x00,0xcf,0xf6,0xf6,
  0x24,0x00,0xcf,0x24,0x01,0x02,0xdb,0xdb,0x02,0xdb,0xdb,0xff,0x03,0xbb,0x9f,0xb5,
  0x01,0xbb,0xbb,0xa4,0x03,0x24,0x00,0x00,0x00,0xdb,0x24,0x00,0x24,0xdb,0x24,0x02,
  0x00,0xe7,0x00,0x24,0x00,0x00,0xdb,0x00,0x00,0x03,0x24,0x03,0xfb,0xf3,0x02,0x03,
  0x02,0x02,0x02,0x02,0x02,0x01,0x01,0xfe,0x01,0xd1,0xd1,0xcf,0x00,0x00,0xcf,0x24,
  0x24,0x00,0xe1,0x24,0xc3,0x00,0x01,0x01,0x02,0xdb,0x03,0xa4,0x02,0x03,0xa4,0xa4,
  0xa4,0x02,0xbb,0x9f,0xf3,0x02,0x00,0x00,0x00,0x00,0x00,0x24,0xb8,0x24,0x24,0x00,
  0xc3,0xf6,0x00,0x00,0x00,0xdb,0xdb,0x24,0x24,0x24,0x02,0xa4,0x9f,0x03,0x03,0x03,
  0x02,0x02,0x02,0x02,0x03,0x02,0x01,0xcf,0xb8,0xd1,0xf6,0xe8,0xc3,0x02,0xcf,0xe1,
  0x24,0x00,0xcf,0x00,0x00,0x00,0xb8,0x01,0xbb,0x9f,0xb5,0xb7,0x01,0x03,0x03,0xf3,
  0x9f,0x02,0x01,0x01,0x9f,0x03,0x03,0x24,0x24,0x01,0x24,0x00,0xcf,0xcf,0x00,0x24,
  0x24,0xf6,0xd1,0xdb,0x00,0xdb,0x00,0x24,0x24,0x01,0x03,0x02,0x01,0x03,0x02,0x03,
  0x01,0x01,0xf2,0x02,0x01,0x9f,0x01,0xcf,0xcf,0x00,0xcf,0xf6,0x01,0xc3,0xcf,0xe1,
  0x01,0x00,0x00,0x01,0x24,0x24,0xcf,0x00,0xa4,0x9f,0xa4,0x02,0xdb,0x01,0x01,0xbb,
  0x03,0x03,0x01,0x01,0x9f,0xf3,0xa4,0xa4,0xdb,0xff,0xdb,0xd1,0xcf,0x00,0x24,0xdb,
  0xe7,0x24,0x00,0xdb,0xdb,0x00,0x24,0x24,0x24,0x02,0x03,0x01,0x03,0x03,0xf7,0x02,
  0x02,0xf2,0xf2,0xf1,0x02,0xf7,0x03,0x01,0x01,0xd1,0xdb,0x00,0x01,0x01,0x02,0xdb,
  0x02,0xdb,0x00,0x00,0x00,0xe7,0x24,0x02,0xa4,0xbb,0x02,0x01,0xdb,0x01,0xfe,0xa4,
  0xa4,0x9f,0xfe,0x01,0x01,0xbb,0x9f,0xd5,0xbb,0x01,0x24,0x00,0x00,0x24,0x24,0x24,
  0x24,0x00,0xf6,0x24,0xdb,0x00,0x00,0x24,0xdb,0xa4,0xbb,0x03,0xdb,0x01,0x02,0x02,
  0x01,0xf7,0xf7,0x01,0xd1,0x01,0x01,0x01,0x9f,0xe8,0x01,0x01,0x00,0xf6,0xdb,0x02,
  0xdb,0x00,0x01,0xc3,0x00,0x24,0x01,0xbb,0xa4,0xa4,0xbb,0xa4,0x03,0x00,0xf6,0x03,
  0x01,0x01,0x9f,0x01,0xa4,0x02,0x01,0x01,0xa4,0x01,0xdb,0x03,0x00,0x00,0x00,0xdb,
  0x00,0xff,0xdb,0xcf,0xe7,0xe7,0x00,0xa4,0x01,0xa4,0x9f,0x03,0x02,0xb5,0x02,0xb7,
  0x01,0xc3,0x02,0xf1,0xdb,0x03,0x01,0x02,0x03,0x03,0x01,0x00,0x24,0x24,0x00,0xdb,
  0x01,0x01,0x02,0xcf,0xe7,0xbb,0xbb,0x03,0x9f,0xdb,0xdb,0x00,0x01,0x01,0x01,0x01,
  0xbb,0x9f,0xbb,0x9f,0x9f,0x9f,0xfe,0x03,0x01,0x03,0x01,0x00,0x00,0x24,0x00,0xdb,
  0xdb,0x24,0xff,0xff,0xe7,0x00,0x02,0xdb,0xdb,0xdb,0xdb,0x03,0x03,0xdb,0x03,0xdb,
  0xb5,0x02,0xb5,0x9f,0x03,0x03,0x01,0x01,0x03,0x03,0x03,0xcf,0xf6,0x00,0xdb,0xdb,
  0x01,0x01,0xdb,0x24,0x02,0x02,0x9f,0xa4,0x02,0xa4,0x02,0x02,0xbb,0x01,0xa4,0x03,
  0x9f,0x03,0xbb,0x9f,0x03,0xfb,0x01,0xfc,0x01,0x02,0x03,0xdb,0x00,0x00,0xf6,0xdb,
  0x24,0x00,0x01,0xdb,0x9f,0x9f,0x03,0x01,0x02,0xdb,0xdb,0x02,0x01,0x02,0x03,0xa4,
  0x02,0x03,0x03,0x02,0x03,0xb5,0x01,0xf7,0x01,0x03,0x03,0x03,0x01,0x00,0x01,0x02,
  0x01,0x01,0xdb,0x03,0x03,0x02,0xa4,0xa4,0xb5,0xa4,0xbb,0x03,0xbb,0xfe,0x01,0xa4,
  0xb5,0x03,0x9f,0x03,0x01,0xbb,0x03,0x01,0xd1,0x01,0x03,0xa4,0x01,0x24,0x24,0x00,
  0x00,0x02,0xe1,0x02,0x03,0x03,0x03,0x03,0xdb,0xb5,0xdb,0x03,0xb5,0xdb,0x03,0xa4,
  0x02,0x01,0x01,0x01,0x02,0x03,0x03,0x03,0x01,0x03,0x9f,0x03,0x03,0x03,0x02,0x02,
  0x02,0x01,0x01,0x03,0xdb,0xbb,0x02,0xbb,0x02,0xdb,0xbb,0xbb,0x01,0x01,0xd1,0xa4,
  0x03,0x03,0x03,0xf3,0xdb,0x01,0xdb,0x01,0x01,0x03,0x03,0x00,0x01,0x01,0xcf,0xcf,
  0x24,0x24,0x02,0x03,0x01,0xf3,0x03,0xa4,0x03,0xa4,0x03,0xdb,0xb7,0x03,0x02,0xdb,
  0x02,0x02,0x01,0xb7,0x01,0x03,0x02,0x01,0xa4,0xfb,0x03,0xa4,0x9f,0x03,0xd1,0x00,
  0x01,0x03,0x03,0x03,0x02,0x9f,0xbb,0x01,0x03,0x01,0x02,0xfe,0x02,0x02,0x01,0x03,
  0xfc,0x01,0x01,0x03,0x9f,0x01,0x03,0x02,0x01,0x02,0x01,0xdb,0xdb,0x01,0xf6,0xcf,
  0xcf,0xb5,0x03,0x01,0xa4,0xf3,0xa4,0xb7,0xb5,0xa4,0xa4,0xa4,0x02,0x02,0x03,0x03,
  0xb5,0xb5,0x03,0x02,0x02,0x01,0x00,0x01,0x03,0x03,0x03,0xa4,0x03,0x03,0x02,0x00,
  0x03,0xf3,0x9f,0x03,0x03,0x03,0xbb,0x01,0xa4,0xb7,0x01,0xd1,0x01,0x00,0x9f,0xfc,
  0x01,0x01,0x03,0xdb,0x03,0x9f,0x01,0x01,0x01,0xdb,0x01,0x01,0x00,0x00,0xdb,0xb8,
  0xf7,0x02,0x01,0x03,0x9f,0x9f,0x03,0xa4,0xa4,0xa4,0x9f,0xa4,0xb5,0xa4,0x02,0xb5,
  0xdb,0x01,0xb5,0x01,0x01,0x03,0x02,0x03,0xfb,0xfc,0x03,0xbb,0x02,0x03,0x03,0x01,
  0x01,0x01,0x03,0xfb,0x03,0xdb,0xbb,0x00,0xdb,0x00,0xd1,0x00,0xd1,0x03,0x03,0x03,
  0x9f,0x01,0x03,0x01,0x03,0x03,0x01,0xfc,0xdb,0x03,0xbb,0xdb,0x01,0xf6,0x02,0x00,
  0x00,0x02,0xdb,0x02,0x03,0xa4,0xa4,0xa4,0xa4,0xa4,0xdb,0x01,0xa4,0x02,0xbb,0x03,
  0x03,0x01,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0xfc,0x01,0x03,0x01,0x02,0x02,0x01,
  0x01,0x01,0x03,0x03,0xfb,0x9f,0x02,0x01,0x01,0xdb,0xfc,0xd1,0x03,0x03,0x9f,0x01,
  0x01,0x03,0x03,0x03,0x03,0x02,0xdb,0xbb,0xfb,0xa4,0xbb,0x9f,0xb5,0x03,0xdb,0x00,
  0x00,0x02,0x02,0xdb,0x01,0xa4,0x02,0xb7,0xa4,0x03,0xb5,0x01,0x02,0x01,0x02,0x03,
  0x02,0xdb,0xf7,0x02,0x02,0xfb,0x01,0xf3,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x00,
  0xc3,0x01,0xb7,0xf3,0x03,0xf3,0x03,0x01,0xd1,0x01,0x03,0xe8,0x03,0x03,0xdb,0x03,
  0xfc,0x03,0x03,0x01,0x01,0xdb,0x02,0x01,0xdb,0xdb,0x02,0x03,0x02,0x01,0xdb,0x00,
  0x24,0xf6,0x00,0x00,0x02,0xb5,0x02,0x02,0xb5,0x01,0x02,0x03,0x03,0x03,0x02,0xa4,
  0x02,0x02,0x02,0x02,0x03,0xf3,0xfb,0x02,0x01,0x03,0x9f,0x03,0xa4,0x03,0x00,0xd1,
  0xb5,0x02,0xdb,0xbb,0xf3,0x03,0x9f,0x9f,0xa4,0x01,0xa4,0x01,0xb5,0x03,0x01,0x02,
  0x03,0xfe,0x01,0x03,0x03,0x01,0x01,0x01,0x9f,0x03,0x01,0xa4,0x01,0x00,0x00,0x01,
  0x00,0xf6,0x00,0xdb,0xdb,0xb5,0x03,0xb5,0xb5,0x02,0xb5,0x03,0x03,0xb5,0x02,0x02,
  0x02,0x02,0x02,0x02,0x03,0xf3,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x00,0xc3,0x01,
  0x01,0x01,0x00,0x24,0x00,0x03,0xdb,0xa4,0x02,0xf3,0x02,0x01,0x03,0x01,0xb5,0x01,
  0x9f,0x9f,0x01,0x01,0xa4,0x02,0x01,0x03,0xa4,0x9f,0x01,0xa4,0x00,0xf6,0x24,0x00,
  0xb8,0xc3,0xe1,0xe1,0x02,0x03,0xb5,0xa4,0x02,0x02,0x02,0x00,0xb5,0xb5,0xa4,0x03,
  0x01,0xa4,0x02,0x03,0x02,0x03,0x01,0x00,0x03,0x03,0x03,0x03,0xdb,0x00,0xdb,0x02,
  0x02,0x01,0xdb,0x24,0x24,0x02,0xb5,0xdb,0xfc,0xfe,0x00,0x00,0x02,0x9f,0xa4,0x03,
  0x03,0xa4,0x01,0xff,0xa4,0x02,0xb5,0x02,0xdb,0x01,0xdb,0x24,0x24,0xdb,0x00,0x00,
  0x02,0xcf,0xe1,0xe1,0x24,0xdb,0x02,0xa4,0x03,0x01,0x02,0x00,0xb5,0xa4,0xb5,0xa4,
  0x03,0xb5,0xb5,0x02,0x02,0x03,0x01,0x02,0x03,0x01,0x02,0x01,0xdb,0x01,0x01,0x02,
  0x02,0x02,0x00,0xc3,0x00,0xf6,0xdb,0xdb,0x03,0x03,0x00,0x01,0xff,0x03,0xb7,0x01,
  0x00,0x03,0x02,0x02,0xa4,0x02,0xdb,0x02,0x02,0x03,0x24,0x00,0x24,0xe1,0x00,0x00,
  0x02,0xd1,0x01,0xcf,0x24,0x00,0x02,0xb5,0x01,0x02,0x9f,0xb7,0xb5,0x02,0x01,0x9f,
  0x02,0xa4,0xb5,0xa4,0xa4,0x9f,0x01,0x03,0x01,0x03,0xc3,0x24,0xcf,0x24,0x00,0x01,
  0x03,0xb5,0x02,0xa4,0x01,0x01,0x00,0xbb,0xa4,0x01,0x01,0x01,0xd1,0x03,0x03,0x03,
  0x01,0x03,0x02,0x03,0xa4,0x03,0xbb,0x01,0x03,0xdb,0x01,0x24,0xcf,0xfa,0x24,0x00,
  0x02,0xe1,0xb8,0x00,0x01,0x00,0xdb,0x01,0x02,0xb5,0x03,0x03,0x02,0x02,0xb5,0xb5,
  0x03,0x03,0xb5,0x03,0xb5,0xfb,0xf3,0x9f,0x03,0xc3,0xb8,0xcf,0xd1,0x01,0xd1,0x00,
  0x00,0xb7,0xb8,0x02,0xb8,0x24,0x02,0x01,0x01,0x01,0xdb,0xfc,0x03,0x03,0x02,0x03,
  0xbb,0x01,0x03,0xdb,0xa4,0x9f,0x9f,0x01,0x00,0x01,0x24,0x24,0x24,0x24,0xe1,0x00,
  0x00,0xb8,0x02,0x02,0x00,0x24,0x00,0xb8,0x01,0x02,0xb5,0x02,0x02,0x02,0x02,0x03,
  0x03,0x03,0xa4,0x02,0x9f,0x9f,0x9f,0x9f,0xdb,0xd1,0x00,0xcf,0x01,0x01,0xcf,0x00,
  0xdb,0x03,0x01,0xdb,0x00,0x00,0x03,0x03,0x00,0x01,0xa4,0x02,0x02,0x03,0x01,0x01,
  0x01,0x9f,0x03,0xa4,0xa4,0x03,0xbb,0x00,0xdb,0x01,0x00,0x00,0xf6,0xb8,0xfa,0xf6,
  0xf6,0x24,0xdb,0x00,0x24,0xc3,0xb8,0x00,0xcf,0x00,0xef,0x02,0x02,0x00,0x02,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0xf6,0x01,0x00,0xc3,0x00,0xa4,
  0x01,0xb7,0x02,0x00,0xcf,0x00,0x03,0x01,0x24,0x01,0xff,0x01,0x01,0x01,0x02,0x03,
  0x01,0xbb,0xa4,0xf3,0xa4,0x9f,0xdb,0xb8,0x24,0xf6,0xf6,0x01,0x00,0xe1,0xe7,0x24,
  0x24,0x00,0x01,0x00,0xf7,0x00,0x01,0x24,0xcf,0xcf,0xd1,0x02,0x02,0x02,0x03,0x02,
  0x03,0x03,0x03,0x01,0x02,0x01,0x02,0xd1,0x01,0x03,0x24,0x00,0x01,0xdb,0xff,0x01,
  0x00,0xdb,0x00,0x02,0x00,0x02,0xfb,0xdb,0x00,0x02,0x00,0x01,0x01,0xc3,0xb5,0xdb,
  0xdb,0x02,0x01,0x03,0x02,0x00,0x00,0xcf,0xcf,0xcf,0xd1,0xe8,0x00,0x00,0x24,0xfd,
  0xcf,0xe7,0xe7,0x00,0xe1,0x24,0xdb,0x01,0xb8,0xc3,0xcf,0x02,0x02,0x02,0x02,0xdb,
  0xdb,0x03,0xb5,0x03,0x02,0x01,0xdb,0x00,0x02,0xfe,0x01,0xcf,0x00,0xc3,0xdb,0xc3,
  0x00,0x00,0x00,0xb7,0x01,0x03,0x00,0xd1,0x00,0x00,0xf6,0x00,0xfc,0xdb,0xb5,0x03,
  0xa4,0x02,0x01,0xbb,0x24,0x00,0x00,0x00,0xf6,0xf6,0x00,0xf6,0xdb,0x00,0xe7,0x24,
  0xcf,0x24,0x00,0x24,0x00,0x00,0x00,0x00,0x00,0xc3,0xcf,0x00,0xb5,0x03,0x02,0x01,
  0x02,0x03,0x02,0x00,0x00,0xc3,0x01,0xb8,0x00,0x00,0x02,0x01,0x00,0x24,0x00,0x00,
  0x00,0xdb,0x01,0xdb,0x02,0x02,0xb8,0xe1,0x00,0xdb,0xdb,0x01,0x01,0x00,0x03,0xdb,
  0xbb,0x02,0x00,0x00,0xdb,0xdb,0xdb,0x00,0xf6,0x01,0xf6,0xf6,0x01,0xdb,0x24,0xf8,
  0x24,0x24,0x00,0xdb,0x00,0x24,0x00,0xb8,0xe1,0x01,0x00,0xdb,0x00,0xb5,0x02,0x02,
  0xa4,0x03,0xdb,0x02,0x00,0x00,0x01,0x01,0xdb,0xc3,0x02,0x02,0x02,0x24,0x00,0x00,
  0x01,0x00,0xc3,0x01,0x00,0x01,0x00,0xe7,0x00,0x01,0xdb,0x00,0x00,0x24,0x00,0x03,
  0x00,0x03,0x00,0x24,0x01,0xdb,0xdb,0xfc,0xd1,0xfe,0xbb,0x01,0x24,0x00,0x24,0x24,
  0xfa,0x00,0xdb,0x00,0x00,0x01,0x01,0x00,0x02,0x02,0x01,0xdb,0x00,0x01,0x03,0xb5,
  0xa4,0x01,0x01,0x00,0x24,0xe1,0xf1,0x01,0x02,0x24,0x00,0xdb,0xf7,0xc3,0x03,0x01,
  0xcf,0x00,0x02,0xdb,0xb5,0xb5,0x01,0xc3,0x01,0xb8,0xcf,0xb8,0x01,0x02,0xdb,0x02,
  0x02,0x24,0x24,0x01,0x24,0x00,0xd1,0xf6,0xd1,0xbb,0x01,0x24,0x24,0x24,0x00,0xe7,
  0xe7,0xe1,0xdb,0xdb,0xdb,0xdb,0x02,0x01,0x00,0x03,0x01,0xf7,0x00,0xdb,0xdb,0x02,
  0x01,0x00,0x01,0x02,0x02,0x00,0x24,0x00,0xc3,0x24,0x00,0xc3,0xc3,0x00,0xdb,0x01,
  0xdb,0xb8,0x00,0x00,0xa4,0xdb,0x24,0x00,0x00,0x01,0x24,0xb8,0x24,0xc3,0x01,0x02,
  0x02,0x01,0x02,0xcf,0x01,0xd1,0xfe,0xd1,0xfe,0xfe,0x01,0x00,0x00,0x01,0x00,0xf6,
  0x24,0x01,0xdb,0xdb,0x02,0xdb,0x01,0x01,0x02,0x03,0x00,0x00,0xdb,0xdb,0xdb,0x01,
  0x02,0x24,0xdb,0xdb,0x00,0x01,0x00,0x00,0xb8,0xe1,0x24,0x01,0xfc,0x01,0x00,0x02,
  0x02,0x02,0x00,0xdb,0x01,0x01,0x02,0x02,0xd1,0x00,0xc3,0xc3,0x24,0x00,0x01,0xb5,
  0x01,0x02,0x01,0x01,0xfe,0x00,0x00,0xcf,0xfe,0xd1,0x01,0x01,0x01,0x00,0x01,0x24,
  0xe7,0xb8,0x24,0x24,0xd1,0xe1,0xdb,0xdb,0xdb,0x01,0x02,0x01,0x02,0x00,0xb5,0x9f,
  0xdb,0x01,0x00,0x00,0xdb,0x00,0x24,0x00,0xe1,0xb8,0xb8,0x00,0x02,0x02,0x03,0x02,
  0x02,0x01,0x01,0x03,0x01,0x02,0x01,0xf6,0xfa,0xb8,0xcf,0x00,0x24,0x03,0x03,0x01,
  0x03,0x03,0xa4,0xcf,0x24,0x00,0x00,0x24,0x00,0xdb,0x01,0xdb,0x24,0xcf,0x00,0x24,
  0x24,0xe1,0x24,0x00,0x24,0x01,0xfc,0x02,0xdb,0xa4,0x00,0x01,0x24,0xdb,0x03,0xb5,
  0x03,0xdb,0x00,0x01,0x00,0x24,0x00,0xe7,0xb8,0xb8,0x00,0x03,0x01,0x01,0xcf,0xd1,
  0xc3,0x00,0x03,0x01,0x02,0xb8,0x00,0xd1,0xb8,0xcf,0xcf,0x00,0x00,0x02,0x03,0x9f,
  0x03,0xa4,0x9f,0xb5,0x00,0x24,0x00,0x00,0x24,0x00,0x00,0xdb,0x00,0x24,0xe1,0x24,
  0xc3,0x24,0x00,0xe7,0x02,0x01,0xfc,0x01,0xbb,0x01,0x03,0x00,0xdb,0x01,0xa4,0xb7,
  0x02,0xa4,0x03,0xfe,0x00,0x00,0x01,0x24,0x24,0x00,0x00,0x02,0x02,0xb8,0xcf,0xf2,
  0x01,0x02,0xc3,0xd1,0x01,0x00,0x01,0xb8,0xcf,0xcf,0xcf,0xdb,0x03,0x9f,0x9f,0x03,
  0x03,0x03,0xf3,0x9f,0x01,0xf6,0xb8,0x24,0x00,0xb8,0x00,0x24,0x24,0x00,0x00,0x00,
  0xc3,0xb8,0x9a,0x00,0x01,0x01,0x00,0x01,0xdb,0xdb,0xdb,0x01,0x03,0x03,0xa4,0x02,
  0x02,0xa4,0x02,0xf7,0xdb,0xd1,0x24,0x00,0x24,0x00,0x24,0x00,0x02,0xc3,0x00,0x00,
  0x02,0xf1,0x02,0x02,0x02,0x01,0xd1,0xcf,0xcf,0xcf,0xc3,0x01,0x9f,0x03,0x03,0xf3,
  0xfb,0xfb,0xf3,0x9f,0xa4,0xbb,0x00,0x24,0x24,0x00,0xb8,0xc3,0x02,0xc3,0x02,0x00,
  0x00,0x00,0x00,0xd1,0x24,0x01,0x00,0xdb,0x00,0x24,0xdb,0x02,0x03,0xa4,0xa4,0xdb,
  0xdb,0xa4,0x9f,0x03,0x00,0x03,0x00,0x00,0x00,0x00,0xa4,0x02,0xc3,0x02,0x00,0xc3,
  0x02,0xf2,0x02,0x01,0xf7,0xd1,0xd1,0xd1,0xc3,0x02,0x03,0x02,0x01,0x9f,0xf3,0xfb,
  0x03,0x9f,0xa4,0xa4,0xbb,0x01,0xc3,0x24,0xdb,0x00,0x24,0x00,0x00,0xf1,0xcf,0xb8,
  0x02,0x01,0x02,0x01,0xc3,0xd1,0xcf,0x00,0x01,0x03,0xb7,0xb5,0xa4,0xa4,0x03,0x00,
  0xdb,0x03,0xbb,0x03,0x01,0x02,0x02,0x00,0xdb,0xff,0xdb,0x02,0x02,0x00,0x02,0xc3,
  0x02,0x00,0x00,0x01,0x02,0xc3,0xd1,0xd1,0x01,0xfb,0x9f,0x01,0xfc,0x01,0x03,0x03,
  0x03,0x03,0xb5,0x03,0x01,0x01,0x01,0xdb,0x00,0x01,0xc3,0xf1,0xb8,0x00,0xcf,0x00,
  0x01,0xdb,0xdb,0xb8,0xcf,0x00,0x00,0x00,0x03,0x03,0xb5,0x03,0xb5,0x02,0xa4,0x03,
  0xdb,0xdb,0x03,0x02,0xdb,0x02,0xbb,0xb5,0x00,0x02,0x01,0x03,0x02,0x00,0xc3,0x00,
  0x02,0x00,0xa4,0xdb,0x01,0x02,0xf1,0x03,0xbb,0x03,0xfb,0x03,0x03,0x03,0x01,0xa4,
  0x01,0x9f,0x03,0x03,0x02,0xb5,0xa4,0xb5,0x02,0x24,0xcf,0xb8,0xcf,0xe1,0xe1,0xf1,
  0x00,0x00,0x00,0x01,0x00,0x24,0xb8,0x01,0x03,0xa4,0x03,0x03,0x03,0x03,0xdb,0xb7,
  0x00,0x03,0xa4,0xa4,0x02,0x01,0x03,0x02,0x01,0x00,0x02,0x02,0x00,0x02,0xcf,0x00,
  0x01,0x01,0x02,0xb5,0x02,0xcf,0x02,0xb5,0x03,0x02,0x03,0x02,0x03,0xdb,0x03,0x9f,
  0xfb,0x02,0x02,0xb5,0x03,0x02,0xa4,0xb7,0x00,0x00,0xb8,0xb8,0xb8,0xb8,0xf1,0x00,
  0xb8,0xe1,0x02,0x01,0x02,0x00,0xdb,0x02,0x03,0x03,0xa4,0x02,0xa4,0xa4,0xb7,0x03,
  0xa4,0x02,0x03,0xa4,0x03,0x01,0x02,0x03,0xa4,0x03,0x02,0x00,0xb8,0xb8,0x00,0x00,
  0x01,0x01,0x00,0x01,0xc3,0x03,0x03,0x02,0x03,0x03,0x02,0xa4,0x01,0x02,0x9f,0xb5,
  0xb5,0x03,0x02,0xdb,0x01,0x02,0x01,0x02,0x00,0x02,0xdb,0xe1,0xb8,0xb8,0x24,0x01,
  0xcf,0xdb,0x01,0x02,0x02,0x02,0xdb,0x03,0xb5,0x03,0x02,0x02,0xb5,0xb5,0xa4,0xb5,
  0xb5,0xb7,0xb5,0x02,0x03,0x02,0xb5,0x03,0x02,0xa4,0xb5,0x02,0xc3,0xb8,0x24,0x01,
  0xc3,0xe1,0xdb,0xcf,0xd1,0x03,0x02,0x02,0x01,0xdb,0x01,0x02,0x02,0x02,0x03,0xf3,
  0xb5,0x01,0x03,0xa4,0x02,0x02,0x01,0xf1,0x01,0xa4,0x03,0x02,0xe7,0xcf,0xcf,0xb8,
  0xe1,0x24,0x00,0x00,0xdb,0xbb,0x03,0xdb,0xa4,0x02,0x02,0x03,0xa4,0xb5,0xb5,0xb5,
  0xdb,0x02,0xdb,0xbb,0xa4,0x03,0x03,0xb5,0x9f,0xb5,0xa4,0xb5,0xf1,0xf1,0xb8,0x01,
  0xb8,0x00,0xdb,0x02,0x02,0x03,0x03,0x03,0x00,0x00,0x01,0x03,0x03,0xdb,0x9f,0x9f,
  0xa4,0xa4,0xdb,0xb5,0x02,0x03,0x01,0x02,0xb7,0x03,0xa4,0x01,0x00,0xb8,0xcf,0xb8,
  0xcf,0x01,0xcf,0xf1,0xb5,0xa4,0xbb,0xa4,0x03,0x03,0x02,0xb5,0xa4,0xa4,0x02,0x01,
  0x02,0xa4,0xa4,0xb5,0xa4,0x02,0xef,0xb5,0xbb,0x03,0x03,0xa4,0x03,0x02,0xf1,0xc3,
  0xdb,0xdb,0x02,0x03,0xb5,0x9f,0x01,0x02,0x03,0x01,0xfc,0x01,0x02,0xb7,0x02,0x02,
  0xdb,0xa4,0x03,0x02,0x01,0xa4,0x01,0x01,0xa4,0x9f,0x03,0x02,0x03,0xf1,0xcf,0xb8,
  0x01,0xb8,0x01,0x03,0x9f,0xf3,0x03,0x02,0xb5,0xb5,0x01,0x01,0x02,0xb5,0x01,0x01,
  0xdb,0xa4,0xb5,0xdb,0x03,0x03,0x02,0x03,0x9f,0xfb,0xbb,0xfb,0xb5,0x02,0xdb,0x00,
  0xe8,0xa4,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xb7,0xdb,0x02,0x01,0x03,
  0x01,0x00,0xa4,0xb5,0xa4,0xdb,0xc3,0xdb,0xbb,0x03,0xa4,0x02,0x02,0x02,0x01,0xcf,
  0xcf,0xf2,0x03,0x03,0x01,0xfb,0xb5,0x03,0xa4,0xa4,0xb5,0xb5,0xb5,0xb5,0xb7,0xb5,
  0xb5,0x02,0x02,0xa4,0x02,0x03,0x02,0x03,0x02,0x02,0x02,0x02,0xb5,0x02,0x02,0xf2,
  0x02,0x02,0x03,0xb5,0x03,0xa4,0x9f,0x03,0xb5,0x03,0x02,0x02,0x01,0x01,0x03,0x00,
  0x02,0x02,0xb7,0xa4,0xa4,0x02,0x01,0xa4,0xdb,0x02,0x03,0x00,0xa4,0x02,0x00,0x01,
  0xc3,0x02,0xdb,0x03,0x03,0x03,0x03,0x03,0xb7,0xa4,0x03,0xb5,0x02,0x02,0xb5,0xb5,
  0x03,0x02,0x02,0x02,0x01,0xa4,0x03,0xbb,0x9f,0x01,0x02,0x02,0x02,0x03,0x03,0x02,
  0x01,0x03,0x02,0x03,0x03,0x03,0xb5,0x03,0x03,0xef,0x02,0xb5,0xb5,0x02,0x02,0x02,
  0xb7,0xb5,0xa4,0xa4,0x03,0x02,0x00,0xdb,0xdb,0xdb,0x01,0x02,0x01,0xdb,0x01,0x00,
  0x02,0x02,0x02,0xdb,0xa4,0x03,0xb5,0x03,0xb5,0xb7,0xa4,0xb5,0x02,0x02,0xa4,0xb5,
  0xf1,0xa4,0x00,0x01,0xb5,0x03,0x03,0x03,0xb5,0x01,0x01,0x01,0x01,0x03,0x03,0xf2,
  0xc3,0x02,0x02,0x01,0x01,0x03,0x03,0x03,0xa4,0xfb,0x03,0x01,0xdb,0xdb,0xb5,0xb5,
  0xb5,0xdb,0x02,0xa4,0x02,0xb7,0xdb,0x00,0xb7,0x02,0x02,0x02,0xb7,0x00,0x01,0xb8,
  0xf7,0x00,0x02,0x03,0xb5,0xb5,0xbb,0xfb,0x03,0x03,0x02,0x02,0x02,0xb5,0xb5,0x02,
  0x03,0xdb,0xf2,0x02,0xa4,0x03,0x01,0x02,0x03,0x01,0x02,0x02,0x03,0x01,0x02,0xb8,
  0xc3,0x02,0x01,0x01,0x01,0x03,0x03,0x03,0xa4,0xb5,0x02,0x01,0x02,0x02,0xdb,0x01,
  0x01,0x01,0x01,0x01,0x01,0x02,0x02,0xdb,0x02,0x03,0x03,0x03,0x00,0xb8,0x02,0x02,
  0x00,0x00,0x00,0xb5,0xb5,0xa4,0xa4,0x02,0xa4,0xb5,0xdb,0x02,0xf7,0x02,0xb5,0x01,
  0x01,0xbb,0x01,0x03,0xf3,0x01,0x02,0x02,0x03,0x02,0x03,0x02,0x02,0x01,0xc3,0xb8,
  0x24,0x01,0xc3,0x01,0x01,0x03,0x03,0x03,0x03,0x03,0xf7,0x02,0x02,0xb5,0x02,0x02,
  0xd1,0xdb,0x00,0x01,0x02,0x02,0xb5,0x01,0x02,0x03,0x01,0x03,0xdb,0xb8,0x02,0x02,
  0x00,0x00,0x00,0x02,0x9f,0xa4,0xa4,0x03,0x03,0x03,0xf2,0xf1,0x02,0xf2,0xf1,0x03,
  0x03,0x03,0x02,0x03,0x03,0xb5,0x01,0xb5,0x03,0xb5,0xbb,0x03,0xb7,0x01,0x01,0x00,
  0xdb,0xbb,0xdb,0xdb,0x02,0xbb,0x03,0x02,0x02,0x02,0xc3,0x02,0x02,0x02,0xdb,0xc3,
  0xc3,0x03,0x00,0xdb,0x02,0x02,0x01,0x01,0x03,0x01,0xb5,0xdb,0x01,0x24,0x24,0x00,
  0x24,0x00,0x00,0xdb,0x02,0x02,0x03,0xdb,0x02,0x02,0x02,0x02,0x02,0xf2,0x02,0x02,
  0xb5,0x03,0x02,0xb5,0xa4,0xb5,0x03,0xb5,0xf3,0x02,0x02,0xdb,0xb8,0xc3,0xd1,0xf6,
  0x01,0x00,0xdb,0xdb,0xdb,0xdb,0x03,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0xc3,0xb8,
  0x02,0xdb,0x02,0x02,0x01,0xc3,0x03,0x01,0x02,0x02,0x01,0x01,0x03,0xdb,0x00,0xdb,
  0xdb,0xe7,0xcf,0x24,0xb8,0xdb,0x03,0x03,0x03,0x02,0x01,0x02,0xdb,0x02,0x02,0x03,
  0xfb,0xdb,0x02,0xb5,0xb5,0xb5,0x03,0x03,0xa4,0xfb,0xf7,0xc3,0xc3,0xcf,0xcf,0xe8,
  0xf6,0xcf,0xdb,0xdb,0xe7,0x24,0xdb,0xb5,0x02,0x03,0x02,0x03,0x02,0x02,0x02,0xc3,
  0x00,0xb5,0x00,0x00,0x02,0x01,0xa4,0x03,0xb5,0x00,0xc3,0xdb,0xa4,0x24,0x24,0xdb,
  0xb8,0xe1,0xe1,0x24,0x00,0x00,0xdb,0x03,0x02,0xef,0xfb,0x02,0x02,0x03,0x02,0x03,
  0xb5,0xdb,0x03,0xb5,0x02,0x03,0x03,0xa4,0x03,0xf7,0xc3,0x00,0xcf,0xe7,0xf6,0xb8,
  0x00,0x00,0x00,0xdb,0x24,0xb8,0xe1,0x02,0xa4,0x01,0xbb,0xdb,0x02,0x03,0xdb,0x01,
  0x01,0xb5,0xb7,0xb5,0xdb,0x01,0x01,0xb5,0xdb,0xc3,0x02,0xdb,0x00,0x00,0x00,0xdb,
  0xe1,0x24,0xe7,0xe7,0x00,0xdb,0x00,0xdb,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0xf2,
  0x03,0x03,0xdb,0x02,0x02,0x02,0x03,0xdb,0x02,0x24,0x01,0xd1,0xd1,0xf6,0xcf,0x01,
  0xf6,0xdb,0x00,0xb7,0xdb,0x00,0x00,0x00,0x03,0xd1,0x01,0xa4,0xb5,0x02,0x02,0x01,
  0x03,0x02,0x01,0x02,0x02,0x03,0xa4,0xdb,0x24,0x01,0x00,0xdb,0x00,0x00,0x24,0xdb,
  0x00,0x24,0xb8,0x00,0xdb,0x00,0x02,0x01,0x02,0xa4,0x03,0x02,0xef,0x01,0x01,0x02,
  0xb5,0xdb,0xb5,0xdb,0x03,0xdb,0x01,0x02,0xc3,0xcf,0x00,0x00,0x24,0xcf,0xf6,0xd1,
  0xdb,0xc3,0xdb,0xdb,0xdb,0xdb,0x24,0x00,0xcf,0xf6,0x01,0xdb,0x01,0x01,0x01,0x01,
  0x01,0xa4,0x01,0x01,0xdb,0xf3,0xb7,0xb8,0xb8,0x24,0xdb,0xdb,0x01,0x24,0x00,0xdb,
  0x01,0x24,0xdb,0x01,0xdb,0xd1,0x00,0x02,0x00,0x03,0xa4,0x02,0x02,0x02,0xdb,0x01,
  0x02,0xb5,0x03,0x02,0x03,0x01,0x02,0x24,0xc3,0xcf,0xf6,0xe7,0x00,0xcf,0xf6,0xd1,
  0xfc,0xc3,0x24,0x24,0x24,0xcf,0xe1,0xcf,0x24,0x01,0x03,0x03,0x03,0x03,0x03,0x01,
  0xbb,0x03,0x02,0x01,0x02,0x02,0x00,0xcf,0x00,0xcf,0x24,0xdb,0xb8,0x24,0xe1,0xdb,
  0xdb,0xc3,0x00,0xd1,0xe8,0x00,0x00,0x00,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x00,
  0xdb,0xb5,0x01,0x00,0x02,0xdb,0xc3,0x00,0x01,0x00,0xcf,0xd1,0xd1,0x24,0xd1,0x24,
  0xb8,0x01,0x24,0xcf,0x00,0x24,0x00,0x00,0x00,0xe1,0xdb,0xa4,0xb5,0xb5,0x02,0x01,
  0x02,0x01,0xa4,0xdb,0x03,0x24,0xc3,0xc3,0x00,0xe1,0x00,0x24,0xdb,0xe7,0x24,0x24,
  0x24,0x01,0x24,0xf6,0xd1,0x00,0x00,0x01,0x01,0xdb,0x01,0x03,0xb5,0x03,0x02,0x01,
  0x01,0x01,0x01,0x01,0xdb,0xe1,0xe7,0x24,0xb8,0x24,0xe1,0x24,0x00,0x01,0x01,0xd1,
  0x00,0xdb,0x00,0xcf,0xcf,0xf6,0xf6,0xf6,0xd1,0x01,0x00,0x00,0x02,0xa4,0x00,0x02,
  0x03,0x03,0xa4,0x03,0x02,0xdb,0x00,0xcf,0xb8,0xb8,0xb8,0xdb,0xf4,0x24,0xb8,0x00,
  0x24,0x24,0x00,0x24,0x00,0x24,0xc3,0x00,0x00,0xdb,0x01,0x00,0x02,0xb5,0x03,0x02,
  0xd1,0x00,0xdb,0x01,0x24,0xb8,0xc3,0x00,0x24,0xb8,0x00,0x00,0x24,0x01,0x00,0x01,
  0xcf,0xdb,0x03,0x00,0xf6,0xf8,0x00,0xd1,0xe8,0xfe,0x00,0xc3,0x00,0xdb,0x01,0x01,
  0xfb,0xa4,0xb5,0xdb,0x24,0xb8,0xc3,0xcf,0xc3,0x24,0x00,0xdb,0xdb,0x02,0x24,0xdb,
  0x24,0xb8,0x24,0x00,0xdb,0x00,0xcf,0x01,0x00,0x00,0x01,0x01,0x02,0x01,0x01,0x02,
  0x03,0x01,0x00,0xb8,0xe1,0x01,0x24,0xb8,0xb8,0xb8,0x00,0x24,0xf6,0x01,0x01,0xf6,
  0x00,0x01,0xc3,0x24,0x24,0xf6,0x24,0x01,0x00,0xf6,0x24,0x24,0xb8,0xe1,0x01,0x01,
  0x03,0x9f,0x03,0x00,0x02,0xcf,0xcf,0xcf,0x01,0x00,0x24,0xdb,0xdb,0xdb,0xa4,0xd5,
  0x00,0x01,0x24,0xdb,0xdb,0x00,0xf6,0xd1,0xf6,0x24,0x00,0x00,0xdb,0x24,0xdb,0x02,
  0xdb,0xcf,0xe1,0xb8,0x24,0xe7,0xe1,0x24,0x00,0x00,0xd1,0x01,0x01,0x24,0x00,0x24,
  0xdb,0x02,0xcf,0xe1,0xfa,0x00,0xe7,0x01,0x02,0x24,0xcf,0x24,0x00,0x01,0x01,0x01,
  0x03,0xdb,0x01,0x00,0x00,0xcf,0xcf,0xf8,0xdb,0x03,0xdb,0x01,0xdb,0xdb,0xb7,0xdb,
  0x00,0x03,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x02,0x00,0xf1,0x00,
  0x24,0x24,0xb8,0x00,0x00,0x24,0xfa,0x24,0xb8,0xdb,0x01,0xe8,0x01,0xdb,0xe1,0x00,
  // 4: ground mud
  0xaa,0x9b,0x08,0x8b,0xeb,0xa3,0xca,0x9b,0x49,0x8b,0x00,0x00,0x08,0x00,0x10,0x00,
  0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,
  0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,
  0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,
  0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,
  0x2c,0xa4,0x6a,0x9b,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,
  0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,
  0x08,0x24,0x0b,0xa4,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,
  0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,
  0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,
  0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,
  0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,
  0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,
  0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,
  0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,
  0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,
  0x69,0x93,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,
  0x18,0x67,0x00,0x80,0x6a,0x8b,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,
  0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x10,0x83,
  0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,
  0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,
  0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,
  0x18,0xa1,0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,
  0x18,0xa3,0x89,0x93,0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,
  0x10,0xa5,0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,
  0x10,0xa7,0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x4c,0xa4,0x00,0xc1,
  0xc8,0x7a,0x6c,0xb4,0x08,0xc1,0x10,0xc1,0x18,0xc1,0xec,0xa3,0x00,0xc2,0x08,0xc2,
  0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x0b,0x9c,0x18,0xc3,0x00,0xc4,
  0x08,0xc4,0x10,0xc4,0x18,0xc4,0xe8,0x82,0x00,0xc5,0x29,0x7b,0x08,0xc5,0x2a,0x8b,
  0x10,0xc5,0x18,0xc5,0x00,0xc6,0x08,0xc6,0x10,0xc6,0x8a,0x8b,0x18,0xc6,0x00,0xc7,
  0x08,0xc7,0x89,0x8b,0xac,0xb4,0x6b,0xa4,0xa8,0x72,0xab,0xa3,0x29,0x93,0x4b,0x9c,
  0x2c,0x9c,0xce,0xbc,0xa7,0x82,0xc7,0x7a,0xa7,0x7a,0x89,0xa3,0xca,0x8b,0x08,0xe1,
  0x02,0x28,0x00,0xe5,0xe3,0x80,0xe5,0xe5,0x00,0x03,0x00,0x04,0x00,0x28,0x29,0x04,
  0x02,0x04,0xf3,0xf2,0xdd,0x28,0x28,0x39,0xdd,0xd1,0x02,0x39,0x00,0xf3,0x39,0x00,
  0x00,0x39,0xb9,0xe5,0x00,0xb9,0x00,0x39,0xe5,0x02,0x00,0xe3,0x04,0x02,0x02,0x80,
  0xb9,0xe5,0xe7,0xd0,0xf4,0x8a,0x00,0x01,0x04,0x03,0xce,0x03,0x02,0x04,0x03,0xb9,
  0xb9,0x02,0x04,0x29,0xe3,0xe3,0xf4,0xb9,0x00,0xe5,0x04,0xe5,0x00,0x00,0x29,0x00,
  0xf1,0x01,0x02,0xce,0xdd,0xce,0x39,0x03,0x02,0x02,0xce,0x29,0x03,0x39,0x39,0x29,
  0xb9,0xe3,0xb9,0xd5,0x00,0x29,0x39,0x8a,0xb9,0x03,0xb9,0x29,0x03,0x03,0x00,0x00,
  0x03,0x01,0xf4,0xe5,0x8a,0x03,0xdd,0xd0,0xf1,0xe5,0x28,0xdd,0x03,0x00,0x00,0x8a,
  0x00,0x02,0x04,0x04,0x29,0x80,0x80,0x00,0x02,0x03,0x03,0x39,0x00,0x01,0x8a,0x80,
  0x39,0xdd,0x00,0x02,0x03,0x80,0x00,0xb9,0x00,0x04,0x04,0x01,0xb9,0xb9,0x02,0x04,
  0xe3,0x04,0xe3,0x29,0x29,0x8a,0x29,0x29,0x03,0x00,0x00,0x28,0x00,0x00,0x00,0x29,
  0x80,0x01,0xf4,0x04,0x03,0x00,0x80,0x80,0xdd,0x04,0x80,0x00,0x03,0x03,0x03,0x00,
  0xd5,0xd1,0x00,0xb9,0x04,0xd1,0x02,0x00,0xce,0xd1,0xd1,0x02,0x04,0x00,0x00,0x04,
  0x00,0x80,0x00,0x03,0xe3,0x01,0x04,0x00,0x04,0xb9,0x01,0x04,0x01,0xb9,0x28,0x29,
  0xe7,0x8a,0x04,0x04,0x8a,0xe5,0x04,0x02,0x03,0x80,0x00,0x80,0x04,0x00,0xb9,0x80,
  0x04,0xb9,0x00,0xd0,0xe3,0x8a,0x80,0xe3,0x04,0x03,0x03,0x02,0x03,0xb9,0x00,0x02,
  0xfd,0x00,0x29,0xb9,0xb9,0xb9,0xf5,0x39,0x02,0x28,0xd5,0x39,0x39,0x03,0x00,0x01,
  0x04,0xe3,0x00,0xed,0xb9,0x00,0xdd,0x03,0x00,0xb9,0x00,0x39,0x39,0x39,0x28,0x02,
  0x8a,0xdd,0xdd,0x03,0xe7,0x8a,0xb9,0x03,0x03,0x03,0x02,0x02,0x00,0x00,0x00,0x03,
  0xb9,0x00,0xe3,0xe5,0xdd,0x00,0x8a,0xe3,0xb9,0x02,0x03,0x02,0x02,0x02,0x00,0xb9,
  0x02,0x03,0x00,0xfb,0x01,0xb9,0x00,0x28,0x39,0xb9,0xd1,0xce,0x03,0x03,0x02,0x04,
  0xf3,0xe3,0xe3,0x04,0x00,0x03,0x03,0x04,0x04,0x01,0x03,0x04,0xdd,0x03,0x02,0x03,
  0xf4,0xdd,0xdd,0x04,0xe3,0x03,0x04,0x29,0x28,0x39,0x39,0x00,0x80,0x00,0x80,0x04,
  0x29,0x00,0xf4,0xd0,0x39,0x03,0xb9,0x80,0x80,0x80,0x04,0xdd,0x39,0x39,0x28,0x00,
  0x39,0x00,0x03,0x80,0x01,0xd1,0xd1,0x02,0x03,0x39,0xd1,0xd1,0x00,0xed,0xdd,0xf1,
  0x00,0x03,0x01,0xfb,0xe5,0x00,0xb9,0x29,0x03,0x39,0x03,0xb9,0x01,0x39,0x03,0x03,
  0x8a,0x39,0x39,0x00,0x00,0x03,0x04,0x80,0x28,0x03,0x00,0xb9,0x01,0xb9,0xb9,0x04,
  0x04,0x04,0xd0,0xfb,0x03,0xb9,0x00,0x80,0xb9,0x03,0x28,0xce,0x28,0x39,0x28,0x28,
  0xb9,0xb9,0x00,0x02,0x03,0x39,0xb9,0x39,0x00,0xd1,0xd1,0xdd,0xf1,0x00,0x39,0x04,
  0x03,0x03,0xb9,0x03,0x03,0xdd,0x04,0x03,0xdd,0x00,0x03,0x03,0xf1,0xdd,0x04,0x00,
  0x00,0x02,0x00,0xb9,0xe5,0x04,0xb9,0xb9,0x39,0x03,0x03,0x00,0xb9,0x04,0x04,0x80,
  0xb9,0x8a,0xf4,0xd0,0x00,0xb9,0x03,0xb9,0x39,0x02,0x39,0xce,0xf3,0x02,0x39,0x39,
  0x00,0x03,0x80,0xdd,0x04,0xfd,0xb9,0xf3,0x39,0x03,0x39,0xe3,0xb9,0x04,0x39,0x00,
  0x00,0x03,0x03,0x03,0x00,0x80,0xb9,0x00,0xb9,0x04,0xb9,0x04,0x04,0xb9,0xd0,0xd0,
  0x04,0xf7,0x03,0x04,0x8a,0x29,0x80,0x00,0xdd,0x04,0xd0,0x8a,0x04,0x04,0x00,0xb9,
  0xb9,0x04,0x04,0x04,0xe3,0xdd,0x03,0x03,0x02,0x00,0x80,0x03,0x39,0x02,0xdd,0x02,
  0x00,0x39,0xd1,0x03,0x00,0x39,0x39,0xdd,0x03,0x03,0x03,0xb9,0xe3,0x04,0x02,0x02,
  0x03,0x01,0x04,0x04,0x03,0xb9,0x04,0x39,0x03,0x03,0x04,0xe3,0x03,0xdd,0x00,0x00,
  0x03,0xdd,0xb9,0x04,0x01,0xe3,0xf4,0xe3,0xe5,0xe3,0xe3,0x80,0x00,0x00,0x03,0x00,
  0xb9,0x03,0xb9,0x04,0xe5,0x01,0x03,0xf1,0xe3,0xe3,0x01,0x03,0xf3,0x39,0xd1,0x39,
  0x04,0x39,0x39,0xdd,0xf1,0xdd,0xdd,0xfe,0x03,0x03,0xe5,0x04,0xf4,0xf1,0x03,0xb9,
  0x00,0xed,0xf1,0xfc,0xf4,0x01,0x03,0x39,0x39,0x02,0x03,0x00,0xdd,0xdd,0x04,0x28,
  0x28,0x03,0x00,0xe3,0xe5,0x04,0xb9,0x00,0x00,0x03,0x03,0x00,0xb9,0x03,0x00,0x00,
  0x00,0x04,0x04,0x04,0xb9,0xe3,0xd0,0xf4,0xd0,0x00,0x04,0x04,0xfe,0xdd,0x28,0x03,
  0x00,0x03,0xf3,0xdd,0x03,0x00,0x04,0x03,0xdd,0x03,0x04,0xe3,0x04,0xfb,0x04,0x00,
  0x04,0xf1,0x04,0x00,0x03,0x03,0x00,0xdd,0x01,0x03,0x04,0xe3,0x03,0x00,0x03,0xb9,
  0x04,0x00,0x80,0xf4,0xe5,0x04,0x04,0xdd,0xdd,0x39,0x03,0x04,0xe3,0x04,0x04,0x04,
  0x00,0x00,0x02,0x03,0xb9,0x80,0x04,0x04,0x03,0xdd,0x04,0xb9,0x03,0xb9,0x03,0xdd,
  0x39,0x39,0xdd,0xdd,0x04,0xe3,0x01,0xe3,0x03,0xe3,0xfb,0xb9,0x03,0x01,0xb9,0xb9,
  0x03,0xd0,0x00,0xb9,0x04,0x00,0x00,0x00,0xf1,0x03,0xe5,0xe3,0x03,0xb9,0x03,0xe3,
  0x03,0x03,0x04,0xf4,0x00,0x04,0x80,0x03,0x03,0x00,0x03,0xe3,0xe3,0x00,0xb9,0xb9,
  0xb9,0x00,0xb9,0x00,0xb9,0x39,0xb9,0x00,0x39,0x03,0x80,0x39,0xf3,0xd1,0x00,0x00,
  0x03,0x03,0x00,0x28,0x39,0xb9,0x03,0x01,0x00,0xf3,0xd1,0xf1,0x03,0xe3,0xb9,0xb9,
  0xf4,0xd0,0x04,0xe3,0xe3,0x01,0x29,0xb9,0x00,0x03,0x04,0xe3,0x39,0x03,0x00,0x03,
  0x04,0x04,0xe3,0xb9,0xe3,0x00,0xb9,0x80,0x00,0x28,0x03,0xb9,0x80,0x04,0x04,0x80,
  0x00,0x00,0x04,0x80,0x00,0xd1,0xd1,0x00,0x01,0x00,0x03,0xf3,0x39,0x39,0x03,0xdd,
  0x28,0xdd,0xdd,0x03,0x80,0xb9,0x00,0x00,0x02,0x39,0x39,0xe3,0xe3,0x80,0x04,0x29,
  0xe5,0xb9,0xb9,0x00,0x03,0x00,0xb9,0x00,0x03,0x39,0x03,0xe3,0xdd,0x03,0x03,0x03,
  0x80,0xb9,0x8a,0xe5,0x04,0x8a,0x04,0x02,0x03,0x03,0x04,0xb9,0x04,0xd0,0xf1,0x80,
  0x02,0x80,0x03,0xd1,0xd1,0x02,0xd1,0x02,0x02,0x04,0x80,0x39,0x03,0x00,0xed,0x39,
  0x02,0x03,0x02,0x03,0xd0,0x04,0xf4,0x04,0xdd,0x00,0x04,0xf1,0x39,0xe3,0x04,0x29,
  0xed,0xb9,0x03,0x03,0xb9,0x01,0xb9,0x39,0xd1,0x00,0x80,0xb9,0xdd,0x04,0xf4,0xed,
  0x04,0x00,0xf1,0xe3,0xe5,0x29,0x00,0x28,0xd5,0xdd,0x02,0x00,0xe3,0x00,0x00,0x03,
  0x02,0x00,0x00,0x02,0x39,0xd1,0x28,0xdd,0x02,0xe3,0xfb,0x03,0x39,0x39,0x80,0xf3,
  0x03,0x00,0x02,0x03,0x29,0xf4,0xdd,0x39,0x03,0x01,0xdd,0x03,0x03,0x80,0xce,0x8a,
  0xf1,0xb9,0x04,0x04,0xb9,0x80,0x00,0x28,0xdd,0x03,0xb9,0xb9,0x03,0xe5,0xe3,0x00,
  0x00,0x04,0x04,0xb9,0x01,0x04,0x00,0x00,0x03,0x28,0xdd,0xb9,0xd0,0x80,0xdd,0xdd,
  0x80,0x04,0xd0,0xe3,0x00,0xe3,0xfb,0x04,0x80,0x03,0x03,0xd1,0x28,0x39,0xd5,0x28,
  0x39,0x28,0x02,0x29,0x80,0x02,0x28,0x03,0x39,0xf3,0xd1,0x80,0x80,0xb9,0x39,0x04,
  0x00,0xdd,0x00,0x00,0x03,0x00,0x04,0x03,0x39,0xb9,0x03,0x03,0x8a,0x04,0x00,0x03,
  0xb9,0x29,0x8a,0x29,0xe3,0x04,0xb9,0x29,0xe5,0x04,0x04,0x04,0x04,0x04,0x00,0x03,
  0x80,0xd0,0xf4,0xf4,0xd0,0x80,0xb9,0x00,0x03,0xce,0xdd,0xd1,0x03,0x00,0x00,0x28,
  0x28,0x28,0x03,0x01,0x00,0x04,0x01,0xb9,0x02,0x39,0xb9,0x01,0x00,0x03,0x00,0x02,
  0xd0,0xe3,0xdd,0x02,0x02,0x00,0xdd,0x39,0x00,0x80,0x39,0x03,0x02,0xd0,0x04,0x00,
  0x04,0x00,0x8a,0x29,0xb9,0x29,0xe3,0xb9,0x04,0x04,0xe5,0x04,0xe3,0xb9,0x03,0x00,
  0xe3,0xe3,0xb9,0xce,0x03,0x39,0x39,0x39,0x28,0x28,0xb9,0xe3,0x04,0x29,0x03,0xdd,
  0xd5,0x02,0x00,0x03,0x00,0xfb,0x04,0x02,0x03,0xe3,0x03,0xb9,0x39,0xb9,0x39,0xb9,
  0xb9,0x00,0x00,0xe3,0xd0,0xfb,0xb9,0x00,0x00,0x00,0xed,0xb9,0x00,0xd0,0x00,0x00,
  0xe3,0xe3,0x04,0xe3,0x04,0xd0,0x04,0xed,0xe5,0x00,0xf8,0x00,0xf4,0xe5,0xe7,0xd0,
  0x8a,0xb9,0x03,0x03,0x04,0x04,0xd1,0x28,0x29,0x80,0xb9,0xb9,0x03,0x80,0xe5,0x03,
  0x28,0xd1,0xf3,0xf3,0x39,0x03,0x02,0x00,0x03,0x03,0x03,0x03,0xce,0x04,0x00,0x00,
  0x04,0x00,0xe3,0xe3,0xf4,0xe3,0x02,0x02,0x03,0xb9,0xe5,0x04,0x8a,0x04,0xb9,0x00,
  0x29,0x29,0x04,0x01,0xd0,0xd0,0xf4,0xe5,0xe7,0x39,0xce,0xd5,0x04,0x29,0xe3,0x29,
  0xe3,0xe5,0xf4,0x8a,0x00,0x8a,0x8a,0x39,0x00,0xce,0x03,0xf1,0x04,0x04,0x03,0x28,
  0x28,0x39,0x03,0x39,0x39,0x01,0xd0,0x03,0x00,0x00,0x00,0x04,0x29,0x29,0x29,0x80,
  0x29,0x80,0xd0,0x04,0x04,0xb9,0x80,0x03,0xb9,0x00,0xd0,0x04,0xb9,0xf4,0xe3,0x03,
  0xb9,0xb9,0x29,0x8a,0xf4,0xf4,0x29,0x00,0x00,0x00,0x8a,0x8a,0xb9,0xe5,0x03,0x04,
  0x04,0xb9,0x04,0x03,0x03,0xed,0xed,0xe5,0x00,0xdd,0x04,0x71,0xfb,0x04,0x03,0x39,
  0x03,0x39,0xf3,0xd1,0x39,0x02,0x00,0x28,0xce,0x28,0x04,0x04,0x00,0xe3,0xe5,0xe3,
  0xe3,0xe3,0xfc,0x04,0x01,0xdd,0x03,0x00,0x03,0x04,0xd0,0x00,0xe3,0x29,0xdd,0x04,
  0xd0,0xd0,0x29,0xe3,0xd0,0x80,0x80,0xb9,0x28,0x00,0x28,0x02,0xb9,0x04,0x8a,0x8a,
  0x04,0xb9,0x00,0x29,0x04,0x8a,0x04,0x8a,0xe5,0xfb,0xe3,0x04,0x02,0x02,0x00,0xdd,
  0x39,0x39,0xf3,0x39,0x00,0x00,0xfc,0xd0,0x00,0x39,0x39,0x03,0x04,0xfc,0x04,0xd0,
  0xd0,0x04,0xf4,0x04,0xe3,0x80,0x28,0x00,0x39,0x29,0x04,0x03,0x00,0xb9,0xb9,0x29,
  0x29,0x00,0x01,0x04,0x01,0x02,0x00,0x8a,0x02,0x00,0xd1,0xd1,0x03,0x04,0x00,0x8a,
  0xe5,0x04,0x04,0x00,0x02,0xdd,0xdd,0xdd,0xb9,0xb9,0xd1,0x39,0xd5,0x00,0x03,0x03,
  0xb9,0x00,0x03,0x03,0x00,0xf4,0xe3,0xb9,0xdd,0x00,0xdd,0x01,0x04,0x02,0x00,0x80,
  0x01,0xe3,0x03,0x39,0x03,0x03,0xb9,0x29,0x00,0x03,0x03,0x39,0x00,0xe3,0x29,0x00,
  0x00,0x29,0xd0,0x80,0x02,0x00,0x00,0xb9,0x29,0x00,0x00,0xf5,0x29,0x8a,0x8a,0xe5,
  0xd0,0xf6,0x29,0x03,0x03,0xce,0x39,0x28,0xd1,0x03,0xd1,0x28,0x02,0x28,0xce,0xd5,
  0x28,0xce,0xdd,0x00,0x03,0x00,0x28,0x03,0xdd,0x8a,0xed,0x04,0x00,0xb9,0x04,0x04,
  0xf4,0x01,0x80,0x01,0x03,0x03,0x01,0xe5,0x04,0x03,0x29,0x28,0x39,0x03,0x00,0x29,
  0x01,0x04,0x00,0x03,0x80,0x29,0x02,0x29,0x00,0xb9,0xb9,0x00,0xb9,0x00,0x8a,0xe3,
  0x01,0x29,0x04,0xb9,0x80,0x28,0xce,0x03,0xd1,0x39,0x00,0x00,0x03,0xf2,0xd1,0xf9,
  0xce,0xf9,0x02,0x28,0x00,0x00,0x28,0x8a,0xb9,0xb9,0x00,0x00,0xd0,0x71,0xd0,0xd0,
  0x80,0x39,0xb9,0xf1,0x80,0xe3,0xb9,0xe3,0x39,0x00,0x00,0x28,0x28,0xb9,0x01,0xd0,
  0xb9,0x03,0x03,0xb9,0x29,0x00,0x00,0xb9,0xb9,0x29,0x28,0x28,0x00,0x02,0x00,0xf6,
  0x29,0x29,0x00,0x01,0xb9,0x03,0xf3,0x28,0x28,0x80,0xe7,0x03,0x28,0xf9,0xd1,0xd1,
  0xd1,0x28,0x39,0xd1,0x02,0x8a,0x03,0xdd,0x28,0x03,0x02,0xd1,0xdd,0x04,0xb9,0x80,
  0x00,0xb9,0x80,0x03,0x03,0x02,0x00,0x04,0x03,0x03,0xb9,0xd1,0x00,0x01,0x29,0x29,
  0x29,0xe3,0xb9,0x01,0x80,0x00,0x00,0x29,0x00,0x04,0xdd,0x03,0x00,0x02,0x80,0x29,
  0x00,0x80,0x00,0x04,0xb9,0xb9,0xb9,0x00,0x04,0xb9,0x29,0x02,0xd1,0xd1,0x02,0x28,
  0x28,0x28,0xd5,0xd1,0x28,0x39,0xd1,0xce,0xf9,0xf9,0xce,0xd1,0x28,0x39,0x02,0x02,
  0xed,0x80,0x29,0x01,0x04,0x04,0xe3,0x04,0x03,0xb9,0x39,0x02,0x00,0x03,0x02,0x80,
  0xfb,0x01,0x29,0x00,0x01,0x01,0xfd,0x29,0x03,0x04,0x00,0xb9,0x01,0xb9,0x00,0x00,
  0x80,0xe3,0x00,0x80,0x8a,0x04,0x00,0x00,0x02,0x00,0x00,0x03,0x00,0x00,0xd1,0xf2,
  0xd1,0x00,0x28,0x28,0xce,0xd1,0xce,0xd1,0xd1,0xce,0x28,0xf3,0x39,0xb9,0x39,0x28,
  0x29,0x00,0x01,0xd0,0xe3,0x80,0x00,0x03,0x00,0xd0,0x8a,0xe3,0x29,0xb9,0x04,0xd0,
  0xf4,0xd0,0x00,0x29,0xfd,0x00,0x00,0x00,0x00,0xe3,0x03,0x39,0x04,0x80,0xb9,0xb9,
  0x04,0xfc,0x02,0x04,0xe3,0x04,0x02,0x03,0xb9,0xe7,0xdd,0x03,0xce,0x03,0x03,0x28,
  0x02,0x8a,0x28,0xce,0xf2,0x39,0xd1,0xdd,0xdd,0x03,0x39,0x03,0x03,0xdd,0x03,0x28,
  0x00,0xb9,0x04,0x00,0x04,0x80,0xb9,0xf4,0x04,0x03,0x00,0xf4,0xe3,0xd0,0x04,0xe3,
  0xe5,0x00,0x29,0x39,0x02,0x00,0x02,0xf5,0x02,0xe3,0x00,0x39,0x02,0x00,0x80,0xb9,
  0x00,0x04,0x29,0xe3,0x04,0xb9,0x39,0xe7,0x03,0x04,0x03,0x28,0xce,0x28,0x02,0x00,
  0x00,0xd5,0xd1,0xd1,0xd1,0x00,0x01,0xb9,0x8a,0x03,0x28,0x02,0x00,0x02,0x29,0xd5,
  0x29,0x00,0x04,0x29,0x04,0x04,0xb9,0xb9,0x03,0xb9,0x28,0x00,0xb9,0x29,0x00,0x04,
  0x00,0xb9,0x29,0x03,0x03,0x80,0xd1,0x02,0xf5,0xd0,0x29,0x02,0x02,0x03,0x80,0x80,
  0x01,0x01,0x80,0xe3,0x04,0x02,0xdd,0x29,0xb9,0xb9,0xd1,0xf9,0xce,0x28,0xce,0x02,
  0x03,0xd1,0x28,0x02,0x00,0x29,0x03,0x00,0x02,0x28,0x28,0xce,0x03,0x00,0x00,0x03,
  0x8a,0x03,0xb9,0x29,0xe3,0xe5,0x8a,0x8a,0x04,0xd0,0xd5,0x01,0x04,0xf1,0x04,0x29,
  0x03,0x04,0x80,0x00,0x29,0x00,0x00,0x29,0xf4,0x29,0x39,0x02,0x02,0xb9,0xb9,0x01,
  0x04,0x04,0x01,0xd0,0xfb,0x80,0xe3,0xb9,0x03,0xd1,0xd1,0xd1,0x00,0x00,0x28,0xb9,
  0x03,0x02,0x28,0xf9,0x00,0xb9,0x39,0x02,0x39,0x28,0x39,0xd5,0x29,0x04,0x00,0x00,
  0x29,0x29,0x80,0x80,0xb9,0xb9,0xe3,0x03,0xe7,0xe3,0x29,0x04,0x80,0xf1,0xd0,0xe3,
  0xe3,0x04,0xb9,0xb9,0x04,0xd0,0x04,0xe5,0x8a,0x02,0x02,0xd5,0x00,0x80,0x00,0x04,
  0x01,0x04,0x80,0xb9,0x04,0xe3,0x04,0x80,0x03,0xce,0x00,0xd5,0x00,0xce,0xf9,0xce,
  0x28,0xd1,0x28,0x02,0x28,0xb9,0x03,0x28,0x39,0xce,0xf8,0x00,0xb9,0x04,0x03,0x03,
  0x29,0x00,0x04,0xe3,0x8a,0x03,0x00,0x00,0xf4,0xe5,0x28,0x39,0x28,0xdd,0xe5,0x8a,
  0x03,0x03,0x00,0x04,0xd0,0xe3,0x04,0x04,0x29,0x00,0x03,0x29,0x00,0x03,0x29,0x39,
  0x80,0x04,0xe5,0x04,0x00,0x8a,0x8a,0x04,0xf3,0xdd,0x28,0x28,0x00,0x28,0xd1,0x28,
  0x00,0x39,0x02,0xb9,0x00,0x00,0xb9,0x04,0x29,0x8a,0xb9,0x29,0xe5,0x00,0x03,0x03,
  0x03,0x29,0x8a,0x29,0x04,0x8a,0x03,0x04,0xe5,0xf1,0x03,0xb9,0x00,0xe7,0x00,0x39,
  0x39,0x00,0x29,0x04,0x00,0x03,0x03,0x03,0x00,0x00,0x00,0x02,0x29,0xe5,0x00,0x04,
  0x04,0xe3,0x04,0xf1,0x00,0x03,0x00,0x00,0x39,0xce,0x28,0xd5,0x00,0x28,0x02,0x00,
  0x03,0x02,0x03,0x00,0x02,0x00,0x80,0x39,0x00,0x28,0x02,0x29,0xb9,0x29,0x00,0xdd,
  0x03,0x00,0xb9,0x00,0x03,0x00,0x29,0xe5,0xf4,0xe5,0x03,0xb9,0x03,0xdd,0x04,0x8a,
  0x8a,0xfe,0x04,0x8a,0x8a,0xed,0xd5,0x28,0x00,0xb9,0x00,0x02,0x02,0x00,0x29,0x80,
  0x04,0x8a,0x29,0x01,0x00,0xb9,0x02,0x39,0x02,0xdd,0xb9,0x04,0xb9,0xd5,0x00,0x02,
  0x28,0x02,0x02,0x29,0x03,0xb9,0xb9,0x28,0x02,0xb9,0x00,0x02,0x00,0x28,0x8a,0x8a,
  0xd5,0xb9,0x29,0x00,0x03,0x29,0x8a,0x29,0x04,0x00,0x03,0x00,0x03,0x29,0x80,0x29,
  0x29,0x00,0x8a,0xdd,0x00,0x00,0xb9,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x28,0x03,
  0x04,0x29,0x04,0xe3,0x00,0x00,0x28,0x02,0x00,0x00,0xb9,0x04,0x28,0x28,0xdd,0x28,
  0x00,0x28,0x39,0x29,0xe3,0x03,0x03,0x00,0xdd,0xf5,0x00,0x02,0x29,0x02,0x00,0xed,
  0xe7,0x8a,0x8a,0x00,0x29,0x8a,0x04,0xe3,0x04,0xb9,0x00,0xed,0x00,0x03,0x00,0xb9,
  0x03,0x00,0xed,0x02,0xce,0xdd,0x03,0x02,0x00,0xd5,0x00,0x80,0x02,0x29,0x02,0x04,
  0x00,0x03,0xe3,0xd0,0x00,0xdd,0x28,0xdd,0x29,0xb9,0xe5,0xd1,0x02,0x28,0x03,0x02,
  0x03,0x02,0x02,0x02,0x28,0xd1,0x39,0x00,0x02,0x02,0x00,0xe3,0x04,0x00,0x00,0x04,
  0x29,0xe3,0xe3,0x8a,0x04,0xe5,0x29,0xb9,0xe5,0xed,0x8a,0x8a,0x29,0xdd,0x03,0x02,
  0x00,0x8a,0xb9,0x00,0x04,0x00,0xd5,0x00,0x03,0xd1,0x02,0x29,0x00,0x29,0x00,0x04,
  0xdd,0xb9,0x04,0x03,0x03,0xdd,0x02,0x00,0xb9,0x00,0x39,0x28,0x02,0xd1,0x03,0xd5,
  0xd5,0x02,0x02,0x39,0x00,0x02,0xb9,0x03,0x02,0xb9,0x29,0x03,0xb9,0x03,0x29,0xe3,
  0xb9,0x04,0x29,0xe5,0xe5,0xe3,0xdd,0x8a,0x01,0xd0,0xe5,0xe3,0x8a,0xb9,0x00,0x03,
  0x8a,0x04,0xe3,0x29,0x03,0xd5,0x02,0x00,0xb9,0x02,0x28,0xd5,0x29,0x80,0x00,0x00,
  0x28,0x00,0x00,0x00,0x8a,0x03,0xd5,0x02,0x28,0x03,0xd1,0x02,0x28,0xf8,0x03,0x00,
  0xd1,0x28,0x03,0xe3,0xd0,0xe5,0xb9,0x29,0x00,0x00,0xb9,0x00,0x8a,0xb9,0x29,0x8a,
  0xb9,0x29,0xb9,0xe7,0x8a,0xdd,0x00,0x00,0xe3,0xe3,0x28,0x39,0x39,0x03,0x39,0x03,
  0xb9,0x8a,0x03,0x03,0x03,0xd5,0x28,0x02,0x03,0x28,0x28,0x28,0x02,0x03,0x04,0x04,
  0x00,0x04,0x03,0x00,0x04,0xed,0x03,0x00,0x00,0x02,0xd5,0x02,0xd5,0x02,0x02,0x00,
  0x39,0x03,0xe3,0xb9,0xe5,0x8a,0x8a,0xe3,0x00,0xd5,0x00,0xb9,0x8a,0x00,0x29,0x8a,
  0x03,0x29,0xd0,0x04,0xb9,0xb9,0x03,0x28,0x29,0xe3,0x00,0x29,0x8a,0x00,0xdd,0x8a,
  0x8a,0xe7,0x00,0x00,0x39,0x28,0xce,0x28,0xd5,0x28,0xd5,0x02,0x03,0x02,0x00,0x00,
  0xfc,0x04,0x80,0x8a,0xe3,0x04,0xf1,0xdd,0x8a,0xb9,0xd5,0xd5,0x00,0x03,0xce,0x02,
  0x04,0xe5,0x04,0x80,0x03,0xb9,0xe3,0xe3,0x00,0xb9,0x8a,0x00,0x03,0xb9,0x03,0x00,
  0xb9,0x8a,0x8a,0x28,0x28,0xf3,0xe7,0xe5,0x04,0x00,0x00,0xdd,0x00,0xdd,0x03,0x8a,
  0x00,0x03,0x00,0x28,0x28,0x02,0x28,0xf5,0xd1,0xd5,0xd5,0x29,0x00,0xb9,0xb9,0x28,
  0x04,0xf6,0x00,0x29,0xe5,0x04,0xb9,0x00,0x28,0xb9,0x03,0x02,0x03,0x39,0xdd,0x29,
  0x28,0xe3,0xe5,0x29,0xf6,0xfc,0x04,0xe5,0xb9,0x00,0x00,0x03,0x8a,0xdd,0x28,0xe5,
  0x8a,0xe5,0x03,0xb9,0x8a,0x00,0x29,0x00,0x8a,0xe5,0xe5,0x01,0xb9,0xb9,0x8a,0xb9,
  0x00,0x00,0x00,0x28,0x28,0x00,0x03,0x02,0x02,0xd5,0xd5,0xd5,0x29,0x29,0x29,0x00,
  0xe5,0xd0,0xe3,0xe5,0x04,0xb9,0xb9,0x00,0x8a,0xe7,0x00,0xf5,0x39,0xb9,0x02,0x03,
  0x00,0xe5,0xe5,0xfb,0x04,0x00,0xe7,0x04,0x39,0xd5,0x28,0xb9,0xdd,0x03,0xd5,0xe5,
  0xd5,0x8a,0xdd,0xe3,0xd0,0xb9,0x00,0x39,0x8a,0x29,0x03,0x03,0xed,0xe3,0x00,0x80,
  0xb9,0x28,0x00,0x02,0x03,0x00,0x02,0x02,0x29,0x28,0x28,0x28,0xd5,0x29,0x04,0xb9,
  0x8a,0x01,0x01,0xe3,0x29,0x04,0xed,0x04,0x03,0x29,0x8a,0x8a,0x03,0xd1,0x00,0x03,
  0x03,0x04,0xe5,0x80,0x02,0x39,0xb9,0x00,0x03,0x39,0xdd,0xd5,0xb9,0x00,0xd1,0xb9,
  0x00,0x00,0xb9,0xce,0xd5,0x03,0xdd,0x00,0xdd,0x00,0x00,0xb9,0x04,0x00,0x00,0xb9,
  0x00,0x04,0xe7,0xb9,0xb9,0x29,0x00,0xd5,0x00,0x00,0x29,0xd5,0x00,0x00,0x04,0x03,
  0x04,0x8a,0x8a,0xe5,0x00,0x04,0x00,0x00,0xe5,0x00,0x8a,0xd0,0x8a,0x8a,0x29,0x04,
  0x39,0x02,0x04,0x00,0x80,0x04,0x04,0xd5,0x03,0xb9,0xce,0x28,0x00,0xd1,0x28,0x29,
  0xe3,0x00,0xdd,0xb9,0x00,0x00,0x39,0x39,0xdd,0x01,0x00,0x29,0xb9,0x03,0xdd,0x03,
  0x29,0xe3,0x00,0x00,0xd5,0x00,0x00,0xd5,0x03,0x28,0xd1,0x39,0xb9,0x00,0xe3,0x04,
  0xe5,0xb9,0x04,0x04,0x80,0x04,0x00,0x04,0x29,0x8a,0xe3,0xe3,0x04,0x04,0x00,0x28,
  0x28,0x02,0x03,0x04,0xb9,0x03,0x02,0x00,0xd5,0xb9,0x03,0x28,0x03,0xd1,0x28,0x04,
  0xe3,0x04,0x00,0x00,0xdd,0xdd,0x28,0x00,0x00,0x28,0x39,0xb9,0x02,0x02,0x00,0x39,
  0x80,0x80,0x02,0x8a,0x29,0x29,0x29,0xd1,0x28,0xce,0xce,0xce,0xce,0x02,0x00,0xe7,
  0xb9,0x04,0xe3,0x00,0x00,0xb9,0x00,0x00,0x04,0x8a,0x00,0x29,0x29,0x29,0x28,0x28,
  0x03,0x03,0x00,0x04,0x02,0x02,0x03,0x02,0x28,0xd5,0xdd,0xb9,0x03,0x28,0x00,0x01,
  0x00,0x03,0x03,0xe3,0xd0,0xe5,0xb9,0x29,0x80,0x00,0x02,0x29,0x00,0xe3,0x29,0x03,
  0x00,0xdd,0x00,0x29,0xdd,0xd5,0x39,0xce,0x39,0x02,0x28,0xd1,0x28,0x02,0x03,0x03,
  0x00,0x04,0x00,0x29,0x01,0x00,0xdd,0xb9,0xb9,0x29,0x00,0x29,0x00,0x03,0x02,0x03,
  0xd1,0x00,0x29,0x00,0x02,0x28,0x00,0x29,0x00,0x8a,0x00,0xb9,0x8a,0xb9,0xd5,0x29,
  0x8a,0x29,0x03,0x8a,0xe5,0x04,0xed,0x00,0x03,0xb9,0x04,0x29,0x04,0x29,0x02,0x02,
  0x00,0x8a,0x03,0x02,0xd5,0x03,0x00,0x39,0x28,0x28,0x28,0x03,0x02,0x28,0x03,0xdd,
  0x03,0xe3,0xb9,0xb9,0x03,0x00,0x00,0x03,0x00,0x03,0x00,0x80,0x29,0x28,0x03,0x02,
  0x28,0x03,0x03,0xd1,0x00,0xe3,0x04,0xf5,0x03,0x00,0x28,0x00,0x02,0xdd,0x00,0x04,
  0x00,0xe5,0x29,0xb9,0x29,0x00,0x04,0x04,0x80,0x80,0x80,0xe3,0x29,0x02,0xd1,0x28,
  0x00,0x28,0x80,0x00,0x03,0x28,0x03,0xce,0x28,0x02,0x28,0x00,0xdd,0xb9,0x28,0x39,
  0x00,0xb9,0x03,0xed,0x04,0xb9,0x00,0xb9,0x80,0x03,0x00,0x00,0x00,0xf5,0x03,0x00,
  0x28,0x02,0x00,0xd5,0xe5,0x04,0xe3,0x02,0xd5,0xd1,0x28,0x28,0x39,0xd5,0x02,0x80,
  0xdd,0xdd,0x8a,0xe5,0x8a,0x03,0xb9,0x00,0xe3,0xe3,0x00,0x29,0xd1,0x28,0xf5,0x00,
  0x00,0x00,0xdd,0x28,0xdd,0x02,0x00,0x28,0xd5,0x04,0xb9,0x28,0xce,0x28,0x03,0x00,
  0x8a,0x28,0xf8,0x03,0x04,0xb9,0xe3,0x00,0x00,0xce,0x39,0x02,0x02,0x00,0xce,0x28,
  0x29,0xe3,0xb9,0x00,0x00,0x00,0x00,0xb9,0xd5,0x02,0xdd,0x02,0x39,0x28,0x28,0x03,
  0xd1,0xb9,0x29,0x29,0xb9,0x8a,0x00,0x00,0xf4,0xe5,0x00,0xb9,0x00,0x28,0x39,0x00,
  0xb9,0x03,0x29,0xb9,0xb9,0x00,0x29,0x03,0x28,0x03,0x28,0xd1,0xce,0xb9,0x8a,0xb9,
  0x03,0x03,0xdd,0x04,0xb9,0x8a,0x04,0x00,0x28,0x00,0x39,0x28,0xce,0x28,0x00,0xb9,
  0x03,0xb9,0x01,0x03,0x00,0x29,0x80,0x02,0xdd,0x29,0x28,0x28,0x00,0x28,0x02,0x80,
  0x02,0x04,0x00,0xce,0x02,0xb9,0x8a,0xe5,0x00,0x39,0xd1,0x29,0x00,0x28,0x28,0x04,
  0xf5,0x02,0x00,0x03,0x03,0x00,0x01,0xdd,0x03,0x29,0x00,0x80,0x00,0x01,0x04,0x39,
  0x29,0x04,0xb9,0x04,0x03,0x8a,0xe5,0xe5,0xf1,0x39,0x03,0x00,0x00,0x29,0x29,0x02,
  0x02,0xd5,0x39,0x02,0x80,0x02,0x02,0x00,0x39,0xd1,0xd1,0x28,0x03,0x03,0xdd,0x39,
  0x03,0x29,0x29,0x03,0x39,0xdd,0x29,0xb9,0xd1,0xd1,0xd1,0x28,0x03,0xd1,0x8a,0x04,
  0x29,0x39,0x00,0xb9,0x29,0x00,0x00,0x29,0x00,0x03,0x39,0x04,0x04,0x00,0x03,0x03,
  0x29,0x03,0x03,0x03,0x80,0xe5,0xb9,0xe3,0x00,0xdd,0x39,0x00,0x29,0xe7,0x02,0x02,
  0x02,0xd1,0x02,0x39,0x39,0x39,0x04,0x00,0x28,0xd1,0xd1,0x00,0x03,0xdd,0xdd,0x03,
  0xdd,0x03,0x00,0xb9,0x8a,0xe3,0xb9,0xdd,0x39,0x39,0x02,0x03,0x29,0x8a,0x00,0x02,
  0xd1,0x02,0x02,0x00,0x04,0x29,0xf6,0x29,0x00,0x04,0x04,0xb9,0xed,0x04,0x29,0x04,
  0xb9,0x03,0x00,0x00,0xb9,0xe3,0x02,0xdd,0xb9,0x00,0xce,0x00,0xe7,0xe7,0x8a,0x02,
  0x02,0x39,0x03,0x00,0x39,0x29,0x29,0xdd,0x28,0xce,0xd1,0x28,0x28,0xd1,0xdd,0xdd,
  0x00,0x03,0x28,0xdd,0x29,0x04,0xdd,0x03,0x8a,0x28,0x28,0x02,0x80,0x00,0xd1,0x03,
  0xe3,0xe5,0x29,0x29,0xb9,0xb9,0x29,0x04,0x29,0x29,0x03,0x00,0x00,0x04,0xb9,0x03,
  0x80,0x00,0x00,0x00,0x03,0x04,0x80,0xdd,0x00,0x39,0x00,0x39,0xe7,0x03,0x00,0x02,
  0x29,0x39,0x39,0x03,0x03,0xfa,0x80,0x28,0x28,0x00,0x39,0xce,0xb9,0x29,0xb9,0x28,
  0x03,0x00,0xdd,0x02,0x00,0xed,0xb9,0x04,0x04,0x39,0x39,0x29,0x39,0x00,0x04,0x00,
  0xe7,0xed,0x8a,0x29,0x00,0x00,0x00,0x29,0x00,0x02,0x00,0x80,0x00,0x04,0x04,0x00,
  0x00,0x04,0x04,0x04,0xfc,0xfb,0xe3,0x00,0x28,0x00,0xf7,0x8a,0x04,0x28,0x03,0xf5,
  0x02,0x39,0x00,0x80,0xe3,0x02,0x39,0xd1,0xd1,0x03,0x03,0x39,0x03,0x28,0x39,0x39,
  0x03,0x29,0x80,0xb9,0xe3,0x00,0xdd,0xb9,0x00,0x39,0xce,0xdd,0xd0,0x8a,0xe5,0xe5,
  0xe5,0x03,0x29,0x04,0x8a,0xb9,0x29,0x00,0x29,0x00,0x00,0xf5,0x00,0x80,0x01,0x04,
  0x00,0x29,0x29,0xe3,0x04,0xb9,0x04,0x03,0xdd,0x03,0x00,0xb9,0x29,0x29,0xb9,0x03,
  0x03,0x00,0x00,0xf6,0xb9,0x00,0x39,0x03,0xce,0xd1,0x00,0x04,0xdd,0xd1,0x39,0xdd,
  0x28,0x03,0x02,0xe5,0x39,0x00,0x00,0x29,0x28,0xce,0xb9,0xf4,0x04,0x04,0x8a,0xb9,
  0xdd,0xce,0x8a,0xe3,0xe3,0x29,0xd5,0x00,0x03,0x00,0x29,0xe7,0xe3,0x04,0x29,0xb9,
  0xf6,0x02,0x00,0x00,0x39,0xb9,0x00,0x04,0xe3,0x29,0x03,0xed,0xb9,0x04,0xe3,0xdd,
  0x39,0xce,0xb9,0x00,0x03,0xb9,0xb9,0x03,0x28,0xd5,0xe3,0x29,0x8a,0xd5,0x03,0x01,
  0x03,0x02,0x8a,0x39,0x03,0x00,0x00,0x03,0x28,0x28,0xb9,0x04,0xe5,0x00,0x39,0x03,
  0xdd,0x02,0x04,0x00,0x03,0x03,0x00,0x8a,0xdd,0x03,0x00,0x80,0x80,0xd0,0xe5,0x28,
  0x02,0x02,0x29,0x29,0x02,0x00,0x80,0x80,0x03,0xdd,0x02,0x04,0x80,0x00,0x04,0x02,
  0x00,0x28,0x00,0x00,0x80,0x03,0x00,0x00,0x8a,0x29,0x80,0x02,0x04,0x29,0x29,0xdd,
  0x29,0xe3,0x03,0xdd,0x03,0xdd,0x00,0x04,0xb9,0x00,0xdd,0x39,0x03,0xdd,0x28,0x39,
  0x00,0x03,0xdd,0x00,0x03,0xb9,0xb9,0xdd,0x00,0x8a,0x29,0xf6,0x00,0xb9,0xd0,0xd0,
  0x28,0x28,0x28,0xdd,0x00,0x04,0xe3,0xe3,0x02,0xed,0xb9,0xdd,0x00,0x39,0x02,0x00,
  0x00,0x03,0x03,0xb9,0x04,0xdd,0x80,0x03,0x01,0x39,0xb9,0xdd,0xe3,0x28,0xe3,0xdd,
  0xdd,0x03,0xd0,0xb9,0xb9,0xce,0x28,0x02,0x02,0xd1,0xf2,0x00,0x00,0x39,0x02,0x00,
  0x03,0x29,0x00,0x02,0xe5,0x03,0x29,0x03,0x00,0x02,0x00,0x01,0xb9,0x80,0x00,0x00,
  0x04,0x80,0x29,0x00,0xb9,0x80,0x01,0x80,0xb9,0xe3,0xdd,0xdd,0x03,0x80,0x00,0x02,
  // 5: ground ice
  0x1d,0x7e,0xfd,0x7d,0x5d,0x86,0xfc,0x85,0x1d,0x86,0xbc,0x7d,0x00,0x00,0x08,0x00,
  0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,
  0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,
  0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0xbb,0x75,0x10,0x07,0x9b,0x6d,0x18,0x07,
  0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,
  0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,
  0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,
  0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,
  0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,
  0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,
  0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,
  0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,
  0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,
  0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,
  0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,
  0x00,0x66,0x08,0x66,0x10,0x66,0x9e,0x8e,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,
  0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,
  0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,
  0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,
  0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,
  0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,
  0x18,0xa1,0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,
  0x18,0xa3,0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,
  0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,
  0x18,0xa7,0x7e,0x86,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x00,0xc1,0x08,0xc1,
  0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,
  0x10,0xc3,0xdc,0x6d,0x18,0xc3,0x00,0xc4,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,
  0x08,0xc5,0x10,0xc5,0x9d,0x8e,0x18,0xc5,0x00,0xc6,0x08,0xc6,0x10,0xc6,0x18,0xc6,
  0x00,0xc7,0x08,0xc7,0x7a,0x6d,0x10,0xc7,0x5d,0x7e,0x18,0xc7,0x00,0xe0,0x08,0xe0,
  0xbc,0x6d,0xde,0x96,0x10,0xe0,0x18,0xe0,0xbe,0x96,0x00,0xe1,0x08,0xe1,0x18,0xe1,
  0xde,0x8e,0xdb,0x75,0x7e,0x7e,0x5d,0x96,0x10,0xe5,0x18,0xe5,0x18,0xe6,0x10,0xe7,
  0xd9,0x05,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x01,0x05,0x03,0x04,0x03,0x05,0x05,
  0x05,0x05,0x05,0x04,0x04,0x00,0xd9,0x01,0x01,0x00,0x00,0xd9,0x02,0x00,0xd9,0xd9,
  0xd9,0x01,0x01,0x01,0x01,0x02,0x00,0x02,0x02,0x02,0xc9,0x02,0x02,0x02,0x02,0xc9,
  0xc9,0xc9,0x00,0x02,0x01,0x05,0x01,0xd9,0x00,0x04,0x00,0x04,0x04,0x01,0x01,0x05,
  0x05,0x01,0x04,0x04,0x04,0x01,0x05,0x05,0x05,0x05,0x01,0x01,0x04,0x01,0x05,0x05,
  0xd9,0x05,0x01,0x01,0x00,0x00,0x01,0xd9,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,
  0x00,0x04,0x01,0x00,0x04,0x02,0x02,0x02,0x02,0xe2,0x02,0x02,0x02,0x02,0xc9,0xc9,
  0x02,0xc9,0xec,0x00,0x00,0x02,0x02,0x02,0x02,0x04,0x02,0x02,0x04,0x01,0x01,0x05,
  0x01,0x01,0x02,0x04,0x04,0x04,0x05,0x05,0x04,0x04,0x05,0x04,0x04,0x01,0x04,0x05,
  0x05,0x05,0x05,0x02,0x00,0x01,0x02,0x00,0x02,0x00,0xec,0xd9,0x00,0xec,0x02,0x00,
  0x00,0x00,0x00,0x02,0x02,0x02,0xec,0x02,0xe2,0x02,0x83,0x02,0x02,0x02,0x02,0x01,
  0x00,0x00,0x02,0x02,0x02,0x83,0x83,0x02,0x02,0x02,0x83,0x83,0x04,0x01,0xd9,0x01,
  0x01,0x01,0x00,0x04,0x01,0x03,0x03,0x05,0x05,0x03,0x01,0x04,0x04,0x04,0x04,0x05,
  0x01,0x01,0x03,0x01,0x01,0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x02,0xec,0xec,0x00,
  0x02,0x02,0x00,0x02,0x00,0x02,0xc9,0x83,0x83,0xc9,0x02,0x04,0x04,0x02,0x83,0x00,
  0x00,0x83,0x01,0x02,0x00,0x02,0xf8,0x02,0x83,0xf1,0xf1,0x83,0x02,0x04,0x01,0x04,
  0x01,0x01,0x01,0x01,0x01,0x04,0x01,0x05,0x01,0x05,0x05,0x01,0x03,0x03,0x05,0x03,
  0x04,0x04,0x04,0x01,0x04,0x04,0x00,0x02,0x00,0xec,0xec,0xec,0xec,0xec,0x02,0x02,
  0x02,0x02,0xc9,0x02,0x02,0x02,0x83,0x83,0x83,0x83,0x02,0x02,0xe2,0x02,0x83,0x02,
  0x02,0xe2,0x00,0x05,0x02,0x83,0x83,0x83,0x83,0xf1,0x83,0x83,0x02,0x02,0x00,0x01,
  0xd9,0x05,0x01,0x01,0x05,0x04,0x04,0x05,0x03,0x04,0x03,0x00,0x04,0x04,0x01,0x00,
  0x02,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc9,0x02,0x02,0x02,0xc9,
  0xc9,0xc9,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x83,0x83,
  0x00,0x02,0xf4,0x00,0x02,0x02,0x02,0xc9,0xf8,0xf4,0xe2,0x83,0x83,0x02,0x00,0x01,
  0x01,0xd9,0x01,0x05,0x03,0x05,0x05,0x05,0x05,0x04,0x05,0x05,0x01,0x04,0x04,0x03,
  0x02,0x02,0x02,0x01,0x00,0x02,0x02,0x02,0x02,0xec,0xc9,0x83,0x83,0x83,0x83,0x83,
  0xc9,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc9,0x00,0x02,0x02,0x83,0x83,0x02,0x02,
  0x00,0x02,0x02,0x83,0xf4,0x00,0x02,0x83,0xf1,0xf1,0xf1,0x83,0x83,0x02,0x00,0x01,
  0x00,0x05,0x04,0x01,0x05,0x05,0x03,0x05,0x05,0x05,0x05,0x04,0x04,0x01,0x04,0x04,
  0x04,0x04,0x04,0x00,0x02,0x00,0x02,0x02,0xc9,0xc9,0xf8,0xf8,0x83,0xf8,0xc9,0xc9,
  0x83,0xc9,0x02,0x02,0x02,0x02,0x00,0x02,0xe2,0x02,0x02,0xe2,0x83,0x83,0x83,0x83,
  0x02,0x83,0x02,0x00,0x02,0x83,0x02,0x02,0xf1,0xf1,0xf1,0xc9,0x02,0x02,0x02,0x01,
  0x00,0x05,0x01,0x01,0x05,0x01,0x05,0x05,0x05,0x05,0x03,0x03,0x03,0x04,0x03,0x04,
  0x04,0x04,0x04,0x02,0x02,0xec,0xec,0x83,0xf8,0xf1,0xf8,0x83,0xf1,0xf8,0xc9,0xc9,
  0x83,0x83,0x02,0x02,0x02,0x00,0x00,0x00,0x83,0xf1,0x02,0x02,0x02,0x02,0x02,0xe2,
  0x02,0x02,0x02,0x00,0x01,0x83,0x83,0x04,0xc9,0xf1,0xf1,0xc9,0xc9,0x02,0x02,0x00,
  0xd9,0x05,0x04,0x04,0x04,0x04,0x01,0x05,0x03,0x04,0x03,0x03,0x03,0x04,0x04,0x04,
  0x04,0x04,0x03,0x01,0x00,0x00,0x02,0x83,0x83,0x83,0xf1,0x02,0x83,0xc9,0xc9,0xc9,
  0xf8,0xf8,0x00,0x02,0x01,0x01,0x04,0x04,0x02,0x83,0xf4,0xe2,0xe2,0x02,0x04,0x04,
  0x00,0x04,0x01,0x04,0x02,0x00,0x83,0x02,0x83,0xf1,0xf1,0xc9,0x02,0x02,0x02,0x04,
  0x01,0x04,0x02,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x00,0xec,0x01,0x00,0x83,0xc9,0xc9,0x83,0x02,0x02,0xc9,0xc9,0xc9,
  0x83,0x83,0x02,0x00,0x01,0x01,0x04,0x01,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x04,
  0x04,0x04,0x03,0x04,0x02,0x02,0x04,0xe2,0x04,0x83,0xc9,0x02,0x02,0x83,0x02,0x04,
  0x04,0x00,0x02,0x01,0x04,0x01,0x01,0x04,0x01,0x03,0x04,0x05,0x04,0x04,0x04,0x04,
  0x02,0x00,0xc9,0x00,0x00,0x00,0x00,0x01,0x02,0x83,0x02,0x02,0xc9,0x02,0xc9,0x83,
  0x83,0x02,0x01,0x01,0x01,0x01,0xd9,0x04,0x04,0x03,0x04,0x04,0x01,0x04,0x03,0x03,
  0x01,0x01,0x04,0x01,0x02,0x02,0x02,0x83,0x83,0x04,0x83,0x02,0x83,0x83,0x83,0x04,
  0x04,0x00,0x00,0x05,0x04,0x00,0x00,0x04,0x04,0x04,0x04,0x01,0x04,0x04,0x04,0x00,
  0x00,0x00,0x01,0x01,0xd9,0xd9,0x00,0x00,0xd9,0x00,0xec,0x02,0x02,0xec,0x83,0xc9,
  0xc9,0xc9,0x01,0x01,0x01,0x01,0x05,0x01,0x03,0x04,0x05,0x04,0x04,0x05,0x05,0x01,
  0x04,0x05,0x04,0x04,0x04,0x02,0x02,0x83,0x83,0x02,0xe2,0x02,0x83,0x02,0x02,0x00,
  0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x02,0x04,0x02,0x02,0x02,0x00,0x00,
  0x01,0x01,0x01,0x01,0xf0,0xd9,0xd9,0xd9,0xd9,0x00,0x00,0x00,0x02,0x02,0xc9,0x02,
  0x02,0x00,0x01,0x00,0x01,0x01,0x01,0x05,0x05,0x03,0x01,0x04,0x04,0x03,0x04,0x05,
  0x03,0x03,0x00,0x04,0x04,0x04,0xe2,0x02,0xc9,0x02,0x02,0x83,0xc9,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x02,0x03,0x01,0x04,
  0x05,0x05,0x05,0x01,0xd9,0x24,0xd9,0x05,0x01,0x01,0x01,0x00,0x02,0x02,0xc9,0x02,
  0x02,0x00,0x01,0x00,0xd9,0x01,0x01,0x05,0x05,0x01,0x05,0x04,0x01,0x04,0x01,0x01,
  0x04,0x04,0x04,0x04,0x04,0x04,0x02,0xe2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,
  0x02,0x83,0x83,0x83,0xe2,0xe2,0x02,0x02,0x00,0x00,0x04,0x04,0x04,0x03,0x05,0x05,
  0x05,0x05,0x05,0xd9,0x05,0xd9,0x05,0xd9,0xd9,0x00,0x00,0x00,0x02,0x02,0x02,0x00,
  0x02,0x04,0x00,0x01,0xd9,0x01,0x01,0xd9,0x05,0x01,0x03,0x03,0x05,0x04,0x01,0x05,
  0x05,0x04,0x04,0x04,0x02,0x04,0x02,0xc9,0x02,0xec,0x00,0x00,0x00,0x00,0x00,0x00,
  0x83,0xe2,0x83,0x83,0xe2,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x03,0x05,
  0x05,0xd9,0xd9,0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0x00,0x00,0x02,0x02,0xc9,0x02,
  0x02,0x00,0x00,0x01,0x00,0x04,0x05,0x01,0x03,0x04,0x04,0x05,0x05,0x04,0x05,0x04,
  0x04,0x02,0x04,0x04,0x04,0x04,0x04,0xe2,0x02,0x02,0x01,0x01,0x01,0x02,0x00,0x00,
  0x02,0x02,0xe2,0x83,0xe2,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x05,0x05,
  0xd9,0x01,0x01,0x05,0x05,0x05,0xf0,0x05,0x05,0xd9,0x00,0x00,0xfa,0x02,0x02,0x02,
  0xd9,0x00,0xd9,0xd9,0x01,0x01,0x01,0x03,0x03,0x04,0x01,0x04,0x05,0x05,0x01,0x05,
  0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x00,0x00,0x05,0x01,0x00,0x02,0x00,0x02,
  0x04,0x04,0x02,0x83,0x83,0x83,0x02,0xe2,0x02,0x04,0x04,0x05,0x05,0x05,0x05,0x05,
  0xd9,0xd9,0x05,0x05,0x05,0xf0,0x05,0x05,0x05,0x00,0x02,0x01,0xec,0x83,0x02,0x00,
  0x00,0x01,0xd9,0x00,0x00,0x04,0x01,0x01,0x04,0x03,0x01,0x05,0x05,0x05,0x01,0x05,
  0x05,0x01,0x01,0x01,0x04,0x04,0x04,0x02,0x00,0x02,0xd9,0x04,0x04,0x01,0x01,0x00,
  0x04,0x01,0x04,0x02,0x02,0x02,0x04,0x02,0x02,0x04,0x03,0x05,0x05,0x05,0xd9,0xf0,
  0xd9,0x05,0x05,0x05,0x26,0x26,0x01,0x05,0x05,0x00,0x00,0x00,0x00,0x83,0x02,0x00,
  0x00,0xd9,0xd9,0xd9,0x01,0x01,0x05,0xd9,0x05,0x05,0x05,0x05,0x05,0x03,0x01,0x05,
  0x05,0x05,0x03,0x01,0x01,0x01,0x02,0x00,0x00,0x00,0x01,0x04,0x04,0x04,0x01,0x01,
  0xd9,0x01,0x01,0x01,0x02,0x02,0x04,0x02,0x02,0x04,0x04,0x05,0x05,0x05,0x24,0xd9,
  0x05,0x24,0xd9,0x05,0x05,0x05,0x05,0x01,0x01,0x01,0xd9,0x00,0x02,0x02,0x02,0x02,
  0x00,0xd9,0xd9,0xd9,0xd9,0xd9,0x01,0x05,0x05,0x05,0x04,0x05,0x05,0x05,0x01,0x03,
  0x05,0x03,0x05,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x04,0x04,0x01,0x04,0x04,0x01,
  0x03,0x05,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x04,0x01,0x05,0x00,0x05,0x05,0x24,
  0x05,0x05,0xf0,0x05,0x05,0x04,0x05,0x01,0x01,0x00,0x01,0x00,0xc9,0x02,0xe2,0xc9,
  0x00,0xd9,0x01,0x05,0x05,0x05,0x01,0x01,0x01,0x05,0x01,0x05,0x03,0x05,0x05,0x05,
  0x05,0x05,0x04,0x05,0x00,0x00,0x00,0xec,0x00,0x04,0x04,0x04,0x04,0x04,0x04,0x05,
  0x05,0x05,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x04,0x04,0x01,0x01,0xd9,0xd9,0x05,
  0x05,0x24,0x26,0x05,0x01,0x01,0x05,0x05,0x01,0x01,0xd9,0x01,0x00,0x02,0x02,0x00,
  0x04,0x01,0xd9,0xd9,0x01,0x03,0x01,0x01,0x05,0x05,0x01,0x05,0x04,0x01,0x05,0x04,
  0x05,0x05,0x04,0x01,0x04,0x02,0xec,0x02,0x04,0x01,0x01,0x01,0x04,0x05,0x01,0x05,
  0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x01,0x00,0x83,0x83,0x02,0x00,0x00,0x00,0xd9,
  0x24,0x24,0x24,0x05,0x05,0x05,0x01,0x05,0x00,0x01,0x01,0x00,0x02,0x00,0x02,0x00,
  0x04,0x01,0x01,0x01,0x04,0x01,0x05,0x01,0x03,0x01,0x01,0x05,0x04,0x04,0x05,0x01,
  0x05,0x04,0x04,0x04,0x02,0x00,0x02,0x02,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x03,
  0x03,0x05,0x03,0x01,0x04,0x04,0x01,0x00,0x01,0x02,0x02,0x03,0x01,0x05,0x05,0x00,
  0xd9,0x05,0x24,0x24,0x05,0x01,0x05,0x05,0x05,0xd9,0x00,0x01,0x02,0xec,0x02,0x00,
  0x01,0xd9,0x05,0x03,0x05,0x05,0x01,0x01,0x04,0x04,0x04,0x02,0x02,0x04,0x04,0x04,
  0x05,0x04,0x04,0x04,0x02,0xc9,0x02,0x02,0x02,0x04,0x04,0x05,0x05,0x05,0x05,0x05,
  0x05,0x01,0x04,0x04,0x01,0x04,0x01,0x00,0x00,0x05,0x01,0x00,0x01,0x01,0x05,0x01,
  0xec,0x00,0x05,0x24,0x03,0x01,0x05,0x26,0xd9,0xd9,0x01,0x00,0x00,0x02,0x04,0x03,
  0x05,0x05,0xd9,0x05,0x05,0x05,0x04,0x04,0x83,0x04,0x04,0x04,0x04,0x04,0x02,0x02,
  0x02,0x04,0x04,0x04,0x04,0xc9,0x02,0x02,0x04,0x04,0x01,0x05,0x01,0x05,0xd9,0x05,
  0x03,0x05,0x05,0x04,0x05,0x01,0x01,0x00,0x01,0x00,0x05,0x00,0x01,0xd9,0x05,0x05,
  0xd9,0x01,0x01,0x05,0x05,0x05,0x05,0x05,0xd9,0x01,0xd9,0x00,0xd9,0x00,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x01,0x03,0x04,0x04,0x01,0x04,0x04,0x02,0x04,0x04,
  0x04,0x04,0x02,0xc9,0x02,0xec,0x02,0x02,0x01,0x04,0x04,0x04,0x01,0x04,0x04,0x05,
  0x05,0x05,0x05,0x01,0x01,0x04,0x01,0x01,0x04,0x04,0x01,0x00,0x00,0xec,0x00,0x24,
  0x05,0xd9,0x00,0x01,0x05,0x05,0x05,0x01,0x01,0xd9,0xd9,0x00,0xd9,0x24,0x24,0x05,
  0xd9,0x03,0x05,0x05,0x05,0x01,0x04,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x01,0x04,0x00,0x00,0x02,0x02,0x04,0x01,0x04,0x04,0x04,0x04,0x01,0x03,0x04,
  0x05,0x05,0x05,0x01,0x04,0x04,0x04,0x00,0x01,0x00,0x02,0x02,0x00,0x00,0x02,0x02,
  0x02,0x00,0x00,0x00,0xd9,0xd9,0x01,0xd9,0x01,0x00,0x00,0x00,0x00,0x01,0xd9,0xd9,
  0x05,0x05,0x05,0x05,0x05,0x01,0x04,0x04,0x04,0x01,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x02,0x04,0x00,0x00,0x02,0x00,0x02,0x04,0x04,0x01,0x04,0x04,0x05,0x05,0x05,
  0x05,0x05,0x05,0x01,0x01,0x04,0x01,0x04,0x00,0x00,0x02,0x02,0x02,0xe2,0x83,0x02,
  0x02,0x02,0x01,0xd9,0xf0,0x01,0xd9,0x01,0x00,0x00,0x00,0x00,0xd9,0xd9,0x01,0x00,
  0x01,0x01,0x05,0x01,0x03,0x03,0x03,0x05,0x01,0x04,0x04,0x04,0x04,0x04,0x01,0x05,
  0x00,0x00,0x01,0x00,0x02,0xe2,0x00,0x00,0x02,0x04,0x04,0x04,0x03,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x01,0x04,0x01,0x02,0x02,0x02,0xe2,0xe2,0x02,0x83,0x83,0x83,
  0x00,0x00,0x00,0x00,0xd9,0x05,0xd9,0xd9,0x00,0x00,0x00,0x00,0xd9,0x01,0x01,0x01,
  0x01,0x01,0x05,0x01,0x04,0x04,0x01,0x05,0x05,0x01,0x01,0x04,0x04,0x01,0x01,0x01,
  0x04,0x02,0x01,0x02,0x83,0xf8,0x02,0x00,0x04,0x00,0x04,0x01,0x05,0x05,0x05,0x05,
  0x04,0x01,0x01,0x01,0x01,0x01,0x00,0x02,0x02,0x83,0x83,0x83,0x83,0x83,0xf1,0xf8,
  0x02,0x02,0x00,0xd9,0x00,0xd9,0xd9,0xd9,0x00,0x00,0x00,0xd9,0xd9,0xd9,0x01,0x04,
  0x01,0x04,0x00,0x03,0x04,0x05,0x01,0x01,0x00,0x04,0x01,0x01,0x05,0x05,0x01,0x05,
  0x01,0x00,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x04,0xd9,0x05,0xd9,0x01,
  0x02,0x00,0x01,0xd9,0xd9,0x01,0x01,0x04,0x02,0xf4,0x83,0x83,0x83,0xc9,0xe2,0xf4,
  0x02,0xe2,0x02,0x05,0xd9,0xd9,0x01,0x00,0x00,0x00,0x00,0xd9,0xd9,0xd9,0x00,0x00,
  0x04,0x00,0x01,0x02,0x01,0x01,0x04,0x05,0x01,0x05,0x01,0x01,0x01,0x03,0x01,0x05,
  0x01,0xd9,0x05,0x00,0x83,0x83,0x02,0x02,0x02,0x00,0x00,0x02,0x02,0x00,0x01,0x00,
  0xe2,0x01,0x05,0x00,0x00,0x01,0xec,0x02,0x02,0x02,0xe2,0x02,0x02,0x02,0xe2,0xe2,
  0xe2,0x04,0x02,0x00,0xd9,0xd9,0x00,0x02,0x00,0x00,0xd9,0xd9,0xd9,0xd9,0x04,0x00,
  0x02,0x01,0x01,0x00,0x01,0x05,0x01,0x03,0x05,0x05,0x05,0x01,0x02,0x01,0x01,0x05,
  0x01,0x00,0x02,0xf4,0x83,0xf1,0xf4,0x83,0x83,0x83,0xe2,0x83,0x02,0x01,0x02,0x00,
  0xd9,0x01,0x01,0x01,0x01,0x00,0x02,0x02,0xe2,0x83,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x00,0x00,0x00,0xd9,0x01,0x00,0x00,0xd9,0xd9,0xd9,0x02,0x00,0x02,
  0x02,0x00,0x00,0xec,0x00,0x01,0x01,0x01,0x00,0x00,0x02,0x02,0x04,0x02,0xec,0xec,
  0xc9,0x02,0x02,0xe2,0xf4,0xe2,0xe2,0x02,0x83,0x02,0x83,0xf4,0x83,0x00,0x02,0x02,
  0x05,0x01,0x01,0x04,0x05,0x05,0x01,0x00,0x04,0x02,0xe2,0x02,0x02,0x02,0x02,0x02,
  0x00,0x02,0x02,0x02,0x00,0x00,0xd9,0xd9,0x00,0xd9,0xd9,0x00,0xd9,0x02,0x02,0x02,
  0x04,0x00,0x04,0x02,0x02,0xec,0x00,0x01,0x01,0x02,0xe2,0xe2,0xec,0xec,0x02,0x00,
  0x02,0x00,0x00,0x02,0xe2,0x02,0x02,0x02,0x02,0x00,0x02,0x00,0x01,0x01,0x01,0xd9,
  0x05,0x01,0x01,0x04,0x05,0x01,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0xe2,0x02,0x02,
  0x00,0x00,0x02,0x02,0x00,0x02,0x02,0x00,0x00,0x01,0x01,0xd9,0xf0,0x01,0x00,0x00,
  0x00,0x00,0x04,0x00,0x00,0x00,0x02,0x00,0x01,0x02,0x83,0x02,0x02,0x00,0x02,0x01,
  0x00,0x01,0x01,0x00,0x83,0x02,0x02,0x00,0x00,0x02,0x00,0x04,0x05,0x01,0x01,0x05,
  0x04,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x02,
  0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x01,0xd9,0x24,0x05,0xd9,0xd9,0xd9,0x00,0x04,
  0x00,0xd9,0x00,0xd9,0x01,0x00,0x00,0x01,0x00,0x02,0x02,0x02,0x02,0x00,0x00,0xd9,
  0x02,0x02,0xd9,0x00,0x02,0xe2,0x00,0x00,0x04,0x00,0x01,0x00,0x04,0x05,0x05,0x04,
  0x05,0x03,0x05,0x01,0x05,0x05,0x04,0x04,0x03,0x03,0x04,0x04,0x01,0x01,0x02,0x02,
  0x00,0x02,0x00,0x00,0x00,0xe2,0x02,0x05,0x26,0x26,0x05,0xd9,0xd9,0x26,0x05,0x05,
  0xd9,0x05,0x05,0x05,0xd9,0xd9,0x00,0x05,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0xd9,
  0xd9,0x01,0x02,0x01,0x00,0x02,0x02,0x00,0x00,0xd9,0x05,0x01,0x05,0xd9,0x04,0x01,
  0x03,0x04,0x05,0x05,0x01,0x05,0x03,0x03,0x03,0x03,0x03,0x05,0x01,0x01,0x01,0x00,
  0x02,0x02,0x00,0x00,0xd9,0x00,0x05,0x05,0x03,0x24,0x24,0x24,0x26,0x26,0x05,0x05,
  0xd9,0x05,0x05,0x05,0x05,0x05,0xd9,0xd9,0xd9,0xd9,0xd9,0x00,0x02,0x02,0x00,0x00,
  0xd9,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x01,0x00,0x05,0x05,0x05,0x05,0x05,0x05,
  0x01,0x04,0x01,0x01,0x05,0x05,0x05,0x05,0x03,0x04,0x05,0x05,0x01,0x05,0x01,0x01,
  0x02,0x00,0xd9,0xd9,0x01,0x05,0x24,0x03,0x02,0x05,0x24,0x24,0x26,0x05,0x26,0x05,
  0x05,0x05,0xd9,0x05,0x26,0x24,0xd9,0x26,0xd9,0xd9,0xd9,0x01,0x00,0x02,0x00,0x00,
  0xd9,0xd9,0xd9,0x01,0xd9,0x00,0x00,0x00,0x00,0x00,0x05,0x05,0x05,0x05,0x05,0x01,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x01,0x01,0x01,
  0x00,0x00,0x00,0x01,0x01,0x05,0x24,0x02,0x03,0x05,0x05,0x26,0x05,0x05,0x24,0x24,
  0x26,0x05,0x05,0x26,0x24,0xf0,0xd9,0xd9,0xd9,0xd9,0x00,0xd9,0x00,0x02,0x00,0x00,
  0xd9,0xd9,0xd9,0xd9,0x05,0xd9,0x01,0x00,0x01,0x01,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x24,0x05,0x05,0x05,0x01,0x01,0xd9,
  0x01,0x00,0x00,0x01,0x01,0x05,0x24,0x01,0x03,0x05,0x26,0x24,0x26,0x05,0x05,0x24,
  0x26,0xd9,0x26,0x26,0x26,0x24,0xd9,0xd9,0xd9,0x01,0x01,0xd9,0x00,0x00,0xd9,0xd9,
  0xd9,0xd9,0xd9,0x01,0x01,0x05,0xd9,0xd9,0x00,0x00,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0xd9,0xd9,
  0x01,0x01,0xd9,0x00,0x03,0x05,0x24,0xd9,0x03,0x05,0x24,0x24,0x26,0x26,0x05,0x05,
  0xd9,0x26,0x26,0x24,0x26,0x24,0x26,0x26,0xd9,0x26,0x01,0x01,0xd9,0xd9,0xd9,0x05,
  0x26,0x26,0x05,0x05,0x01,0xd9,0xd9,0x05,0xd9,0x01,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x03,0x05,0x05,0x05,0x05,0x05,0x05,0x24,0x24,0x05,0x01,0x05,0x24,0xd9,
  0xd9,0x00,0x01,0x02,0x02,0x24,0x26,0x26,0xd9,0x24,0x24,0x05,0x24,0x24,0x26,0x26,
  0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,0x26,0xd9,0x00,0xd9,0xd9,0xd9,0xd9,0xd9,
  0xd9,0x26,0x26,0x05,0x05,0xf0,0x26,0x05,0xd9,0x00,0x05,0x05,0x05,0x24,0x05,0x05,
  0x05,0x05,0x05,0x03,0x03,0x04,0x05,0x01,0x05,0x05,0x24,0x24,0xd9,0xd9,0x05,0x26,
  0xd9,0x02,0x00,0x02,0x02,0x01,0xf9,0x24,0xf9,0x24,0x26,0x24,0x24,0x24,0x26,0x24,
  0x26,0x26,0x24,0xea,0x26,0xea,0x26,0x24,0xd9,0x01,0x00,0xd9,0xd9,0xd9,0x26,0x26,
  0x05,0x26,0x26,0x05,0xf0,0x05,0xd9,0x05,0xd9,0x00,0x00,0x05,0x24,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x03,0x05,0x05,0x24,0x05,0x05,0x05,0x05,0x05,0x05,0x26,
  0x01,0x04,0x03,0x02,0xfb,0x01,0x05,0x24,0x26,0x26,0x26,0xd9,0xd9,0x24,0x26,0x26,
  0x24,0x26,0x26,0x26,0x26,0x26,0x26,0x05,0xd9,0xd9,0xd9,0xd9,0x01,0xd9,0x05,0x26,
  0x05,0x26,0x26,0xd9,0xd9,0xf0,0x05,0xd9,0x05,0x05,0x00,0x05,0x05,0x05,0x24,0x05,
  0x05,0x05,0x05,0x03,0x05,0x05,0x05,0x24,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x26,
  0x26,0x00,0x02,0x02,0x00,0x01,0xd9,0xea,0xea,0xea,0x26,0x26,0x24,0xd9,0xd9,0x24,
  0x26,0x24,0x24,0x24,0x26,0x24,0x26,0x26,0x26,0xd9,0x05,0x01,0x01,0xd9,0x05,0x26,
  0x26,0x26,0x26,0xd9,0xd9,0xd9,0xd9,0x05,0x05,0x05,0x01,0x01,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x03,0x05,0x05,0x05,0x24,0x05,0x01,0x05,0x05,0x24,0x05,0x24,0x26,
  0xd9,0x01,0x02,0x01,0x01,0x04,0x00,0xd9,0x26,0x26,0x24,0x26,0x24,0x26,0x05,0x26,
  0x24,0x26,0x26,0x26,0x26,0x24,0x05,0x24,0x26,0xd9,0xd9,0x01,0x05,0x26,0x05,0x26,
  0x26,0x05,0x05,0x01,0xd9,0x01,0x05,0x05,0x05,0xd9,0x01,0x05,0x05,0x01,0x01,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0x24,0xea,0x24,
  0x00,0x02,0x02,0xd9,0xd9,0x01,0x01,0x00,0x01,0xd9,0x24,0x26,0x24,0x24,0x24,0xea,
  0xea,0x26,0x26,0x26,0x26,0x24,0x24,0x26,0x05,0x05,0x01,0xd9,0x26,0x26,0x26,0x05,
  0x26,0x26,0x05,0x01,0xd9,0x01,0x05,0x24,0x26,0x24,0xd9,0x05,0x01,0x05,0x24,0x05,
  0x05,0x05,0x03,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x26,0xea,0xea,
  0xd9,0x02,0x02,0x01,0xd9,0xd9,0x01,0xd9,0x01,0x05,0x24,0x24,0x24,0x24,0x24,0xea,
  0x26,0x24,0x26,0x24,0x26,0x05,0x24,0x24,0x26,0x26,0x01,0xf0,0x26,0x05,0xf0,0x26,
  0x24,0x05,0x05,0xd9,0x01,0x05,0x05,0x26,0x24,0x24,0x05,0x05,0xd9,0x01,0xd9,0x24,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0x26,0x24,0x24,0x24,
  0x24,0x04,0x04,0xd9,0xd9,0xd9,0xd9,0x01,0x05,0x05,0x05,0xf9,0x24,0xea,0xea,0x26,
  0x26,0x24,0x24,0x26,0x24,0x24,0x05,0x24,0x26,0x26,0x00,0xd9,0x00,0xf0,0x26,0x26,
  0x26,0x26,0x05,0x05,0x05,0x05,0x26,0x24,0x24,0x26,0x24,0x26,0x24,0xd9,0xd9,0x05,
  0x05,0x05,0x05,0x24,0x24,0x24,0x24,0x05,0x05,0xd9,0x24,0x05,0x24,0x05,0x24,0x01,
  0xd9,0x04,0x03,0x01,0x01,0x01,0xd9,0x05,0x24,0x24,0x24,0x24,0x26,0xea,0xea,0x26,
  0x24,0x26,0x26,0x26,0x05,0x05,0x05,0x24,0x26,0x26,0xd9,0xf0,0x01,0xd9,0x26,0x26,
  0x26,0x26,0x05,0x05,0x24,0x26,0x26,0x26,0x26,0x24,0x26,0x05,0x24,0x26,0xd9,0x05,
  0x05,0x05,0x05,0x05,0x24,0x24,0x24,0x00,0x05,0x05,0x05,0x24,0x24,0x24,0x03,0xd9,
  0x01,0x01,0x03,0x04,0x03,0x03,0xd9,0x26,0x26,0x24,0x26,0x26,0x24,0xea,0x26,0x26,
  0x26,0x24,0x24,0x24,0x24,0x24,0x24,0x05,0x26,0x26,0xd9,0xd9,0xd9,0xf0,0x26,0x26,
  0x26,0x26,0x24,0x05,0x24,0x05,0x26,0x24,0x26,0x24,0x26,0x24,0xd9,0x05,0x24,0x05,
  0x24,0x05,0x05,0x24,0x24,0x05,0x05,0x01,0x05,0x05,0x05,0x05,0x05,0x05,0x02,0x01,
  0x01,0x05,0xd9,0x01,0x04,0x01,0xd9,0xd9,0x26,0x26,0x26,0x26,0x26,0x24,0x26,0x05,
  0x05,0x24,0x24,0x24,0x24,0x05,0x24,0x24,0x26,0xd9,0xd9,0xd9,0xd9,0x26,0x26,0xd9,
  0x26,0x26,0x24,0x26,0x05,0xd9,0x26,0x05,0x24,0xea,0x24,0x24,0x24,0x26,0x24,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0x02,0x04,
  0x00,0x00,0x01,0x01,0x04,0x01,0xd9,0xd9,0xd9,0x26,0x26,0x26,0x26,0x24,0x24,0x05,
  0x05,0x24,0x05,0x26,0x24,0x24,0x24,0x26,0x26,0xd9,0xd9,0xd9,0xd9,0xf0,0x26,0x26,
  0xd9,0xd9,0xd9,0x24,0x05,0x05,0x05,0x05,0x05,0x24,0x24,0x26,0x24,0xea,0x26,0xd9,
  0x01,0xd9,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x01,0x00,0x01,0xd9,0xd9,0x00,0x00,
  0x04,0x02,0x01,0x01,0x04,0x00,0x02,0xd9,0xd9,0x26,0x05,0x24,0x26,0x24,0x26,0x26,
  0x24,0x24,0x24,0x24,0x26,0x26,0x05,0x05,0xd9,0x00,0x00,0xd9,0xd9,0xd9,0xf0,0xf0,
  0x26,0xd9,0xd9,0xd9,0x05,0x24,0x05,0x05,0x24,0x24,0x05,0x24,0x26,0x05,0x26,0x26,
  0x05,0x26,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0x05,0x05,0x01,0x00,0x01,0x00,
  0x01,0x02,0x02,0x00,0x00,0x00,0x02,0x02,0xd9,0xd9,0xd9,0x24,0x24,0x24,0x26,0x26,
  0x24,0x05,0x05,0x05,0x05,0x24,0x26,0xd9,0xd9,0x00,0xd9,0xd9,0xd9,0xd9,0x26,0xf0,
  0x26,0xd9,0x01,0xd9,0xd9,0x05,0x05,0x24,0x24,0x26,0x05,0x24,0x26,0x26,0x05,0x05,
  0x05,0x05,0x26,0x26,0x24,0x24,0x24,0x05,0x24,0x05,0x05,0x05,0x05,0x01,0x05,0x01,
  0x02,0x00,0x02,0x00,0x00,0x02,0x02,0x00,0x00,0xd9,0xd9,0x26,0x26,0x26,0x26,0x26,
  0x24,0x05,0x05,0x05,0x05,0x05,0xd9,0x05,0xd9,0x01,0xd9,0xd9,0xd9,0x00,0xd9,0xd9,
  0x01,0xf0,0x01,0x00,0xd9,0x01,0x05,0x05,0x24,0x24,0x26,0x26,0x26,0x26,0x05,0x05,
  0xd9,0x05,0x26,0x24,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x24,
  0x05,0x05,0x05,0x05,0x00,0x00,0x02,0x02,0x00,0x01,0xd9,0xd9,0xd9,0x26,0x26,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x01,0xd9,0xd9,0x00,0x00,0xd9,0xd9,0x01,
  0x00,0xd9,0x01,0x01,0x01,0x01,0xd9,0x05,0x05,0x05,0x26,0x26,0x05,0x05,0x05,0x04,
  0xd9,0xd9,0xd9,0x05,0x26,0x05,0x05,0x05,0x24,0x05,0x05,0x05,0x04,0x05,0x05,0x05,
  0x05,0x05,0x05,0xd9,0xd9,0x01,0x00,0x02,0x00,0x00,0x00,0xd9,0xd9,0xd9,0x26,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0xd9,0xd9,0x00,0xd9,0xd9,0x00,0x00,0x00,0xd9,0x00,
  0x00,0x00,0x02,0x00,0xd9,0xd9,0xd9,0x05,0x05,0x05,0x05,0x05,0xd9,0xd9,0xd9,0xd9,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x01,0x05,0x05,0x01,0x24,0xd9,0xd9,0x00,0x00,0x00,0x00,0x00,0x00,0xd9,0xd9,0x24,
  0x05,0x24,0x24,0x05,0x24,0x05,0x05,0x01,0x01,0x00,0x01,0x02,0x00,0x00,0xd9,0xd9,
  0x00,0x00,0x02,0x04,0x01,0xd9,0x05,0xd9,0x05,0xd9,0x05,0x01,0x01,0x01,0xd9,0xd9,
  0x05,0x01,0xd9,0x05,0x01,0x05,0x05,0x05,0x05,0x05,0x05,0x03,0x05,0x05,0x05,0x01,
  0x05,0x05,0x05,0x05,0x01,0x01,0x01,0x05,0x00,0x00,0x00,0xec,0x00,0xd9,0xd9,0xd9,
  0x05,0x05,0x05,0xd9,0x05,0x01,0x01,0x01,0x00,0x02,0x00,0x00,0x02,0x00,0x00,0x02,
  0xec,0x02,0x02,0x01,0x01,0xd9,0x05,0x01,0x01,0xd9,0x00,0x01,0x04,0x01,0x00,0x01,
  0x05,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x03,0x05,0x05,0x03,0x01,0x05,0x05,
  0x05,0xd9,0x05,0x05,0x05,0x01,0xd9,0x01,0x00,0x00,0x00,0x00,0xec,0xd9,0xd9,0xd9,
  0x05,0x05,0x01,0x05,0x01,0x01,0x00,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x83,0x83,
  0xc9,0x02,0x02,0x00,0x01,0xd9,0x01,0xd9,0xd9,0x01,0x01,0x04,0x00,0x01,0x00,0x00,
  // 6: ground accel
  0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,
  0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,
  0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,
  0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,
  0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,
  0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,
  0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,
  0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,
  0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x28,0x42,0x00,0x41,0x08,0x41,0x10,0x41,
  0x18,0x41,0x00,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,
  0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,
  0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,
  0x00,0x60,0x08,0x60,0x10,0x60,0x01,0x69,0x18,0x60,0x08,0x61,0x10,0x61,0x18,0x61,
  0x00,0x62,0x08,0x62,0x10,0x62,0x28,0x6b,0x18,0x62,0x00,0x63,0x10,0x63,0x18,0x63,
  0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,
  0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,
  0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x28,0x8b,0x10,0x81,
  0x18,0x81,0x00,0x82,0x31,0x8c,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x10,0x83,
  0x18,0x83,0x09,0x84,0x00,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,
  0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,
  0x00,0xa0,0x01,0xaa,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0xcf,0x7b,
  0x10,0xa1,0x18,0xa1,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x09,0xa3,0x00,0xa3,0x10,0xa3,
  0x08,0xac,0x18,0xa3,0x00,0xa4,0x10,0xa4,0x18,0xa4,0xae,0x73,0x29,0xa5,0x00,0xa5,
  0x10,0xa5,0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,
  0x10,0xa7,0x18,0xa7,0x0a,0x73,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x00,0xc1,
  0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0x08,0xc3,
  0x00,0xc3,0xea,0x7a,0x28,0xc5,0x28,0xcc,0x10,0xc3,0x18,0xc3,0x00,0xc4,0x10,0xc4,
  0x18,0xc4,0x00,0xc5,0x10,0xc5,0x18,0xc5,0x09,0xce,0x00,0xc6,0x10,0xc6,0x18,0xc6,
  0x00,0xc7,0x08,0xc7,0x10,0xc7,0xcf,0x83,0x18,0xc7,0x00,0xe0,0xc4,0xe2,0x04,0xeb,
  0x44,0xec,0x8a,0x52,0x08,0xe0,0x62,0xb2,0x10,0xe0,0x84,0xeb,0x6c,0x73,0x18,0xe0,
  0x45,0xb2,0x07,0xdd,0x26,0xe5,0x2b,0xef,0xc6,0xdc,0x22,0xa2,0x62,0x89,0xc3,0x82,
  0x66,0xbc,0xcb,0x5a,0xa1,0x91,0x84,0xe2,0xaa,0xd5,0x88,0xe5,0xc9,0xe5,0xe3,0xeb,
  0xf0,0xe6,0xf4,0xf3,0xf1,0xeb,0xf6,0xeb,0xf8,0xfd,0xff,0xe6,0xed,0xf2,0xfe,0xff,
  0xe6,0xff,0xf1,0xfc,0xd0,0xfa,0xfa,0xf7,0x6b,0xf9,0xf9,0xee,0xa7,0xa7,0xa7,0xa7,
  0xa7,0xa7,0xa7,0xa7,0xee,0xee,0xee,0xc2,0xf7,0xf6,0xfa,0xd0,0xfc,0xd2,0xff,0xe6,
  0xff,0xfe,0xfd,0xed,0xe6,0xed,0xfd,0xf1,0xeb,0xf6,0xf5,0xf4,0xf3,0xf2,0xed,0xf0,
  0xe6,0xe8,0xfd,0xf3,0xf4,0xa1,0x68,0xf7,0x86,0xb2,0xe6,0xfb,0xed,0xfd,0xfe,0xff,
  0xe6,0xed,0xf8,0xb0,0xae,0xf7,0xfa,0xf8,0xb6,0x69,0xee,0xf9,0xb5,0xa7,0xa7,0xa7,
  0xb5,0xb5,0xb5,0xb5,0xf9,0xf9,0xe9,0x6b,0xf7,0xf6,0x69,0xf7,0xb2,0xf8,0xed,0xe6,
  0xff,0xfd,0xfd,0xff,0xfb,0xe6,0xd6,0xb2,0xf7,0x68,0xf5,0xff,0xf3,0xfe,0xff,0xe6,
  0xf4,0xfd,0xf3,0xfd,0xd0,0xfa,0x69,0x6b,0x6b,0xf7,0xa1,0xeb,0xff,0xfd,0xf1,0xed,
  0xeb,0xa1,0x86,0x86,0xd1,0x8b,0xa1,0xd2,0xdc,0xf7,0xc2,0xc2,0xb5,0xa7,0xa7,0xb5,
  0xb5,0xb5,0xb5,0xb5,0xf9,0xf9,0xf7,0xf7,0xb0,0x89,0x69,0x6b,0xd1,0x86,0xa1,0xeb,
  0xe6,0xf1,0xfe,0xff,0xcb,0xeb,0xf5,0x69,0x69,0x69,0x89,0xd0,0xf1,0xf3,0xfd,0xf4,
  0xf3,0xf3,0xfd,0xff,0xf5,0xd1,0xd1,0xd1,0xd1,0xd1,0x8b,0xf5,0xd0,0xfd,0xf8,0xeb,
  0xf6,0xb3,0xd1,0xee,0xd1,0xf6,0xd0,0xdc,0xdc,0xf7,0xc2,0xc2,0xb5,0xa7,0xa7,0xa7,
  0xa7,0xa7,0xb5,0xb5,0xf9,0xe9,0xf7,0xf7,0xb6,0xeb,0xf6,0xc2,0x6b,0x6b,0x69,0xf6,
  0xa1,0xf8,0xfe,0xd0,0xa1,0x8b,0x69,0x69,0x6b,0xd1,0xee,0xf5,0xd0,0xfd,0xf3,0xf3,
  0xf1,0xe8,0xd0,0xf5,0xfa,0xf5,0xf5,0x86,0xad,0xaa,0x8b,0x86,0xae,0xb0,0xf5,0x8b,
  0x69,0xb8,0xee,0xee,0x8b,0xfa,0xd6,0xfc,0xdc,0xf7,0x6b,0xee,0xf9,0x44,0x44,0x44,
  0xf9,0xe9,0xf9,0xf9,0xe9,0xe9,0x86,0xf7,0xd2,0xd0,0xfa,0x6b,0x6b,0xc2,0xc2,0x6b,
  0x63,0xf0,0xb2,0xae,0xf5,0x8b,0xfa,0xf7,0x86,0xad,0xf0,0xfa,0xf5,0xd0,0xe8,0xf1,
  0xeb,0xf5,0xaa,0x6b,0xf5,0xd0,0xf1,0xd3,0xd0,0xeb,0xeb,0xd0,0xf8,0xf7,0x45,0x46,
  0xf9,0x8a,0x69,0xc2,0xf6,0xeb,0xe8,0xf3,0xd2,0xf7,0xd1,0x6b,0x69,0xf9,0xee,0xe9,
  0xee,0xee,0xee,0xe9,0x69,0x69,0xf7,0xf7,0xb6,0xd3,0xa1,0xf6,0x6b,0xc2,0x69,0xc2,
  0x69,0x6b,0x86,0xb2,0xd0,0xeb,0xeb,0xd0,0xf8,0xd2,0xed,0xa1,0xee,0xf6,0xf5,0xeb,
  0xf5,0xf7,0xee,0xd1,0xad,0xd3,0xfd,0xe8,0xe7,0xfb,0xed,0xf4,0xfe,0xae,0xc2,0x69,
  0xe3,0x8a,0xc2,0x6b,0xfa,0xd0,0xf4,0xfc,0xf8,0xf5,0xf7,0x89,0x89,0x8b,0xc2,0x69,
  0xee,0xee,0x6b,0x86,0x86,0x86,0x86,0xb2,0xb0,0xf8,0xd0,0xfa,0x8b,0xc2,0xc2,0xc2,
  0x6b,0x69,0xad,0xf1,0xf4,0xed,0xfb,0xe7,0xe8,0xfd,0xf1,0xae,0xee,0x69,0x6b,0xf7,
  0xeb,0xf7,0xe3,0xd1,0x86,0xf8,0xe8,0xed,0xfb,0xe7,0xe8,0xfe,0xfd,0xeb,0x8b,0xc2,
  0x8a,0xee,0xc2,0x8b,0xa1,0xff,0xf4,0xd2,0xd6,0xd6,0xd6,0xf8,0xf8,0xb0,0x6b,0xc2,
  0xee,0x69,0xf7,0xf7,0x8e,0xb2,0xae,0xae,0xb2,0xf8,0xff,0xa1,0x8b,0xc2,0xc2,0xc2,
  0xc2,0x69,0xa1,0xf1,0xfe,0xe8,0xe7,0xfb,0xed,0xe8,0xf1,0xb0,0xd1,0x69,0x8b,0xeb,
  0xf1,0xae,0xee,0x69,0x86,0xd0,0xe7,0xfb,0xe7,0xff,0xfd,0xfe,0xe8,0xeb,0x8b,0xe3,
  0xe3,0xb5,0xee,0xf5,0xd0,0xff,0xf1,0xf1,0xf2,0xfc,0xfc,0xf3,0xfe,0xfc,0xf8,0x6b,
  0x6b,0xf7,0xb6,0xdc,0xfc,0xfc,0xd2,0xf8,0xf8,0xf4,0xe8,0xd0,0xfa,0xd1,0xd1,0xc2,
  0xc2,0xd1,0xa1,0xe8,0xfe,0xfd,0xe8,0xe7,0xfb,0xe7,0xe6,0xad,0xd1,0x69,0x86,0xf8,
  0xfd,0xf8,0xad,0x69,0x8b,0xeb,0xfb,0xe7,0xff,0xfd,0xf3,0xf1,0xed,0xf0,0xf5,0x8a,
  0xc2,0xe3,0x86,0xeb,0xe8,0xd2,0xf3,0xf3,0xfe,0xfc,0xdc,0xf1,0xf8,0xeb,0xf5,0xf5,
  0xf5,0xf5,0xae,0xf8,0xf8,0xd2,0xdc,0xdc,0xf3,0xf3,0xfd,0xff,0xeb,0x86,0x6b,0xf9,
  0xee,0x8b,0xa1,0xed,0xf1,0xfe,0xfd,0xe8,0xe7,0xfb,0xcb,0xaa,0xd1,0xf7,0xae,0xfd,
  0xff,0xe7,0xeb,0xf6,0x8b,0xeb,0xe7,0xe8,0xfd,0xfe,0xfd,0xff,0xe6,0xe6,0xeb,0xd1,
  0xe3,0xe3,0xf0,0xf8,0xfc,0xf3,0xf3,0xfd,0xf4,0xff,0xed,0xeb,0xf5,0xf0,0x69,0xf6,
  0xf6,0x69,0x69,0xfa,0xf5,0xd0,0xd6,0xe8,0xd2,0xfc,0xf3,0xfc,0xf8,0xf5,0xc2,0xb5,
  0xee,0xf0,0xf0,0xe6,0xed,0xf1,0xfe,0xfd,0xe8,0xed,0xeb,0x8b,0x8b,0xa1,0xe6,0xff,
  0xe6,0xfb,0xf0,0xa1,0x8b,0xcb,0xf4,0xfd,0xfe,0xf1,0xff,0xe6,0xe6,0xed,0xae,0xd1,
  0xc2,0xee,0xb0,0xfc,0xfc,0xfc,0xf1,0xff,0xd0,0xa1,0xfa,0xf5,0x8b,0x6b,0x8b,0xeb,
  0xd0,0xf0,0xd1,0x6b,0x68,0xf6,0xfa,0xeb,0xd0,0xe8,0xf1,0xfc,0xde,0xb0,0x86,0xb5,
  0xee,0x86,0xed,0xcf,0xe6,0xed,0xf1,0xfe,0xfe,0xf4,0xd0,0xf0,0xf5,0xeb,0xfb,0xe6,
  0xed,0xed,0xff,0xd0,0xae,0xb0,0xfd,0xf1,0xe8,0xff,0xe6,0xe6,0xed,0xd3,0xb0,0x86,
  0xee,0x86,0xd2,0xd2,0xf1,0xd3,0xf0,0xf5,0xf5,0xf0,0x69,0x69,0x69,0xc2,0xae,0xf8,
  0xd2,0xb2,0x6b,0x69,0xc2,0x69,0x69,0x89,0xfa,0xf5,0xcf,0xf4,0xd2,0xfc,0x86,0xee,
  0xee,0xb0,0xd3,0xff,0xcf,0xe6,0xed,0xe8,0xf1,0xfd,0xf8,0xae,0xd0,0xff,0xed,0xed,
  0xf2,0xfd,0xfd,0xfe,0xb0,0x86,0xae,0xeb,0xeb,0xeb,0xfb,0xed,0xd3,0xd2,0xd3,0xf0,
  0xd1,0xae,0xf8,0xed,0xf0,0xf5,0xf5,0x86,0xd1,0xd1,0xc2,0x69,0x6b,0x91,0xb0,0xfc,
  0xfc,0xf8,0x86,0xc2,0xc2,0xc2,0x69,0x6b,0x6b,0x8b,0xf6,0xaa,0xcf,0xd3,0x86,0x86,
  0x86,0xf8,0xd2,0xd3,0xed,0xcf,0xf0,0xeb,0xeb,0xeb,0xf7,0xae,0xf1,0xfd,0xfd,0xf2,
  0xfe,0xfe,0xf1,0xf8,0xf5,0x69,0x69,0x63,0x8b,0x8b,0xf0,0xae,0xb0,0xd3,0xd3,0xad,
  0xf0,0xf0,0xeb,0xf5,0x86,0x86,0xee,0xb5,0xc2,0xe9,0xf9,0xee,0xee,0xad,0xd0,0xff,
  0xff,0xff,0xeb,0xd1,0xb5,0xee,0xf9,0xe9,0xc2,0xf9,0xb5,0xd1,0x8b,0xf5,0xf0,0xf0,
  0xf7,0xb0,0xd3,0xb0,0xad,0xf0,0x8b,0x8b,0x69,0x69,0x69,0xf7,0xf8,0xf1,0xfe,0xfe,
  0xff,0xff,0xed,0xeb,0x89,0x69,0x6b,0xc2,0xe3,0xe3,0xee,0xd1,0xad,0xae,0xb6,0xb0,
  0xeb,0xf5,0x86,0xe3,0xb5,0xee,0xee,0xc2,0xf9,0xf9,0xe9,0x69,0x89,0xa1,0xfb,0xe6,
  0xe6,0xe6,0xeb,0x8b,0x69,0xe9,0xf9,0xa7,0xb5,0xee,0xa7,0xa7,0xb5,0xc2,0x8b,0xf5,
  0xad,0xb6,0xb0,0xad,0xd1,0xee,0xc2,0xee,0xc2,0x69,0x69,0x68,0xa1,0xe6,0xff,0xff,
  0xe6,0xe6,0xeb,0xfa,0x6b,0xee,0xa7,0xee,0xee,0xee,0xa7,0xa7,0xb5,0xd1,0xf7,0xed,
  0xcf,0xf7,0x91,0x86,0xe3,0xb5,0xc2,0xe9,0x69,0x44,0x69,0x69,0xfa,0xeb,0xe6,0xed,
  0xed,0xed,0xd0,0xa5,0x69,0x69,0x44,0x44,0xb5,0xe3,0xb5,0xee,0xee,0x86,0x86,0xaa,
  0xf5,0x86,0xee,0xf9,0xee,0xb5,0xee,0xe9,0xc2,0xc2,0xee,0xc2,0x8b,0xa1,0xe6,0xe6,
  0xff,0xed,0xa1,0x86,0xe3,0xee,0xe9,0xee,0xee,0xee,0xb5,0xd1,0x8b,0xfa,0xf0,0xf5,
  0xf5,0xed,0xf8,0xb0,0x86,0x86,0x8b,0x69,0x69,0xe9,0xe9,0xf7,0xeb,0xe8,0xf4,0xf1,
  0xf2,0xf2,0xf4,0xd0,0xfa,0x45,0x44,0x69,0x45,0x69,0x86,0x86,0x91,0xb0,0xcf,0xaa,
  0xfa,0xf5,0x8b,0xd1,0xc2,0xee,0xf9,0xee,0xc2,0xc2,0xc2,0xee,0xd1,0xa1,0xe6,0xff,
  0xf1,0xf8,0xf7,0x6b,0x6b,0x69,0xc2,0xc2,0xc2,0xd1,0x8b,0xf5,0xb0,0xd3,0xcf,0x8b,
  0xf0,0xd3,0xfc,0xfc,0xd6,0xeb,0xf5,0xf5,0xf5,0xf5,0xf7,0xae,0xf1,0xfe,0xfe,0xfe,
  0xfe,0xfe,0xfe,0xfd,0xae,0x68,0x89,0xfa,0xf6,0xf5,0xeb,0xed,0xd2,0xfe,0xd2,0xf5,
  0x8b,0xaa,0xf8,0xad,0xf0,0x8b,0xc2,0xb5,0xc2,0xc2,0xc2,0xc2,0xc2,0xf7,0xf8,0xf1,
  0xfc,0xf8,0x86,0x69,0x6b,0xd1,0xc2,0x8b,0xfa,0xf5,0xf8,0xfc,0xd2,0xed,0xfa,0x86,
  0x91,0xb6,0xfc,0xe8,0xed,0xfb,0xe6,0xe6,0xd6,0xf8,0xb0,0xae,0xf4,0xe8,0xe8,0xe8,
  0xe8,0xe8,0xe8,0xf4,0xae,0xb2,0xf8,0xd6,0xed,0xe6,0xfb,0xe6,0xe8,0xfc,0xd2,0x86,
  0xc2,0xfa,0xcf,0xd3,0xfc,0xf8,0xf5,0x8b,0xd1,0x6b,0xc2,0xe9,0x6b,0x6b,0xb2,0xfc,
  0xd0,0xf5,0x6b,0xc2,0xd1,0x8b,0xfa,0xf5,0xcf,0xe8,0xfc,0xfc,0xd3,0xcf,0xf0,0xb5,
  0x8a,0xb0,0xe8,0xed,0xe6,0xe7,0xff,0xf1,0xfe,0xfd,0xf8,0xeb,0xeb,0xe6,0xe7,0xe6,
  0xe6,0xe6,0xe6,0xeb,0xf5,0xd0,0xf1,0xfe,0xfd,0xe8,0xed,0xe6,0xe6,0xff,0xb0,0x6b,
  0xee,0x8b,0xaa,0xff,0xfc,0xfc,0xd3,0xeb,0xfa,0xf6,0x69,0x6b,0xc2,0xd1,0xf7,0xd0,
  0xfa,0x8b,0x69,0x8b,0xfa,0xa1,0xd0,0xff,0xe8,0xfc,0xf3,0xf1,0xd3,0xaa,0x86,0xe3,
  0xee,0xae,0xe6,0xe6,0xed,0xe8,0xfd,0xfe,0xf1,0xff,0xeb,0xfa,0xf6,0xeb,0xe6,0xe7,
  0xe7,0xe7,0xeb,0xf6,0x89,0xeb,0xff,0xe8,0xfe,0xfe,0xe8,0xed,0xe6,0xe6,0xeb,0xb5,
  0xc2,0xc2,0xf5,0xcf,0xf1,0xf3,0xfc,0xe8,0xff,0xd0,0xa1,0xf6,0x8b,0x6b,0x69,0xfa,
  0xfa,0xf7,0xf5,0xae,0xd6,0xe8,0xf4,0xf1,0xd2,0xf3,0xfc,0xd3,0xcf,0xf5,0xee,0xb5,
  0xc2,0xa1,0xeb,0xe6,0xe8,0xfe,0xfe,0xf4,0xff,0xe7,0xeb,0xf6,0x63,0xeb,0xd0,0xe8,
  0xe8,0xff,0xeb,0x8b,0x8b,0xa1,0xfb,0xed,0xe8,0xfe,0xfe,0xe8,0xed,0xcb,0xf5,0xe3,
  0xee,0xc2,0x8b,0xf0,0xe8,0xfc,0xf3,0xdc,0xd2,0xf4,0xf8,0xd0,0xae,0xf5,0xf7,0x89,
  0xf7,0xf7,0xd2,0xdc,0xfc,0xf3,0xf3,0xd2,0xdc,0xf3,0xf1,0xed,0xf5,0x86,0xc2,0xc2,
  0x69,0xf6,0xa1,0xff,0xfd,0xfe,0xe8,0xed,0xe6,0xe6,0xfb,0xeb,0x45,0x86,0xb0,0xfd,
  0xfd,0xf8,0xee,0xd1,0xf0,0xeb,0xe6,0xfb,0xed,0xe8,0xfe,0xfd,0xff,0xeb,0x8b,0xee,
  0xa7,0xc2,0xc2,0xf5,0xe6,0xf4,0xf3,0xdc,0xd2,0xfc,0xf3,0xfc,0xdc,0xd2,0xf7,0xf7,
  0xc2,0x91,0xb6,0xfc,0xd2,0xd2,0xb0,0xd6,0xfc,0xf3,0xe8,0xcf,0xf5,0xd1,0xf9,0xe9,
  0x45,0xd1,0xa1,0xf4,0xfe,0xf4,0xed,0xe6,0xe7,0xff,0xd3,0xb0,0x45,0x6b,0xae,0xd0,
  0xd0,0xf7,0xee,0xc2,0x86,0xd6,0xff,0xe7,0xfb,0xed,0xe8,0xfe,0xf1,0xeb,0x8b,0xc2,
  0xb5,0xee,0xc2,0xf7,0xeb,0xff,0xfc,0xfc,0xf8,0xf8,0xd2,0xd2,0xfc,0xb6,0x91,0xee,
  0xf9,0x6b,0xf7,0xf7,0xf7,0xf7,0xae,0xae,0xfc,0xfc,0xff,0xa1,0x8b,0xc2,0xf9,0x44,
  0x24,0xe3,0xa1,0xf1,0xfd,0xff,0xe6,0xe6,0xff,0xf1,0xfd,0xb2,0x45,0x69,0x63,0xf5,
  0xf6,0x68,0xe3,0xc2,0x86,0xf1,0xf1,0xff,0xe7,0xfb,0xff,0xf1,0xfd,0xeb,0x45,0xc2,
  0xc2,0xa7,0xc2,0xd1,0xfa,0xed,0xd2,0xf3,0xb0,0xae,0x86,0xae,0xf7,0xf7,0x6b,0xf9,
  0xe9,0xf9,0xc2,0x69,0x6b,0x6b,0x86,0xb0,0xfc,0xf1,0xe6,0xfa,0xc2,0xee,0xf9,0x45,
  0xf9,0xee,0xf7,0xf1,0xe8,0xe6,0xfb,0xed,0xf4,0xfe,0xf4,0xeb,0x63,0x89,0xfa,0xf5,
  0xfa,0xfa,0xad,0x69,0xf0,0xf8,0xfe,0xf1,0xed,0xfb,0xfb,0xff,0xf1,0xf7,0x45,0xe9,
  0xc2,0xe3,0xc2,0x6b,0x89,0xeb,0xf4,0xf3,0xb0,0x86,0x6b,0x6b,0x6b,0xc2,0xf9,0xe9,
  0xe9,0x44,0xf9,0x44,0xee,0xd1,0x86,0xf8,0xfe,0xf4,0xf0,0x8b,0xb5,0xee,0xe9,0x45,
  0xe3,0x68,0xb0,0xae,0xeb,0xa1,0xeb,0xd0,0xf8,0xf8,0xd0,0xf5,0xfa,0xeb,0xd0,0xff,
  0xff,0xed,0xeb,0xfa,0xf6,0xeb,0xeb,0xf8,0xeb,0xa1,0xa5,0xa1,0xae,0xae,0x68,0x69,
  0x69,0xb5,0xee,0x6b,0xd1,0xfa,0xd6,0xf3,0xb6,0xf7,0x6b,0x6b,0xc2,0xf9,0xe9,0xe9,
  0xee,0xee,0xee,0xb5,0xc2,0x6b,0xf7,0xd2,0xfc,0xed,0xf5,0xd1,0xb5,0xee,0x69,0x63,
  0xaa,0xeb,0xd2,0xeb,0xf5,0x89,0x8b,0xf7,0x86,0xf7,0xf6,0xfa,0xeb,0xe8,0xfd,0xfe,
  0xf3,0xfe,0xf4,0xd0,0xf5,0x89,0x86,0xd1,0xd1,0xd1,0xd1,0xf6,0xeb,0xf8,0xae,0xfa,
  0x69,0xee,0xe3,0xd1,0xc2,0x89,0xd0,0xdc,0xdc,0xf7,0x6b,0xf9,0xee,0xee,0xf9,0xee,
  0xb5,0xb5,0xa7,0xa7,0x44,0x69,0xf7,0xd2,0xfc,0xeb,0x8b,0xee,0xc2,0xc2,0xf0,0xf6,
  0xeb,0xf4,0xfe,0xff,0xeb,0xa5,0xf6,0x69,0x69,0x69,0xf6,0xeb,0xe8,0xfe,0xf3,0xfe,
  0xfd,0xfe,0xf3,0xf2,0xeb,0xf6,0x69,0x69,0x69,0x63,0xf6,0xa1,0xe6,0xfd,0xf1,0xd0,
  0xa5,0xf5,0xe3,0xc2,0xc2,0x8b,0xa1,0xd2,0xfc,0xf7,0x69,0xe9,0xa7,0xb5,0xee,0xb5,
  0xb5,0xb5,0xb5,0xa7,0x44,0xe9,0xd1,0xb0,0xd2,0xf5,0xd1,0xee,0x86,0x86,0xeb,0xeb,
  0xff,0xfd,0xfe,0xff,0xfb,0xcb,0xeb,0x86,0x63,0x68,0xf5,0xed,0xfe,0xfe,0xf4,0xe8,
  0xff,0xe8,0xfe,0xfe,0xff,0xf5,0x63,0x45,0xf7,0xae,0xcb,0xf0,0xed,0xfd,0xfd,0xff,
  0xeb,0xcb,0xae,0x86,0x86,0xc2,0xf5,0xb0,0xb6,0xf7,0xc2,0xe9,0xb5,0xb5,0xb5,0xb5,
  0xee,0xa7,0xa7,0xa7,0xe9,0xf9,0xe9,0x91,0xf7,0xf0,0x8b,0xeb,0xd2,0xd2,0xed,0xe6,
  0xff,0xfe,0xfd,0xed,0xfb,0xed,0xf4,0xf8,0xf5,0xf6,0xa1,0xf4,0xf3,0xfd,0xed,0xe6,
  0xe6,0xed,0xf2,0xf3,0xf2,0xeb,0x63,0xfa,0xb0,0xf8,0xed,0xfb,0xed,0xf2,0xfe,0xff,
  0xe6,0xed,0xd3,0xd2,0xeb,0x8b,0xf7,0xf7,0x91,0xf9,0xf9,0xe9,0xa7,0xb5,0xb5,0xb5,
  0xb5,0xa7,0xa7,0xe3,0x44,0xe9,0xf9,0xc2,0xf7,0xf6,0xf5,0xed,0xfc,0xf1,0xed,0xe6,
  0xff,0xfe,0xfd,0xed,0xfb,0xed,0xf1,0xf1,0xeb,0xf6,0xf5,0xf2,0xf3,0xf2,0xed,0xf0,
  0xf0,0xe6,0xe8,0xf3,0xf2,0xeb,0xfa,0xeb,0xf1,0xf1,0xff,0xfb,0xed,0xf2,0xfe,0xff,
  0xe6,0xed,0xf1,0xfc,0xed,0xf5,0xf6,0xf7,0x6b,0xc2,0xc2,0xe9,0xe3,0xa7,0xa7,0xa7,
  0xb5,0xb5,0xb5,0xe3,0x44,0xf9,0xf9,0xc2,0xf7,0xf7,0x8b,0xeb,0xb0,0xd2,0xed,0xe6,
  0xff,0xfd,0xfd,0xed,0xfb,0xe7,0xd3,0xb0,0xf5,0x63,0xf5,0xe8,0xf3,0xfd,0xff,0xe6,
  0xe6,0xed,0xf2,0xf3,0xf4,0xeb,0x63,0xf7,0xae,0xf8,0xe7,0xfb,0xed,0xf2,0xfe,0xff,
  0xe6,0xed,0xf8,0xb6,0xeb,0x8b,0xf6,0x86,0x6b,0xe9,0xf9,0xe9,0xa7,0xa7,0xb5,0xa7,
  0xb5,0xb5,0xb5,0xa7,0xe9,0xf9,0x6b,0x91,0xb0,0xf5,0x6b,0x6b,0x86,0xf7,0xeb,0xf0,
  0xed,0xfd,0xfe,0xff,0xfb,0xcb,0xeb,0xf7,0x69,0x69,0xfa,0xd0,0xdc,0xf3,0xf2,0xe8,
  0xe8,0xf2,0xfe,0xfe,0xed,0xf5,0x69,0xd1,0x86,0xad,0xeb,0xcb,0xff,0xfd,0xfd,0xff,
  0xeb,0xeb,0xae,0x86,0xd1,0xc2,0xf5,0xb0,0x91,0xf7,0xe9,0x44,0xa7,0xa7,0xa7,0xb5,
  0xa7,0xa7,0xb5,0xa7,0xe9,0xf9,0x86,0x86,0xb6,0xeb,0x8b,0xc2,0xee,0xc2,0x8b,0xf6,
  0xeb,0xf8,0xfe,0xff,0xa1,0xfa,0x8b,0x69,0x69,0xc2,0xd1,0xa1,0xff,0xfe,0xf3,0xfe,
  0xfe,0xf3,0xfe,0xe8,0xeb,0x8b,0x69,0x69,0x69,0x69,0xf6,0xa5,0xd0,0xfe,0xf1,0xeb,
  0xf6,0xad,0xc2,0xee,0xee,0x8b,0xeb,0xb0,0x91,0xf7,0x44,0xe9,0xa7,0xa7,0xa7,0xb5,
  0xe9,0xe9,0xf9,0xe9,0xf9,0xc2,0x86,0xf7,0xd2,0xd0,0xf6,0xc2,0xc2,0xc2,0xc2,0x6b,
  0xf6,0xeb,0xf8,0xae,0xfa,0x69,0x8b,0x8b,0x86,0x86,0x86,0xf5,0xeb,0xff,0xf1,0xfe,
  0xfe,0xf1,0xff,0xeb,0xfa,0x8b,0xad,0x86,0x86,0x86,0x8b,0x8b,0xf5,0xf8,0xeb,0xf6,
  0xc2,0x8a,0xee,0xe3,0x6b,0xf6,0xd0,0xd2,0x91,0xf7,0x44,0xf9,0xf9,0xf9,0xf9,0xf9,
  0xe9,0xf9,0xf9,0xc2,0x69,0x6b,0x86,0xf7,0xd2,0xe8,0xf5,0x8b,0xd1,0xc2,0x69,0xc2,
  0x44,0xc2,0x86,0xae,0xeb,0xa5,0xa1,0xeb,0xf8,0xf8,0xd0,0xf5,0x86,0xf5,0xeb,0xd0,
  0xd0,0xeb,0xf5,0x8b,0xfa,0xeb,0xf8,0xf8,0xd0,0xeb,0xa1,0xa1,0xae,0xf7,0x69,0x46,
  0xe3,0xe3,0xc2,0x69,0x8b,0xf5,0xff,0xb6,0x86,0xf7,0x6b,0x69,0x69,0xf9,0xf9,0xe9,
  0xee,0xf9,0xd1,0x86,0x86,0x86,0xad,0xb2,0xd2,0xf8,0xd0,0xf6,0xd1,0xb5,0xe9,0xf9,
  0x69,0x69,0x86,0xd2,0xe8,0xe6,0xfb,0xe7,0xe8,0xfe,0xf1,0xeb,0xee,0x68,0x8b,0xf6,
  0xfa,0x8b,0xd1,0xd1,0xf0,0xf8,0xfe,0xf4,0xed,0xfb,0xe6,0xe8,0xfd,0xae,0x69,0xe9,
  0x8a,0xc2,0xee,0xc2,0xfa,0xd0,0xe8,0xb6,0xb2,0xf7,0xf7,0x86,0xf7,0xf7,0xe9,0xf9,
  0xf9,0x6b,0x91,0x91,0x91,0xb2,0xae,0xae,0xd2,0xd2,0xff,0xf5,0x8b,0x69,0xe9,0xc2,
  0xc2,0x69,0xeb,0xf1,0xfd,0xff,0xe7,0xe6,0xff,0xf4,0xfd,0xae,0xd1,0x69,0x69,0xf5,
  0xf5,0x8b,0xee,0xd1,0x86,0xf1,0xf4,0xff,0xe6,0xe6,0xff,0xfd,0xfd,0xeb,0x8b,0xe9,
  0x8a,0xc2,0xc2,0x8b,0xf5,0xed,0xd2,0xd2,0xae,0xae,0xb2,0x91,0xb0,0x91,0x6b,0xf9,
  0xf7,0xf7,0xb6,0xfc,0xfc,0xfc,0xd2,0xf8,0xdc,0xfc,0xe8,0xe6,0xfa,0x69,0xb5,0xc2,
  0xc2,0x69,0xa1,0xf4,0xfe,0xf1,0xff,0xe6,0xe6,0xed,0xff,0x86,0xd1,0x69,0xf7,0xd0,
  0xd6,0x86,0xd1,0xd1,0x86,0xd0,0xed,0xe6,0xe6,0xed,0xf1,0xfe,0xf1,0xeb,0x8b,0x8a,
  0xb5,0xee,0xc2,0xfa,0xeb,0xff,0xfc,0xfc,0xf8,0xd2,0xfc,0xfc,0xfc,0xb6,0xf7,0xf7,
  0xf6,0x89,0xf5,0xeb,0xd6,0xd3,0xf1,0xd2,0xdc,0xf3,0xf1,0xed,0xf5,0x8b,0xc2,0xc2,
  0xc2,0x8b,0xa1,0xff,0xf1,0xfe,0xf4,0xed,0xe6,0xfb,0xfb,0xaa,0xd1,0x8b,0xae,0xfd,
  0xfd,0xf8,0xf7,0x8b,0x8b,0xeb,0xfb,0xe6,0xed,0xf4,0xfe,0xfd,0xff,0xcc,0xaa,0x8a,
  0xee,0xc2,0x8b,0xf5,0xed,0xf1,0xf3,0xfc,0xd2,0xf1,0xf8,0xff,0xae,0xf5,0xf6,0x89,
  0xfa,0x69,0x6b,0x8b,0xfa,0xa1,0xd0,0xff,0xe8,0xfc,0xfc,0xd3,0xcf,0xf5,0xc2,0xd1,
  0xc2,0xf5,0xeb,0xe6,0xff,0xfd,0xfe,0xf4,0xff,0xe7,0xeb,0x8b,0xd1,0xa1,0xd0,0xff,
  0xff,0xed,0xa1,0x8b,0x86,0xa1,0xe6,0xff,0xf4,0xfe,0xfd,0xe8,0xe6,0xcc,0xf0,0xd1,
  0xc2,0xc2,0xf5,0xf0,0xe8,0xfc,0xfc,0xf4,0xff,0xe6,0xeb,0xfa,0x89,0x6b,0x6b,0xf6,
  0xd0,0xf5,0xc2,0x6b,0xc2,0x8b,0xf6,0xf5,0xd0,0xd3,0xfc,0xd2,0xd3,0xf0,0x86,0xc2,
  0xc2,0xf7,0xd0,0xfb,0xe7,0xff,0xfd,0xfe,0xf2,0xe8,0xeb,0x8b,0x8b,0xa1,0xfb,0xe6,
  0xe6,0xfb,0xeb,0xf5,0x8b,0xeb,0xff,0xf2,0xfe,0xfd,0xe8,0xe7,0xe6,0xed,0xae,0xee,
  0xee,0xd1,0xf5,0xed,0xf1,0xfc,0xf1,0xe6,0xf5,0xf6,0x8b,0x6b,0xd1,0xd1,0x86,0xae,
  0xfc,0xb0,0xd1,0xd1,0x6b,0xc2,0xd1,0x69,0xf6,0xeb,0xf8,0xfc,0xd3,0xcf,0xf0,0xc2,
  0x6b,0xb2,0xe8,0xed,0xe6,0xe7,0xff,0xf4,0xfe,0xfe,0xf8,0xf5,0xa1,0xe6,0xe7,0xe7,
  0xe7,0xe7,0xe7,0xcb,0xf5,0xae,0xfd,0xfe,0xf4,0xff,0xe6,0xe6,0xed,0xf1,0xb0,0x86,
  0xc2,0x86,0xcf,0xd3,0xfc,0xf8,0xeb,0xfa,0x8b,0xd1,0xc2,0xc2,0xc2,0xd1,0xf7,0xd2,
  0xfd,0xf8,0x86,0xd1,0xee,0xc2,0xc2,0xf9,0xee,0xd1,0xad,0xb0,0xb0,0xd3,0xaa,0x6b,
  0x86,0xf8,0xf1,0xe8,0xed,0xfb,0xe6,0xe6,0xd0,0xf8,0xb0,0xb2,0xf1,0xf1,0xe8,0xe8,
  0xe8,0xf4,0xf1,0xf1,0xb2,0xb0,0xae,0xd0,0xe6,0xe6,0xcc,0xed,0xf1,0xfc,0xd2,0xae,
  0xc2,0xaa,0xd3,0xf8,0xb0,0xae,0x8b,0xc2,0xee,0xc2,0xc2,0xd1,0xc2,0xf7,0xb0,0xd2,
  0xff,0xed,0xeb,0xd1,0xb5,0xee,0xc2,0xc2,0xee,0xa7,0xb5,0xee,0x91,0xae,0xf0,0x8b,
  0xf5,0xf8,0xd2,0xd2,0xd0,0xeb,0xa5,0xf6,0xfa,0x86,0x69,0xf7,0xd2,0xfe,0xfe,0xfe,
  0xfe,0xfe,0xfe,0xf1,0xae,0x69,0x8b,0x8b,0xf6,0xf6,0xeb,0xf8,0xd2,0xd2,0xd3,0xf0,
  0x8b,0xf0,0xf8,0x91,0x91,0xc2,0xc2,0xb5,0xc2,0xc2,0xc2,0xc2,0xd1,0xeb,0xd0,0xff,
  0xe6,0xe6,0xeb,0xf6,0x69,0xf9,0xb5,0xa7,0xee,0xb5,0xe3,0xa7,0xee,0xc2,0x86,0xfa,
  0xaa,0xcf,0xf0,0xf7,0xf7,0xf7,0xd1,0xc2,0xc2,0x69,0x69,0x8b,0xeb,0xff,0xf4,0xf4,
  0xe8,0xe8,0xff,0xeb,0xf7,0x69,0x69,0x69,0xee,0xe3,0xd1,0x86,0xf7,0xb0,0xad,0xaa,
  0xf0,0xf5,0x86,0xee,0xc2,0xb5,0xc2,0xc2,0xee,0xf9,0xe9,0x69,0xf6,0xa1,0xfb,0xe6,
  0xff,0xff,0xed,0xa1,0x8b,0x69,0x44,0xf9,0xe3,0xe3,0xee,0xc2,0x6b,0x86,0xf7,0xf5,
  0xaa,0x8b,0xd1,0x69,0x69,0xee,0xd1,0xc2,0xee,0xee,0xee,0x6b,0xf6,0xf0,0xe6,0xe6,
  0xe6,0xe6,0xf0,0xf6,0x69,0xc2,0xee,0xa7,0xe3,0xb5,0xb5,0xee,0xee,0xc2,0x86,0xaa,
  0xf8,0x86,0x86,0x6b,0xee,0xb5,0xee,0xc2,0xe9,0x44,0x44,0x69,0xf5,0xe6,0xff,0xff,
  0xfe,0xfd,0xf1,0xd6,0xf5,0x69,0x69,0x69,0x63,0xd1,0xad,0x86,0xb0,0xb0,0xb6,0xb0,
  0xf5,0xf5,0x86,0x6b,0xe9,0xee,0xee,0xc2,0xee,0xee,0xee,0xb5,0xd1,0xa1,0xe6,0xe6,
  0xe6,0xe6,0xeb,0x86,0xb5,0xa7,0xb5,0xf9,0xb5,0xee,0xc2,0xee,0xd1,0xfa,0xa1,0xf0,
  0x86,0xb6,0xb0,0xb0,0x86,0xf0,0x8b,0x69,0x69,0x69,0x69,0xf7,0xeb,0xf1,0xfd,0xfe,
  0xfd,0xfd,0xfe,0xfe,0xb2,0xf7,0xf7,0xf5,0xa1,0xa1,0xf0,0xed,0xd3,0xd2,0xb0,0x86,
  0xd1,0xf0,0xf0,0xf5,0xf7,0x69,0xc2,0xb5,0xee,0xf9,0xee,0xb5,0xee,0xad,0xd0,0xe8,
  0xe8,0xd3,0xae,0xd1,0xd1,0xb5,0xa7,0xb5,0x69,0x6b,0x86,0xfa,0xaa,0xcf,0xf8,0x86,
  0x86,0xb0,0xd2,0xd3,0xed,0xf0,0xa1,0xa1,0xf5,0xf7,0xf7,0xae,0xfd,0xfe,0xfd,0xfd,
  0xed,0xed,0xff,0xff,0xae,0xae,0xd2,0xf1,0xe8,0xed,0xe6,0xe6,0xed,0xd3,0xb0,0x86,
  0xe9,0x91,0xb0,0xed,0xcf,0xf5,0xf6,0x69,0xe9,0xee,0xa7,0xb5,0xb5,0xd1,0xb0,0xfe,
  0xfe,0xb0,0x86,0xd1,0xc2,0xee,0xee,0x6b,0x8b,0xfa,0xf0,0xcf,0xd3,0xf8,0x86,0xc2,
  0xd1,0xb0,0xd3,0xed,0xe6,0xe6,0xed,0xe8,0xf1,0xd2,0xb0,0xae,0xd0,0xff,0xff,0xed,
  0xe6,0xfb,0xfb,0xa1,0xf5,0xeb,0xf4,0xfe,0xfe,0xf1,0xed,0xe6,0xe6,0xed,0xae,0x6b,
  0xb5,0x91,0xb6,0xfc,0xd3,0xf4,0xf0,0xf5,0xf6,0x86,0xd1,0xb5,0xa7,0xee,0x86,0xf8,
  0xd3,0xae,0x6b,0xc2,0xc2,0x6b,0xf6,0xf6,0xa1,0xe6,0xd3,0xfc,0xfc,0xb6,0x86,0xee,
  0xee,0xf7,0xed,0xe6,0xe6,0xed,0xe8,0xfe,0xfe,0xf1,0xd0,0xf5,0xa1,0xcb,0xfb,0xfb,
  0xed,0xe7,0xeb,0xfa,0x8b,0xeb,0xed,0xe8,0xfd,0xfe,0xf1,0xed,0xe6,0xcf,0xf0,0xc2,
  0xc2,0xc2,0xae,0xd2,0xfc,0xfc,0xf4,0xff,0xd0,0xf5,0xfa,0x8b,0xc2,0xee,0x8b,0xeb,
  0xeb,0x8b,0xc2,0x69,0xf6,0xfa,0xeb,0xe6,0xff,0xf4,0xfc,0xfc,0xd2,0xae,0xd1,0xa7,
  0xee,0xf0,0xf0,0xe6,0xed,0xf4,0xfe,0xfe,0xe8,0xed,0xcb,0xf6,0xf6,0xa1,0xe6,0xed,
  0xfd,0xd6,0xad,0x8b,0x8b,0xeb,0xfb,0xe7,0xe8,0xfd,0xfe,0xf1,0xed,0xf0,0xaa,0xe3,
  0xee,0xc2,0x86,0xeb,0xd2,0xf3,0xf3,0xdc,0xf1,0xe8,0xd6,0xeb,0xa1,0xf7,0x6b,0xf6,
  0xf6,0x69,0xf6,0xfa,0xd0,0xd0,0xd6,0xf4,0xf1,0xf3,0xfe,0xf1,0xd0,0xf5,0xee,0xa7,
  0xee,0x8b,0xa1,0xed,0xe8,0xfe,0xfe,0xe8,0xed,0xfb,0xeb,0xf5,0x63,0xf5,0xd0,0xfd,
  0xd2,0xb0,0xe3,0xd1,0xf0,0xd0,0xe7,0xfb,0xe7,0xe8,0xfd,0xfe,0xe8,0xeb,0x86,0xa7,
  0xa7,0xb5,0x6b,0xf5,0xe6,0xf4,0xfc,0xf3,0xf3,0xf3,0xdc,0xd2,0xf8,0xb2,0xae,0x89,
  0x89,0x8e,0xae,0xf8,0xd2,0xd2,0xfc,0xdc,0xfc,0xfc,0xf4,0xed,0xf5,0xd1,0xc2,0xee,
  0xc2,0x63,0xa1,0xe8,0xfe,0xfe,0xe8,0xed,0xfb,0xe7,0xe6,0xf5,0x68,0x68,0xb0,0xf1,
  0xeb,0xf5,0xe3,0x69,0x86,0xf8,0xff,0xed,0xfb,0xe7,0xe8,0xfe,0xf1,0xeb,0xd1,0xee,
  0xe3,0xee,0xc2,0x8b,0xf5,0xff,0xd2,0xd2,0xd2,0xdc,0xf3,0xf3,0xf3,0xfc,0xb0,0x6b,
  0xc2,0x91,0xb6,0xb6,0xb6,0xb0,0xb0,0xb2,0xf8,0xd2,0xff,0xeb,0x8b,0xee,0xb5,0xc2,
  0x69,0x45,0xf5,0xf1,0xfe,0xe8,0xed,0xfb,0xe7,0xff,0xf8,0x86,0x69,0x45,0x86,0xeb,
  0xf5,0x68,0xe3,0x69,0x86,0xf1,0xfd,0xe8,0xe7,0xfb,0xed,0xf4,0xfe,0xae,0x69,0xd1,
  0xa7,0xf9,0xc2,0xd1,0xfa,0xd0,0xf4,0xdc,0xf4,0xae,0xae,0xae,0xb2,0x92,0x6b,0xee,
  0xf9,0x6b,0xf7,0xf7,0xf7,0x8e,0xae,0xae,0xd2,0xf8,0xed,0xf6,0xd1,0xee,0xc2,0xe9,
  0x45,0x69,0xad,0xd2,0xf1,0xed,0xfb,0xe7,0xff,0xfd,0xf1,0xae,0x69,0x45,0xf7,0xf6,
  0xeb,0xf5,0x86,0x8b,0xf0,0xff,0xfd,0xf1,0xed,0xcb,0xcb,0xd0,0xf8,0xf7,0x69,0xc2,
  0xee,0xe3,0xee,0xc2,0x8b,0xeb,0xe8,0xf3,0xd2,0xf7,0x86,0xd1,0xf7,0x6b,0xc2,0xe9,
  0xe9,0xe9,0xc2,0xc2,0x6b,0xd1,0x86,0xae,0xd2,0xe8,0xeb,0xf6,0xc2,0xc2,0xc2,0xe9,
  0x45,0xe3,0xf7,0xb0,0xd0,0xcb,0xcb,0xe6,0xf1,0xfd,0xe8,0xa1,0x6b,0xf6,0xf5,0xeb,
  0xf2,0xe8,0xd0,0xf5,0xfa,0xa1,0xeb,0xb0,0xae,0xf5,0xaa,0xf0,0xae,0xae,0x89,0x69,
  0xc2,0x8a,0xd1,0xd1,0xd1,0xfa,0xd6,0xfc,0xdc,0xf7,0xd1,0xee,0xee,0xc2,0xf9,0xe9,
  0xe9,0xe9,0x44,0xf9,0xee,0xc2,0xf7,0xf7,0xd2,0xd6,0xfa,0x69,0xc2,0xd1,0xc2,0x69,
  0x45,0x86,0xf7,0xae,0xf5,0xfa,0xf5,0xf5,0xb0,0xae,0xeb,0xf6,0xf5,0xeb,0xff,0xf4,
  0xf3,0xf3,0xf1,0xed,0xf5,0x8b,0x6b,0xc2,0xd1,0xd1,0x8b,0xf5,0xeb,0xd2,0xf8,0xa1,
  0x89,0xb3,0xd1,0xd1,0xd1,0x8b,0xd0,0xdc,0xdc,0x8e,0xd1,0xee,0xb5,0xa7,0xee,0xb5,
  0xb5,0xb5,0xa7,0xa7,0xf9,0x69,0xf7,0x91,0xd2,0xd0,0xf6,0xc2,0x69,0x6b,0x69,0xf6,
  0xf6,0xeb,0xd2,0xd0,0xf5,0x8b,0x69,0x69,0x69,0x69,0x69,0xf5,0xd0,0xf1,0xf3,0xf3,
  0xf2,0xfe,0xf3,0xfd,0xd0,0xfa,0x69,0x69,0x69,0xf7,0xf5,0xeb,0xed,0xfd,0xf1,0xe6,
  0xa1,0xf0,0x86,0xd1,0xc2,0x69,0xf5,0xd2,0xdc,0xf7,0x69,0xc2,0xee,0xa7,0xb5,0xb5,
  0xb5,0xa7,0xb5,0xa7,0xe9,0xe9,0xf7,0x91,0xd2,0xf5,0x8b,0xee,0xc2,0xf7,0xf5,0xf5,
  0xd0,0xf1,0xfe,0xff,0xeb,0xa1,0xf6,0x69,0x69,0x69,0xf6,0xeb,0xf1,0xf3,0xfe,0xf2,
  0xed,0xff,0xfd,0xf3,0xe8,0xf5,0x63,0x63,0xf7,0xae,0xe6,0xfb,0xed,0xfd,0xfe,0xff,
  0xe6,0xd0,0xb2,0x86,0xf7,0x6b,0xf7,0xb0,0xb6,0x6b,0xee,0xe9,0xee,0xb5,0xb5,0xb5,
  0xee,0xb5,0xb5,0xa7,0xf9,0xe9,0x44,0x6b,0x86,0xf6,0xc2,0xf7,0xf7,0xb2,0xd0,0xfb,
  0xff,0xfd,0xfd,0xed,0xfb,0xe6,0xeb,0xf7,0x63,0x45,0xf5,0xff,0xf3,0xfe,0xe8,0xed,
  0xf0,0xe6,0xf4,0xf3,0xf1,0xeb,0x89,0xf5,0xf8,0xf1,0xff,0xfb,0xed,0xf2,0xfe,0xff,
  0xe6,0xed,0xf1,0xfc,0xae,0xfa,0x89,0xf7,0x6b,0xc2,0xc2,0xe9,0xee,0xa7,0xa7,0xb5,
  0xb5,0xb5,0xa7,0xa7,0xf9,0xee,0xf9,0xe9,0xf7,0x89,0xf6,0xae,0xd2,0xf1,0xff,0xe6,
  0xff,0xfe,0xfd,0xed,0xfb,0xed,0xf1,0xf8,0xeb,0xf6,0xf5,0xf4,0xf3,0xf2,0xed,0xf0,
  // 7: ground fan
  0x2c,0x63,0x4d,0x6b,0x08,0x42,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,
  0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x00,0x03,
  0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,0x00,0x05,
  0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,
  0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,
  0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,
  0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,
  0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,
  0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,
  0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,
  0x10,0x43,0x18,0x43,0x8c,0x73,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,
  0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0xaa,0x5a,0x08,0x46,0x10,0x46,0x18,0x46,
  0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0xae,0x7b,0x10,0x60,
  0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x11,0x84,
  0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,
  0x10,0x64,0x6d,0x73,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x69,0x52,0x18,0x65,
  0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,
  0x00,0x80,0x08,0x80,0x10,0x80,0x2b,0x6b,0x18,0x80,0xc7,0x41,0x00,0x81,0x08,0x81,
  0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,
  0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x6e,0x6b,0x18,0x84,0x00,0x85,0x08,0x85,
  0x10,0x85,0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,
  0x10,0x87,0xeb,0x62,0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,
  0xcb,0x5a,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,0xec,0x62,0x11,0xa5,
  0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,0x08,0xa4,
  0x10,0xa4,0x6e,0x73,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x18,0xa5,0x48,0x4a,0x00,0xa6,
  0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0x00,0xc0,
  0x08,0xc0,0x10,0xc0,0x18,0xc0,0xed,0x83,0x00,0xc1,0xef,0x7b,0x08,0xc1,0x07,0x4a,
  0x10,0xc1,0x6f,0x94,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0x2d,0x73,
  0x00,0xc3,0xed,0x7b,0xa6,0x39,0x08,0xc3,0x10,0xc3,0xeb,0x72,0x18,0xc3,0x00,0xc4,
  0x8a,0x62,0xec,0x5a,0x8f,0x94,0xc7,0x39,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,
  0xb0,0x9c,0x0d,0x5b,0x08,0xc5,0x10,0xc5,0xd4,0x8c,0x16,0x95,0xb4,0x8c,0x56,0x95,
  0x08,0x52,0xc8,0x39,0xc7,0x51,0x6a,0x42,0x97,0x9d,0x89,0x52,0xa7,0x41,0x2e,0x8c,
  0xf1,0xf1,0xf1,0x01,0x00,0x00,0x00,0x00,0xf1,0xf1,0x00,0x01,0x01,0x00,0x01,0x01,
  0x00,0x00,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0xb6,0xb6,0x01,0xb6,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0xb6,0xb6,0xb6,0xb6,0x00,0xb6,0x00,0x00,0x00,0xb6,0x00,
  0xc1,0x6f,0x7e,0x01,0x01,0x01,0x01,0x01,0x9c,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0xc1,0xc1,0xc1,0x9c,0x66,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0x66,0xc1,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xc1,0x01,0xc1,0x01,0x01,0x01,0x01,0x01,
  0x00,0x01,0x01,0x01,0x00,0x01,0x00,0x00,0xb0,0xb6,0x00,0x00,0x00,0x6f,0x7e,0x00,
  0x01,0x66,0xb0,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0xc1,0x01,0xc1,0x01,0xc1,0x01,0x01,0xc1,0xc1,0x01,0xc1,0x01,
  0xc1,0x01,0x01,0x00,0x00,0x01,0x01,0x00,0x01,0x00,0x00,0x00,0xb6,0x00,0x00,0x01,
  0x01,0x00,0xb6,0xb6,0x00,0x00,0x00,0x00,0x00,0xb0,0xb0,0x00,0x00,0x66,0xfd,0x00,
  0x01,0x01,0x01,0x5c,0xc6,0xfb,0x7e,0xfd,0xfb,0x7e,0xfb,0x7e,0x7e,0xfd,0xfd,0xfd,
  0xfd,0xfd,0xfd,0xfd,0xfd,0x5c,0x7e,0x7e,0x7e,0xfd,0xfb,0xfd,0x7e,0xfd,0xb0,0x7e,
  0x7e,0xfb,0xfb,0xc6,0xc6,0x02,0x02,0xfb,0x7e,0xfb,0x7e,0xc6,0x7e,0xc6,0x02,0x02,
  0x02,0xc6,0x02,0xc6,0xc6,0x7e,0xfb,0x7e,0x7e,0x02,0x7e,0xfd,0xb6,0x00,0x00,0x00,
  0x01,0x01,0x00,0x02,0x6f,0x6f,0xd9,0xea,0xf4,0x6f,0xff,0xd9,0xf0,0xb7,0xf4,0xf0,
  0xea,0xd9,0xd9,0x6f,0xd9,0x6f,0xf6,0xea,0xd9,0xd5,0x00,0xa9,0xb6,0x00,0x00,0xe9,
  0xe9,0x00,0x00,0x00,0xdf,0x66,0x6f,0xd9,0xd9,0xd9,0xd9,0x6f,0xd9,0x6f,0xd9,0xd9,
  0xd9,0xd9,0xd9,0xd9,0xea,0xea,0xf0,0xea,0x6f,0x6f,0x6f,0x02,0x5c,0x00,0x01,0x00,
  0x01,0x9c,0x00,0x02,0x6f,0xf7,0xf7,0xf7,0xf7,0xb7,0xb7,0xb7,0xb7,0xf7,0xf7,0xc5,
  0xb7,0xc5,0xf6,0xf0,0xf7,0xc5,0xf4,0x6f,0x01,0x00,0x01,0x01,0xe9,0xb0,0xb0,0xe9,
  0xb0,0xb0,0xb0,0x00,0x9c,0x66,0x9c,0xd5,0xf6,0xc5,0xfc,0xfc,0xb7,0xf4,0xb7,0xb7,
  0xf0,0xf6,0xf4,0xf0,0xf0,0xf4,0xb7,0xc5,0xf7,0xc5,0xf6,0x02,0x5c,0x00,0x00,0x00,
  0x01,0xc1,0x01,0x02,0x6f,0xf4,0xb7,0xc5,0xf7,0xc5,0xb7,0xb7,0xc5,0xb7,0xb7,0xf0,
  0xb7,0xb7,0x6f,0xd9,0x6f,0x00,0x9c,0x01,0x00,0xfd,0x02,0x8d,0x02,0x8d,0xe2,0xeb,
  0x8d,0x02,0x8d,0xe2,0x02,0xfd,0xb0,0x00,0x9c,0x9c,0x6f,0xb7,0xc5,0xb7,0xb7,0xb7,
  0xb7,0xb7,0xb7,0xb7,0xf0,0xb7,0xb7,0xc5,0xc5,0xf6,0xd5,0x8d,0x5c,0x00,0x00,0x01,
  0x9c,0xc1,0x01,0x02,0x6f,0x6f,0x6f,0xb7,0xf7,0xb7,0xb7,0xb7,0xb7,0xb7,0xb7,0xf0,
  0xb7,0xf0,0x66,0x00,0x9c,0x00,0x5c,0xc6,0xc6,0xe2,0xe2,0xe2,0xb0,0xb0,0x02,0xeb,
  0xe2,0x02,0xeb,0x8d,0xeb,0x02,0x7e,0xb0,0x00,0xe9,0x9c,0xd5,0x6f,0xea,0xf0,0xf0,
  0xb7,0xb7,0xb7,0xf0,0xf0,0xf4,0xb7,0xf0,0xd9,0xf6,0xc1,0x8d,0xb0,0x01,0x9c,0x01,
  0x9c,0x01,0x01,0x02,0x6f,0x6f,0x6f,0x6f,0xb7,0xf0,0xf0,0xf0,0xf0,0xf7,0xb7,0xf0,
  0xd9,0x01,0x01,0x00,0xfd,0x8d,0xe2,0x8d,0xeb,0xfb,0xf1,0x6f,0x6f,0x6f,0xd5,0x6f,
  0xd5,0xd5,0xd5,0x66,0x00,0xe9,0xb0,0xb0,0xb0,0xfd,0xe9,0x01,0x9c,0xd5,0x6f,0xf4,
  0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0x6f,0xd5,0x6f,0x79,0x8d,0xb0,0x01,0x9c,0x01,
  0xc1,0x9c,0x01,0x02,0x66,0x66,0x66,0xd5,0x6f,0x6f,0x6f,0xc0,0xf0,0xb7,0xc5,0xd5,
  0x00,0x01,0x5c,0xc6,0xe2,0xe2,0x02,0x66,0xf4,0xf4,0xd5,0x52,0x52,0x8b,0x79,0x8b,
  0x8b,0x79,0x8b,0x66,0x66,0xd5,0x66,0xd5,0xf1,0xfb,0xb0,0x00,0x00,0x9c,0xd5,0xd5,
  0xd9,0x6f,0xd9,0x6f,0xd9,0x6f,0x6f,0xd5,0x6f,0x6f,0xdf,0x8d,0xe9,0x00,0x9c,0x01,
  0xc1,0xc1,0x01,0x02,0xd5,0xd5,0xd5,0x66,0xd5,0x66,0xd9,0xf4,0xc5,0xf0,0x9c,0x01,
  0xa9,0xc6,0xc6,0xc6,0xfb,0xf6,0xf7,0x6f,0x52,0xa9,0x52,0x8b,0x8b,0x66,0x52,0x52,
  0x52,0x52,0x79,0x8b,0xa9,0x8b,0x73,0x66,0x66,0x6f,0x9c,0xb0,0x00,0xb0,0x00,0x9c,
  0xd5,0x6f,0xb7,0xf7,0xf0,0x66,0x66,0xd5,0x66,0x6f,0xdf,0xfe,0xe9,0x01,0x9c,0x01,
  0xc1,0xc1,0x01,0x02,0xd5,0x6f,0x6f,0x66,0x66,0x6f,0xf7,0x66,0xd9,0xc5,0x00,0xb0,
  0x8d,0x8d,0xd7,0x78,0xf7,0x66,0x52,0x52,0x8b,0x66,0xd3,0xff,0xd9,0xea,0xea,0xf0,
  0xf0,0xf0,0xb7,0xf0,0xf0,0xff,0x66,0x73,0x01,0x00,0xd5,0xc1,0xb0,0xfd,0x7e,0x01,
  0x01,0xf7,0x6f,0x79,0xb7,0xd5,0x66,0xd5,0x6f,0xd5,0xc1,0x02,0xe9,0x9c,0x01,0x01,
  0xc1,0x01,0x01,0x02,0x66,0x6f,0xc0,0x66,0xd5,0xd9,0xd5,0xf8,0xb6,0xdf,0x5c,0x8d,
  0x8d,0xc6,0xfb,0xf4,0x8b,0x52,0x52,0xd3,0xd3,0xff,0xd3,0xff,0xd9,0xd3,0xea,0xff,
  0xd9,0xf0,0xd9,0xb7,0xb7,0xb7,0xb7,0xd9,0xd3,0x8b,0x01,0x52,0xa9,0x5c,0x02,0xd7,
  0x00,0xf0,0xb6,0x01,0xe8,0x79,0x66,0x66,0xf6,0x6f,0x79,0xc6,0xe9,0x01,0x9c,0x9c,
  0xc1,0xc1,0x9c,0xc6,0x66,0x6f,0xc0,0xd9,0x66,0xd5,0x52,0xfd,0xe2,0xfd,0xf8,0x8d,
  0xe2,0xe2,0x9c,0x8b,0x52,0x52,0x52,0xe1,0xd3,0x52,0xd3,0x52,0x8b,0x8b,0x8b,0x8b,
  0x8b,0x8b,0x8b,0x8b,0x52,0xea,0xea,0xf3,0xb7,0xb7,0x52,0x8b,0xb0,0x02,0xc6,0x8d,
  0x02,0xd5,0xf8,0xe2,0xb0,0xc1,0x66,0xd5,0x6f,0x6f,0xdf,0xeb,0xe9,0x01,0x9c,0x01,
  0xc1,0xc1,0xc1,0x02,0x66,0xd5,0x6f,0xd5,0x66,0x66,0x79,0xfd,0xc6,0xfd,0xf8,0xf8,
  0xe2,0x8d,0x00,0x52,0xd3,0x52,0xd3,0xe1,0x8b,0x66,0x8b,0xe1,0xd3,0xd3,0xd9,0xd9,
  0xf0,0xd9,0xf0,0xff,0x52,0x8b,0x66,0x52,0xb7,0xf3,0xb7,0xd9,0xfd,0xe2,0x8d,0x8d,
  0x8d,0x7e,0xe8,0xc6,0xb0,0x66,0x66,0xd5,0xd5,0xd5,0x00,0xeb,0xe9,0x9c,0x9c,0xc1,
  0xc1,0xc1,0x01,0x02,0x66,0xd5,0xd5,0xd5,0x66,0x79,0xc1,0x00,0x7e,0x7e,0xf8,0x5c,
  0x02,0xd7,0xfd,0x79,0x52,0xe1,0x8b,0x8b,0xe1,0xe1,0xd3,0xd3,0xff,0xd9,0xea,0xea,
  0xf0,0xb7,0xb7,0xf3,0xb7,0xf0,0xd9,0x79,0xa9,0xd3,0xf0,0x01,0x8d,0x8d,0xd7,0x8d,
  0xe2,0x8d,0xfd,0xa9,0xc1,0x6f,0x66,0x66,0x66,0x66,0x00,0xeb,0xe9,0x9c,0x01,0xc1,
  0xc1,0x9c,0x01,0x02,0x66,0xd5,0xd5,0xd5,0x66,0x79,0x79,0x5c,0xf8,0x7e,0x7e,0x7e,
  0xfd,0x7e,0xc6,0x5c,0x66,0x8b,0xe1,0xd3,0x52,0xd3,0xd3,0xe1,0x66,0x8b,0x66,0x8b,
  0x8b,0xe1,0x52,0xd9,0xf0,0xf3,0xb7,0xf3,0xd3,0x01,0x01,0x7e,0x8d,0x8d,0xd7,0xd7,
  0x8d,0xc6,0xc6,0xa9,0xa9,0xd5,0x6f,0x66,0x66,0x66,0x00,0x8d,0xe9,0x01,0x01,0xc1,
  0xc1,0xc1,0xc1,0x02,0x66,0xd5,0xd5,0xd5,0x79,0x79,0xe8,0xc6,0x8d,0x02,0xc6,0x7e,
  0x7e,0x7e,0x7e,0xd7,0xfd,0x79,0x66,0x66,0x79,0xe1,0x01,0x52,0x66,0x66,0xd3,0xff,
  0xff,0xd3,0xe1,0x52,0x8b,0x52,0xb7,0xea,0xb7,0x79,0xa9,0xc6,0xc6,0x8d,0xd7,0xd7,
  0xc6,0x02,0x5c,0xb0,0x00,0x66,0x6f,0x6f,0x66,0x66,0xdf,0xeb,0xe9,0x00,0x01,0xc1,
  0xc1,0xc1,0x9c,0x02,0x66,0xd5,0xd5,0x66,0x79,0xb0,0x7e,0x8d,0x8d,0xc6,0xc6,0xfb,
  0x7e,0x7e,0xc6,0xc6,0xd7,0xc6,0x01,0x52,0x8b,0x8b,0x52,0xd3,0xd3,0xd9,0xff,0xff,
  0xd9,0xd9,0xd9,0xff,0xd9,0x52,0xe8,0x8b,0x66,0x01,0xb0,0xc6,0x8d,0x8d,0xc6,0x5c,
  0xc6,0x7e,0xb0,0xb0,0x00,0x01,0xd5,0x6f,0x66,0x66,0x79,0x02,0xe9,0xf1,0x01,0xc1,
  0xc1,0xc1,0x9c,0x02,0x66,0xd5,0xd5,0x01,0x00,0x7e,0xfd,0xc6,0xb0,0xb0,0xc6,0xc6,
  0xc6,0xc6,0xfd,0xc6,0x8d,0xe2,0xc6,0x66,0xd3,0xd9,0xd3,0xd9,0xd9,0xff,0xd9,0xd9,
  0xd9,0xea,0xd9,0xf0,0xd9,0xea,0xd9,0x01,0x00,0x5c,0xfd,0x02,0x02,0x02,0x7e,0xfd,
  0xf1,0xf6,0x66,0x66,0x00,0xa9,0xe9,0x6f,0x6f,0x66,0xdf,0x8d,0xe9,0x01,0x01,0xc1,
  0xc1,0xc1,0x9c,0x02,0x66,0xd5,0xc1,0x66,0x5c,0xc6,0xa9,0x6f,0x00,0x52,0x8b,0xfd,
  0xc6,0x02,0xd7,0xc6,0xe2,0xe2,0xe2,0x8d,0xd5,0xd3,0xff,0xd3,0xd3,0xff,0xd3,0xff,
  0xd3,0xff,0xff,0xd9,0xff,0xd9,0x79,0x00,0xfd,0xfd,0x7e,0x02,0x02,0xc6,0x7e,0x9c,
  0xb7,0xd3,0xa9,0xa9,0xd5,0x00,0xfb,0x9c,0x6f,0x66,0xdf,0x8d,0xe9,0x01,0x01,0xc1,
  0xc1,0xc1,0x9c,0x02,0x66,0xd5,0x01,0x01,0xfd,0x7e,0xd5,0x73,0x66,0x52,0xd3,0x00,
  0x5c,0x02,0x8d,0x02,0xeb,0x8d,0xe2,0xe2,0x02,0xd5,0xd3,0xff,0xd3,0xd3,0xd3,0x52,
  0xd3,0xff,0xd3,0xff,0xd9,0x66,0x01,0xb0,0x5c,0xfd,0xc6,0x02,0x02,0xc6,0x78,0xf6,
  0xd9,0xf0,0x52,0x00,0xe1,0x01,0xb0,0xe9,0x6f,0x6f,0x00,0x8d,0xe9,0x01,0x01,0xc1,
  0xc1,0xc1,0x9c,0x02,0x66,0x66,0xc1,0x00,0x5c,0xb6,0x6f,0x79,0x52,0x52,0x66,0x66,
  0x00,0x5c,0xc6,0x02,0x02,0x02,0xeb,0xe2,0xe2,0xc6,0x66,0x52,0x52,0x52,0x8b,0x52,
  0x8b,0x8b,0x79,0x66,0x01,0xb6,0x5c,0x5c,0xfd,0xfb,0x7e,0xfd,0xfb,0x9c,0xf7,0x00,
  0xd3,0xea,0xd9,0x8b,0x01,0x66,0xb0,0xfd,0x78,0xf6,0x00,0x02,0xb6,0x01,0x9c,0xc1,
  0xc1,0xc1,0xc1,0x02,0x66,0x01,0x01,0xb0,0x5c,0x6f,0x01,0x52,0x8b,0x52,0x52,0x52,
  0x52,0x8b,0x5c,0x8d,0xe2,0x02,0x02,0x8d,0xe2,0xe2,0xc6,0x01,0x52,0x52,0x52,0xd3,
  0x52,0x66,0x8b,0xa9,0x7e,0x7e,0xb0,0x7e,0xfd,0x7e,0xfd,0xfd,0x78,0xf5,0xea,0x52,
  0xe8,0xea,0xea,0x52,0x5c,0x01,0x00,0xb0,0x9c,0xf4,0x00,0x02,0xf1,0x9c,0x01,0xc1,
  0xc1,0xc1,0x01,0x02,0x66,0x01,0x01,0x5c,0x5c,0x66,0x52,0x8b,0x79,0x8b,0x52,0x66,
  0x8b,0x66,0xa9,0xb6,0x02,0x8d,0x8d,0x8d,0x02,0x8d,0xd7,0xc6,0x66,0x52,0x52,0x8b,
  0x8b,0xd3,0x00,0xc6,0xd7,0x8d,0xc6,0x02,0xc6,0x5c,0xb0,0x9c,0xf4,0x52,0xf0,0xd9,
  0x79,0x52,0xf0,0xff,0x5c,0x79,0x00,0xfb,0x9c,0xf4,0xd5,0x02,0x00,0x9c,0x01,0xc1,
  0xc1,0xc1,0xc1,0x02,0x79,0x66,0x00,0x5c,0x00,0xd5,0x52,0x79,0xa9,0x52,0x52,0x8b,
  0x79,0x8b,0x8b,0x00,0xb0,0x02,0x02,0xeb,0xc6,0xc6,0x8d,0xe2,0x8d,0xd7,0x8d,0xe2,
  0xe2,0xe2,0xd7,0x8d,0xc6,0x8d,0xd7,0x7e,0xfd,0xb0,0x78,0xf4,0xff,0xa9,0xd3,0xf0,
  0x52,0x00,0xd9,0xd9,0x79,0xfd,0x79,0x02,0xfd,0x6f,0x6f,0x02,0xf1,0x9c,0x9c,0xc1,
  0xc1,0xc1,0x01,0x02,0x01,0xd5,0x5c,0xb0,0x9c,0x52,0x66,0xe8,0x79,0x52,0x01,0x79,
  0x8b,0x66,0x8b,0x8b,0x01,0xb0,0xc6,0x7e,0x7e,0xc6,0xd7,0xd7,0xd7,0xd7,0xd7,0x8d,
  0x8d,0x8d,0x8d,0xc6,0xd7,0x02,0x7e,0xfd,0xa9,0x9c,0xf4,0xd9,0xff,0x52,0x79,0xd9,
  0xff,0x5c,0xff,0xd9,0x52,0x01,0x8b,0xc6,0x02,0x6f,0xf4,0x8d,0xf1,0x9c,0x01,0xc1,
  0xc1,0xc1,0x01,0x8d,0x01,0xc1,0x5c,0xb0,0x66,0x66,0x52,0x01,0x01,0x66,0x52,0x8b,
  0x66,0x52,0x8b,0x79,0x8b,0x00,0x5c,0xc6,0xfd,0xc6,0xd7,0xd7,0x8d,0xd7,0xd7,0x8d,
  0xd7,0xd7,0xd7,0xd7,0xd7,0xc6,0x7e,0x5c,0x9c,0xf4,0xd3,0xd9,0xff,0xff,0x5c,0xd3,
  0xea,0x01,0x52,0xd9,0xd3,0x5c,0x52,0xf1,0x8d,0x6f,0xf4,0xfe,0x00,0x9c,0xc1,0xc1,
  0xc1,0xc1,0x9c,0x02,0x79,0xc1,0x00,0xe9,0x66,0x52,0x52,0xa9,0x52,0xd3,0x8b,0xa9,
  0xe1,0x52,0x00,0x00,0x01,0x8b,0x01,0x5c,0xc6,0xc6,0xd7,0xc6,0x8d,0xe2,0xe2,0xe2,
  0xe2,0xd7,0xd7,0x8d,0xc6,0xc6,0x7e,0x00,0xf4,0x8b,0xd3,0xd3,0xff,0xd3,0x00,0xd3,
  0xd3,0x52,0x52,0xff,0xff,0x00,0x00,0x00,0x5c,0x9c,0xf4,0xfd,0xf1,0x9c,0x9c,0xc1,
  0xc1,0x66,0xc1,0x02,0xc1,0xc1,0x00,0xe9,0x66,0x52,0x52,0xa9,0xe1,0x52,0xa9,0x01,
  0x52,0x01,0xa9,0xa9,0x52,0x52,0xd3,0xa9,0xc6,0x8d,0xd7,0x8d,0xe2,0xeb,0x00,0x01,
  0x5c,0xe2,0x8d,0xd7,0x8d,0xd7,0x7e,0xf6,0x52,0xa9,0x52,0xff,0xd3,0xff,0x52,0x8b,
  0xff,0x52,0xa9,0xff,0xd3,0x01,0xb0,0x01,0xfd,0x5c,0x6f,0xe9,0x00,0x01,0x01,0xc1,
  0xc1,0xc1,0xc1,0x8d,0xc1,0x01,0xb0,0x00,0xe1,0xe1,0xa9,0x79,0x52,0x52,0x8b,0x8b,
  0xd3,0xa9,0x8b,0x52,0x8b,0xd3,0xe1,0x52,0x7e,0x8d,0xd7,0x8d,0x8d,0x00,0xe8,0xd7,
  0x6e,0x00,0xe2,0x8d,0xd7,0x8d,0xc6,0xe1,0x52,0xa9,0xd3,0xd3,0xff,0xd3,0x52,0x79,
  0xd3,0xd3,0xa9,0xd3,0xff,0x8b,0xa9,0x01,0x5c,0xfd,0x6f,0x9c,0xe9,0x9c,0x9c,0xc1,
  0xc1,0xc1,0xc1,0x02,0xc1,0x01,0x00,0x01,0x66,0xd3,0x79,0x5c,0xd3,0xe1,0xa9,0x79,
  0xd3,0x79,0xa9,0xa9,0x66,0xe1,0xd3,0x00,0x7e,0xd7,0xd7,0xe2,0x00,0x7e,0xc6,0xd7,
  0x7e,0xd7,0xb0,0xe2,0xd7,0xd7,0x02,0x79,0xe1,0xa9,0x52,0xd9,0xd3,0xd3,0xe1,0xe8,
  0xd9,0xd3,0xa9,0xd3,0xd3,0x66,0x5c,0x52,0x5c,0xb0,0x6f,0x9c,0xe9,0x01,0x9c,0xc1,
  0xc1,0xc1,0xc1,0x8d,0xc1,0x01,0xb0,0x00,0xd3,0xd3,0xe8,0x79,0xd3,0x52,0x00,0x52,
  0xd3,0xa9,0x00,0x52,0x8b,0xff,0xd3,0x01,0xc6,0xd7,0xd7,0xe2,0x01,0xc6,0x7e,0x8b,
  0x7e,0xd7,0x01,0xe2,0xd7,0xc6,0xc6,0x66,0x52,0xa9,0x79,0xd3,0xff,0xd3,0x8b,0x66,
  0xff,0xd3,0xa9,0xd3,0xd3,0x8b,0x00,0x8b,0x5c,0xa9,0x6f,0x9c,0x00,0x9c,0x9c,0xc1,
  0xc1,0x66,0x9c,0x02,0xc1,0x00,0xa9,0x00,0xe1,0xd3,0x79,0x5c,0x66,0xe1,0xa9,0x01,
  0xd3,0x01,0xa9,0xa9,0x79,0xff,0xd3,0x8b,0x7e,0xc6,0xd7,0xe2,0x01,0xe8,0xe8,0x00,
  0xfd,0xd7,0xd5,0xe2,0xd7,0x8d,0xc6,0x66,0x79,0xa9,0x52,0x66,0xd3,0xd3,0xd3,0xa9,
  0xd3,0xd3,0xa9,0xe1,0xd3,0x01,0xb0,0x01,0x5c,0x01,0x6f,0x9c,0xf1,0x9c,0xc1,0xc1,
  0xc1,0xc1,0x01,0x02,0xc1,0x01,0xb0,0x00,0xe1,0xff,0x01,0xa9,0xd3,0x52,0xa9,0x52,
  0xd3,0x00,0x00,0x01,0x52,0xd3,0xff,0x79,0x00,0x7e,0xc6,0xd7,0xc6,0xc1,0xfd,0x8b,
  0x7e,0xf5,0x00,0x8d,0xd7,0xd7,0xc6,0x79,0x52,0x00,0x79,0xd3,0xe1,0xd3,0x8b,0x01,
  0xff,0x52,0x00,0xd3,0xd3,0x52,0xa9,0x01,0x00,0x9c,0x6f,0x00,0xf1,0x9c,0x9c,0xc1,
  0xc1,0xc1,0x01,0x02,0x9c,0x01,0xb0,0xb6,0x66,0xff,0x8b,0x01,0x01,0xd3,0x00,0x5c,
  0xd9,0x8b,0xa9,0x01,0xa9,0x66,0xd9,0x79,0x5c,0x7e,0x7e,0x7e,0x7e,0xc6,0xc1,0x6f,
  0xca,0x9c,0x02,0x8d,0xd7,0x02,0xc6,0x8b,0xdf,0x8b,0x52,0x52,0xd3,0xe1,0x8b,0x66,
  0xe1,0xe1,0xa9,0xd3,0xff,0xa9,0x01,0x01,0x01,0x66,0xf6,0xe9,0x00,0x9c,0x9c,0xc1,
  0xc1,0xc1,0x01,0x02,0x9c,0x9c,0x00,0xb0,0x66,0xff,0x66,0x5c,0x66,0xff,0xa9,0x79,
  0xd3,0x79,0xa9,0xb0,0x66,0xa9,0x66,0xb0,0x5c,0x5c,0x7e,0x7e,0xd7,0x8d,0x8d,0xe2,
  0xe2,0xe2,0xc6,0xd7,0xd7,0xd7,0xd7,0x7e,0x00,0x52,0x8b,0x66,0x52,0xe1,0x01,0x52,
  0xff,0x8b,0x79,0xd3,0xe1,0x79,0xa9,0x01,0x01,0xd5,0xf4,0x02,0x01,0x9c,0x9c,0x9c,
  0xc1,0xc1,0x9c,0x02,0x01,0xc1,0xb0,0xfd,0x01,0xd9,0x52,0x79,0xa9,0xff,0xe1,0xe8,
  0xd5,0xd9,0xa9,0x01,0x5c,0x9c,0xa9,0x5c,0xfd,0x5c,0x5c,0x5c,0xd7,0x8d,0x02,0x8d,
  0xd7,0xe2,0xd7,0xc6,0xf8,0xf8,0xf8,0xf8,0xb0,0x00,0x79,0x52,0x66,0x52,0xa9,0x66,
  0xd3,0x79,0x52,0xd3,0xd3,0xe8,0x79,0x01,0x66,0xd5,0xf6,0x8d,0x00,0x9c,0x9c,0x9c,
  0xc1,0xc1,0xc1,0x02,0xdf,0x66,0x01,0x00,0x01,0xff,0xd9,0x5c,0x00,0xd3,0xd3,0x00,
  0xa9,0xea,0x01,0xa9,0x01,0x00,0x5c,0x5c,0x5c,0xb0,0x00,0x5c,0x5c,0x7e,0xc6,0x02,
  0xc6,0x7e,0xfd,0x7e,0x7e,0x7e,0xc6,0xd7,0xe2,0x7e,0x00,0x66,0x52,0x01,0x52,0x52,
  0x66,0x8b,0xd3,0xd3,0x8b,0x00,0x00,0xc1,0xd5,0x6f,0x6f,0xd7,0x01,0xc1,0xc1,0xc1,
  0xc1,0xc1,0x9c,0x02,0x79,0x66,0x9c,0xb0,0xe9,0xd3,0xea,0x52,0x01,0x52,0xf0,0x52,
  0xa9,0xd9,0xd3,0x01,0x5c,0x7e,0x5c,0xfd,0x5c,0xa9,0x00,0x00,0xb0,0x5c,0x5c,0x5c,
  0x7e,0xb0,0x5c,0x5c,0x7e,0x7e,0x7e,0xd7,0xe2,0xf8,0xc6,0x01,0x79,0x01,0xe1,0xe1,
  0x66,0x01,0xd3,0xd3,0x8b,0x00,0x01,0x66,0x9c,0x6f,0x6f,0x02,0xf1,0x9c,0xc1,0x9c,
  0xc1,0xc1,0x9c,0x02,0x66,0x66,0x00,0xe9,0x00,0x66,0xff,0xd3,0x5c,0xa9,0xd9,0x52,
  0x8b,0x52,0xff,0xb0,0x02,0xd7,0xfd,0xfd,0xfd,0xa9,0xb0,0x00,0xf4,0x66,0xc1,0xd5,
  0x6f,0x6f,0x79,0x5c,0x5c,0xd7,0xc6,0xc6,0xc6,0xc6,0x8d,0x7e,0x00,0xa9,0x52,0x52,
  0x73,0x52,0x52,0x8b,0xa9,0x8b,0x01,0x01,0x66,0xf6,0xc1,0x02,0xf1,0xc1,0xc1,0x9c,
  0xc1,0xc1,0xc1,0x02,0x79,0x66,0x66,0x00,0xe9,0x01,0xff,0xd9,0xc1,0xa9,0xe1,0xf0,
  0x8b,0xc1,0x00,0xc6,0xf8,0xc6,0xc6,0x7e,0x7e,0xb0,0x00,0xf7,0xb7,0xb7,0xf0,0xd9,
  0xff,0xd9,0xd3,0x79,0x5c,0x5c,0xfd,0xc6,0xd7,0x5c,0x7e,0xc6,0xb0,0x79,0xd3,0x01,
  0x79,0x52,0xe1,0x79,0xa9,0x66,0xc1,0xc1,0xd5,0xf4,0xb6,0x02,0x00,0x9c,0xc1,0xc1,
  0xc1,0xc1,0xc1,0xc6,0x66,0x66,0xd5,0x00,0xb0,0x00,0x52,0xea,0xff,0xa9,0x01,0xf0,
  0xd5,0x00,0xc6,0xc6,0xfd,0x5c,0x5c,0xc6,0x7e,0x00,0xf4,0x8b,0x66,0xd9,0xd9,0xea,
  0xff,0xd3,0xff,0x8b,0x01,0x00,0xb0,0x7e,0xc6,0xa9,0xa9,0xfd,0xb0,0xa9,0x01,0x79,
  0x52,0xe1,0x52,0xa9,0x66,0x00,0x01,0x66,0x6f,0xf6,0xb6,0x02,0xf1,0x9c,0xc1,0x9c,
  0xc1,0xc1,0xc1,0xc6,0x79,0xd5,0x66,0xc1,0xb0,0x5c,0x00,0xd9,0xd9,0x52,0x00,0x79,
  0x5c,0xf8,0x5c,0x02,0xd7,0xc6,0x7e,0x5c,0xc1,0xf5,0x8b,0x52,0x79,0x8b,0x79,0x01,
  0x01,0x79,0xa9,0x01,0x8b,0xd5,0x9c,0x5c,0x00,0x00,0x5c,0xb0,0xb0,0xb0,0x00,0x01,
  0x66,0x52,0x01,0x01,0xa9,0x9c,0xc1,0xc1,0xf4,0x66,0xe5,0x02,0x00,0x9c,0x9c,0xc1,
  0xc1,0xc1,0xc1,0xc6,0x79,0x66,0x66,0xd5,0x01,0x7e,0x02,0x01,0xb7,0xd3,0x01,0x5c,
  0xd7,0xd7,0xc6,0x8d,0x8d,0xd7,0xd7,0x00,0xf4,0x00,0x01,0x52,0x52,0x52,0x52,0x52,
  0x52,0x52,0x01,0x52,0x00,0x8b,0xd5,0x66,0xc1,0x01,0x7e,0xfb,0xfb,0xfb,0xfd,0x01,
  0x79,0x52,0x01,0x00,0xb0,0x00,0x01,0xd5,0xf4,0x79,0xe5,0x02,0x00,0x9c,0xc1,0x9c,
  0xc1,0xc1,0xc1,0xc6,0x66,0x66,0x66,0xd5,0x66,0x5c,0xc6,0xd7,0xfd,0x00,0x5c,0xc6,
  0xd7,0xd7,0xe2,0xe2,0xf8,0x8d,0xb0,0xca,0xea,0x8b,0x00,0x8b,0x8b,0x52,0x8b,0x66,
  0x52,0x8b,0x66,0xa9,0x8b,0x52,0x8b,0xd5,0xd5,0x01,0xb0,0xb0,0xfb,0xfb,0x02,0xc6,
  0xb0,0x00,0xb0,0x5c,0x02,0x00,0x66,0xf5,0x66,0xdf,0x8b,0x02,0x00,0xc1,0x66,0xc1,
  0xc1,0x66,0xc1,0xc6,0x79,0x66,0x66,0x66,0xd5,0x01,0xa9,0xf8,0xd7,0xc6,0xc6,0xf8,
  0xd7,0xe2,0xe2,0xe2,0xd7,0xb0,0xf4,0xff,0xb7,0xb7,0xb7,0x66,0x52,0x8b,0x66,0x8b,
  0x8b,0x66,0xa9,0x66,0x52,0x79,0x66,0x52,0x66,0x66,0xb0,0xb0,0xfb,0xfb,0xc6,0x02,
  0xc6,0x7e,0xb0,0xfd,0xf1,0x01,0x6f,0xf4,0x79,0xdf,0xe5,0xd7,0xf1,0x9c,0xc1,0x9c,
  0x9c,0xc1,0x9c,0xc6,0x79,0x66,0x66,0x66,0xd5,0xd5,0x00,0x7e,0xd7,0xd7,0xc6,0xf8,
  0xe2,0xe2,0xfd,0x7e,0x00,0xf7,0x8b,0x8b,0x79,0x52,0xb7,0xf3,0xc9,0xb7,0xf3,0xf0,
  0xea,0xea,0xd5,0xd3,0x66,0xe1,0x52,0x66,0x52,0xd5,0x9c,0xb0,0xb0,0xb0,0xc6,0x7e,
  0xfd,0x5c,0xb0,0x5c,0xa9,0xd5,0xf7,0x79,0x79,0xdf,0xb6,0x02,0x00,0x9c,0x9c,0x9c,
  0xc1,0xc1,0x9c,0xc6,0x79,0x66,0xd5,0xd5,0x66,0xd5,0xd5,0xb6,0xf8,0xc6,0xf8,0xd7,
  0xc6,0xe2,0xd7,0xb6,0xf4,0xb7,0xb7,0xd9,0x00,0x52,0x8b,0x52,0x79,0xd3,0xd3,0xd9,
  0xd3,0xd3,0x52,0x79,0x52,0x52,0x52,0x52,0x66,0x01,0xd5,0x9c,0xb0,0xfd,0x7e,0xfd,
  0x00,0x00,0x00,0xb0,0x79,0xf5,0x6f,0x79,0x79,0x79,0xb6,0x02,0x00,0x9c,0xc1,0x9c,
  0xc1,0x66,0xc1,0xc6,0x66,0x66,0x66,0xd5,0xd5,0xd5,0x6f,0x9c,0x5c,0x7e,0xc6,0xd7,
  0xd7,0xd7,0xf8,0x6f,0x8b,0x01,0xd9,0xf3,0xb7,0xd9,0xd3,0x8b,0x52,0x52,0x52,0x52,
  0x79,0x52,0x52,0x66,0x52,0x66,0xe1,0x79,0x52,0x66,0x52,0x66,0xe9,0xc6,0xc6,0x7e,
  0xfd,0x5c,0x00,0xc1,0xf6,0xf5,0x79,0x79,0xdf,0x79,0xb6,0x02,0x00,0x9c,0x9c,0x9c,
  0x9c,0xc1,0xc1,0x02,0x79,0x66,0x66,0xd5,0xd5,0x66,0xf0,0xb7,0xfc,0x6f,0xc6,0xc6,
  0xc6,0x8d,0x5c,0xea,0x8b,0x00,0xa9,0xa9,0xea,0xd9,0xf3,0xb7,0xb7,0xff,0xf0,0xd3,
  0xd3,0xff,0xd3,0xe1,0xe1,0xe1,0x8b,0x52,0x8b,0x8b,0x52,0x8b,0x00,0xfd,0xc6,0xa9,
  0xb0,0x6f,0xb7,0xf7,0xfc,0xdf,0xdf,0x66,0x79,0x79,0x8b,0xd7,0x00,0x9c,0x01,0xc1,
  0x01,0xc1,0xc1,0xc6,0x66,0x66,0xd5,0x6f,0xd5,0x6f,0xb7,0xdf,0x79,0xb7,0xa9,0x5c,
  0xfd,0xc6,0x00,0xca,0xfc,0xf0,0x52,0x52,0x8b,0x79,0x52,0xff,0xf0,0xd9,0xb7,0xf0,
  0xea,0xea,0x52,0xff,0x52,0x52,0x66,0x52,0x66,0x52,0x52,0x66,0x00,0xa9,0xa9,0xa9,
  0x01,0xb7,0x79,0xb6,0xf0,0x66,0xdf,0x79,0x79,0x79,0xe5,0xd7,0x00,0x9c,0x9c,0xc1,
  0x01,0xc1,0xc1,0xc6,0x66,0xff,0x6f,0x6f,0xd9,0xd9,0x66,0x00,0xa9,0xe8,0x00,0xa9,
  0xa9,0xb0,0xf8,0xff,0xca,0xca,0xca,0xd9,0x66,0x8b,0x79,0x8b,0x8b,0x66,0x8b,0x66,
  0x52,0x8b,0xd5,0x8b,0x52,0x66,0x8b,0x66,0x52,0x52,0x79,0x00,0xa9,0x5c,0x5c,0x00,
  0xc1,0xd9,0xf8,0xb6,0x7e,0xdf,0x79,0x79,0x66,0x66,0x8b,0xfe,0xf1,0x9c,0x9c,0x9c,
  0x01,0xc1,0xc1,0xc6,0x66,0x6f,0x6f,0xd5,0xd5,0x66,0xd3,0xd7,0xe2,0xf1,0x6f,0x66,
  0xa9,0x7e,0x7e,0xfd,0x00,0xea,0xca,0xca,0xca,0xb7,0xea,0xd3,0x52,0x8b,0x79,0x8b,
  0x8b,0x66,0x8b,0x66,0x52,0xe1,0x66,0x52,0x66,0x52,0x8b,0xa9,0x5c,0xfd,0x7e,0x01,
  0xf4,0xd9,0xf8,0xe2,0xd7,0xdf,0x79,0x79,0x66,0xdf,0xb6,0x8d,0x00,0x01,0x01,0x01,
  0x01,0xc1,0xc1,0xc6,0x66,0xd5,0xd5,0xd5,0x66,0x66,0x00,0x5c,0x5c,0xe5,0x79,0xf6,
  0xd5,0x01,0xe8,0xe8,0x5c,0xa9,0xb0,0xd9,0xb7,0xce,0xca,0xca,0xca,0xb7,0xb7,0xb7,
  0xf0,0xf0,0xd5,0xea,0xff,0xd5,0xd5,0x79,0x01,0x7e,0xc6,0xd7,0x7e,0x7e,0x66,0xfc,
  0xf5,0x8b,0x8b,0x5c,0x5c,0xe5,0xdf,0x79,0x66,0xdf,0xdf,0xd7,0xb6,0x9c,0x01,0x9c,
  0x01,0x9c,0x9c,0xc6,0x66,0xd5,0xd5,0xd5,0x66,0xdf,0xdf,0xe5,0xe5,0xe5,0xe5,0xdf,
  0x6f,0x6f,0x79,0x00,0xb0,0xa9,0xfd,0x5c,0x00,0x01,0xd3,0xf0,0xd9,0xc9,0xf0,0xb7,
  0xf0,0xff,0xd9,0x52,0x52,0x8b,0x00,0x66,0x00,0xeb,0xb0,0xe9,0x66,0xf4,0xca,0x6f,
  0x79,0xdf,0xe5,0xe5,0xe5,0xe5,0xdf,0x79,0x79,0x79,0xdf,0x02,0x00,0x01,0x9c,0x01,
  0x01,0x9c,0x01,0xc6,0x66,0xd5,0x6f,0x66,0xdf,0xb6,0xe5,0xe5,0x8b,0xe5,0xdf,0xdf,
  0xe5,0xc1,0x6f,0x6f,0xd5,0x79,0x7e,0x5c,0x7e,0x5c,0xa9,0x6f,0x9c,0x66,0x01,0x01,
  0x00,0x79,0x00,0xfd,0xe9,0xe9,0x00,0x00,0xb0,0xb0,0x66,0x6f,0xca,0xfc,0x79,0xdf,
  0x79,0x79,0xdf,0xe5,0xdf,0xe5,0xe5,0xdf,0x66,0x66,0x00,0x02,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0x02,0x79,0xd5,0x66,0xdf,0xb6,0x8b,0x8b,0x8b,0x8b,0xe5,0xe5,0xdf,
  0xdf,0xdf,0x8b,0x79,0x6f,0x6f,0xd5,0xc1,0x01,0x5c,0x5c,0x01,0x01,0x5c,0xc6,0x02,
  0xc6,0xb0,0x5c,0x9c,0xa9,0xa9,0xa9,0x6f,0xfb,0x6f,0xca,0xfc,0x79,0xdf,0xdf,0xdf,
  0x00,0xdf,0xdf,0xe5,0xe5,0xdf,0x8b,0x8b,0xdf,0x79,0xdf,0x02,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0x02,0xdf,0xdf,0xe8,0xe8,0xe8,0x5c,0xe8,0x5c,0xe8,0x8b,0x8b,0x8b,
  0xe5,0x8b,0xe8,0xe8,0xe8,0x00,0xd5,0xf6,0xf6,0x6f,0xd5,0xd5,0xd5,0x79,0xa9,0xb0,
  0xf1,0x00,0x9c,0x6f,0x6f,0x6f,0xf5,0xca,0xfc,0xf5,0xdf,0x8b,0x8b,0x8b,0xe5,0xe5,
  0xa9,0xe8,0x00,0xa9,0xe8,0xe8,0xe8,0xe8,0xa9,0xb6,0x00,0x02,0x00,0x01,0x01,0x01,
  0x01,0x00,0xb6,0xc6,0xa9,0xfa,0xe8,0x02,0x7e,0x7e,0x7e,0x7e,0x7e,0x7e,0x7e,0xe8,
  0x7e,0x7e,0x7e,0x7e,0x7e,0x7e,0x7e,0x7e,0xf1,0xd5,0x6f,0xf6,0xf5,0xf5,0xf5,0xf4,
  0xf5,0xf7,0xca,0xca,0xfc,0xf7,0xf4,0x01,0x7e,0x5c,0xe8,0xa9,0xe8,0xe8,0xe8,0x7e,
  0x7e,0x7e,0x5c,0x7e,0x5c,0x7e,0x6e,0x6e,0xf8,0xf8,0x7e,0x02,0x00,0x01,0x01,0x01,
  0x01,0x01,0x00,0xfb,0x02,0x02,0xf9,0xeb,0xeb,0xf9,0xf9,0xeb,0xf9,0xeb,0xeb,0x02,
  0xe2,0xfe,0x8d,0xe2,0xe2,0xe2,0xe2,0xe2,0xeb,0xe2,0xe2,0xeb,0xe2,0xfb,0xf1,0xf1,
  0xf1,0xf1,0xfb,0xe2,0xe2,0xe2,0xeb,0xe2,0xeb,0xeb,0xeb,0xeb,0xeb,0xe2,0xe2,0xeb,
  0x02,0xeb,0x02,0x02,0xeb,0xeb,0xeb,0xeb,0xeb,0xeb,0xeb,0x02,0xb0,0x00,0x01,0x01,
  0x66,0x9c,0xb0,0xf1,0xf1,0xf1,0xf1,0x00,0xb6,0x00,0x00,0xb0,0xb0,0xb6,0xb0,0xb6,
  0xa9,0xb6,0xb6,0x00,0xb6,0xa9,0xb0,0xb0,0xb0,0xb6,0xb6,0xb6,0xb6,0x00,0xb6,0xb6,
  0xb0,0xe9,0xb6,0xb6,0xb6,0xb6,0xb0,0xb0,0xb6,0xb0,0xb6,0xb6,0x00,0x00,0xb6,0x00,
  0x00,0x00,0xb6,0x00,0xb6,0x00,0x00,0x00,0x00,0x00,0x00,0xb6,0xb0,0xd5,0x5c,0x01,
  0xc1,0xa9,0xfd,0xf1,0x01,0xf1,0xf1,0x00,0xf1,0x00,0xb6,0x00,0xb6,0x00,0xb6,0xb6,
  0xb6,0xb6,0x00,0x00,0xb6,0xe9,0xb0,0xb0,0xb0,0xb6,0x00,0x00,0x00,0xe9,0xb6,0xb6,
  0x00,0xe9,0x00,0x00,0xb6,0xb0,0xb6,0xb6,0xb6,0x00,0xb6,0xb6,0x00,0x00,0xb6,0x00,
  0x00,0xb6,0xb6,0x00,0x00,0x00,0xb6,0x00,0x00,0x00,0x00,0x00,0xb6,0x66,0xc6,0x01,
  0x01,0x00,0x00,0x01,0x00,0xb6,0x00,0x00,0x00,0x00,0xb6,0xe9,0xb6,0xe9,0xb6,0xb6,
  0xb6,0xb6,0xb6,0xb6,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0xb0,0x00,0xb6,0xe9,0xb0,0xb0,
  0xb0,0xe9,0xe9,0x00,0xb6,0xb6,0xb6,0xb0,0xb0,0xb0,0xe9,0xb6,0xb6,0xb6,0xb6,0x00,
  0xb6,0xb6,0xb6,0xb6,0xb6,0x00,0x00,0x00,0x00,0x00,0x00,0xf1,0xf1,0xf1,0x00,0x00,
  // 8: sky 1, part A
  0xde,0xc6,0xba,0x5c,0x7a,0x54,0x7a,0x4c,0x00,0x00,0x3f,0xd7,0x08,0x00,0x10,0x00,
  0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0xbe,0xbe,
  0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,
  0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,
  0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,
  0x10,0x22,0xfb,0x6c,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,
  0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,
  0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,
  0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,
  0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,
  0x08,0x44,0xbc,0x8d,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x9d,0xb6,
  0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,0xfd,0x9d,0x18,0x46,0x7b,0x7d,0x00,0x47,
  0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,0x5f,0xd7,0x18,0x60,
  0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,
  0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,
  0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,
  0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,
  0x00,0x81,0x08,0x81,0x5c,0xae,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,
  0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,
  0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,
  0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x3b,0x6d,0x18,0x87,0x00,0xa0,0x08,0xa0,
  0x10,0xa0,0x18,0xa0,0x9c,0x7d,0x3f,0xcf,0x00,0xa1,0x3c,0xa6,0x08,0xa1,0x10,0xa1,
  0x18,0xa1,0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,
  0x18,0xa3,0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,
  0x9f,0xe7,0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,
  0x10,0xa7,0xbe,0xc6,0x18,0xa7,0x1b,0x75,0x00,0xc0,0x7d,0xb6,0x08,0xc0,0x10,0xc0,
  0x18,0xc0,0x7e,0xb6,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,
  0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x18,0xc3,0x00,0xc4,0x08,0xc4,
  0x10,0xc4,0xfe,0xd6,0x7f,0xe7,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x10,0xc5,0x18,0xc5,
  0x00,0xc6,0x08,0xc6,0x10,0xc6,0x18,0xc6,0x00,0xc7,0x08,0xc7,0x1d,0x96,0x10,0xc7,
  0x18,0xc7,0x08,0xe0,0x10,0xe0,0xdd,0xc6,0xdc,0x8d,0x00,0xe1,0x08,0xe1,0x00,0xe7,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,
  0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x02,0x03,0x03,0x03,0x02,
  0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x03,0x03,0x02,
  0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x03,0x02,0x03,0x03,
  0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x03,0x03,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x31,0x01,0x01,0x01,0x01,0x01,0x31,0x31,0x31,0x31,0x31,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x03,0x03,0x03,0x02,0x03,0x03,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x02,0x01,
  0x01,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x31,0xac,
  0xd3,0x31,0x31,0xac,0xac,0xac,0x31,0x31,0x31,0x31,0xac,0xac,0xd3,0xac,0xac,0x31,
  0x31,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x31,0x31,0x31,0x31,0x31,0xac,0xd3,0xd3,0x66,
  0x66,0xd3,0xac,0xd3,0xd3,0xac,0xac,0xd3,0xd3,0xd3,0xd3,0x66,0xb2,0x66,0xd3,0xac,
  0xac,0xd3,0xac,0x31,0x31,0x31,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x31,0x31,0x31,0x01,0x31,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x31,0x31,0x31,0xd3,0xac,0xac,0xac,0xac,0xd3,0xd3,0x66,0xb2,0x59,0x59,
  0xb2,0xb2,0x66,0x66,0x66,0x66,0xb2,0x59,0x59,0x59,0xb2,0x59,0x66,0xb2,0xb2,0xb2,
  0x66,0x66,0x66,0x66,0x66,0xd3,0x31,0x31,0x31,0x01,0x31,0x31,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x31,0x01,0x31,0x31,0x31,0x31,0xac,0xac,0xd3,0x01,0xd3,0xac,0xac,0xac,
  0x31,0x31,0x31,0x31,0x31,0x31,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x31,0x31,0xac,0xac,0xd3,0xac,0xd3,0xac,0xd3,0x66,0x66,0xb2,0x59,0x59,0xfc,0x59,
  0x59,0xfc,0xfc,0x59,0x59,0x59,0x64,0x64,0xfc,0xfc,0x59,0xb2,0xb2,0x59,0xfc,0xfc,
  0xfc,0xfc,0x59,0x59,0x59,0xb2,0x66,0xd3,0xd3,0xac,0xd3,0xac,0xd3,0x31,0x31,0xd3,
  0xd3,0xac,0xd3,0x31,0x01,0x01,0x01,0x31,0x31,0x01,0x01,0x31,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x31,0xac,0xac,0xac,0xac,0x31,0x31,0x31,
  0xd3,0xd3,0xd3,0xac,0xac,0xd3,0xac,0xd3,0x66,0x66,0xd3,0xd3,0x66,0xd3,0xd3,0xd3,
  0xd3,0xd3,0xd3,0xac,0xd3,0xd3,0xac,0xac,0x31,0x31,0x01,0x31,0x01,0x01,0x01,0x01,
  0x31,0xd3,0xd3,0xac,0xac,0xd3,0xd3,0xac,0xd3,0xb2,0x59,0xfc,0x59,0xb2,0xfc,0x64,
  0xf6,0x64,0xb5,0xf6,0xb5,0x92,0x92,0x92,0x92,0xf6,0xf6,0xfc,0xfc,0x64,0xf6,0xb5,
  0xb5,0x64,0x64,0x64,0x64,0xfc,0x59,0xb2,0xb2,0x66,0xd3,0x66,0xd3,0xac,0xd3,0xac,
  0xd3,0xd3,0xd3,0xac,0xd3,0xac,0xac,0xd3,0xd3,0xac,0xac,0xd3,0xac,0xac,0xd3,0xac,
  0x01,0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x31,0x01,0x31,0x31,0x31,0x31,
  0x31,0x31,0x31,0x01,0x31,0xac,0xac,0xac,0xd3,0xd3,0xd3,0xd3,0xd3,0xac,0xac,0xd3,
  0xd3,0x66,0x66,0x66,0x66,0x66,0xb2,0x59,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0x66,0xb2,
  0xb2,0xb2,0x66,0xb2,0x66,0x66,0x66,0xd3,0xd3,0xac,0xac,0xd3,0xac,0x01,0xac,0xac,
  0xd3,0x66,0xac,0xd3,0x66,0xd3,0xd3,0xb2,0x59,0x59,0x59,0x59,0xfc,0xfc,0xf6,0x92,
  0xb5,0x92,0xd5,0x92,0xd5,0x5f,0x5f,0x5f,0xd5,0x92,0xd5,0x92,0xb5,0xb5,0x92,0x5f,
  0xd5,0xb5,0xb5,0xb5,0xb5,0xb5,0xf6,0x64,0xfc,0x59,0x66,0x66,0x66,0x66,0x66,0x66,
  0xd3,0xd3,0x66,0x66,0xd3,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0xd3,0x66,0xd3,
  0xd3,0xd3,0xac,0xd3,0xd3,0xac,0xd3,0xac,0xd3,0xac,0x31,0xd3,0xac,0x31,0x01,0x31,
  0xac,0xd3,0xd3,0xd3,0xac,0xd3,0xd3,0x66,0x66,0x66,0x66,0x66,0x66,0xb2,0xb2,0x59,
  0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0x59,0x59,0xb2,0x59,0xfc,0x59,0xfc,0xfc,0x59,0x59,
  0x59,0xfc,0x59,0xfc,0xfc,0xfc,0x59,0x59,0xb2,0xb2,0xb2,0x66,0x66,0x66,0x66,0xd3,
  0xac,0xd3,0xd3,0xac,0x66,0x66,0xb2,0x59,0xb2,0x59,0x59,0x59,0xfc,0x64,0xb5,0xb5,
  0x92,0xd5,0x5f,0x5f,0x5f,0x0f,0xb3,0xb3,0x00,0xb3,0x00,0x00,0xd5,0xd5,0x5f,0x0f,
  0x5f,0xd9,0x92,0xd9,0xd9,0xd9,0x92,0xb5,0xb5,0xfc,0xb2,0x66,0x66,0xb2,0xb2,0xb2,
  0x66,0x66,0x66,0x66,0x66,0x66,0xb2,0xb2,0xb2,0xb2,0x66,0x66,0xb2,0x66,0x66,0xb2,
  0x66,0x66,0xd3,0x66,0xd3,0xd3,0xac,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0xac,0xac,0xd3,
  0x66,0xd3,0xac,0xd3,0x66,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0x59,0xfc,0xfc,0xfc,0x64,
  0x64,0x64,0xfc,0x59,0x64,0x64,0xfc,0xfc,0x64,0x64,0x64,0x64,0xb5,0xb5,0xb5,0xb5,
  0xb5,0xb5,0xf6,0xf6,0x64,0x64,0xfc,0x59,0xfc,0xfc,0xfc,0x59,0xb2,0xb2,0xb2,0xd3,
  0xb2,0x66,0x66,0xb2,0x59,0x59,0x59,0x59,0x59,0x59,0xfc,0x59,0x64,0x64,0xb5,0x92,
  0x92,0xd9,0x5f,0x0f,0xd1,0xb3,0x05,0x6e,0x6e,0x6e,0x05,0xb3,0x00,0x00,0x00,0x00,
  0x0f,0x0f,0x5f,0x5f,0xd5,0xd5,0xd5,0x92,0x92,0xb5,0xb5,0x64,0xb2,0xb2,0xb2,0xb2,
  0xb2,0x66,0x66,0xb2,0x66,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,
  0x66,0x66,0x66,0x66,0x66,0x66,0x66,0xd3,0xd3,0xd3,0x66,0x66,0x66,0x66,0x66,0x66,
  0xb2,0xb2,0x66,0xb2,0x59,0xfc,0xfc,0x59,0x59,0xfc,0xfc,0xfc,0xfc,0x64,0x64,0x64,
  0x64,0x64,0x64,0xb5,0xb5,0xfc,0x64,0xf6,0xb5,0xb5,0x92,0x92,0x92,0xd9,0x92,0x92,
  0x92,0x92,0x92,0xb5,0x92,0xb5,0xb5,0xb5,0xb5,0xb5,0xf6,0xb5,0x64,0xfc,0x59,0x59,
  0xfc,0xfc,0x59,0x59,0x59,0x59,0x59,0xfc,0x59,0x59,0xfc,0xfc,0xf6,0x64,0xb5,0x92,
  0x92,0xd5,0x0f,0x00,0xb3,0x05,0x6e,0x6e,0xea,0xea,0xea,0xe9,0xea,0x05,0xb3,0x00,
  0x00,0x00,0x00,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,0xd9,0xf6,0xb2,0x59,0xb2,0xb2,
  0xfc,0xb2,0xb2,0xb2,0x66,0xb2,0xb2,0xb2,0x59,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,
  0xb2,0xb2,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0xb2,0xb2,
  0x59,0x59,0x59,0x59,0x59,0x59,0x59,0x64,0x64,0x64,0x64,0x64,0x64,0xb5,0xb5,0xb5,
  0xb5,0x92,0x92,0x92,0x64,0xf6,0x92,0x92,0x92,0xd9,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,
  0x5f,0x5f,0x5f,0xd9,0x5f,0xd5,0x92,0xd9,0x92,0x92,0x92,0x92,0xb5,0xb5,0xfc,0xfc,
  0x64,0xfc,0xf6,0xf6,0x64,0xfc,0xf6,0xf6,0xf6,0xf6,0xfc,0x64,0xb5,0x92,0x5f,0x0f,
  0x0f,0xd1,0x00,0x00,0xb3,0x05,0x6e,0x6e,0xe9,0xea,0xea,0xc8,0xea,0x6e,0x05,0xb3,
  0xb3,0x00,0x00,0x00,0x00,0xd1,0x0f,0x0f,0xd1,0x0f,0x5f,0x64,0xfc,0x59,0xfc,0xfc,
  0x59,0xb2,0xb2,0xb2,0x59,0xb2,0x59,0x59,0x59,0x59,0x59,0x59,0x59,0xfc,0xfc,0xfc,
  0xfc,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,0x66,0xb2,0x66,0xb2,0xb2,0xb2,0x59,
  0xfc,0xfc,0xfc,0xfc,0x64,0x64,0xf6,0xb5,0xb5,0xb5,0xb5,0xb5,0x92,0x92,0x92,0x92,
  0x92,0x92,0x92,0xb5,0xb5,0x92,0xd9,0xd5,0x5f,0x5f,0x0f,0xd1,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x5f,0x5f,0x5f,0xd9,0xd5,0xd9,0xd9,0xd5,0xd9,0xd9,0x92,0x92,0xb5,0xb5,
  0xb5,0xb5,0x92,0x92,0xb5,0xb5,0xb5,0x92,0x92,0x92,0xd5,0xd5,0x0f,0x0f,0x00,0xb3,
  0x00,0x00,0xb3,0xb3,0xb3,0x05,0x05,0x6e,0xe9,0xc8,0xc8,0xc8,0xe9,0x6e,0x05,0x05,
  0xb3,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x00,0x00,0x0f,0x5f,0x92,0x64,0x59,0xf6,0xfc,
  0x59,0x64,0x59,0xfc,0x59,0x59,0x59,0x59,0x59,0x59,0x64,0xfc,0x59,0xfc,0xfc,0x59,
  0x59,0x59,0x59,0xfc,0x59,0x59,0x59,0xb2,0xb2,0x59,0x59,0xb2,0xfc,0x59,0x59,0xfc,
  0x64,0x64,0x64,0x64,0x64,0xb5,0xb5,0xb5,0x92,0x92,0x92,0x92,0xd9,0xd5,0x92,0x92,
  0xd9,0x92,0x92,0x92,0x92,0xd9,0x5f,0xd5,0x5f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x00,
  0x0f,0x0f,0x5f,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,0x0f,0x5f,0xd9,0xd9,0x92,0x92,
  0xd5,0xd5,0xd9,0xd5,0xd9,0x92,0x92,0x5f,0x0f,0x0f,0x00,0x00,0x00,0x00,0xb3,0xb3,
  0xb3,0xb3,0xb3,0x05,0x05,0x05,0x05,0x05,0x6e,0x6e,0xe9,0xe9,0x6e,0xe9,0x6e,0x6e,
  0x05,0x05,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x00,0x00,0xd1,0x5f,0x92,0xb5,0x92,0xb5,
  0x64,0x64,0x64,0xf6,0x64,0xfc,0xfc,0x64,0x64,0xb5,0xf6,0xb5,0xb5,0xb5,0x64,0xf6,
  0x64,0xfc,0x59,0x59,0xfc,0xfc,0x59,0x59,0xfc,0x59,0xfc,0x64,0x64,0x64,0x64,0xb5,
  0xb5,0xb5,0x92,0xb5,0x92,0x92,0x92,0x92,0x92,0xd9,0xd5,0xd5,0x5f,0x5f,0xd9,0x5f,
  0x5f,0x5f,0x5f,0x0f,0x0f,0x0f,0x5f,0x0f,0xd1,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x00,
  0x0f,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0xd5,0xd5,0xd5,
  0x5f,0x5f,0x0f,0x0f,0x0f,0x5f,0x0f,0x0f,0x00,0xb3,0x05,0x05,0xb3,0xe9,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x6e,0x6e,0x6e,0x05,0x6e,0x6e,0x6e,0x6e,
  0x6e,0x05,0x05,0x05,0xb3,0xb3,0xb3,0xe9,0xe9,0xb3,0x00,0x0f,0x5f,0xd9,0xd5,0xd9,
  0x92,0xb5,0xb5,0x92,0x92,0x92,0xb5,0x64,0xb5,0x92,0x92,0x92,0x92,0xb5,0xb5,0xb5,
  0xf6,0xf6,0xf6,0xf6,0x64,0x59,0x59,0x64,0x59,0xfc,0xf6,0xb5,0xb5,0x92,0x92,0x92,
  0x92,0xd9,0xd5,0x5f,0x0f,0x5f,0x5f,0x5f,0x0f,0x0f,0x0f,0x0f,0x0f,0xd1,0x0f,0x0f,
  0x0f,0x0f,0x00,0x0f,0xd1,0xd1,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x00,0x0f,0x0f,0x0f,
  0x00,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,
  0x5f,0x0f,0x00,0x00,0x00,0xb3,0xb3,0x05,0xb3,0xb3,0xb3,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0xb3,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x6e,0x6e,0x6e,0x6e,0x05,
  0x05,0x05,0x05,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x0f,0x0f,0x0f,
  0x5f,0xd5,0xd5,0xd9,0xd5,0x92,0x92,0x92,0xd5,0x5f,0x0f,0x0f,0x5f,0xd5,0xd9,0x92,
  0x92,0xb5,0xb5,0xb5,0x64,0xfc,0x64,0x64,0x64,0x64,0x64,0xb5,0x92,0xd9,0xd5,0x5f,
  0x0f,0xd1,0x0f,0x0f,0x00,0x00,0xd1,0xd1,0xd1,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0xb3,0x00,0x0f,0x0f,0x00,0x0f,0x00,0x00,0x00,0x00,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x00,0x0f,0x0f,0x0f,0x5f,0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0x0f,0x5f,
  0x0f,0x00,0x00,0xb3,0xb3,0xb3,0x05,0x05,0x05,0xb3,0x05,0x05,0x6e,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xe9,0x05,0xe9,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0xb3,0xb3,0x00,0xb3,0x00,0x00,0x00,0x00,0x00,0x00,
  0x0f,0x0f,0x5f,0x0f,0x0f,0x5f,0xd5,0x5f,0x0f,0x00,0x0f,0x00,0x00,0x0f,0x0f,0x0f,
  0x0f,0xd9,0x92,0x92,0x92,0x92,0xb5,0x92,0xb5,0xb5,0xb5,0x92,0xd9,0x5f,0x0f,0x0f,
  0x0f,0xd1,0x00,0x00,0xb3,0x00,0x00,0x00,0x00,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x00,
  0x00,0x00,0x0f,0x0f,0x0f,0x0f,0x00,0xb3,0x00,0x00,0x0f,0x0f,0x0f,0x0f,0x00,0x0f,
  0x00,0xd1,0x0f,0x00,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0xd9,0xd9,0x5f,0x5f,
  0x0f,0x00,0x00,0xb3,0xb3,0xb3,0x05,0x6e,0x6e,0x6e,0xea,0xe9,0x6e,0x6e,0x05,0x05,
  0x6e,0x05,0x05,0x6e,0x05,0x05,0x05,0x05,0x05,0xe9,0xe9,0xe9,0x05,0x05,0x05,0x05,
  0xe9,0xb3,0xe9,0x05,0x05,0xb3,0xb3,0xb3,0x00,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x0f,
  0x0f,0x0f,0x0f,0xd1,0x0f,0x0f,0x0f,0x00,0x00,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,
  0x00,0x00,0x0f,0x0f,0x0f,0x5f,0xd5,0xd5,0xd5,0xd5,0xd5,0xd9,0x5f,0x5f,0x0f,0x0f,
  0x0f,0x0f,0x00,0xb3,0x00,0xb3,0x00,0x00,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x0f,0x0f,0x00,0xb3,0xb3,0xb3,0xb3,0x00,0x0f,0xd1,0x0f,0x0f,0x0f,0x00,
  0x00,0x00,0xb3,0xb3,0x00,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0xd9,0xd5,0x5f,0x5f,0x0f,
  0x0f,0x0f,0x00,0x00,0x00,0xb3,0x05,0x6e,0xe9,0xea,0xea,0x6e,0x05,0x6e,0x6e,0x6e,
  0x6e,0x05,0x6e,0x6e,0x6e,0x05,0x6e,0x6e,0x05,0x05,0x05,0xe9,0xe9,0xb3,0xe9,0xe9,
  0x00,0x00,0x00,0xe9,0xe9,0x05,0x05,0xb3,0x00,0x00,0xb3,0xb3,0xb3,0xb3,0x00,0x00,
  0xd1,0x00,0x00,0x00,0x00,0x00,0xb3,0x05,0xb3,0xb3,0x05,0x05,0x05,0x05,0x05,0xb3,
  0xb3,0xb3,0xb3,0xb3,0xb3,0x00,0xd1,0x0f,0xd1,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x00,0x00,0xb3,0xb3,0xb3,0xb3,0xb3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0xd1,0xd1,0x0f,0x00,0xb3,0xb3,0xb3,0xb3,0xb3,0x00,0x0f,0x0f,0x0f,0x00,0xb3,0x00,
  0x00,0xb3,0xb3,0xb3,0xb3,0x00,0xd1,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0x0f,
  0x0f,0x0f,0x0f,0x00,0x00,0xb3,0xb3,0x6e,0x6e,0x6e,0x6e,0xe9,0x6e,0xea,0x6e,0x6e,
  0x05,0x6e,0x05,0x05,0x6e,0x05,0x05,0x05,0x05,0x05,0x05,0xe9,0xe9,0xb3,0xb3,0xe9,
  0xe9,0xe9,0xb3,0xe9,0x05,0x05,0x05,0xe9,0xb3,0x00,0x00,0x00,0xb3,0x00,0x00,0x00,
  0xd1,0x00,0x00,0x00,0x00,0xb3,0x05,0x05,0xb3,0x05,0x6e,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0xb3,0x05,0x05,0x05,0x00,0x00,0x00,0x00,0x00,0x0f,0x0f,0x0f,0xd1,0xd1,
  0x0f,0xd1,0x00,0xb3,0xb3,0xb3,0x00,0xd1,0xb3,0x00,0xb3,0x00,0x00,0xb3,0xb3,0xb3,
  0x00,0x00,0xb3,0x00,0x00,0xb3,0xb3,0xb3,0xb3,0x00,0x00,0xd1,0xd1,0x00,0x00,0x00,
  0xb3,0xb3,0xb3,0x00,0x00,0x00,0x00,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0x0f,
  0x0f,0x0f,0x0f,0xd1,0xd1,0x00,0xb3,0x05,0x05,0x05,0x6e,0x05,0x05,0xe9,0xe9,0x6e,
  0x05,0xe9,0x05,0x05,0x05,0x05,0x05,0x05,0xe9,0x05,0x05,0xe9,0xe9,0xe9,0xe9,0xe9,
  0xe9,0xe9,0x05,0x05,0xe9,0xe9,0xe9,0x05,0x05,0xe9,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0xb3,0x05,0x05,0x05,0x05,0x6e,0x6e,0x6e,0x6e,0x6e,0x05,0x05,
  0x6e,0x6e,0x05,0x05,0x05,0xb3,0x00,0xd1,0x0f,0x0f,0x5f,0x5f,0x0f,0x0f,0x0f,0x0f,
  0x0f,0x0f,0x00,0xb3,0xb3,0x00,0x00,0x00,0x00,0x00,0xb3,0x00,0xb3,0xb3,0xb3,0xb3,
  0xb3,0xb3,0xb3,0xb3,0xb3,0x05,0xb3,0xb3,0xb3,0x00,0xb3,0x0f,0x00,0x00,0x00,0x00,
  0x00,0xb3,0x0f,0x0f,0x0f,0x0f,0x5f,0xd5,0xd5,0xd9,0xd5,0x5f,0x5f,0xd5,0xd5,0x5f,
  0x0f,0x0f,0x0f,0x5f,0x0f,0x0f,0x00,0xb3,0x05,0xb3,0x05,0xb3,0x05,0xb3,0x05,0x05,
  0xb3,0xe9,0xb3,0xe9,0xb3,0xb3,0xe9,0xe9,0x05,0xe9,0xe9,0x05,0x05,0x05,0xe9,0xe9,
  0x05,0x05,0x05,0x6e,0x6e,0x05,0x05,0x05,0x05,0x05,0xe9,0xb3,0x00,0x00,0x00,0xb3,
  0x00,0x00,0x00,0x00,0xe9,0x05,0xea,0xc8,0xc8,0xea,0xea,0xea,0xea,0x6e,0x05,0x6e,
  0x05,0x05,0xe9,0x00,0x00,0x00,0xd1,0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x0f,0x0f,
  0x0f,0xd1,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,
  0xb3,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb3,0x00,0xb3,0xb3,0xb3,0x00,0x00,0x00,
  0x00,0x00,0x0f,0xd1,0x0f,0x5f,0x5f,0xd5,0xd5,0x5f,0x5f,0xd5,0xd9,0x5f,0xd5,0xd5,
  0xd5,0xd5,0x5f,0x0f,0x00,0x0f,0x00,0x00,0xb3,0xb3,0xb3,0x00,0x00,0xb3,0xb3,0xe9,
  0x00,0x00,0xb3,0x05,0x05,0xe9,0xc8,0xe9,0xe9,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xea,
  0xe9,0xe9,0x05,0x05,0x05,0x6e,0x05,0x05,0x05,0x05,0xe9,0x05,0x05,0x05,0xe9,0xe9,
  0x00,0x00,0x00,0x6e,0xc8,0xea,0xc8,0xc8,0xc8,0xc8,0xc8,0xea,0xea,0xea,0x6e,0x6e,
  0x6e,0x05,0xe9,0x00,0x00,0xd1,0xd1,0x00,0x00,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,
  0x5f,0x5f,0x0f,0xd1,0x00,0xb3,0xb3,0xb3,0xb3,0xb3,0x05,0x05,0xb3,0xb3,0x05,0x6e,
  0x05,0x05,0x05,0x6e,0x05,0x6e,0x05,0x6e,0xb3,0xd1,0xb3,0xb3,0x00,0xd1,0x0f,0xd1,
  0x0f,0xd1,0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0xd5,0xd9,0xd9,0xd5,0x92,0x92,0xd9,0xd5,
  0xd5,0x92,0xb5,0x92,0x0f,0xe9,0x05,0x05,0xe9,0xea,0xea,0xe9,0xea,0xe9,0xe9,0xb3,
  0xb3,0xe9,0xe9,0xea,0xc8,0xea,0xc8,0xc8,0xc8,0xc8,0xc8,0xe9,0xe9,0xea,0xe9,0xe9,
  0x05,0x05,0xe9,0xe9,0x05,0x05,0x05,0x05,0xe9,0xe9,0xea,0x05,0xe9,0xe9,0x00,0xb3,
  0x00,0x05,0xc8,0xea,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xea,0xc8,0xea,0x6e,0x6e,
  0x6e,0x05,0x05,0x05,0xb3,0x00,0xd1,0xd1,0x5f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,0x5f,
  0x5f,0x5f,0x0f,0x00,0xb3,0xe9,0x05,0x05,0x05,0x05,0x6e,0x05,0xb3,0x05,0x6e,0x05,
  0x6e,0x05,0x05,0xb3,0xb3,0x05,0x05,0xb3,0xb3,0x00,0x0f,0x0f,0x0f,0x00,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x5f,0x5f,0x5f,0x5f,0x5f,0xd5,0xd9,0x92,0xd5,0xd5,0x92,0xd5,0x92,
  0xb5,0x92,0x92,0x92,0xd1,0x05,0xe9,0xc8,0xc8,0xc8,0xc8,0xc8,0xea,0xea,0xea,0x05,
  0x00,0xe9,0xc8,0xc8,0xc8,0xea,0xc8,0xc8,0xc8,0xc8,0xe9,0x05,0x05,0x6e,0xe9,0xe9,
  0xe9,0xe9,0xe9,0xe9,0xe9,0xea,0xe9,0xe9,0xe9,0x05,0xe9,0x05,0xe9,0xe9,0xe9,0xe9,
  0xe9,0xea,0xea,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xc8,0xea,0x6e,0xea,0xe9,0x6e,
  0x05,0x05,0xe9,0xe9,0xfb,0x00,0x00,0x00,0xd1,0xd1,0x0f,0x5f,0x5f,0x5f,0x0f,0x0f,
  0x0f,0x0f,0x0f,0x0f,0x00,0xb3,0x05,0x05,0x05,0x05,0x05,0x05,0xb3,0x05,0x6e,0x6e,
  0x6e,0x05,0xb3,0x05,0xb3,0x05,0x6e,0x05,0xb3,0xb3,0x00,0xd1,0x0f,0xd1,0xd1,0xd1,
  0x0f,0x5f,0x5f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x5f,0x5f,0xd5,0x92,0x92,0xd5,0x92,
  // 9: sky 1, part B
  0x3a,0xae,0x5c,0xb6,0x9c,0xbe,0xba,0x9d,0xfe,0xce,0x00,0x00,0x08,0x00,0x10,0x00,
  0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x5e,0xdf,0x00,0x02,0x08,0x02,
  0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0xbf,0xe7,0x10,0x05,0x18,0x05,0x00,0x06,
  0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x1e,0xdf,0x10,0x07,0x18,0x07,
  0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,
  0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,
  0x00,0x24,0x08,0x24,0xfa,0xa5,0x3c,0xae,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,
  0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,
  0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,
  0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,
  0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,
  0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,
  0x7c,0xae,0x10,0x47,0x18,0x47,0xbd,0xce,0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,
  0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0xfd,0xce,
  0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,
  0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,
  0xda,0x95,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,
  0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,
  0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x39,0x8d,0x18,0x83,0x00,0x84,
  0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x00,0x86,0x08,0x86,
  0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,
  0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0xbe,0xce,0x10,0xa1,0x18,0xa1,0x00,0xa2,
  0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,
  0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,0x18,0xa5,0x00,0xa6,
  0x08,0xa6,0x10,0xa6,0x99,0x95,0x18,0xa6,0xfe,0xd6,0x00,0xa7,0x08,0xa7,0x10,0xa7,
  0x79,0x8d,0x18,0xa7,0x00,0xc0,0xbe,0xbe,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x00,0xc1,
  0x08,0xc1,0xfc,0x9d,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x7b,0xbe,0x08,0xc2,0x10,0xc2,
  0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x18,0xc3,0x00,0xc4,0x08,0xc4,0x10,0xc4,
  0x18,0xc4,0x7f,0xe7,0x00,0xc5,0xb9,0xa5,0x08,0xc5,0x10,0xc5,0x3a,0xb6,0x18,0xc5,
  0x7e,0xb6,0x00,0xc6,0x7c,0xc6,0x08,0xc6,0x3f,0xcf,0x10,0xc6,0x18,0xc6,0xfd,0xd6,
  0x9c,0x85,0x5a,0x8d,0x18,0xe1,0x99,0x8d,0x1d,0x96,0x08,0xe2,0x18,0xe2,0x18,0xe4,
  0x3b,0xf0,0xb4,0x0d,0xe9,0xf4,0xe9,0xe9,0xe9,0xe9,0x0d,0x0d,0x0d,0x1c,0x1c,0xe9,
  0x0d,0x1c,0x1c,0x1c,0x1c,0xe9,0x0d,0x0d,0x1c,0x1c,0x0d,0x0d,0x1c,0x1c,0x1c,0x1c,
  0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,0x1c,0x1c,0xe9,0x0d,0x0d,0xf4,0x04,0xcc,0x1c,
  0xe9,0x1c,0xe9,0xe9,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,0xe9,0xe9,0x1c,0xe9,0x0d,
  0xe9,0xcc,0x04,0x04,0x04,0x04,0xb4,0xd3,0xd3,0xd3,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
  0xf0,0xf0,0xd3,0xd3,0xb4,0xf4,0xf4,0xf4,0xf4,0xf4,0xe9,0xf4,0xf4,0xf4,0xf4,0xe9,
  0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0x04,0x04,0x04,0xf4,0x04,
  0x04,0xb4,0xd3,0xf0,0xb4,0xb4,0xb4,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0x3b,0x3b,
  0xb4,0xf4,0x0d,0xe9,0x1c,0xe9,0x1c,0x1c,0x1c,0xe9,0x0d,0x1c,0x1c,0x1c,0x1c,0x1c,
  0xe9,0x1c,0x1c,0x1c,0x1c,0x0d,0x0d,0xf4,0x0d,0xe9,0x0d,0x1c,0x1c,0x1c,0x1c,0x1c,
  0x1c,0x1c,0x1c,0x1c,0x1c,0x0d,0x0d,0xcc,0x0d,0xe9,0x1c,0x0d,0xf4,0xcc,0x0d,0x1c,
  0x1c,0x1c,0x0d,0x0d,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,
  0xf4,0xf4,0xb4,0xb4,0xb4,0x04,0xd3,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,
  0xf0,0xf0,0xf0,0xf0,0xd3,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xe9,0xf4,0xe9,0xf4,0xe9,0xf4,0xf4,0x04,0xb4,0x04,
  0xb4,0xd3,0xd3,0xd3,0xd3,0x04,0x04,0xd3,0xf0,0xf0,0xf0,0xd3,0xf0,0xf0,0xf0,0xf0,
  0xe9,0xe9,0xe9,0x1c,0x1c,0xe9,0xe9,0x1c,0x1c,0xe9,0xe9,0xe9,0x1c,0x1c,0x1c,0x1c,
  0x1c,0x1c,0x1c,0x1c,0x1c,0x0d,0xf4,0xf4,0x0d,0x0d,0xe9,0x1c,0x1c,0x1c,0xe9,0x1c,
  0x1c,0x1c,0x1c,0x1c,0x0d,0x0d,0x04,0x6b,0x6b,0xcc,0xf4,0x0d,0xf4,0x04,0x0d,0x1c,
  0x1c,0x1c,0x0d,0xf4,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x0d,
  0xf4,0xf4,0xb4,0xb4,0xd3,0xb4,0xb4,0xd3,0xd3,0xd3,0xf0,0xf0,0xf0,0x3b,0x3b,0x3b,
  0xf0,0x3b,0xf0,0xf0,0xf0,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xe9,0xf4,0xe9,0xf4,0xf4,0xf4,0xe9,0x0d,0xf4,0xf4,0xf4,0xf4,0xf4,0xd3,0xf0,
  0xf0,0xd3,0x04,0x04,0xb4,0x04,0x04,0xb4,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xd3,0x04,
  0xe9,0xe9,0x1c,0x1c,0x1c,0xe9,0x1c,0x1c,0x1c,0x1c,0xe9,0x1c,0x1c,0x1c,0x1c,0x1c,
  0x1c,0x1c,0xe9,0x1c,0x0d,0xf4,0x04,0x04,0xf4,0x0d,0x1c,0x1c,0x1c,0x1c,0x0d,0x0d,
  0xe9,0x1c,0x0d,0x0d,0x0d,0x0d,0xcc,0x04,0xb4,0x04,0x6b,0x04,0x04,0xf4,0x1c,0x1c,
  0x1c,0x1c,0x0d,0xcc,0x0d,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,0x0d,0xe9,
  0xf4,0xcc,0x04,0x04,0x04,0x04,0x6b,0x04,0x04,0x04,0xd3,0xf0,0xf0,0x3b,0x3b,0x3b,
  0x3b,0x3b,0x3b,0xf0,0xf0,0xd3,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0x0d,0xe9,0xf4,0xe9,0xf4,0xe9,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0x04,0x04,0x04,
  0xd3,0xd3,0xb4,0x04,0x04,0xb4,0xd3,0xd3,0xd3,0xd3,0xf0,0xf0,0x3b,0x04,0xf4,0xe9,
  0xe9,0x1c,0x1c,0x1c,0xe9,0x0d,0xe9,0x1c,0x1c,0xe9,0xcc,0x0d,0x1c,0x1c,0xe9,0x0d,
  0xf4,0xf4,0xf4,0x0d,0x0d,0xf4,0xf4,0xcc,0xcc,0xcc,0xe9,0xe9,0xe9,0x0d,0x0d,0x0d,
  0xcc,0xf4,0xf4,0x04,0x04,0x04,0x6b,0x04,0xcc,0x6b,0x6b,0x6b,0x6b,0xe9,0x1c,0x1c,
  0x1c,0x1c,0xe9,0xf4,0x0d,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,0xe9,0x0d,0x0d,
  0xe9,0x0d,0xf4,0xcc,0x04,0x04,0x04,0x6b,0x6b,0x04,0xd3,0xf0,0x3b,0x3b,0x3b,0x3b,
  0x3b,0x3b,0x3b,0x3b,0x3b,0xf0,0xd3,0x04,0xb4,0x04,0xf4,0xf4,0xe9,0xf4,0xe9,0xf4,
  0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xe9,0xf4,0xf4,0xf4,0x04,0x04,
  0x04,0xd3,0xb4,0x04,0x04,0x04,0xd3,0xd3,0xd3,0xd3,0xd3,0x3b,0xf0,0xf4,0x0d,0x1c,
  0x1c,0x1c,0x1c,0x0d,0xcc,0xf4,0x0d,0x0d,0x1c,0xe9,0x0d,0x0d,0x1c,0xe9,0x25,0x6b,
  0x04,0x6b,0x6b,0x04,0xf4,0xcc,0x04,0x04,0xf4,0xf4,0x0d,0x0d,0xf4,0x04,0x04,0x04,
  0x04,0x04,0xcc,0x6b,0x6b,0x6b,0xb4,0x6b,0x04,0x04,0xb4,0xd3,0xd3,0x1c,0x0d,0xf4,
  0x0d,0x0d,0x0d,0x0d,0xe9,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,0xf4,0x0d,0xe9,0xe9,
  0xe9,0x0d,0xf4,0x04,0x04,0xcc,0x04,0x04,0x04,0x04,0xd3,0x02,0x01,0x3b,0x3b,0x3b,
  0x3b,0x3b,0xf0,0xf0,0x3b,0xf0,0xd3,0xb4,0x04,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0xf4,0xe9,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,0xd3,0xf4,0xd3,0xd3,0x04,0x0d,0xe9,0x1c,
  0x1c,0x1c,0x0d,0xf4,0x6b,0x6b,0x6b,0x6b,0x04,0x0d,0xf4,0xf4,0xcc,0x04,0xb4,0xd3,
  0x01,0x01,0x02,0xb4,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0xf4,0x25,0x04,0xb4,0xd3,0xd3,
  0x6b,0x6b,0x6b,0x6b,0x6b,0xf4,0x04,0xcc,0x04,0xb4,0x02,0xd3,0x6b,0xe9,0x6b,0x6b,
  0x04,0xf4,0x0d,0x0d,0xe9,0x1c,0xe9,0x1c,0x1c,0x1c,0x1c,0xe9,0xf4,0xe9,0xe9,0x0d,
  0x0d,0x0d,0xf4,0x6b,0x6b,0x04,0xd3,0xd3,0xd3,0xd3,0xd3,0x02,0x01,0xf0,0x3b,0x3b,
  0xf0,0xf0,0x01,0xf0,0x3b,0xf0,0xf0,0xd3,0xb4,0x04,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xe9,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0x04,0x04,0xd3,0x04,0xf4,0xf4,0xf4,0x04,0xf4,0xf4,0xf4,0x0d,0x0d,0x1c,0x1c,
  0x0d,0x0d,0xcc,0xf4,0x04,0xd3,0xd3,0xd3,0xd3,0xb4,0xd3,0xf0,0x02,0x01,0x01,0x01,
  0xf0,0x3b,0x3b,0x3b,0x02,0x02,0xd3,0xd3,0xd3,0x6b,0x6b,0x6b,0xd3,0x01,0x01,0x01,
  0x01,0x01,0x6b,0xb4,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0xd3,0xd3,0x6b,0xf4,0x6b,0xd3,
  0xb4,0x6b,0xf4,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0xe9,0xcc,0x25,0x0d,0x0d,0x0d,
  0x0d,0x0d,0x25,0xcc,0xd3,0x01,0xf0,0x01,0x02,0x02,0x02,0xd3,0x01,0x02,0x01,0x01,
  0x01,0x01,0x3b,0xf0,0xf0,0xf0,0xf0,0xf0,0xd3,0xb4,0xb4,0x04,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0x04,0xf4,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0x04,0x04,0x04,0xf4,0xe9,0xf4,0xf4,0xf4,0xe9,0x1c,0x0d,0x0d,0x1c,0x1c,
  0xf4,0x04,0x04,0x04,0xb4,0x01,0x68,0x01,0xf0,0x3b,0x3b,0x3a,0x3a,0x3b,0x3b,0x3a,
  0x3a,0x3b,0x3b,0x3a,0x3a,0x3a,0x3b,0x3b,0x3b,0x01,0x3b,0x3b,0x3b,0x3b,0x01,0x3b,
  0x3b,0x01,0xd3,0xd3,0x02,0x02,0xd3,0x02,0x6b,0xd3,0xd3,0xd3,0xd3,0xcc,0xcc,0x6b,
  0xd3,0xd3,0x6b,0x0d,0x0d,0xe9,0x0d,0xe9,0x0d,0x0d,0xf4,0x25,0xf4,0x0d,0xf4,0x25,
  0x0d,0x0d,0xf4,0xcc,0xd3,0x01,0x01,0x01,0x01,0xf0,0x01,0x01,0x02,0x01,0x01,0x01,
  0x01,0xf0,0x3b,0xf0,0xf0,0xf0,0xf0,0xf0,0xd3,0x04,0x04,0x04,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0x04,0x04,0x04,0xf4,0x04,0x04,0xf4,0xf4,0x04,0x04,0xf4,0xf4,0xf4,0xf4,
  0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0x0d,0xe9,0x0d,0xe9,0x1c,0xe9,0x1c,0x1c,0x1c,0x0d,
  0x6b,0x04,0x04,0x6b,0x04,0xd3,0x3b,0x3b,0x3b,0x3b,0xd9,0x03,0xd9,0x3a,0xd9,0x03,
  0x03,0x3a,0x3a,0x3b,0x3b,0x3b,0x3b,0x01,0x01,0x01,0x3b,0x3a,0xd9,0x3a,0x3b,0x3b,
  0x3b,0x01,0x01,0x02,0xd3,0xd3,0xd3,0xd3,0x02,0x02,0xd3,0x6b,0x6b,0x25,0x04,0x6b,
  0x02,0x01,0xd3,0x04,0xcc,0x04,0x6b,0x04,0x04,0xcc,0xcc,0x04,0xcc,0xf4,0x0d,0x1c,
  0x0d,0xcc,0x6b,0x6b,0x6b,0xd3,0x6b,0x3b,0x3b,0x3b,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0xf0,0x01,0x01,0x6b,0x6b,0xcc,0xcc,0xcc,0xcc,0xf4,0xf4,0xf4,0xf4,
  0x04,0xf4,0x04,0xf0,0xf0,0xb4,0x04,0xd3,0xd3,0xd3,0xd3,0xb4,0xb4,0x04,0x04,0xd3,
  0xd3,0x04,0x04,0x04,0xf4,0xf4,0xf4,0x0d,0x1c,0x1c,0x1c,0x0d,0x1c,0x1c,0x1c,0xcc,
  0x25,0x0d,0xe9,0x6b,0xb4,0xd3,0x01,0xd9,0x3b,0x3b,0xd9,0x03,0x03,0xd9,0xd9,0xd9,
  0x3b,0x3a,0x3b,0x3b,0x3b,0x3b,0x3b,0x01,0x02,0xd3,0x01,0x3b,0xd9,0x3b,0x3b,0x01,
  0x01,0x01,0x01,0xd3,0x6b,0x02,0x01,0x02,0xd3,0xd3,0xd3,0xb4,0x6b,0x25,0x25,0x6b,
  0x6b,0xd3,0xd3,0x6b,0x6b,0x6b,0x04,0x6b,0xb4,0x6b,0x6b,0x6b,0x04,0x25,0x0d,0xe9,
  0x0d,0x6b,0x6b,0x6b,0xd3,0xd3,0x6b,0x01,0x3b,0x3b,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x02,0x02,0x01,0xf0,0x01,0xd3,0x6b,0xcc,0x25,0xf4,0xf4,0xcc,0x04,0xf4,0x04,
  0x04,0xd3,0xd3,0xb4,0xf0,0xd3,0x04,0x04,0xd3,0x04,0xd3,0xd3,0xd3,0xf0,0xf0,0xd3,
  0xd3,0xd3,0xd3,0xf0,0xf4,0xf4,0xcc,0x0d,0x1c,0x1c,0xe9,0x1c,0x1c,0x1c,0x0d,0x0d,
  0xcc,0x04,0xf4,0x04,0x04,0xd3,0x3b,0x3b,0xd9,0xd9,0x03,0x3a,0x3a,0x3b,0x3b,0x3b,
  0x3b,0x3b,0x3b,0x3a,0x3b,0x3b,0x3b,0x01,0x02,0x02,0x01,0x3b,0x3a,0x3a,0x3b,0x3b,
  0x3b,0x01,0x3b,0x01,0x02,0x01,0x01,0x01,0xd3,0x6b,0x04,0x6b,0x04,0xf4,0xf4,0x6b,
  0x6b,0x02,0x6b,0x6b,0x6b,0xb4,0x04,0xb4,0x6b,0xcc,0x6b,0x6b,0x6b,0xcc,0x0d,0x25,
  0x04,0x6b,0x6b,0x6b,0xd3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x3b,0xee,0x3b,0x01,
  0x01,0x02,0xd3,0xd3,0x01,0x01,0xd3,0x6b,0x6b,0xcc,0x04,0xf4,0x04,0x04,0xf4,0x04,
  0x04,0x04,0xd3,0xd3,0xf0,0xf0,0xf0,0xd3,0xd3,0xd3,0xd3,0xd3,0xf0,0xf0,0x3b,0x3b,
  0x3b,0xf0,0x3b,0x3b,0x04,0x0d,0x0d,0x0d,0xe9,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x0d,
  0xcc,0x6b,0x6b,0x6b,0x04,0xd3,0x3b,0x03,0x88,0xd9,0x03,0x3a,0x3b,0x3b,0xd9,0xd9,
  0xd9,0xd9,0x3a,0x03,0x3a,0xd9,0xd9,0x3b,0x3b,0x3b,0x3b,0xd9,0xd9,0x03,0x3a,0x3a,
  0x3a,0x3b,0x3b,0x3b,0x3b,0x01,0x01,0x01,0x01,0x6b,0x6b,0x6b,0x6b,0x6b,0x04,0x6b,
  0x6b,0x6b,0x6b,0x6b,0xd3,0xd3,0xd3,0xd3,0x6b,0x6b,0x6b,0x6b,0x04,0xf4,0x6b,0x6b,
  0x6b,0x02,0x6b,0xd3,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x3b,0x3b,0x3b,0x3b,0x01,
  0x01,0x01,0x01,0x01,0x01,0x3b,0x02,0xb4,0x6b,0x6b,0xcc,0xcc,0x04,0xf4,0xcc,0x04,
  0x04,0xcc,0xcc,0xb4,0xd3,0xf0,0xf0,0xd3,0xd3,0xd3,0xd3,0xd3,0xf0,0x3b,0x3b,0x3b,
  0xfc,0xfc,0x3b,0xf0,0x04,0xe9,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x1c,0x0d,
  0x6b,0x6b,0x04,0x6b,0x04,0xd3,0x3b,0xd9,0x88,0x88,0xd9,0x3b,0x3b,0x03,0x03,0x88,
  0x88,0x03,0x03,0xd9,0xd9,0xd9,0x88,0x03,0x3a,0x3a,0xd9,0x03,0x03,0x03,0xd9,0xd9,
  0x03,0x03,0x3a,0x3a,0x3a,0x3b,0x3b,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x6b,0x6b,
  0x02,0x02,0x6b,0xd3,0xd3,0x02,0x01,0x01,0x01,0xd3,0xd3,0x02,0x6b,0x6b,0x02,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x3b,0x3b,0x00,0x3b,0x3a,0x3a,0x3a,0x3b,0x3b,
  0x01,0x01,0x3b,0x01,0x3b,0x3b,0x01,0xd3,0xb4,0x6b,0xb4,0x6b,0x04,0x04,0x04,0xb4,
  0xb4,0x6b,0xb4,0x04,0xb4,0xf0,0xf0,0xf0,0xd3,0xf0,0xf0,0x3b,0x3b,0x3b,0xfc,0xfc,
  0x3b,0xf0,0xf4,0xe9,0x0d,0x0d,0x0d,0x1c,0xe9,0x1c,0x0d,0xe9,0xe9,0x0d,0xcc,0xcc,
  0x04,0x04,0x6b,0x6b,0x02,0x02,0x02,0x01,0x3b,0x3a,0x3b,0x3b,0x3a,0x03,0x88,0x88,
  0x88,0xf9,0xfb,0x88,0x88,0x03,0x03,0x03,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x03,0x03,
  0x03,0x3a,0x01,0x02,0xd3,0xd3,0x01,0x01,0x01,0x01,0x02,0x02,0x6b,0xd3,0xcc,0x6b,
  0x02,0xd3,0x6b,0xd3,0x6b,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x3b,0x3a,0x3a,0x3b,0x01,0x3b,0x3a,0x3a,0x3a,0x3a,0x3a,0xd9,0xd9,0x3a,0x3b,
  0x01,0x01,0x3b,0x3b,0x3b,0x01,0x01,0x01,0x02,0xb4,0xd3,0x6b,0xcc,0xcc,0x6b,0xd3,
  0x01,0x02,0xb4,0x04,0xd3,0x02,0x3b,0x3b,0xf0,0xd3,0xd3,0x3b,0x3b,0x3b,0xf0,0xf0,
  0x3b,0xd3,0xf4,0x0d,0x0d,0x0d,0x0d,0xcc,0xe9,0x1c,0x0d,0xf4,0xcc,0xcc,0xf4,0xcc,
  0xf4,0x04,0x6b,0x6b,0xd3,0x01,0x02,0x01,0x01,0x3b,0x3b,0xf0,0x3b,0x03,0x88,0x88,
  0xf9,0xd0,0x88,0x03,0x03,0xd9,0x3a,0x3a,0x3a,0x3a,0x03,0x3a,0x3b,0x3b,0x3a,0x88,
  0x3a,0x01,0x6b,0x6b,0xd3,0x6b,0x6b,0x6b,0x01,0x01,0x01,0x02,0x6b,0x02,0xcc,0xcc,
  0x02,0x6b,0x6b,0x6b,0x02,0x02,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x01,0x01,0x01,
  0x3b,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x03,0x03,0x03,0x03,0x3a,0x3a,0x3a,
  0x3b,0x3b,0x01,0x3b,0x01,0x01,0x01,0x01,0x01,0x01,0xd3,0x6b,0x6b,0x04,0xb4,0x04,
  0x01,0x01,0x02,0xb4,0xd3,0xd3,0xf0,0x6b,0xd3,0xf0,0xf0,0xf0,0xf0,0xd3,0xd3,0xd3,
  0xd3,0x04,0xf4,0xf4,0xf4,0xf4,0xe9,0x0d,0x1c,0x1c,0x1c,0x0d,0xcc,0x04,0xf4,0xcc,
  0x04,0x6b,0x6b,0x6b,0xb4,0x02,0x01,0x68,0x01,0x3b,0x3a,0x3a,0x3a,0x88,0x88,0xd9,
  0x88,0xfb,0x03,0x03,0x03,0x3a,0x3b,0x3b,0x3b,0x3a,0x03,0x03,0x3a,0x3a,0x3a,0x3a,
  0x3a,0x01,0x01,0xd3,0xd3,0x02,0x6b,0x6b,0x02,0x02,0x02,0x6b,0x04,0x77,0x25,0xcc,
  0x6b,0x6b,0x6b,0xd3,0x02,0x01,0x01,0x01,0x3b,0x3b,0x3a,0x3a,0x3b,0x3b,0x3b,0x3a,
  0x3a,0x03,0x3a,0x03,0x03,0x03,0x88,0x03,0x03,0x03,0x03,0x03,0x03,0x3a,0x3a,0x3a,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x6b,0x04,0x04,
  0x01,0x00,0x01,0x01,0x01,0x3b,0x3b,0xf0,0xf0,0x3b,0x3b,0xf0,0xf0,0xf0,0xf0,0xf0,
  0xd3,0x04,0xd3,0x04,0xf4,0x04,0xf4,0x0d,0x1c,0xe9,0x1c,0x0d,0xcc,0x04,0x6b,0x04,
  0x6b,0x6b,0x6b,0x6b,0x02,0x01,0x3b,0x3b,0x3a,0x3b,0x3a,0x03,0x88,0x88,0xd0,0x88,
  0x88,0xd0,0x88,0x03,0x03,0x03,0x3a,0x3a,0x3a,0x3a,0x03,0x03,0x03,0x3a,0x3b,0x3b,
  0x01,0x01,0x01,0x02,0x6b,0x02,0x01,0x01,0x01,0x02,0x6b,0x6b,0x6b,0x04,0xcc,0xcc,
  0x6b,0x02,0x01,0x01,0x01,0x01,0x01,0x00,0x3a,0x3a,0x3a,0x3a,0x3b,0x3a,0x3a,0x03,
  0x03,0x03,0x88,0x03,0x03,0x03,0x88,0x88,0x88,0x88,0x88,0x03,0x03,0x3a,0xd9,0x3a,
  0x3a,0x3b,0x01,0x3b,0x01,0x01,0x01,0x01,0x01,0x01,0xd3,0x02,0x04,0xf4,0x04,0xcc,
  0x6b,0x01,0x00,0x01,0xd3,0xf0,0x3b,0x3b,0xd9,0xd9,0xd9,0x3b,0x3b,0x3b,0x3b,0x3b,
  0x3b,0xf0,0xf0,0xf4,0xf4,0xf4,0xf4,0x0d,0xe9,0xe9,0x0d,0xcc,0xf4,0x04,0x04,0x6b,
  0x6b,0x01,0x01,0x01,0x01,0x00,0x3a,0x03,0x03,0xd9,0x3a,0x3a,0x88,0x88,0x88,0x03,
  0x03,0x03,0x03,0x3a,0x3a,0xd9,0x03,0x3a,0x03,0x88,0x88,0x88,0x3a,0x3b,0x01,0x02,
  0x02,0x6b,0x02,0x6b,0xcc,0xd3,0x01,0x01,0x02,0x02,0x02,0x6b,0x6b,0xd3,0x02,0x6b,
  0x01,0x00,0x00,0x00,0x00,0x3b,0x3b,0x01,0x3b,0x3a,0x03,0x3a,0x3a,0x3a,0x88,0x88,
  0x88,0xd0,0xd0,0xd0,0x88,0x88,0x88,0x88,0x88,0xd0,0xd0,0x03,0xd9,0x3a,0x3a,0x03,
  0x3a,0x3a,0x3a,0x3a,0x01,0x01,0x01,0x01,0x01,0x02,0xb4,0xcc,0xcc,0xb4,0x6b,0x04,
  0xf4,0xf4,0xb4,0xb4,0xd3,0x3b,0xd9,0xf9,0xf8,0xd9,0xd9,0xd9,0xd9,0xd9,0x3b,0xf0,
  0x3b,0x01,0xb4,0x04,0xf4,0xcc,0xcc,0x04,0x0d,0xe9,0xcc,0xcc,0x25,0x04,0x6b,0x6b,
  0x01,0x01,0x3b,0x00,0x3b,0x3a,0x03,0x03,0x3a,0x01,0x3b,0x3a,0x3a,0x3a,0x3b,0x01,
  0x3b,0xd9,0xd9,0x03,0x88,0x03,0x88,0x03,0xd9,0x03,0x88,0x88,0x03,0x3a,0x00,0x01,
  0x01,0x02,0x6b,0x6b,0xcc,0xf4,0x02,0x00,0x01,0x01,0x00,0x01,0x01,0xd3,0x04,0xd3,
  0x3a,0x03,0x3a,0x3a,0x00,0x3b,0x3b,0x3b,0x3b,0x3a,0x3a,0x3a,0x3b,0xd9,0x88,0x88,
  0xd0,0xf9,0xf9,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x3a,0x03,0x03,0x03,
  0x03,0x03,0x03,0x3a,0x3b,0x01,0x01,0x02,0x01,0x3b,0x01,0xb4,0xd3,0xd3,0xb4,0xb4,
  0xb4,0x04,0xf4,0x04,0xf0,0xd9,0xf8,0xd9,0xfc,0xfc,0x3b,0x3b,0xf0,0x01,0xd3,0xd3,
  0x6b,0xb4,0xcc,0x6b,0x04,0x04,0x04,0xcc,0x0d,0xe9,0x0d,0xcc,0x0d,0xf4,0x6b,0x02,
  0x01,0x02,0x01,0x3a,0x03,0x3a,0x00,0x3b,0x3b,0x00,0x3a,0xd9,0x3a,0x00,0x01,0x01,
  0x3b,0x3b,0x3b,0x3b,0x3a,0x03,0x03,0x88,0x88,0x03,0x88,0x03,0x3a,0x3b,0x01,0x01,
  0x02,0x6b,0x04,0x04,0x6b,0x6b,0x6b,0x01,0x01,0x01,0x01,0x02,0x02,0x04,0x04,0x3b,
  0x03,0x03,0x03,0x3a,0x3b,0x01,0x3a,0x3a,0x3a,0x3a,0x03,0x3a,0x3a,0x03,0x88,0x88,
  0x88,0xd0,0xd0,0x88,0x88,0x88,0x03,0x88,0x88,0xd0,0x88,0xca,0x03,0x03,0x3a,0x3a,
  0x03,0x03,0x3a,0x00,0x01,0x02,0x01,0x01,0x00,0x00,0x01,0x01,0x6b,0x04,0x6b,0x6b,
  0xb4,0xd3,0xb4,0xb4,0x3b,0xf8,0xfc,0x3b,0xfc,0xf0,0x04,0xb4,0x04,0x6b,0xb4,0xd3,
  0x02,0xd3,0x6b,0xcc,0xcc,0x04,0x6b,0x6b,0x04,0xcc,0xcc,0xcc,0x77,0x6b,0x02,0x01,
  0x01,0x01,0x00,0x3a,0x03,0x00,0x01,0x00,0x3a,0x3a,0x03,0x3a,0x3a,0x3a,0x3b,0x01,
  0x01,0x3b,0x3b,0x3b,0x3a,0x88,0x3a,0x3a,0x03,0x88,0xd9,0x3b,0x3b,0x00,0x01,0x01,
  0x02,0x02,0x6b,0x6b,0x02,0x02,0x01,0x02,0x6b,0x6b,0x6b,0xd3,0x02,0xd3,0x01,0x03,
  0x03,0x88,0x03,0x3a,0x3b,0x00,0x3a,0x3a,0x88,0x3a,0x03,0x88,0x88,0x88,0xd0,0xf9,
  0xd0,0xd0,0xd0,0x88,0x88,0x88,0x88,0x88,0xd0,0xd0,0xd0,0x03,0x03,0x3a,0x01,0x00,
  0x00,0x00,0x01,0x02,0x6b,0x02,0xdd,0x01,0x01,0x00,0x3a,0x3a,0x00,0xd3,0x04,0x04,
  0xcc,0xd3,0x6b,0x6b,0xf0,0x3b,0x3b,0x3b,0x3b,0xd3,0x04,0xf4,0xcc,0x6b,0x6b,0xd3,
  0x6b,0xb4,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x02,0x6b,0x6b,0x77,0x04,0x6b,0x02,0x01,
  0x00,0x00,0x01,0x01,0x01,0x01,0x00,0x3a,0x3a,0x3a,0x3a,0x03,0x3a,0xd9,0x3a,0x01,
  0x01,0x00,0x01,0x01,0x01,0xd9,0x03,0x03,0x3a,0x3a,0x01,0x3b,0x3b,0x3b,0x01,0x02,
  0x02,0x02,0x6b,0x04,0x6b,0x6b,0x6b,0x6b,0x6b,0x04,0x6b,0xd3,0x02,0x02,0x01,0x3a,
  0x03,0x03,0x03,0x3a,0x3a,0x3a,0x88,0x03,0x88,0x88,0x88,0x03,0x88,0xd0,0xd0,0xd0,
  0x88,0x88,0xd0,0x88,0x88,0x03,0x88,0xd0,0xd0,0xd0,0x88,0x03,0x3a,0x00,0x01,0x01,
  0x02,0x02,0x02,0x6b,0x6b,0xf2,0x01,0x00,0x00,0x03,0x03,0x03,0x3a,0x01,0x6b,0x6b,
  0x6b,0x6b,0xd3,0x01,0x3b,0xf0,0x3b,0x3b,0x3b,0xd3,0xf4,0xcc,0x6b,0xd3,0xd3,0x6b,
  0xcc,0xf4,0x04,0x6b,0x6b,0x02,0xd3,0x02,0x02,0x02,0x02,0x02,0x6b,0x01,0x68,0x01,
  0x3a,0xd9,0x00,0x3a,0x00,0x00,0x00,0x3a,0x3a,0xd9,0x03,0x3a,0x00,0x01,0x01,0xd3,
  0x02,0x01,0x01,0x3b,0x01,0xd9,0x3a,0x3b,0x3b,0x3b,0x3a,0x3b,0x3b,0x3b,0x01,0x01,
  0x02,0x6b,0xcc,0xcc,0x04,0x04,0x77,0x04,0x04,0x6b,0x6b,0xd3,0x01,0x00,0x3a,0x03,
  0x03,0x03,0x03,0x03,0x3a,0x3a,0x03,0x88,0xd0,0x03,0x3a,0x3a,0x03,0xca,0x88,0x3a,
  0x3a,0x03,0xca,0x88,0x88,0x88,0x88,0xd0,0xd0,0x88,0x03,0x3a,0x00,0x01,0x02,0x6b,
  0x6b,0x02,0x02,0xdd,0x01,0x00,0x00,0x3a,0x03,0x03,0x03,0x03,0x3a,0x00,0x01,0x02,
  0x02,0x6b,0xb4,0x3b,0xd9,0xd9,0x3b,0x3b,0xfc,0x04,0xf4,0xcc,0xb4,0xb4,0xd3,0xd3,
  0x6b,0xcc,0x6b,0x02,0x01,0x01,0x01,0x02,0x02,0x02,0xdd,0x01,0x3b,0x00,0x3a,0x3a,
  0x03,0x03,0x3a,0xd9,0x00,0x01,0x01,0x00,0x00,0x01,0x3a,0x01,0x01,0x02,0x02,0xd3,
  0xd3,0x01,0x01,0x00,0x3b,0x3b,0x03,0x3a,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x01,0xd3,
  0x6b,0x04,0xcc,0xcc,0x6b,0x6b,0x6b,0x6b,0xb4,0xd3,0xd3,0xd3,0x01,0x03,0x03,0x03,
  0x03,0x3a,0x3a,0x03,0x03,0xca,0x03,0x03,0x3a,0x3a,0x3a,0x88,0x03,0x03,0x88,0x3a,
  0x3a,0x03,0x88,0x3a,0x3a,0x88,0x88,0xd0,0x88,0x03,0x3a,0x00,0x01,0x01,0x02,0x02,
  0xf2,0xdd,0x01,0x00,0x00,0x3a,0x3a,0x03,0x03,0xca,0xca,0x03,0x3a,0x00,0x00,0x01,
  0x02,0x6b,0xcc,0xf0,0x3b,0x3b,0x3b,0x3b,0x3b,0x04,0xcc,0x6b,0xd3,0xd3,0xb4,0xb4,
  0x6b,0xd3,0xd3,0x01,0x01,0x02,0x02,0x02,0xb4,0xd3,0x01,0x3a,0x3a,0x3a,0x3a,0x3a,
  0x3a,0x03,0xd9,0x3b,0x01,0x01,0x01,0xd3,0x02,0xdd,0x01,0x01,0x00,0x3b,0x00,0x01,
  0x01,0x01,0x00,0x3a,0x3a,0x3b,0x3b,0xd9,0x00,0x3b,0x3b,0x3b,0x3b,0x01,0x01,0x01,
  0x02,0xd3,0x02,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0x02,0x68,0x3a,0x03,0x03,
  0x03,0x03,0x3a,0x03,0xca,0xca,0x03,0x3a,0x88,0x03,0x03,0x88,0x88,0x03,0x03,0x3a,
  0x00,0x3a,0x3a,0x3a,0x3a,0x03,0x88,0xca,0x03,0x00,0x00,0x01,0x01,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0x00,0xee,0x00,0x3a,0x3a,0x3a,0x3a,0x03,0xeb,0x3a,0x3a,0x01,
  0x01,0x02,0x6b,0xd3,0xf0,0xd3,0xd3,0xd3,0x04,0xf4,0x25,0xcc,0xd3,0xd3,0xd3,0xd3,
  0xb4,0x6b,0x01,0x02,0x02,0x6b,0x77,0xcc,0x6b,0x02,0x3a,0x03,0x88,0x3a,0x3b,0x3a,
  0x3a,0x88,0x88,0x03,0xd9,0x00,0x00,0xdd,0x02,0x01,0x02,0x6b,0x01,0x3a,0x3b,0x00,
  0x01,0x00,0x3b,0x00,0x00,0x3b,0x00,0x3a,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0xf0,
  0x01,0x01,0x01,0x01,0xf0,0x01,0x02,0xd3,0x68,0x01,0x3b,0x01,0xd3,0x3a,0x03,0x03,
  0x03,0x03,0x3a,0x3a,0x03,0x03,0xd9,0x88,0x03,0x88,0x03,0x88,0x03,0xca,0x88,0x3a,
  0x3a,0x3a,0x03,0x3a,0x3b,0x3a,0x03,0x3a,0x3a,0x00,0x00,0x01,0x01,0x00,0x00,0x00,
  0x00,0x01,0x01,0x01,0x01,0xf2,0x01,0xee,0x00,0x3a,0x00,0x3a,0x03,0x3a,0x00,0x01,
  0x02,0x6b,0xd3,0x04,0xf4,0xd3,0xd3,0xf4,0x0d,0x0d,0xf7,0x6b,0xd3,0xb4,0xb4,0x6b,
  0x02,0x01,0x02,0x6b,0x6b,0x25,0xe9,0x0d,0x04,0x04,0x01,0x3a,0x3a,0x3a,0x01,0x00,
  0x3b,0x88,0x88,0x3a,0x3a,0x3a,0x00,0x02,0x02,0x01,0x01,0x01,0x01,0x3a,0x3a,0x3a,
  0xd9,0xd9,0x3a,0x3a,0x3b,0x3a,0xee,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0xf0,0x01,
  0xf0,0xf0,0x68,0x01,0x02,0xf0,0xf0,0xf0,0x3b,0x01,0x3b,0xf0,0x68,0x88,0x03,0x03,
  0x03,0x03,0x03,0x03,0x88,0x03,0x03,0x03,0x88,0x03,0xfb,0xca,0xca,0x03,0x3a,0x3a,
  0x00,0x3b,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x00,0x00,0x01,0x00,0x00,0x00,0x00,
  0x00,0x00,0xee,0x00,0xee,0x00,0x3a,0x03,0x03,0x03,0x03,0x03,0x88,0x00,0xee,0x01,
  0x02,0x02,0xd3,0xb4,0x04,0xb4,0xb4,0xf4,0x0d,0x0d,0xcc,0xd3,0x6b,0x6b,0x02,0x01,
  0x01,0xd3,0x6b,0x04,0xcc,0x0d,0x0d,0xf7,0xcc,0x77,0x6b,0xdd,0x00,0x00,0x01,0x00,
  0xd9,0x88,0xfb,0xca,0xca,0x3a,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x3a,0x3a,0x3b,
  0x3a,0x3b,0x00,0x3b,0x00,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x00,0x3b,0x3b,0x3b,0x01,
  0x01,0xf0,0xf0,0x68,0xf0,0x3b,0x3b,0x3b,0x3a,0x3b,0x3b,0x3b,0x3a,0x03,0x3a,0x03,
  0x03,0xca,0xca,0xca,0xca,0x03,0x3a,0x03,0xca,0x03,0x03,0x03,0x3a,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,0x3a,0x00,0x3a,0x3a,0x3a,0x00,0x00,0x00,0x00,0x3a,0x00,0x00,
  0x00,0xee,0x01,0xdd,0xee,0x03,0xca,0xca,0xd0,0xca,0x03,0x03,0x3a,0x00,0x01,0xdd,
  0xf2,0xf2,0x02,0x6b,0x6b,0x6b,0x25,0x0d,0xe9,0x0d,0xcc,0x6b,0x6b,0x02,0x02,0x01,
  0x01,0xd3,0x6b,0x6b,0x77,0x77,0x77,0x77,0x77,0x6b,0x6b,0x6b,0x02,0x01,0xdd,0x68,
  0x68,0x3a,0xca,0x9d,0xca,0x00,0x01,0x01,0x00,0x3a,0x00,0x00,0x3a,0x3a,0x3a,0x00,
  0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3a,0xd9,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,
  0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0xf0,0x3b,0x88,0x03,0x03,0x03,
  0xca,0xca,0xd0,0x03,0x3a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3a,0x3a,0x00,
  0x00,0x00,0xee,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0x3a,0xeb,0xeb,0x3a,0xeb,
  0x00,0x00,0x00,0x03,0xd0,0xd0,0xf9,0xca,0xca,0xeb,0x3a,0x00,0x01,0xf2,0xf2,0xf2,
  0xf2,0x6b,0x04,0xcc,0x77,0xcc,0x0d,0x0d,0x0d,0xcc,0xcc,0x6b,0x6b,0x02,0x02,0x02,
  0x01,0x01,0x02,0x02,0x6b,0x6b,0x6b,0xf2,0x6b,0x02,0x02,0x02,0xdd,0x00,0x00,0x01,
  0x00,0x00,0x00,0xfb,0x03,0x3a,0x3a,0x00,0x3a,0x00,0x00,0x00,0x00,0x00,0x3b,0x3b,
  0x3b,0x3b,0x3b,0x3a,0x3b,0x3b,0x3a,0xd9,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x01,0x3b,
  0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x00,0x00,0x3b,0x3b,0x3a,0x3a,0xd9,0xd9,
  0x3a,0x03,0xca,0xca,0x03,0x3a,0x3a,0x00,0x00,0x00,0x00,0x01,0x01,0x3a,0x3a,0x3a,
  0x3a,0x00,0x00,0x00,0x3a,0xeb,0x03,0xeb,0xeb,0xca,0xca,0xeb,0xeb,0xca,0xca,0x03,
  0x3a,0x03,0xca,0xd0,0x9d,0x9d,0xd0,0xca,0x03,0x00,0xee,0x01,0xf2,0xf2,0x6b,0x6b,
  0xcc,0x25,0x25,0x25,0xcc,0x25,0xcc,0xcc,0x6b,0x6b,0x6b,0xdd,0x02,0x01,0x01,0x01,
  0x01,0x00,0x01,0xdd,0x6b,0x6b,0x6b,0x6b,0x02,0x02,0xdd,0x02,0x01,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,0xd9,0x3a,0x3a,0x00,0x3a,0x00,0x3a,0x00,0x01,0x00,0x3b,0x00,
  0x00,0x3a,0x3a,0x3a,0x3a,0x3a,0x00,0x3b,0x3b,0x01,0x3b,0x3a,0x00,0x68,0x01,0x3b,
  0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x00,0x3b,0x3a,0x3a,0x01,0x01,0x00,0x3a,0x3a,0x00,
  0x3a,0x03,0xca,0x88,0x03,0x88,0x03,0x3a,0x01,0x00,0x00,0x00,0x00,0x3a,0x3a,0x3a,
  0x3a,0x3a,0x3a,0x3a,0xeb,0x03,0xca,0xca,0xca,0xca,0xca,0xca,0x9d,0x9d,0xd0,0x03,
  0xca,0xd0,0x9d,0xd0,0x9d,0x9d,0xd0,0x03,0x00,0xee,0x01,0xdd,0xdd,0x6b,0xcc,0x0d,
  0x0d,0x25,0x77,0xcc,0xcc,0x6b,0x02,0xdd,0x01,0x02,0x02,0xdd,0x01,0x00,0x3a,0x00,
  0x00,0xee,0x00,0x00,0xdd,0x02,0x02,0x01,0x68,0x01,0x00,0x00,0xdd,0xdd,0x00,0xee,
  // 10: sky 1, part C
  0x5a,0xbe,0xdc,0xc6,0xfa,0xad,0x3b,0xa6,0x00,0x00,0x3a,0xb6,0x08,0x00,0x10,0x00,
  0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,
  0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,
  0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,
  0x18,0x06,0xb9,0xa5,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,
  0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,
  0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,
  0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,
  0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,
  0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x5c,0xb6,0x00,0x42,
  0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,
  0x08,0x44,0x10,0x44,0x99,0x95,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,
  0x00,0x46,0x08,0x46,0x7b,0xc6,0x9c,0xbe,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,
  0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,0x39,0x8d,0x18,0x60,0x00,0x61,
  0x9b,0xc6,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,
  0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,
  0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,
  0x00,0x67,0x08,0x67,0x10,0x67,0xbc,0xce,0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,
  0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,
  0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,
  0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x5e,0xdf,0x00,0x86,0x08,0x86,0x1e,0xdf,
  0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,
  0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,
  0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,0x08,0xa4,
  0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,0x18,0xa5,0x00,0xa6,0x08,0xa6,
  0x10,0xa6,0x18,0xa6,0xfc,0xd6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0x7c,0xae,
  0x00,0xc0,0x08,0xc0,0xba,0x9d,0x3d,0xd7,0x10,0xc0,0x18,0xc0,0x00,0xc1,0x08,0xc1,
  0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,
  0x10,0xc3,0xbf,0xe7,0xdb,0xce,0xf9,0xad,0x18,0xc3,0x00,0xc4,0x08,0xc4,0x7c,0xc6,
  0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0xfa,0xb5,0xd9,0x9d,0x10,0xc5,0x37,0x95,
  0x19,0xbe,0x18,0xc5,0x00,0xc6,0x36,0x8d,0x35,0x95,0x10,0xc6,0x35,0x8d,0xfb,0x9d,
  0x18,0xc6,0x7f,0xe7,0x3c,0xae,0x00,0xe0,0x08,0xe0,0xdc,0x8d,0x77,0x9d,0x9b,0xbe,
  0x4e,0x03,0x03,0x03,0x03,0xfa,0x02,0x03,0xfa,0x02,0x03,0x02,0x02,0x03,0x02,0x03,
  0x02,0x21,0x03,0x02,0x03,0xcf,0x63,0x8b,0x4e,0x03,0xfa,0x4e,0x4e,0x4e,0x4e,0x03,
  0x03,0x03,0xfa,0xfa,0x03,0x03,0x02,0x03,0x03,0x03,0xfa,0x4e,0x4e,0x03,0x02,0x03,
  0x02,0xd2,0x5a,0x6d,0x5a,0x5a,0xd2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0xd2,
  0x03,0x03,0x03,0x02,0xd2,0x6d,0x6d,0x6d,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,0xd2,0x5a,
  0x6d,0x6d,0x6d,0x6d,0x6d,0x6d,0x5a,0x03,0x02,0x4e,0xff,0x4e,0x63,0x8b,0xd3,0xf9,
  0xd3,0xa7,0xa7,0x01,0x8b,0x01,0xff,0x02,0x02,0x4e,0xe7,0x4e,0x05,0x02,0x02,0x21,
  0x02,0x05,0x05,0x02,0x03,0x03,0x02,0x02,0x02,0xfa,0x03,0x03,0x4e,0xff,0x03,0x03,
  0x4e,0x4e,0x03,0x02,0x03,0x21,0x02,0x4e,0x03,0x02,0x02,0x02,0x21,0x21,0x03,0xfa,
  0x03,0x02,0x02,0x03,0xcf,0x4e,0x4e,0x01,0x01,0x01,0x4e,0x03,0xcf,0x4e,0xcf,0x4e,
  0xfa,0x03,0x03,0xf7,0x03,0xfa,0xfa,0x03,0x03,0x02,0x03,0x03,0x4e,0xcf,0x4e,0x03,
  0x03,0x02,0xd2,0x6d,0x6d,0x5a,0x02,0x02,0x02,0x21,0x02,0x02,0x02,0x02,0xd2,0x03,
  0xf7,0xf7,0x03,0xd2,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,0x6d,0x5a,0x5a,0x5a,0x02,0xd2,
  0x5a,0xd2,0xd2,0x03,0x5a,0x5a,0xd2,0x03,0x02,0x4e,0x4e,0x4e,0xcf,0x01,0xa4,0xa4,
  0xa7,0xa7,0xa7,0xa7,0x01,0x4e,0x05,0x02,0x02,0x4e,0x05,0x05,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x21,0x21,0xd2,0x02,0x02,0x02,0x03,0xff,0x63,0xff,
  0xcf,0x02,0x02,0x02,0x03,0x02,0x03,0x4e,0x4e,0x03,0x03,0x02,0x02,0x02,0x02,0x03,
  0xd2,0xf7,0x03,0xfa,0x4e,0x02,0xd2,0x03,0x03,0x63,0x63,0x4e,0x4e,0xe7,0x4e,0x03,
  0x03,0x03,0xfa,0xcf,0x4e,0x4e,0xcf,0x8b,0x01,0xe7,0x4e,0x4e,0x4e,0x4e,0x4e,0x02,
  0x02,0x02,0x02,0xd2,0x5a,0xd2,0x21,0x03,0x02,0x21,0x02,0x02,0xd2,0xd2,0xd2,0x03,
  0x03,0x03,0xd2,0x5a,0x5a,0x6d,0x6d,0x6d,0x5a,0x5a,0x21,0x21,0x02,0x02,0x02,0x03,
  0x5a,0xd2,0x03,0x03,0x03,0x02,0x02,0x03,0x4e,0x4e,0x03,0x4e,0xcf,0x01,0xf9,0xe1,
  0xa7,0xa7,0x01,0x01,0x63,0x4e,0x03,0x02,0x05,0x05,0x05,0x05,0x02,0x02,0x03,0x02,
  0x02,0x02,0x21,0x21,0xed,0x21,0xed,0xed,0x02,0x02,0x03,0x03,0x03,0x4e,0xff,0x4e,
  0x03,0x03,0x03,0x4e,0x03,0xcf,0xcf,0xcf,0xff,0x4e,0xcf,0x4e,0x03,0xfa,0x03,0x03,
  0x03,0x02,0x03,0xcf,0xcf,0xd2,0x5a,0xd2,0x02,0xcf,0xa7,0xa7,0xa7,0x01,0x01,0x63,
  0x4e,0xcf,0x4e,0x4e,0xfa,0x03,0x03,0x03,0x4e,0x63,0x01,0x4e,0x4e,0xcf,0xfa,0x02,
  0x21,0x21,0x21,0x21,0xd2,0xd2,0xd2,0x5a,0x5a,0x5a,0x6d,0x6d,0x5a,0xd2,0xd2,0x03,
  0x02,0x21,0x21,0x5a,0x6d,0x6d,0x5a,0x21,0x21,0x21,0x21,0xec,0x02,0x02,0x02,0x03,
  0xd2,0xd2,0xf7,0xd2,0xd2,0x03,0x05,0x03,0x4e,0x03,0x4e,0xe7,0xe7,0xd3,0xa4,0xa4,
  0x8b,0x8b,0x01,0xe7,0xff,0x4e,0x4e,0x05,0x02,0x05,0x05,0x05,0x05,0x05,0x05,0x02,
  0x02,0x21,0xed,0x5a,0x5a,0x5a,0x21,0x02,0x02,0x02,0x03,0x05,0x03,0x4e,0x03,0x03,
  0x4e,0x63,0x4e,0x4e,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x4e,0x4e,0x4e,0xcf,0x4e,
  0xcf,0xcf,0x63,0x01,0x4e,0x5a,0x5a,0x21,0x03,0x4e,0x63,0xd3,0x01,0x63,0x63,0x63,
  0x63,0xcf,0x63,0x03,0x03,0x02,0x02,0x03,0x03,0x63,0x63,0xcf,0x4e,0x02,0x03,0xd2,
  0x5a,0x5a,0x5a,0xd2,0x21,0xd2,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,0x6d,0x5a,0x5a,0x21,
  0x02,0x02,0x21,0x5a,0x6d,0x5a,0x21,0x21,0x21,0x02,0x02,0x02,0xec,0x02,0x02,0x03,
  0xd2,0x5a,0x5a,0xf7,0xd2,0xd2,0x03,0x03,0xff,0x63,0x63,0xe7,0x01,0x8b,0xca,0x8b,
  0x8b,0x01,0x63,0x63,0xff,0x4e,0x05,0x02,0x02,0x02,0x05,0x02,0x05,0xff,0x03,0x02,
  0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0xd2,0x21,0x02,0x02,0x02,0x02,0x02,0x03,0x4e,0x4e,
  0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0xcf,0xcf,0x4e,0x4e,0xcf,
  0x4e,0x4e,0x4e,0x4e,0xfa,0x5a,0x5a,0x02,0x02,0x4e,0xff,0x01,0x63,0xe7,0x4e,0x03,
  0x63,0xa7,0xa7,0x4e,0x4e,0x03,0x02,0x02,0x02,0x03,0x4e,0xcf,0x03,0x03,0x03,0x21,
  0x5a,0x5a,0xd2,0x5a,0x5a,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,0x5a,0x5a,0x6d,0x5a,0x21,
  0x02,0xec,0x02,0x5a,0x21,0x21,0x21,0x02,0x02,0x02,0x02,0x02,0x5a,0x6d,0x6d,0x5a,
  0xf7,0xf7,0x03,0x03,0xd2,0xf7,0x03,0x63,0x8b,0x8b,0xe7,0xe7,0x01,0x01,0x01,0x8b,
  0x8b,0x8b,0x63,0xe7,0xff,0x00,0xff,0x05,0x02,0x02,0x05,0x05,0xff,0x70,0xff,0x02,
  0x21,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x21,0x21,0x03,0x02,0x02,0x03,0x4e,0x4e,0x4e,
  0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x63,0x4e,0xcf,0x4e,0xcf,
  0x03,0x4e,0x02,0x02,0x02,0xed,0x21,0xed,0x21,0x03,0x4e,0x4e,0x01,0x01,0xa7,0x01,
  0x8b,0x63,0x63,0x4e,0x4e,0x03,0x05,0x02,0x02,0x03,0x4e,0x4e,0x03,0x4e,0x03,0x02,
  0x5a,0x5a,0x21,0x6d,0x5a,0x5a,0x5a,0x6d,0x6d,0x6d,0x5a,0x21,0x21,0x21,0x21,0x21,
  0x21,0x21,0x02,0xec,0x02,0x02,0x21,0x21,0x02,0x02,0x21,0x21,0x5a,0x5a,0x03,0x02,
  0xd2,0x03,0x03,0x4e,0x4e,0x4e,0x4e,0x8b,0xca,0x8b,0x70,0x70,0x01,0x8b,0x8b,0x01,
  0x63,0x63,0xff,0xff,0x63,0x00,0x4e,0x4e,0x4e,0x05,0x05,0x4e,0x01,0x8b,0x8b,0x4e,
  0x05,0xed,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0xd2,0xd2,0x02,0x03,0x03,0x4e,0x4e,0x63,
  0x63,0x63,0x63,0x63,0xff,0x4e,0xff,0x4e,0x4e,0x4e,0x63,0x63,0x63,0x4e,0x4e,0x4e,
  0x4e,0x03,0xed,0xed,0x21,0x5a,0x5a,0x5a,0x5a,0x21,0x02,0x4e,0x4e,0x4e,0x4e,0x01,
  0x63,0x63,0x4e,0x63,0xe7,0x02,0x02,0x03,0x02,0xf7,0x4e,0xe7,0xcf,0x4e,0xcf,0xd2,
  0x5a,0x5a,0x5a,0x6d,0x6d,0x5a,0x21,0x5a,0x21,0x21,0x21,0x21,0x02,0xec,0xec,0xec,
  0x02,0x02,0x02,0x21,0x21,0x21,0x21,0x02,0xff,0xe7,0xff,0x4e,0x05,0x05,0x4e,0x4e,
  0x03,0x03,0x03,0x4e,0x63,0x63,0xe7,0x8b,0xca,0x8b,0x01,0x01,0x8b,0x8b,0x01,0x70,
  0x63,0xff,0x00,0xff,0xff,0xff,0xff,0x00,0xff,0x4e,0xff,0x01,0xa7,0xd3,0xd3,0x70,
  0x05,0x21,0x21,0x5a,0x5a,0x5a,0x5a,0x5a,0x21,0x21,0x02,0x03,0x03,0xcf,0xff,0x63,
  0x4e,0x4e,0x4e,0x4e,0x4e,0xcf,0x4e,0x4e,0xff,0xff,0x4e,0x63,0xcf,0xcf,0xcf,0x4e,
  0x03,0x02,0x21,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x21,0x02,0xff,0xff,0x02,0x05,0x4e,
  0x4e,0x01,0x01,0x63,0x01,0x05,0x00,0x03,0xfd,0xcf,0x4e,0x4e,0x4e,0x4e,0x03,0xfd,
  0xfd,0x6d,0x5a,0x5a,0x6d,0x6d,0x5a,0x21,0x5a,0x21,0xec,0x02,0x05,0x02,0x21,0x21,
  0x21,0x21,0x5a,0x6d,0x5a,0x05,0x00,0x4e,0x00,0x05,0x02,0x02,0x21,0x02,0x02,0x03,
  0xd2,0x03,0x03,0x03,0x03,0x4e,0x63,0x01,0x01,0x63,0xe7,0xe7,0x70,0x01,0x70,0x70,
  0xff,0xff,0x63,0x63,0x8b,0x01,0x70,0xff,0x70,0x70,0x01,0xa7,0xd3,0xd3,0x8b,0xff,
  0x02,0x21,0x02,0x02,0xed,0x21,0x21,0xed,0xed,0x02,0x02,0x03,0xcf,0xff,0xff,0x4e,
  0x03,0x02,0x03,0x03,0x03,0xcf,0x4e,0x4e,0xff,0xff,0x4e,0xff,0x63,0x63,0x4e,0x02,
  0x03,0x21,0x5a,0x5a,0x5a,0x5a,0xed,0x02,0x05,0x4e,0x4e,0x00,0x4e,0x02,0x02,0x4e,
  0xa7,0xa7,0x01,0x4e,0x4e,0xf7,0x03,0x03,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xfd,0xfd,
  0xfd,0xf7,0x03,0x21,0x21,0x5a,0x21,0x02,0x21,0xec,0x21,0x5a,0x5a,0x6d,0x6d,0x6d,
  0x6d,0x6d,0x6d,0x6d,0x05,0x05,0x02,0x21,0x21,0x21,0x21,0x5a,0x6d,0x6d,0x5a,0x6d,
  0x6d,0x6d,0x6d,0x6d,0xf7,0x4e,0x63,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
  0xe7,0x8b,0x8b,0x8b,0x8b,0x01,0x01,0x70,0x00,0xe7,0x8b,0xd3,0xd3,0xd3,0x8b,0x01,
  0x05,0x02,0x02,0x02,0x21,0x02,0xed,0x02,0x02,0x02,0x03,0x4e,0xff,0x4e,0x03,0x05,
  0x4e,0xff,0x4e,0x4e,0x4e,0x4e,0xff,0x4e,0x4e,0xff,0xcf,0x4e,0x4e,0x4e,0x63,0x63,
  0xcf,0x02,0x21,0x21,0x02,0x4e,0x05,0x62,0xff,0x70,0x01,0xff,0x05,0x05,0x4e,0x01,
  0xa7,0x63,0xcf,0xcf,0xcf,0x03,0xfa,0xfd,0xcf,0xcf,0xcf,0xcf,0xcf,0x63,0x63,0x03,
  0xfd,0x4e,0x4e,0x00,0xec,0x02,0x21,0x21,0xec,0x02,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,
  0x6d,0x6d,0x6d,0xd2,0x05,0x02,0x21,0x5a,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,0xfd,0xf7,
  0xf7,0xf7,0xfd,0xfd,0xf7,0x4e,0x03,0xcf,0x4e,0x4e,0x63,0xff,0xff,0x4e,0x70,0xff,
  0x8b,0xd3,0xd3,0xd3,0x8b,0x01,0x70,0xff,0x00,0x70,0x8b,0x70,0x01,0xa7,0xd3,0x01,
  0x00,0x05,0x02,0x05,0x05,0x05,0x05,0x05,0x4e,0x05,0x70,0x70,0x01,0xff,0x4e,0x05,
  0x02,0x4e,0xff,0x63,0x63,0xff,0xff,0xff,0xff,0x4e,0xcf,0x4e,0x4e,0x4e,0x63,0x63,
  0x63,0x01,0x01,0x8b,0x70,0xff,0x00,0x4e,0x70,0x8b,0x01,0x01,0x00,0xff,0x01,0x01,
  0x4e,0xcf,0xfd,0xfd,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0x63,0x63,0x4e,0x4e,
  0x01,0x4e,0x63,0x63,0x03,0x02,0x02,0x02,0x02,0x5a,0x6d,0x6d,0x6d,0x6d,0x6d,0x6d,
  0x6d,0x6d,0x6d,0x00,0x4e,0x05,0x02,0x02,0xd2,0x5a,0x6d,0x5a,0x5a,0x6d,0xf7,0xcf,
  0x63,0x03,0xfd,0xcf,0xcf,0x63,0x4e,0xcf,0xcf,0xff,0x01,0x63,0x4e,0x4e,0xff,0x01,
  0x01,0x8b,0xd3,0xca,0x01,0xff,0x05,0x05,0x70,0x01,0xd3,0x8b,0x8b,0x8b,0xca,0xff,
  0xff,0x4e,0x05,0x4e,0xff,0x63,0x70,0x70,0x01,0x01,0x01,0x01,0xff,0x70,0x05,0x02,
  0x05,0x05,0x4e,0xff,0x63,0x4e,0xff,0xcf,0x4e,0x4e,0xcf,0x4e,0x4e,0x63,0x63,0x63,
  0x63,0x63,0x00,0x00,0x05,0x05,0x05,0x62,0x01,0x01,0x70,0xff,0x62,0xff,0x4e,0xcf,
  0xfd,0x03,0xfd,0xfd,0xcf,0x63,0x01,0x01,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,0x01,0xa7,
  0xd3,0xd3,0xd3,0xa7,0xcf,0xf7,0xd2,0xd2,0x6d,0x6d,0x6d,0x6d,0x5a,0x5a,0x5a,0x02,
  0xd2,0xd2,0x21,0x02,0x5a,0x5a,0x21,0x5a,0x5a,0xd2,0x5a,0x6d,0xfd,0xf7,0xf7,0xfd,
  0x03,0xcf,0xfd,0x03,0xcf,0x4e,0x63,0x4e,0x4e,0x8b,0x01,0x63,0xff,0x00,0x02,0xff,
  0x63,0x01,0x63,0x4e,0x05,0x02,0x02,0x05,0xff,0xff,0x8b,0xca,0xca,0x01,0x01,0x8b,
  0xff,0x02,0x02,0x05,0x05,0xff,0xff,0x63,0x01,0x01,0x70,0xff,0x05,0x05,0x02,0x02,
  0x02,0x05,0x4e,0xff,0xff,0x4e,0xcf,0x4e,0x4e,0x4e,0xcf,0xff,0x4e,0x03,0x4e,0x4e,
  0x4e,0x4e,0x05,0x05,0x05,0x05,0xff,0x62,0x62,0x62,0x00,0x02,0x05,0x03,0xfd,0xfd,
  0xfd,0xfd,0x03,0xcf,0x4e,0x01,0x01,0xa7,0xd3,0xf9,0xf9,0xd3,0xd3,0xa7,0x01,0x8b,
  0x01,0xa7,0xa7,0x01,0x03,0xf7,0x5a,0x6d,0x6d,0x6d,0x5a,0x05,0x8b,0xca,0x01,0xff,
  0x05,0x05,0x03,0x05,0x02,0x02,0xd2,0x5a,0x5a,0x5a,0xf7,0xfd,0xfd,0xfd,0xfd,0xfd,
  0xfd,0xfd,0xfd,0xcf,0x4e,0x63,0x63,0x4e,0x63,0x8b,0x63,0xff,0x4e,0x02,0x02,0xcf,
  0x4e,0xff,0x00,0x02,0x21,0x21,0xec,0x05,0xff,0xff,0x01,0x8b,0x01,0x01,0x01,0xff,
  0x02,0x02,0xec,0xec,0xec,0x05,0x4e,0x4e,0xff,0xff,0x05,0x05,0x05,0x02,0x21,0x02,
  0x02,0x02,0x4e,0x4e,0x02,0x05,0x4e,0xff,0x4e,0x4e,0x4e,0x4e,0x4e,0x02,0x21,0x02,
  0x02,0x62,0x00,0x00,0x05,0x70,0x70,0xff,0x05,0x05,0x02,0x02,0x02,0xcf,0xcf,0x03,
  0x03,0x4e,0x4e,0x63,0x8b,0x01,0xa7,0xa7,0xd3,0xd3,0xa7,0x8b,0xa7,0x01,0x63,0x8b,
  0x01,0xa7,0xd3,0x8b,0x4e,0x03,0x02,0x02,0x02,0x02,0xff,0x8b,0x8b,0xca,0x70,0x4e,
  0x02,0x02,0x02,0x02,0x02,0x02,0xd2,0x5a,0x5a,0x03,0x03,0x4e,0xf7,0xfd,0xfd,0xfd,
  0xcf,0x4e,0x4e,0x4e,0x4e,0x01,0x8b,0x63,0xa7,0x8b,0x63,0x05,0x02,0x63,0x01,0x01,
  0x63,0xff,0x4e,0x02,0x21,0xfe,0x21,0x02,0x05,0xff,0x70,0xe2,0x01,0x70,0xff,0x05,
  0x05,0x05,0x02,0x02,0x02,0x02,0x02,0x05,0x05,0x05,0xe3,0x02,0x02,0x02,0xec,0x02,
  0x02,0x02,0x02,0x02,0x03,0x4e,0x4e,0x4e,0x4e,0xff,0x01,0x8b,0x01,0x05,0x02,0xe3,
  0x05,0x00,0x00,0xf0,0x05,0x05,0x05,0xec,0x02,0x4e,0x4e,0x00,0x05,0x00,0x63,0x63,
  0x01,0x01,0x01,0x01,0xa7,0xa7,0xd3,0xd3,0xa4,0xa7,0xa7,0x63,0xe7,0xe7,0xe7,0x63,
  0x63,0x01,0xa7,0x01,0xe7,0x05,0x05,0x05,0x05,0x05,0xff,0x62,0x00,0x00,0x62,0xff,
  0xff,0x4e,0x03,0xd2,0x5a,0xd2,0xd2,0xd2,0xfa,0x03,0x03,0x4e,0x03,0x4e,0xa4,0xe1,
  0xe1,0xf9,0xa4,0xf9,0xa4,0xa7,0xa7,0xd3,0xd3,0x01,0x63,0x4e,0xff,0x01,0x01,0x8b,
  0x01,0xff,0xff,0x05,0xfe,0x5a,0x5a,0x02,0x05,0xf0,0x05,0x00,0xff,0x00,0x05,0x05,
  0x05,0x02,0x02,0xec,0x21,0x02,0x02,0x02,0x02,0x02,0xe3,0x02,0xec,0x02,0x02,0x02,
  0x02,0x02,0x02,0x05,0x05,0x05,0x4e,0xff,0x8b,0x8b,0xa4,0xd3,0x8b,0x00,0x05,0xec,
  0x05,0x00,0x05,0xec,0xec,0xec,0x02,0x05,0x00,0x62,0xe7,0x70,0x70,0xff,0x63,0xe7,
  0xe7,0x01,0x8b,0x01,0xa7,0xa7,0xa4,0xa4,0xa4,0x8b,0x8b,0x62,0x00,0x00,0x00,0x62,
  0xe7,0x8b,0x8b,0x8b,0xe7,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xe3,0x02,0x21,0x00,
  0xff,0x01,0x01,0x70,0x63,0xfa,0xed,0x03,0xfa,0xd3,0xd3,0xd3,0xd3,0xa4,0xe1,0xe1,
  0xf9,0xd3,0xe1,0xe1,0xa4,0xd3,0xa7,0xa7,0xa7,0x8b,0x8b,0xff,0xff,0x8b,0x8b,0x8b,
  0xa7,0x01,0xff,0x00,0x02,0xef,0xfe,0x21,0x21,0x21,0x02,0x05,0x70,0x00,0x05,0x02,
  0xec,0x21,0x21,0x21,0x21,0x21,0x02,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x02,0x02,
  0x02,0x02,0x03,0x4e,0x4e,0x4e,0x01,0x01,0x01,0xd3,0xd3,0x8b,0x8b,0x8b,0xe7,0x05,
  0xec,0x05,0x05,0x21,0x21,0x02,0x05,0x62,0x8b,0x01,0x70,0x62,0x62,0xe7,0x62,0x8b,
  0x8b,0xd3,0xa4,0xa4,0xa4,0xa4,0xf9,0xa7,0x01,0x70,0xe7,0x00,0x00,0x00,0x05,0x05,
  0xe7,0xe7,0x63,0x62,0x62,0x00,0x05,0x05,0x05,0xf0,0x05,0x21,0x21,0xfe,0x21,0x4e,
  0x00,0xe7,0x70,0x8b,0x8b,0x01,0x01,0xd3,0xa4,0xf9,0xa4,0xf9,0xf9,0xa4,0xa4,0xa4,
  0xd3,0xa4,0xf9,0xf9,0xe1,0xe1,0xf9,0x8b,0x01,0xa7,0x8b,0x63,0x70,0xa7,0xa7,0xa7,
  0x8b,0x01,0xff,0xff,0x00,0x02,0x05,0x05,0x02,0xe3,0x05,0x00,0x62,0x00,0x05,0x02,
  0x02,0xed,0x21,0x21,0x21,0x21,0x02,0x21,0x21,0x21,0x21,0x02,0x02,0x02,0x02,0x02,
  0x05,0x4e,0xff,0x63,0x70,0xff,0x70,0x70,0x01,0x01,0x8b,0x01,0xa7,0xa7,0x8b,0x01,
  0x00,0x05,0x02,0x21,0xd2,0x02,0x4e,0xe7,0x8b,0x8b,0x8b,0x01,0x62,0x62,0x01,0x8b,
  0xca,0xca,0x8b,0xca,0xa7,0xa4,0xd3,0x01,0x62,0xff,0x4e,0x4e,0x05,0x05,0x05,0x05,
  0x05,0x4e,0x00,0x8b,0x01,0x8b,0x62,0x4e,0x05,0x05,0x02,0xfe,0x21,0x5a,0x21,0x02,
  0x02,0x02,0x05,0xff,0xff,0xff,0x8b,0xf9,0xe1,0xe1,0xe1,0xe1,0xf9,0xf9,0xf9,0xf9,
  0xf9,0xf9,0xe1,0xe1,0xf9,0xa4,0xd3,0xa7,0x01,0x01,0xa7,0xd3,0xd3,0xd3,0xe1,0xa4,
  0x8b,0x01,0x01,0x62,0x00,0xff,0xff,0x62,0x62,0x70,0x62,0x62,0x00,0x05,0x05,0xec,
  0x21,0x21,0x21,0xfe,0x21,0x21,0xe3,0xe3,0x02,0x21,0xe3,0x05,0xe3,0xec,0x05,0x05,
  0xff,0xff,0x70,0x01,0x01,0x01,0x4e,0x4e,0x4e,0x4e,0xff,0x8b,0xa7,0xa4,0x8b,0x01,
  0x62,0x05,0x02,0x02,0x05,0x05,0x00,0x70,0x62,0x62,0xff,0x62,0x62,0x70,0x62,0x8b,
  0xca,0xd3,0x63,0xe7,0x01,0x8b,0x8b,0x01,0x70,0xff,0x62,0x62,0x05,0x05,0x00,0x00,
  0x00,0xff,0x62,0x62,0x62,0x62,0x62,0x62,0x62,0x05,0x02,0xfe,0x5a,0x5a,0x5a,0x5a,
  0xed,0x02,0x03,0x02,0x4e,0x4e,0x63,0xa7,0xe1,0xe1,0xe1,0xe1,0xf9,0xf9,0xf9,0xf9,
  0xf9,0xe1,0xe1,0xf9,0xd3,0x8b,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa7,0xa4,0xf9,0xa4,
  0x8b,0x01,0x01,0x62,0x00,0x05,0x00,0x00,0xff,0x62,0x00,0x00,0x00,0xf0,0x05,0x05,
  0xe3,0x21,0x21,0xfe,0xed,0xe3,0xe3,0xec,0xe3,0x21,0xe3,0x02,0x21,0x21,0x21,0x05,
  0xff,0x63,0x8b,0xa7,0xa7,0x01,0x70,0x00,0x00,0xff,0x4e,0x70,0xca,0xe1,0xd3,0x01,
  0x01,0x70,0x70,0x62,0x62,0x70,0x70,0x70,0x00,0x62,0xff,0xff,0x00,0x62,0x62,0x8b,
  0xa4,0xf9,0xa7,0x70,0x70,0x01,0x01,0x01,0x01,0x62,0xe7,0x70,0x62,0x62,0x70,0x01,
  0x70,0x62,0x62,0x8b,0x8b,0x8b,0x01,0x01,0x01,0x70,0x4e,0x02,0x5a,0x5a,0x21,0x02,
  0xed,0x02,0x03,0x02,0x02,0x03,0x4e,0x8b,0xd3,0xa4,0xa4,0xf9,0xf9,0xf9,0xf9,0xa4,
  0xe1,0xe1,0xf9,0xd3,0x8b,0xa7,0xa4,0xa4,0xd3,0xd3,0xd3,0x8b,0x8b,0xa7,0xd3,0xd3,
  0xe2,0x62,0xff,0x00,0x00,0x05,0xf0,0x00,0x00,0x00,0x05,0x00,0x00,0xe3,0x05,0x02,
  0x21,0xed,0x21,0x21,0x21,0x02,0xec,0x02,0x02,0x02,0x02,0x02,0x21,0x02,0x02,0x05,
  0xe7,0x8b,0xd3,0xd3,0xa4,0xd3,0x01,0x8b,0xe7,0x01,0x70,0x01,0xa4,0xe1,0xa7,0x01,
  0x62,0x70,0x70,0x70,0x70,0x62,0xff,0x62,0x62,0x00,0x00,0x62,0x62,0x62,0x62,0x8b,
  0xa4,0xf9,0xe1,0xca,0x62,0x62,0x8b,0x8b,0x01,0x8b,0x8b,0xca,0x8b,0x01,0x01,0x8b,
  0xca,0xca,0xca,0xd3,0xca,0xca,0x8b,0x8b,0x01,0x01,0x01,0xff,0x03,0x02,0x03,0x03,
  0x02,0x03,0xff,0x05,0xed,0xed,0x02,0x03,0x05,0xd3,0xa7,0xa4,0xf9,0xf9,0xd3,0xd3,
  0xd3,0xa4,0xa4,0xd3,0x8b,0xa7,0xd3,0xa4,0xd3,0xd3,0xd3,0x8b,0x8b,0x8b,0x8b,0x01,
  0xff,0x00,0x62,0x00,0x05,0x62,0x01,0x70,0x70,0x62,0x62,0xff,0x00,0x05,0x02,0x02,
  0xed,0x21,0x02,0x05,0xe3,0x00,0x05,0x05,0xf0,0x05,0xe3,0x05,0x02,0x05,0x05,0x4e,
  0x63,0xca,0x8b,0x8b,0xd3,0xd3,0x8b,0x8b,0xca,0x01,0x70,0x70,0x8b,0xa4,0xca,0xe2,
  0xe2,0x01,0x62,0x62,0x62,0x00,0x05,0x00,0x00,0x00,0x62,0xff,0x70,0x62,0x62,0x8b,
  0x8b,0xa4,0xa4,0xd3,0x01,0xa7,0xe1,0xd3,0xca,0x8b,0x8b,0x8b,0x8b,0xe2,0x01,0x01,
  0x8b,0xca,0xd3,0xd3,0x8b,0x8b,0x01,0x01,0x01,0x01,0x8b,0x63,0x4e,0x4e,0xcf,0x03,
  0x03,0x03,0x4e,0x05,0x02,0xed,0x02,0x02,0x4e,0xd3,0xd3,0xd3,0xa4,0xa4,0xd3,0xa4,
  0xa4,0xd3,0x8b,0x01,0x8b,0x01,0x8b,0x8b,0xca,0x8b,0x8b,0x01,0xca,0x01,0xff,0xff,
  0x62,0x00,0xff,0x00,0x62,0x8b,0x8b,0x01,0x01,0x8b,0x01,0x01,0x62,0xff,0xff,0x02,
  0x02,0x05,0x00,0x00,0x00,0xff,0x00,0xff,0x00,0xff,0xff,0x00,0x4e,0x05,0x63,0x01,
  0x70,0x01,0x8b,0xa7,0x8b,0x8b,0xca,0xe2,0xca,0x8b,0x01,0x01,0xe2,0x8b,0x8b,0x70,
  0x70,0x62,0x62,0x00,0x05,0x05,0x05,0x05,0x05,0x05,0xf0,0x00,0x00,0x62,0x62,0x62,
  0x01,0xd3,0xa4,0xe1,0xe1,0xf9,0xa7,0xa4,0xa4,0xca,0x62,0x70,0x8b,0xca,0xca,0xca,
  0x8b,0xa7,0xd3,0xca,0x01,0x63,0x70,0xff,0x63,0x01,0x63,0xff,0x02,0x4e,0x4e,0x63,
  0x03,0x02,0x02,0x21,0x21,0x21,0x02,0x03,0xa7,0xa4,0xd3,0xd3,0xd3,0xd3,0xd3,0xd3,
  0xd3,0x8b,0x01,0x01,0x01,0x63,0x70,0x01,0x01,0x8b,0x8b,0x8b,0x8b,0x01,0xff,0x4e,
  0x05,0x05,0x05,0x00,0xe7,0x8b,0xca,0xca,0xca,0xca,0x01,0x62,0x62,0x01,0x01,0xff,
  0x4e,0xff,0x70,0x62,0xe7,0xe7,0x62,0x63,0x70,0xff,0xff,0xff,0xe7,0xe7,0xe7,0x01,
  0x01,0x8b,0xca,0x8b,0x01,0x8b,0x01,0x01,0x8b,0x01,0x01,0xe2,0x01,0xca,0xe2,0x62,
  0xff,0x00,0x62,0x05,0x05,0x00,0x00,0x05,0x00,0xf0,0x05,0x05,0xf0,0xf0,0x00,0x62,
  0x70,0xca,0xa7,0xa7,0x8b,0x8b,0x8b,0x8b,0x8b,0xa4,0x8b,0x01,0xd3,0xa7,0xa7,0xa4,
  0xd3,0xd3,0xa7,0xca,0x70,0xe7,0x4e,0x05,0x62,0xff,0x4e,0x03,0x03,0x4e,0x4e,0x4e,
  0x02,0x02,0x21,0xed,0x5a,0x5a,0xed,0xed,0x03,0xa7,0xca,0x8b,0xd3,0xd3,0xd3,0x8b,
  0xd3,0x8b,0x8b,0x01,0x01,0x70,0xff,0xff,0x62,0x01,0x62,0x01,0x8b,0x70,0x00,0x05,
  0xe3,0xec,0xec,0x05,0x70,0x62,0xff,0x00,0x70,0x01,0x8b,0x01,0x62,0xff,0xff,0x01,
  0x70,0x01,0x01,0x70,0x01,0x01,0x70,0x01,0x70,0x62,0x62,0x62,0xff,0x70,0x70,0x70,
  0x01,0x01,0x8b,0xe7,0x01,0x01,0x62,0x70,0x62,0x62,0x62,0x62,0x8b,0xa7,0x62,0x62,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,0xe3,0xe3,0x21,0xe3,0x02,0xf0,0xf0,0x05,
  0x00,0x00,0x62,0x70,0x62,0x70,0x62,0x62,0x8b,0xca,0xd3,0xd3,0x8b,0xca,0xd3,0xca,
  0x8b,0xa7,0x8b,0x70,0xe7,0xff,0x05,0x05,0x4e,0x4e,0x05,0x02,0x03,0x4e,0x4e,0x03,
  0x02,0xed,0x21,0x21,0x21,0x21,0x05,0xff,0x01,0xd3,0xd3,0xd3,0xa4,0xd3,0xd3,0xd3,
  0xa4,0xca,0x01,0x62,0x00,0x4e,0x00,0xff,0xff,0x00,0xff,0x62,0x62,0x01,0xff,0x4e,
  0x05,0xf0,0xf0,0xf0,0xff,0x62,0xff,0x8b,0x01,0x8b,0xe2,0x01,0xff,0x62,0x62,0x62,
  0x01,0xe7,0xff,0xff,0xff,0xe7,0x62,0xf0,0x00,0x00,0x00,0x62,0x62,0x62,0x62,0x70,
  0xe2,0x62,0x62,0x00,0x00,0x05,0x62,0x00,0x62,0x00,0x05,0x05,0x00,0x00,0xf0,0x00,
  0xff,0x00,0x00,0x00,0x00,0x05,0x05,0x05,0xf0,0x21,0x21,0x21,0x21,0x21,0x21,0xec,
  0x02,0x05,0x05,0x00,0xff,0x01,0x01,0x70,0xff,0xe7,0x01,0xa7,0x8b,0x01,0x01,0x8b,
  0x8b,0x01,0x01,0xe7,0xff,0x00,0x02,0x05,0x05,0x05,0x02,0x03,0x03,0x03,0x02,0x02,
  0x21,0x21,0xed,0x21,0x02,0x4e,0xd3,0xd3,0xa4,0xd3,0xa7,0x8b,0xd3,0xd3,0xd3,0x8b,
  0x01,0x70,0x62,0x00,0x00,0x05,0x4e,0x00,0x05,0x00,0xff,0xff,0xa7,0x01,0x70,0xff,
  0xff,0xff,0xff,0x05,0x05,0x4e,0x8b,0xd3,0xca,0xd3,0x8b,0x70,0x00,0x00,0xff,0x4e,
  0x62,0xff,0xff,0xff,0x00,0x00,0x00,0xe2,0x70,0x70,0x70,0x00,0xff,0x00,0xff,0x01,
  0x00,0x05,0x00,0x00,0x62,0xe3,0x02,0x02,0xec,0x02,0x02,0x21,0x02,0x00,0x01,0x70,
  0x01,0x70,0x05,0x05,0xe3,0xe3,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
  0x21,0xec,0x02,0x05,0xff,0xff,0x05,0x05,0x4e,0x63,0x8b,0x8b,0x8b,0x01,0x70,0x70,
  0x70,0x8b,0x70,0xff,0x4e,0x05,0x02,0x03,0x02,0x02,0x02,0x4e,0x03,0x03,0x02,0x02,
  0x21,0x21,0x21,0x02,0x03,0x4e,0xa7,0xca,0xd3,0xd3,0xa4,0xd3,0xa7,0xca,0x01,0x01,
  0x70,0x62,0xff,0x62,0x00,0x05,0x62,0x70,0xe2,0xe2,0x62,0xff,0xca,0x01,0x62,0x00,
  0x62,0x70,0x00,0x00,0x05,0x05,0x70,0x01,0x8b,0x8b,0xff,0x4e,0xff,0x70,0xff,0x00,
  0x00,0x00,0x05,0x05,0x21,0xec,0x62,0x62,0x62,0x00,0x05,0x05,0x05,0x00,0xff,0x00,
  0x02,0xe3,0x21,0x21,0x21,0x21,0x21,0x02,0x02,0x05,0x02,0x00,0x70,0x8b,0x8b,0x62,
  0x62,0xf0,0x21,0xf0,0x21,0x21,0xfe,0xfe,0xfe,0xfe,0x21,0x21,0xfe,0xfe,0xfe,0xfe,
  0x21,0x21,0x21,0x21,0x02,0xec,0xec,0xec,0x05,0x05,0x63,0xff,0x63,0xe7,0xe7,0xff,
  0xff,0x70,0xe7,0x63,0xff,0x4e,0x4e,0x03,0x02,0xfa,0x03,0x02,0x02,0x03,0x02,0x02,
  0xed,0xfe,0x5a,0x21,0x02,0x03,0xe7,0xa7,0xd3,0xd3,0xd3,0xe1,0xd3,0xca,0xe2,0xe2,
  0x8b,0x01,0x62,0xff,0x70,0x01,0x70,0x62,0x01,0x62,0x62,0x70,0x8b,0x01,0xff,0xff,
  0xff,0x62,0x05,0x05,0xf0,0xf0,0x00,0x62,0x70,0x00,0x05,0x05,0x00,0x62,0x05,0x05,
  0x02,0x21,0x21,0x5a,0xfe,0x21,0x21,0xf0,0xf0,0xe3,0xe3,0x02,0xe3,0xe3,0xec,0x02,
  0x21,0x5a,0x5a,0x5a,0x5a,0x5a,0xfe,0x21,0xec,0x05,0x70,0xd3,0xe2,0x01,0xff,0x05,
  0x21,0xfe,0xfe,0x8b,0xe2,0x05,0xef,0xef,0xef,0xef,0xfe,0x5a,0xfe,0xfe,0xfe,0xfe,
  0x5a,0xfe,0x5a,0xfe,0xfe,0x21,0xec,0x21,0xe3,0x05,0x00,0x4e,0x05,0x00,0x4e,0x05,
  0x00,0xe7,0x63,0x63,0x63,0x63,0x4e,0x03,0xfa,0x03,0x03,0xed,0x03,0xfa,0x02,0x02,
  0xed,0x5a,0x5a,0xfe,0xed,0x4e,0xe7,0xd3,0xa4,0xca,0xca,0xca,0xd3,0xd3,0xca,0x01,
  0x70,0x62,0x70,0xff,0x62,0x70,0xff,0x70,0x63,0x00,0x01,0x01,0xe2,0x70,0x00,0xf0,
  0x00,0x05,0x02,0x02,0xf0,0x05,0x62,0xff,0x00,0x05,0x05,0x00,0x8b,0x70,0x02,0xec,
  0x21,0x5a,0xfe,0xfe,0x5a,0x5a,0x5a,0x5a,0x21,0x21,0x21,0x21,0x21,0x5a,0x5a,0x21,
  0x00,0xe3,0x21,0x02,0x05,0xff,0x70,0x62,0xff,0x00,0x62,0xca,0x62,0x00,0x05,0xe3,
  0x21,0xfe,0x21,0x70,0x62,0xe2,0xe2,0x62,0x21,0xef,0xef,0xef,0xef,0xef,0xef,0xfe,
  0xef,0xef,0xfe,0xef,0xef,0xfe,0xfe,0x21,0x21,0x21,0x02,0x02,0x05,0x00,0xff,0x4e,
  0x63,0x63,0x00,0x4e,0x4e,0x03,0x03,0x03,0x03,0xf7,0x03,0xed,0x03,0x4e,0xcf,0xcf,
  0x63,0x03,0x5a,0x5a,0xed,0x02,0x02,0x8b,0xca,0xe2,0xe2,0xca,0xd3,0xd3,0xd3,0xca,
  0x01,0xff,0x01,0x01,0xe7,0xff,0xff,0x70,0xff,0x62,0xe2,0x70,0x70,0x70,0xff,0x05,
  0x05,0x00,0x00,0x70,0x00,0x00,0x00,0xf0,0x02,0xe3,0x05,0x70,0xe2,0x62,0x00,0x21,
  0x21,0x21,0x21,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x21,0x02,0xfe,0x5a,0xed,0xe3,0x05,
  0x70,0x62,0x70,0x62,0xe2,0xe2,0x01,0x62,0x70,0x05,0x05,0x62,0x00,0xe3,0x05,0xe3,
  0xfe,0xfe,0xe3,0x00,0x00,0x70,0xca,0xca,0x70,0xf6,0xef,0xef,0xef,0xf4,0xef,0xef,
  0xef,0xef,0xef,0xef,0xf3,0xfe,0xef,0xfe,0x5a,0x21,0x21,0x5a,0xed,0x02,0x4e,0x63,
  0x4e,0x4e,0x03,0x03,0x03,0x03,0x03,0xd2,0xf7,0x03,0xfa,0x05,0x03,0x03,0x03,0x02,
  0x03,0xcf,0xcf,0x03,0x03,0xed,0x21,0x05,0x00,0x62,0x8b,0xe2,0x8b,0xca,0x8b,0x01,
  0x62,0x63,0x70,0x62,0xff,0x00,0xff,0x62,0x70,0x62,0x62,0x62,0x70,0xff,0x70,0x70,
  0x00,0x70,0xca,0xca,0x70,0xca,0xff,0x4e,0x00,0x00,0xff,0x8b,0x01,0x62,0x00,0x02,
  0x02,0x21,0x5a,0x5a,0x5a,0x5a,0x21,0x00,0xff,0x01,0xca,0xd3,0x00,0x05,0x05,0x62,
  // 11: sky 1, part D
  0x7b,0xbe,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,0xed,0x83,0x08,0x01,
  0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,
  0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,
  0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,
  0x10,0x07,0x08,0x4b,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,
  0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,
  0x08,0x23,0x10,0x6c,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,
  0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,
  0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,
  0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,
  0x00,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,
  0x08,0x45,0x10,0x45,0x99,0x95,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,
  0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,
  0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,
  0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x18,0x64,0x00,0x65,
  0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,
  0x08,0x67,0x14,0x9d,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,
  0x00,0x81,0x08,0x81,0x10,0x81,0xb9,0xa5,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,
  0x18,0x82,0x00,0x83,0x08,0x83,0x73,0x7c,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,
  0xd5,0x84,0x10,0x84,0xfa,0xa5,0x9b,0xc6,0x18,0x84,0x00,0x85,0x18,0x85,0x08,0x85,
  0x10,0x85,0x3a,0xb6,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,
  0x10,0x87,0x18,0x87,0x00,0xa0,0x7b,0xc6,0x08,0xa0,0x0a,0x7c,0x10,0xa0,0x18,0xa0,
  0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,
  0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,0x08,0xa4,0x1b,0xae,0x10,0xa4,
  0x18,0xa4,0x88,0x5b,0x00,0xa5,0x08,0xa5,0x37,0x8d,0x10,0xa5,0xfc,0xd6,0x18,0xa5,
  0xb3,0x84,0x69,0x5b,0x00,0xa6,0xeb,0x7b,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,
  0x08,0xa7,0x77,0x9d,0x10,0xa7,0x18,0xa7,0x00,0xc0,0xd3,0x94,0x08,0xc0,0x10,0xc0,
  0x18,0xc0,0x00,0xc1,0xb4,0x7c,0xc8,0x63,0x4b,0x4a,0x08,0xc1,0x35,0x8d,0x10,0xc1,
  0xe9,0x5a,0x18,0xc1,0x12,0x6c,0x00,0xc2,0x08,0xc2,0x10,0xc2,0xe7,0x42,0x2a,0x32,
  0xf8,0xad,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x26,0x74,0x10,0xc3,0x18,0xc3,0x3d,0xd7,
  0x96,0xad,0x6c,0x63,0x00,0xc4,0xe7,0x6b,0x07,0x6c,0x08,0xc4,0x10,0xc4,0x29,0x32,
  0x18,0xc4,0x69,0x63,0x89,0x62,0xc8,0x6a,0x00,0xc5,0x08,0x53,0x8b,0x5b,0x86,0x42,
  0xa1,0xab,0x9b,0x9b,0x9b,0x9b,0x9b,0x00,0xab,0x9b,0x9a,0x5a,0x8b,0x8b,0x8b,0xd1,
  0xd1,0xc4,0xd1,0xa1,0x00,0xa1,0xa1,0xa1,0xab,0x00,0x8b,0xc4,0xc4,0xc4,0xc4,0xc4,
  0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0x5a,0x8b,0xbe,0xbe,
  0xbe,0xbe,0xbe,0x9a,0x8b,0x5a,0x8b,0x8b,0x9a,0xbe,0xbe,0xbe,0xbe,0xbe,0xbe,0x9a,
  0x8b,0x8b,0xbe,0x00,0x9a,0x8b,0x8b,0x9a,0xa1,0xa1,0x00,0x9b,0x9b,0x9b,0x00,0xab,
  0x00,0x00,0xa1,0x00,0xa1,0x00,0x00,0x00,0x00,0xab,0x00,0xab,0x9b,0x00,0x9b,0x9b,
  0x9b,0xc6,0xc6,0x9b,0xab,0x9b,0x00,0x00,0x00,0x00,0x00,0xab,0x9b,0x9b,0x9b,0x00,
  0x8b,0x5a,0x8b,0x8b,0xa1,0x9a,0xa1,0xc6,0xc6,0xef,0xef,0xc6,0x9b,0x9b,0xab,0xab,
  0x00,0x9b,0xab,0x9b,0xab,0x9b,0xc6,0x9b,0x9b,0x9b,0x9b,0x9b,0xa1,0xc4,0xe8,0xc4,
  0x98,0x98,0xa1,0xa1,0x00,0x9b,0xa1,0xe8,0xe8,0xe8,0xe8,0x8b,0x98,0x98,0x98,0x98,
  0xc4,0x98,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0x00,0xef,0xef,0xef,0xc6,
  0x00,0xbe,0xbe,0x9a,0x9a,0x5a,0x5a,0x8b,0x9a,0xbe,0xbe,0xbe,0xbe,0xbe,0x9a,0x9a,
  0x8b,0x8b,0x8b,0xbe,0xbe,0x9a,0x9a,0x9a,0x9a,0xa1,0x9a,0xa1,0x00,0xab,0xa1,0xa1,
  0xa1,0xa1,0xa1,0x00,0x00,0x00,0x00,0x00,0x00,0xab,0x00,0xab,0x9b,0x9b,0xc6,0x9b,
  0x9b,0xc6,0xc6,0x9b,0x00,0xab,0x00,0x00,0xa1,0xa1,0x00,0x9b,0x9b,0xab,0xab,0x8b,
  0xc4,0xef,0xef,0xef,0xef,0xef,0x9b,0x8b,0x9b,0xc6,0xef,0xc6,0xc6,0xc6,0xab,0x00,
  0x9b,0x9b,0x00,0xab,0x00,0xab,0xab,0xab,0xc6,0xc6,0x9b,0x9b,0xc6,0x00,0xc4,0xc4,
  0x98,0xa1,0x00,0xa1,0xe8,0xa1,0xe8,0xd1,0xd1,0xd1,0x98,0x98,0x98,0x98,0x98,0x98,
  0x98,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0x9a,0xc6,0xc6,0xef,0xef,0xc6,0x9b,0xc6,
  0x00,0xa1,0xbe,0xbe,0x9a,0x9a,0xbe,0x8b,0x8b,0xbe,0xbe,0x9a,0x9a,0x9a,0x9a,0x9a,
  0x8b,0x8b,0x9a,0xbe,0x00,0x9a,0x9a,0x8b,0x9a,0x9a,0x8b,0x9a,0xa1,0x00,0xa1,0x9a,
  0xe8,0x9a,0xa1,0x00,0xa1,0x00,0xa1,0x00,0x00,0x00,0xab,0x9b,0x9b,0x9b,0x9b,0xc6,
  0xef,0xef,0xc6,0xab,0xab,0xab,0x9b,0x00,0xa1,0xa1,0x00,0xab,0xc6,0x9a,0x5a,0x8b,
  0x9b,0xef,0xef,0xef,0xef,0xef,0xc6,0x9b,0xef,0xef,0xef,0xef,0xef,0xc6,0xc6,0x9b,
  0xa1,0x00,0x9b,0xab,0x00,0xa1,0x00,0x9b,0x9b,0x9b,0x9b,0x00,0xa1,0xa1,0x5a,0x5a,
  0x8b,0x00,0xa1,0xa1,0xe8,0x00,0xe8,0xd1,0xc4,0x98,0x98,0x98,0x98,0x98,0x98,0x98,
  0x98,0x98,0x98,0xc4,0xc4,0xc4,0xc4,0xc4,0x9b,0xef,0xef,0xef,0xc6,0xc6,0xab,0x9b,
  0x00,0x9a,0xa1,0x9a,0x9a,0xbe,0xbe,0xbe,0x9a,0x9a,0x8b,0x5a,0x5a,0x8b,0x9a,0x9a,
  0x9a,0xbe,0xbe,0xbe,0xbe,0x9a,0x5a,0x5a,0x8b,0x9a,0x8b,0x8b,0x9a,0x00,0x9a,0x9a,
  0x9a,0xa1,0xa1,0x00,0x00,0x9a,0x9a,0xab,0x9b,0x9b,0x9b,0xc6,0xc6,0x9b,0xab,0xc6,
  0xef,0xc6,0xab,0x00,0x9b,0x9b,0x9b,0xab,0x00,0x9a,0xa1,0x9b,0x9b,0x00,0x9b,0xc6,
  0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xc6,0x9b,
  0x5a,0xa1,0xa1,0xe8,0x9a,0x9a,0xe8,0xa1,0xa1,0x00,0x9b,0xe8,0x5a,0xc4,0x5a,0x5a,
  0xc4,0x8b,0x9b,0xc6,0x9b,0x9b,0x00,0xa1,0xc4,0x98,0xc4,0xc4,0xd1,0xde,0x98,0x98,
  0x5a,0x9b,0x9b,0x8b,0xc4,0xc4,0xc4,0x5a,0xef,0xc6,0xc6,0x9b,0x00,0x9b,0x00,0x00,
  0xa1,0xbe,0xa1,0xbe,0x9a,0xbe,0x9a,0x9a,0x9a,0x9a,0x8b,0x5a,0x5a,0x5a,0x8b,0x9a,
  0xbe,0xbe,0xbe,0x9a,0x9a,0x8b,0x5a,0x5a,0x8b,0x8b,0x8b,0x9a,0xab,0x9b,0x9b,0x00,
  0xa1,0xa1,0xa1,0x00,0xa1,0xa1,0x00,0xab,0xc6,0xc6,0xc6,0x9b,0x9b,0x9b,0x00,0x00,
  0xa1,0x00,0xa1,0x00,0x00,0x9b,0x00,0x00,0xab,0x9a,0x9a,0x9b,0x9b,0x9b,0xef,0xef,
  0xef,0xc6,0xc6,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xc6,0xc6,0xc6,0x9b,0xab,0x00,
  0xc4,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x8b,0x00,0x00,0xe8,0x8b,0x5a,0x5a,0xc4,
  0x5a,0x9b,0xc6,0xc6,0xc6,0xab,0x9b,0x00,0x98,0xc4,0x00,0x9b,0xab,0xc6,0xa1,0x5a,
  0xef,0xef,0xc6,0xc6,0x5a,0xd1,0xa1,0xe8,0x9b,0xc6,0xab,0x00,0x9b,0x00,0xa1,0xa1,
  0xa1,0x9a,0x9a,0xbe,0x9a,0x8b,0x9a,0x9a,0x8b,0x8b,0x8b,0x5a,0x5a,0x5a,0x5a,0x8b,
  0x9a,0x9a,0x9a,0xbe,0xbe,0x9a,0x8b,0x8b,0x9a,0xa1,0x8b,0xa1,0x9b,0xab,0x9b,0x9b,
  0xa1,0x00,0xa1,0x00,0xa1,0x00,0x9b,0x9b,0x9b,0x9b,0x00,0xab,0xab,0x9b,0x9b,0xc6,
  0xef,0xef,0xc6,0x9b,0x00,0x00,0x00,0x00,0xa1,0x8b,0xab,0xc6,0x9b,0xc6,0xc6,0xc6,
  0xc6,0xc6,0xc6,0xc6,0xc6,0xc6,0x9b,0xc6,0xef,0xef,0xc6,0xef,0xef,0x9b,0x00,0xa1,
  0xa1,0x5a,0x5a,0x8b,0x8b,0x5a,0x5a,0x5a,0x5a,0x00,0x00,0xab,0xa1,0x5a,0x5a,0x9a,
  0x9b,0xc6,0x9b,0x9b,0x00,0xab,0x00,0xab,0x9b,0x00,0x00,0xab,0xab,0x9b,0x9b,0x9b,
  0xc6,0xc6,0xc6,0xc6,0xab,0xab,0xc6,0xef,0xc6,0x9b,0x00,0xa1,0xa1,0xe8,0x9a,0x9a,
  0x9a,0x8b,0x8b,0x9a,0x9a,0x9a,0xbe,0x9a,0x9a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,
  0x9a,0x8b,0x8b,0xbe,0x8b,0x9a,0x8b,0x9a,0x00,0x00,0x00,0x00,0x00,0xa1,0x00,0x00,
  0xa1,0x00,0xa1,0xa1,0x00,0x00,0x9b,0x9b,0x9b,0xab,0x00,0x00,0x9b,0xef,0xef,0xef,
  0xef,0xef,0xef,0xc6,0x00,0x00,0xa1,0x00,0xa1,0x00,0xc6,0x9b,0xc6,0xef,0xef,0x9b,
  0x9b,0x9b,0x9b,0x9b,0xc6,0xef,0xef,0xef,0xef,0xef,0xc6,0xef,0xc6,0x9b,0x00,0xa1,
  0x9a,0x5a,0x8b,0x9a,0xe8,0x9a,0xe8,0x8b,0x00,0x9b,0x00,0x00,0x00,0xa1,0xe8,0xa1,
  0xa1,0xab,0xab,0x00,0xe8,0xa1,0xab,0x9b,0x00,0x00,0xab,0x00,0x9b,0xab,0x9b,0x9b,
  0xc6,0xc6,0xab,0x9b,0x9b,0xc6,0xc6,0xab,0xab,0xab,0xe8,0x00,0x00,0xe8,0xe8,0xe8,
  0x8b,0x8b,0x9a,0xe8,0x8b,0x9a,0xbe,0x9a,0x5a,0x5a,0x5a,0x8b,0x5a,0x5a,0x9a,0x8b,
  0x8b,0x5a,0x5a,0x5a,0x5a,0x5a,0x9a,0xa1,0x00,0x00,0xa1,0x00,0x00,0xa1,0x9a,0xa1,
  0x00,0x00,0xa1,0x00,0x00,0x9b,0xab,0xab,0xab,0x00,0xa1,0x00,0xc6,0xef,0xef,0xc6,
  0xc6,0xc6,0xc6,0xc6,0x00,0x00,0x00,0x00,0x00,0x9b,0x9b,0xc6,0xef,0xc6,0x9b,0x9b,
  0x9b,0x9b,0x9b,0x9b,0xc6,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0x9b,0x00,0x00,
  0xa1,0x5a,0x5a,0x8b,0x9a,0x9a,0x8b,0xa1,0xab,0xa1,0xa1,0x9a,0xa1,0x9a,0x8b,0xe8,
  0xe8,0xe8,0xe8,0x8b,0xe8,0x00,0x9b,0x00,0xab,0xc6,0xc6,0x00,0xe8,0xe8,0xa1,0x00,
  0xab,0xab,0xab,0x00,0xab,0xab,0x9b,0xab,0xab,0xa1,0xe8,0xa1,0xa1,0x8b,0x8b,0xe8,
  0xd1,0xe8,0xe8,0x8b,0x8b,0xd1,0x5a,0x8b,0x8b,0x5a,0x5a,0x8b,0x9a,0x8b,0xbe,0x5a,
  0x5a,0x5a,0x5a,0x5a,0xd1,0x8b,0xa1,0xa1,0x00,0xab,0x00,0xa1,0xa1,0xa1,0x9a,0xa1,
  0x00,0xa1,0xa1,0x00,0xa1,0x00,0x00,0x9b,0x00,0xa1,0xa1,0x00,0x9b,0x9b,0x9b,0x9b,
  0xc6,0xc6,0x9b,0x9b,0x00,0xa1,0x00,0x00,0x00,0x00,0xc6,0x00,0x9b,0xab,0x9b,0xab,
  0x9b,0xab,0x9b,0xc6,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xc6,0x9b,0x00,0x9b,
  0xa1,0x5a,0x5a,0x5a,0x5a,0x8b,0x5a,0x8b,0x8b,0x8b,0x8b,0x5a,0x8b,0x5a,0xc4,0x5a,
  0xd1,0x8b,0xe8,0x8b,0xe8,0xe8,0xe8,0xe8,0x00,0x00,0xab,0xe8,0xe8,0xe8,0xe8,0xa1,
  0xab,0x00,0xab,0xab,0xab,0x9b,0x9b,0x9b,0xa1,0xe8,0xe8,0xe8,0x8b,0xe8,0xe8,0xd1,
  0xd1,0xd1,0x8b,0x8b,0xd1,0xd1,0x5a,0x5a,0x5a,0x5a,0xc4,0x5a,0x5a,0x8b,0x5a,0x5a,
  0x5a,0x8b,0x8b,0x8b,0x8b,0x9a,0x9a,0x9a,0xa1,0x00,0x00,0x00,0x00,0xbe,0xa1,0x00,
  0x00,0x00,0xab,0x9b,0xa1,0xa1,0x00,0xa1,0x00,0xa1,0x9b,0xc6,0xc6,0x9b,0x9b,0x9b,
  0x9b,0x9b,0x00,0x00,0xa1,0x9a,0xa1,0xa1,0xa1,0x9a,0xa1,0x9b,0x00,0xc6,0x9b,0x9b,
  0x00,0x00,0x9b,0xef,0xef,0xc6,0xc6,0xc6,0xc6,0x9b,0x9b,0x9b,0x9b,0xab,0x00,0xa1,
  0x9a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0xc4,0xc4,0xc4,
  0xd1,0xc4,0xd1,0xe8,0xe8,0xe8,0x8b,0xe8,0x8b,0x8b,0x8b,0xe8,0xe8,0xe8,0x8b,0xe8,
  0xe8,0xe8,0xe8,0x00,0xe8,0xe8,0xab,0x00,0xe8,0xe8,0xe8,0x8b,0xe8,0x8b,0xd1,0xd1,
  0xd1,0xc4,0xd1,0xd1,0x5a,0xc4,0x5a,0x5a,0xd1,0x5a,0xc4,0xc4,0x5a,0x8b,0x5a,0x5a,
  0x8b,0x5a,0x5a,0x5a,0x8b,0x8b,0x9a,0x9a,0x9a,0xa1,0xa1,0xa1,0x00,0x00,0xa1,0x00,
  0x00,0x00,0x00,0x00,0xa1,0xbe,0xbe,0xbe,0xbe,0xa1,0x00,0x00,0x00,0xa1,0xa1,0x9b,
  0x9b,0x9b,0x00,0xa1,0xa1,0x00,0xa1,0xa1,0x00,0xa1,0x00,0x00,0x00,0x9b,0x9b,0x9b,
  0xa1,0x00,0x00,0x9b,0x9b,0xc6,0x9b,0x9b,0xab,0x00,0xa1,0x00,0xab,0x00,0xc6,0xab,
  0x8b,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0xc4,
  0xc4,0x8b,0xab,0xab,0x8b,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0x8b,
  0x8b,0xe8,0xe8,0xab,0xab,0xe8,0xe8,0xab,0xab,0xab,0xe8,0x8b,0xe8,0xe8,0xe8,0xe8,
  0xab,0xe8,0x8b,0xe8,0xc4,0xc4,0xc4,0xd1,0xd1,0xc4,0xc4,0x5a,0x5a,0x8b,0x8b,0x8b,
  0x8b,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x8b,0x9a,0x9a,0x9a,0xa1,0x00,0x00,0x00,
  0x9b,0x00,0xa1,0xa1,0xa1,0xbe,0xbe,0xa1,0xbe,0x9a,0xbe,0xa1,0xa1,0xa1,0x00,0x00,
  0x00,0x00,0x00,0xa1,0x00,0x00,0x00,0xa1,0x00,0xa1,0x00,0x00,0x00,0x00,0x9b,0x9b,
  0x00,0xa1,0xa1,0xa1,0xa1,0x9b,0x9b,0x9b,0x00,0xa1,0xa1,0xa1,0x9a,0x9a,0x00,0xa1,
  0x5a,0xa1,0x8b,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0xc4,
  0x98,0xd1,0xab,0x9b,0xe8,0xd1,0xd1,0x81,0xd1,0x81,0xe8,0xd1,0xd1,0xe8,0xe8,0xe8,
  0x8b,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0x8b,0x8b,0xe8,0xe8,0xe8,
  0xe8,0xe8,0xe8,0x8b,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0xde,0x81,
  0xd1,0xd1,0xd1,0xd1,0x5a,0x5a,0x5a,0xd1,0xd1,0xc4,0x5a,0x8b,0x8b,0x8b,0x9a,0x00,
  0xa1,0xa1,0x00,0xbe,0xbe,0x9a,0xbe,0x00,0xbe,0xbe,0xa1,0x00,0x00,0xa1,0xa1,0x00,
  0xbe,0x00,0xbe,0xa1,0x00,0xa1,0x00,0x00,0x00,0x00,0x00,0xbe,0x00,0x9b,0x00,0x00,
  0x00,0x9a,0x8b,0x9a,0xa1,0xa1,0x00,0x9b,0xa1,0xa1,0xa1,0xa1,0x00,0x9a,0xa1,0x9a,
  0xa1,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0xc4,
  0xc4,0xa1,0xab,0x9b,0xe8,0xd1,0xd1,0xd1,0xc4,0xd1,0xd1,0x81,0xd1,0xd1,0xd1,0xd1,
  0xe8,0xe8,0xf0,0xd1,0x8b,0xe8,0xe8,0xe8,0xe8,0xf0,0x8b,0xe8,0xe8,0xe8,0xe8,0xf0,
  0xf0,0x8b,0xd1,0xde,0x98,0xc4,0xc4,0x98,0x98,0xde,0xde,0x93,0x93,0x93,0x93,0x93,
  0xc8,0xde,0x81,0xda,0x93,0xda,0xde,0xde,0xe8,0xde,0xe8,0xde,0xde,0xde,0xde,0xde,
  0xde,0xde,0xde,0xd1,0xde,0xde,0xf0,0xe8,0xe8,0xe8,0x9a,0xa1,0x00,0xa1,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0xa1,0xa1,0xab,0xab,0x00,0x00,0xa1,0x00,0x00,0xa1,0xa1,
  0xa1,0x9a,0x8b,0x8b,0x8b,0x9a,0xa1,0xe8,0x9a,0xa1,0xa1,0xa1,0x00,0x00,0xa1,0xa1,
  0xa1,0xa1,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x5a,0x8b,0x5a,0x8b,0x5a,0xc4,
  0xde,0xd1,0xde,0xde,0x98,0xda,0xda,0xda,0xe8,0xf0,0x81,0x81,0x81,0x81,0xe8,0x81,
  0xda,0xda,0xc8,0xe2,0x93,0x93,0xda,0xde,0xd1,0xd1,0xe8,0xe8,0xe8,0x8b,0xd1,0xde,
  0xda,0x93,0xe2,0x93,0xe2,0x93,0x31,0xe2,0xe2,0x93,0xde,0xc8,0x93,0xde,0x93,0xda,
  0x93,0xda,0xde,0xde,0xde,0xda,0xde,0xde,0x98,0x98,0x98,0xde,0x98,0x8b,0xf0,0x81,
  0xf0,0xde,0xde,0x98,0xc4,0x98,0x98,0x81,0xe8,0xe8,0xe8,0x98,0x98,0xde,0xde,0xde,
  0xde,0x98,0xc4,0xde,0x98,0xc4,0xde,0xde,0xd1,0xf0,0xe8,0xe8,0xe8,0x00,0xa1,0xa1,
  0x9a,0xa1,0xa1,0x8b,0x8b,0x9a,0x8b,0xe8,0xa1,0x00,0x9a,0xc4,0xde,0xde,0xda,0x8b,
  0xda,0xde,0xda,0xda,0x98,0x98,0xde,0xde,0x93,0x98,0x98,0x93,0x93,0xe2,0xe2,0xe2,
  0xc8,0x93,0x93,0x93,0x31,0x93,0xe2,0xe2,0x31,0x98,0xde,0xde,0x98,0x98,0x98,0x93,
  0x93,0x93,0x93,0x93,0x93,0x31,0xe2,0x31,0xe2,0x31,0x93,0x93,0x93,0x31,0xe2,0xe2,
  0xe2,0x93,0x93,0xc8,0xc8,0xde,0xde,0x93,0x93,0x31,0xe2,0x31,0x93,0xe2,0xe2,0x93,
  0x93,0x93,0x93,0x93,0xda,0x98,0x93,0x98,0x93,0x98,0xda,0x98,0xda,0x98,0x98,0x98,
  0x98,0xde,0xde,0xde,0xde,0x98,0x98,0x98,0xde,0xde,0xde,0xde,0xde,0xde,0x98,0x98,
  0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,0x98,0xde,0xf0,0xa1,0xa1,0x00,
  0x00,0xe8,0x00,0xe8,0xe8,0xf0,0xde,0xde,0xde,0x93,0xda,0xf0,0xf0,0xde,0xde,0xc8,
  0x93,0x93,0x93,0xc8,0x81,0xc8,0xc8,0xc8,0x81,0xc8,0xc8,0xf0,0xe8,0xf0,0x81,0xf0,
  0x93,0x81,0x81,0x93,0xc4,0xc4,0x81,0x81,0x93,0xc4,0xc4,0x98,0x98,0xef,0xbe,0x98,
  0xe2,0x31,0x31,0x93,0x93,0x31,0x93,0xc8,0xc8,0x93,0x31,0xc8,0xc8,0x93,0xe2,0xe2,
  0x93,0x93,0xc8,0xc8,0xc8,0x93,0x81,0xd5,0x93,0x31,0x31,0x31,0x31,0x31,0xe2,0xe2,
  0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0xda,0xda,0x98,0x98,0x98,
  0x98,0x98,0xda,0x98,0xde,0xde,0xde,0x98,0x98,0xde,0x98,0xde,0xde,0xde,0x98,0xde,
  0x98,0x98,0xde,0x98,0xde,0x98,0x98,0xde,0xde,0xde,0x98,0x98,0x98,0xda,0x98,0x98,
  0xde,0xde,0xde,0x93,0xe2,0xe2,0xde,0xc8,0x81,0xd5,0x81,0xde,0xc8,0xda,0x93,0x93,
  0x31,0x93,0x93,0x93,0x81,0xf0,0xa1,0xf0,0xa1,0xf0,0xd5,0x81,0xde,0xde,0x81,0x81,
  0x31,0xd5,0xc8,0x93,0x98,0x93,0x98,0xd5,0xc8,0x98,0x93,0x93,0x98,0xef,0x00,0xe8,
  0xf0,0xd5,0x31,0x31,0x93,0xd5,0xde,0x31,0xe2,0x31,0x31,0x31,0xd5,0xd5,0xd5,0x93,
  0x93,0xd5,0xd5,0x81,0xd5,0xc8,0x93,0x93,0x93,0x31,0x31,0x31,0x31,0x93,0x93,0x93,
  0x93,0x93,0x93,0xc8,0xc8,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x98,0xda,0xda,0xda,
  0xda,0xda,0xda,0xda,0xda,0x98,0xde,0x98,0xde,0xde,0xde,0xde,0xde,0xde,0xde,0x8b,
  0xde,0x98,0xde,0xde,0x81,0xde,0xde,0xde,0xde,0x93,0x93,0xe2,0xe2,0xe2,0x93,0xe2,
  0xe2,0xe2,0xe2,0xe2,0xe2,0x31,0xc8,0xda,0xde,0x81,0x81,0xc8,0xda,0xde,0xde,0xda,
  0x81,0xc8,0xc8,0xde,0xc8,0x93,0xd5,0x81,0xd5,0xe8,0xf0,0xc8,0xc8,0xd5,0xd5,0x93,
  0x31,0x93,0x98,0xc4,0x98,0x93,0x93,0x93,0x93,0xc4,0x98,0x98,0xbe,0x9a,0xc4,0xc4,
  0x5a,0x81,0x81,0xd5,0x31,0x31,0xd5,0xd5,0x31,0x31,0x31,0x31,0x93,0x93,0x31,0x31,
  0x31,0x93,0x93,0x31,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x93,0x31,0x93,0x93,0x31,
  0x93,0x93,0x93,0xc8,0xd5,0x81,0x81,0x81,0xd5,0x81,0x81,0xde,0xc8,0xc8,0xde,0xda,
  0xda,0xda,0xda,0x98,0xde,0x98,0xda,0x98,0xde,0x98,0xde,0xde,0xde,0xde,0xd5,0xde,
  0xde,0xc8,0xda,0xe2,0x93,0xe2,0xe2,0xe2,0xe2,0x93,0x93,0xc8,0xc8,0x93,0x31,0x93,
  0x31,0xc8,0xd5,0x93,0x93,0xf0,0x81,0x81,0x81,0xde,0x81,0xe8,0x81,0x81,0xf0,0xf0,
  0x81,0xde,0xd5,0x81,0xf0,0xf0,0xe8,0x81,0x81,0xd5,0xd5,0xc8,0x93,0xc8,0xc8,0x31,
  0xc8,0xe8,0xd5,0x93,0x93,0x93,0x98,0xd1,0xd1,0x98,0x98,0x93,0x98,0x93,0x72,0x9e,
  0x9e,0x98,0x93,0x81,0x81,0xd5,0x31,0xc8,0x93,0x93,0x93,0x98,0x98,0xc4,0xc4,0x93,
  0x98,0x98,0x93,0x93,0x31,0xc8,0x81,0x81,0xd5,0xde,0xde,0x81,0x81,0x81,0xd5,0xd5,
  0x93,0xd5,0x93,0x81,0x81,0x81,0x81,0xd5,0x81,0x81,0x81,0x81,0x81,0xde,0x81,0xf0,
  0xf0,0xa1,0xe8,0xf0,0x81,0xde,0x81,0xde,0xde,0x31,0x93,0x93,0x93,0xc8,0xde,0x93,
  0xda,0xda,0xe2,0xe2,0xe2,0xe2,0xe2,0x31,0xde,0x81,0x81,0xde,0x81,0x81,0xf0,0xf0,
  0x81,0xde,0x98,0x98,0xc8,0x93,0x31,0xde,0x93,0x93,0xd5,0xc8,0x31,0xc8,0xd5,0xf0,
  0xc8,0x81,0xf0,0x93,0x81,0x81,0xd5,0xf0,0xd5,0x81,0x93,0xd5,0xc8,0xc8,0x93,0x31,
  0x31,0x93,0x98,0x93,0x31,0xe2,0x8b,0xc6,0xc6,0x93,0x93,0xe2,0x51,0x51,0x51,0xc4,
  0x98,0xc4,0xc4,0x81,0x81,0xf0,0xf0,0xc8,0xde,0xc8,0x98,0xc4,0xc4,0x9a,0xbe,0x5a,
  0x5a,0x9a,0x93,0xa1,0x8b,0xe8,0xe8,0x5a,0x98,0xc4,0xc4,0x98,0x98,0x99,0x31,0x31,
  0x99,0xf1,0xdc,0xdc,0x99,0xf1,0x93,0x81,0x81,0x93,0x93,0x72,0xf1,0xdc,0xf1,0xf1,
  0x99,0x93,0x93,0x99,0xd5,0xf1,0xdc,0xf1,0xf1,0xdc,0xdc,0xdc,0x51,0xdc,0x72,0xdc,
  0x51,0x72,0x93,0x93,0x98,0xda,0x31,0x81,0xc4,0xc4,0xc4,0xc4,0xd1,0xde,0xd5,0x81,
  0xd1,0x5a,0x5a,0xbe,0xde,0x93,0x31,0x93,0xc8,0x31,0xe2,0x93,0xc8,0xc8,0xc8,0xd5,
  0x93,0x93,0x81,0x81,0xd5,0xc8,0xd5,0xc8,0x81,0xd5,0xc8,0xd5,0x93,0x93,0x93,0x93,
  0xc8,0xc8,0xda,0x31,0xf1,0xe2,0x72,0xa1,0x00,0x98,0x93,0x93,0x51,0x51,0x51,0x72,
  0x51,0x72,0x72,0x93,0x72,0x93,0xe2,0x51,0x93,0x93,0x93,0xc4,0x98,0x9a,0x9a,0x8b,
  0x5a,0xc4,0x98,0xc4,0x98,0xc4,0xc4,0xc4,0x98,0xc4,0x98,0x98,0x98,0x72,0x93,0xf1,
  0xc8,0xc4,0xe7,0xe7,0xdc,0xdc,0xdc,0xdc,0xf1,0xdc,0xdc,0xe7,0xe7,0xe7,0xdc,0xdc,
  0xf1,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xdc,0xf1,0xdc,0xdc,0xdc,
  0xdc,0xe7,0xdc,0xe7,0x98,0x98,0xe8,0x98,0x98,0x98,0x5a,0xc4,0x98,0x98,0xc4,0xc4,
  0x5a,0x5a,0x9a,0xbe,0x81,0xd5,0x81,0x81,0xf0,0x81,0xde,0xde,0xc8,0xde,0x93,0x31,
  0xd5,0x93,0x93,0x93,0x81,0x81,0xf1,0x93,0xdc,0x31,0x72,0xf1,0xdc,0xdc,0xf1,0x72,
  0x51,0xf1,0xf1,0xe6,0xf7,0xf1,0xe7,0x31,0xde,0xf1,0xf1,0xe2,0x51,0x51,0xf1,0xe2,
  0xdc,0xdc,0xdc,0xdc,0xdc,0xdc,0x72,0x51,0x51,0x51,0xdc,0x98,0x93,0x8b,0x8b,0x8b,
  0x5a,0xc4,0x98,0x98,0x93,0x98,0x98,0x93,0x98,0xc4,0x98,0x72,0x72,0x98,0xe8,0x81,
  0xe2,0x93,0xe7,0x2c,0xdc,0xdc,0xe6,0xdc,0xdc,0xdc,0xdc,0xe7,0xe7,0xe7,0xdc,0xdc,
  0xdc,0xdc,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xe7,0xdc,0xdc,0xdc,0xdc,0xdc,0xe7,0xdc,
  0xdc,0xdc,0xdc,0xe7,0xe2,0xe2,0xf0,0x98,0xe6,0x31,0x98,0x98,0x98,0x98,0x93,0x98,
  0xc4,0x5a,0x8b,0x8b,0xde,0x81,0x81,0x81,0x81,0x99,0x99,0x99,0xd5,0xd5,0x81,0x81,
  0x81,0xd5,0xda,0xc8,0x93,0x93,0xdc,0xdc,0xe7,0xe7,0xe7,0xdc,0xdc,0xdc,0xdc,0xdc,
  0xdc,0xe7,0xf1,0xe7,0x2c,0xe7,0xe7,0xe7,0xe7,0xf1,0xe7,0xdc,0x51,0xdc,0x51,0xdc,
  0xdc,0xdc,0xdc,0xf7,0xe6,0xe6,0xdc,0xdc,0xdc,0xdc,0xdc,0x72,0xe2,0x98,0xc4,0xc4,
  0xc4,0x98,0x98,0xf7,0xdc,0x72,0x93,0x93,0x93,0x31,0xf1,0xf1,0xe7,0x98,0x93,0xf1,
  0xe6,0x72,0xe7,0xe7,0xdc,0x4d,0x4d,0xdc,0x4d,0x4d,0xdc,0xdc,0xe7,0xdc,0xe7,0xf7,
  0xdc,0xdc,0xdc,0xdc,0xe7,0xdc,0xdc,0xdc,0xdc,0xe7,0xdc,0xdc,0xf7,0xdc,0xe7,0xe7,
  0xe7,0xdc,0xdc,0xdc,0xe6,0xfd,0xe7,0xe2,0xdc,0xe6,0xfe,0x9e,0x93,0x51,0xe7,0x31,
  0xc4,0x98,0x98,0x98,0xd5,0xd5,0xd5,0xd5,0x99,0xf1,0xf1,0xf1,0xf1,0xf1,0xd5,0xf0,
  0xfd,0xfd,0xfd,0xff,0xe6,0xe6,0x4d,0xdc,0x4d,0x4d,0x4d,0x4d,0xf7,0xdc,0xdc,0xdc,
  0xdc,0xe7,0xfd,0xf7,0x2c,0xdc,0xfa,0x92,0xfa,0xe7,0xe7,0x99,0xdc,0xdc,0xdc,0xf1,
  0xdc,0xcb,0x92,0xf0,0xc6,0xc6,0x06,0xf1,0xdc,0xdc,0x51,0x98,0xe2,0x51,0xdc,0x4d,
  0x51,0x93,0x72,0xdc,0xfa,0xfa,0xf1,0x72,0xe2,0x21,0xff,0x2c,0xe7,0x72,0xf7,0x4d,
  0x4d,0xf7,0xe7,0xe7,0xdc,0x4d,0xe7,0xdc,0xe6,0xe6,0xfd,0xfd,0xe6,0xdc,0xdc,0xe6,
  0xe6,0xfd,0xf1,0xcb,0xdc,0xdc,0xdc,0xdc,0xdc,0xdc,0xe7,0xe7,0xf7,0xdc,0xe7,0xf7,
  0xe6,0xe7,0xe7,0xe7,0xe6,0xf1,0xe6,0xdc,0xf1,0xfd,0xfe,0x72,0xf1,0xf1,0xdc,0xf1,
  0x31,0xf1,0xdc,0x93,0xc1,0xfd,0x21,0x21,0x21,0xfe,0xfe,0xfe,0xfd,0xf1,0xfd,0xfd,
  0xff,0x21,0xff,0xff,0xff,0xdc,0x4d,0x4d,0x4d,0xff,0xff,0x4d,0xff,0xff,0xff,0xff,
  0xff,0xff,0x4d,0xff,0xf7,0x06,0xbf,0x92,0x92,0xfa,0xf9,0xbf,0xe0,0xff,0xe6,0xf9,
  0xcb,0xcb,0x71,0x06,0xcb,0x06,0x92,0x06,0x92,0xcb,0xf1,0xfa,0xfa,0xcb,0xcb,0xcb,
  0x71,0xcb,0xcb,0x71,0x92,0x92,0x92,0x06,0xbf,0x92,0xf9,0x06,0xf1,0xfe,0xfd,0xe0,
  0x4d,0xf7,0xff,0x21,0xff,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,0x21,
  0x21,0xe6,0xf9,0xbf,0xe0,0xfa,0xcb,0xfa,0x99,0xdc,0x4d,0x4d,0x4d,0xe0,0x21,0x21,
  0x21,0xfd,0xfd,0xc1,0xfe,0xdc,0xdc,0xfd,0xfd,0xfd,0xc1,0xc1,0xfd,0xfd,0xfd,0xc1,
  0xc1,0xfd,0xfd,0xfd,0xfd,0xfd,0xc1,0xc1,0xe6,0xe6,0x21,0xfd,0xfd,0xc1,0xfd,0xe6,
  0xff,0xff,0xff,0xff,0xff,0xff,0x4d,0x4d,0x4d,0xff,0xff,0x4d,0x4d,0xff,0x4d,0x4d,
  0x4d,0xff,0xff,0xe0,0xff,0xf9,0xe0,0x4d,0xe0,0x71,0xe0,0xfb,0x92,0xfb,0xfb,0xfb,
  0x71,0xff,0xfd,0x92,0xcb,0x92,0x92,0xfb,0xe0,0xff,0x4d,0x4d,0x4d,0x71,0xcb,0x71,
  0x71,0xfb,0x71,0x71,0xfb,0xe0,0xf9,0xfb,0x06,0xfb,0xe0,0xe0,0xf9,0xf9,0xe0,0xe0,
  0xe0,0xfa,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xf9,0xfd,0xe0,0xfd,
  0xfd,0xfd,0xf9,0xfd,0xc9,0xfd,0xfd,0x21,0xff,0xe0,0xe0,0xe0,0x4d,0xf1,0xf9,0x4d,
  0xe0,0xfd,0xc1,0xe6,0xe0,0x4d,0xdc,0xfd,0xc1,0xc1,0xfd,0xc1,0xdb,0xc1,0xc1,0xc1,
  0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc9,0xc1,0xc1,0xc1,0xc1,0xc9,0xc9,0xfe,0xff,
  0xf9,0xfd,0x21,0x21,0x21,0xff,0x21,0xfd,0xff,0xe6,0xfd,0xe6,0xe6,0xe6,0xe6,0xfd,
  0xe6,0xe6,0x21,0xe6,0xe6,0x21,0xe6,0x21,0xe6,0xe6,0xfd,0xe6,0xe6,0xe6,0xe6,0xe6,
  0xe6,0x21,0x21,0xe6,0x21,0xe6,0xe6,0xe6,0xe6,0x21,0xe6,0xe6,0x21,0x21,0x21,0x21,
  0x21,0xe6,0x21,0xe6,0x21,0x21,0xe6,0xe6,0xe6,0xfd,0xfd,0xfd,0xfd,0xc9,0xc1,0xfd,
  0xc1,0xc1,0xc1,0xc9,0xc1,0xc1,0xc9,0xc1,0xc1,0xdb,0xc1,0xc1,0xc1,0xc1,0xf9,0xf9,
  0xc9,0xf9,0xf9,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xc1,0xfd,0xc9,
  0xfd,0xfd,0xfd,0x21,0xfd,0x4d,0x21,0x21,0xfd,0x21,0x21,0xc9,0xc9,0xc9,0xc1,0xc1,
  0xc1,0xdb,0xc1,0xdb,0xc1,0xc9,0xc1,0xc1,0xc1,0xc9,0xc1,0xc1,0xc1,0xc9,0xc1,0xf9,
  0xf3,0xf3,0xdb,0xdb,0xdb,0xdb,0xdb,0xc1,0xc1,0xc1,0xc1,0xc9,0xc9,0xc9,0xc9,0xc9,
  0xc9,0xc9,0xc9,0xc9,0xf9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xc9,0xf9,
  0xc9,0xf9,0xc9,0xf9,0xf9,0xc9,0xc9,0xc9,0xc9,0xf9,0xc9,0xf9,0xc1,0xc1,0xc1,0xdb,
  0xdb,0xc1,0xc1,0xc1,0xdb,0xc1,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xf3,0xf3,0xf3,0xf3,
  0xf3,0xf4,0xf4,0xf4,0xf4,0xec,0xec,0xec,0xec,0xec,0xec,0xec,0xec,0xec,0xec,0xec,
  0xec,0xf4,0xf4,0xf4,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xdb,0xdb,0xdb,
  0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xdb,0xf3,0xf3,0xf4,0xf4,
  0xf4,0xec,0xec,0xec,0xec,0xf4,0xec,0xec,0xf4,0xf4,0xf4,0xf4,0xf4,0xf3,0xf4,0xf3,
  0xad,0xad,0xad,0xad,0xad,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,
  0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,
  0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xcb,0xad,0xad,0xad,
  0xad,0xcb,0xcb,0xcb,0xcb,0xcb,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
  0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
  0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xcb,0xcb,
  0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
  0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,0xad,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xcb,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0xcb,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0xcb,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xad,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xad,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xad,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x06,
  // 12: sky 2, part A
  0xdc,0xce,0xbc,0xce,0x9c,0xbe,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,
  0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x00,0x03,
  0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,0x00,0x05,
  0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,
  0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,
  0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,
  0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x7b,0xbe,0x10,0x24,0x18,0x24,
  0x00,0x25,0xbb,0xc6,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,
  0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,
  0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x7c,0xbe,0x18,0x41,0x00,0x42,0x08,0x42,
  0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,
  0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,
  0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,
  0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,
  0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,
  0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,
  0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,
  0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,
  0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,
  0x10,0x84,0x18,0x84,0x00,0x85,0x9b,0xbe,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,
  0x08,0x86,0x10,0x86,0x18,0x86,0xdc,0xbe,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,
  0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,
  0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,
  0x00,0xa4,0x08,0xa4,0x10,0xa4,0xfd,0xd6,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,
  0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,
  0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x00,0xc1,0x08,0xc1,0x10,0xc1,
  0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,
  0x18,0xc3,0x00,0xc4,0x3e,0xd7,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,
  0x7c,0xc6,0x10,0xc5,0x18,0xc5,0x00,0xc6,0x08,0xc6,0x1d,0xcf,0x10,0xc6,0xfe,0xce,
  0x18,0xc6,0x00,0xc7,0x08,0xc7,0x10,0xc7,0x18,0xc7,0x00,0xe0,0x08,0xe0,0x10,0xe0,
  0x1d,0xd7,0x18,0xe0,0x08,0xe1,0xbb,0xd6,0xfe,0xd6,0x18,0xe1,0x08,0xe2,0x18,0xe2,
  0x08,0xe3,0x10,0xe3,0x00,0xe4,0x08,0xe5,0xbc,0xd6,0x10,0xe5,0x08,0xe6,0x18,0xe6,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0xa3,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x02,0x02,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x01,0x02,0x02,0x01,0x02,0x02,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x01,0x01,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x4c,0x02,0x4c,0x4c,
  0x4c,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x01,0x02,0x01,0x01,0x02,0x01,
  0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x4c,0x4c,0x02,0x4c,0x02,0x4c,0x4c,0x4c,0x9b,0x4c,0x9b,0x02,0x02,0x02,
  0x02,0x4c,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x9b,0x02,0x02,0x02,0x02,0x02,0xa3,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,
  0x02,0x02,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x01,0xa3,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x9b,
  0x02,0x02,0x9b,0x02,0x4c,0x35,0x4c,0x02,0x02,0x35,0x02,0x02,0x02,0x02,0x35,0x02,
  0x35,0x35,0x4c,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x9b,0x4c,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x01,0x01,
  0x02,0x02,0x02,0x02,0x02,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x39,0x02,0x02,0x02,
  0x4c,0x02,0x4c,0x4c,0x02,0x02,0x02,0x02,0x02,0x02,0x9b,0x02,0x02,0x02,0x02,0x02,
  0x02,0x4c,0x9b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x4c,0x4c,0x4c,0x4c,0x02,0x4c,0x02,0x02,0x02,0x4c,0x02,0x02,0x4c,0x02,0x02,
  0x02,0x35,0x02,0x02,0x35,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x02,0x01,0x02,0xa3,0x02,0x02,0x02,0x9b,0x02,0x9b,0x4c,0x4c,0x02,0x9b,0x02,
  0x9b,0x39,0x9b,0x9b,0x9b,0x4c,0x35,0x4c,0x4c,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x9b,0x9b,0x02,0x02,0x4c,
  0x02,0x02,0x02,0x4c,0x02,0x4c,0x02,0x02,0x02,0x02,0x02,0x02,0x4c,0x02,0x9b,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x9b,
  0x4c,0x4c,0x02,0x4c,0x4c,0x35,0x02,0x02,0x02,0x4c,0x4c,0x35,0x4c,0x4c,0x02,0x02,
  0x02,0x02,0x35,0x35,0x35,0x35,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x02,0x02,0x02,0x02,
  0x01,0x01,0x02,0x02,0x02,0x02,0xa3,0x39,0x02,0x39,0x9b,0x02,0x02,0xe0,0x39,0x02,
  0x39,0x4c,0x4c,0xe0,0x02,0x02,0x4c,0x9b,0x4c,0x4c,0x35,0x35,0x02,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x4c,0x4c,0x39,0x4c,0x4c,0x4c,0x4c,
  0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x02,0x02,0x02,0x02,0x02,0x02,0xa3,0x01,0xa3,0x01,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x9b,0x9b,0x9b,
  0x9b,0x4c,0x9b,0x9b,0x9b,0x35,0x35,0x9b,0x9b,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x02,0x02,0x4c,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xa3,0x02,0x02,0x02,
  0x01,0x01,0xa3,0x02,0xa3,0x01,0xa3,0x01,0xa3,0xa3,0x01,0x01,0x01,0xa3,0x01,0x01,
  0x01,0x02,0x02,0x02,0x02,0x39,0x02,0x02,0x39,0x39,0x02,0x4c,0x9b,0x4c,0x4c,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x4c,0x4c,0x4c,0x4c,0x35,0x35,
  0x35,0x4c,0x4c,0x4c,0x4c,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x02,0x01,0x02,0x02,0x02,0x01,0x02,0x02,0xa3,0x02,0x02,0x02,0x02,0x02,0x4c,
  0x4c,0x4c,0x02,0x4c,0x4c,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x4c,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x4c,0x4c,0x9b,0x4c,0x02,0x02,0x02,0x02,0x01,0x02,0x01,
  0xa3,0xa3,0xa3,0x02,0x02,0x01,0xa3,0x02,0x02,0x02,0x02,0x02,0xa3,0x01,0x01,0x01,
  0x01,0x02,0x02,0xa3,0x01,0xa3,0xa3,0x01,0x01,0x4c,0x4c,0x35,0x4c,0x39,0x39,0x4c,
  0x4c,0x35,0x39,0x39,0x35,0x35,0x35,0x35,0x35,0x4c,0x39,0x39,0x4c,0x4c,0x4c,0x4c,
  0x4c,0x4c,0x39,0x39,0x02,0x02,0x02,0x01,0xa3,0x01,0x01,0xa3,0x01,0xa3,0xa3,0xa3,
  0xa3,0xa3,0xa3,0xa3,0x02,0x39,0x02,0xa3,0xa3,0x02,0x02,0x9b,0x02,0x02,0x9b,0x9b,
  0x4c,0x9b,0x4c,0x35,0x4c,0x9b,0x9b,0x02,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x4c,0x35,0x35,0x35,0x4c,0x39,0x9b,0x9b,0xa3,0x02,0xa3,
  0xa3,0xa3,0x02,0x02,0x01,0xe0,0x02,0xe0,0x39,0x39,0xe0,0xe0,0x02,0x01,0x01,0x02,
  0x01,0x01,0x01,0xa3,0x01,0x01,0x01,0x02,0x39,0x39,0x39,0x39,0x39,0xe0,0x39,0x39,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x39,0x39,0x39,0x39,0x39,0x39,0x4c,0x4c,0x02,
  0x02,0x9b,0x02,0x01,0xa3,0x9b,0x39,0x39,0x39,0x02,0x39,0x39,0x39,0x39,0x02,0x4c,
  0x39,0x02,0x39,0x9b,0x9b,0x39,0x39,0x02,0x02,0x9b,0x4c,0x9b,0x4c,0x4c,0x4c,0x9b,
  0x9b,0x9b,0x4c,0x9b,0x9b,0x9b,0x9b,0x9b,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,
  0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x35,0x4c,0x4c,0x9b,0x39,0x02,0x02,0x02,0x01,
  0xa3,0x02,0x02,0x39,0x39,0xe0,0xe0,0xe0,0xe0,0xe0,0x39,0x01,0x01,0x01,0x01,0x00,
  0x00,0xa3,0xa3,0x01,0xa3,0xa3,0x00,0x01,0x01,0x01,0xe0,0x01,0x01,0x01,0x02,0xe0,
  0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x4c,0xe0,0x39,0x01,0xe0,0x01,0x01,
  0x39,0x39,0x39,0x39,0xe0,0x39,0x39,0x39,0x39,0x39,0x39,0x9b,0x4c,0x4c,0x39,0x39,
  0x39,0x39,0x39,0x9b,0x9b,0x9b,0x39,0x4c,0x4c,0x39,0x9b,0x9b,0x9b,0x4c,0x9b,0x9b,
  0x4c,0x9b,0x9b,0x9b,0x9b,0x4c,0x35,0x35,0x35,0x35,0x4c,0x9b,0x9b,0x9b,0x9b,0x4c,
  0x9b,0x4c,0x9b,0x4c,0x35,0x35,0x4c,0x4c,0x35,0x4c,0x9b,0x35,0x35,0x39,0x35,0x35,
  0x4c,0x39,0x4c,0x35,0x35,0x35,0x4c,0x4c,0x39,0x39,0x39,0x01,0x01,0xa3,0xa3,0x01,
  0x01,0x01,0x00,0x01,0x01,0x02,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,
  0x00,0x00,0x00,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,
  0x02,0x01,0x9b,0xa3,0x02,0x01,0x02,0xe0,0xe0,0x39,0x4c,0x4c,0x39,0x39,0x39,0x39,
  0x39,0x4c,0x4c,0x39,0x4c,0x4c,0x39,0x4c,0x39,0x02,0x39,0x9b,0x9b,0x39,0x02,0x39,
  0x4c,0x4c,0x4c,0x4c,0x35,0x4c,0x35,0x35,0x35,0x35,0x4c,0x39,0x35,0x4c,0x39,0x9b,
  0x9b,0x9b,0x9b,0x9b,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0x4c,0xe0,0x39,0x39,0x39,0xe0,
  0xe0,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x01,0xa3,0xa3,0xa3,0x01,0x01,0x01,0x01,
  0x01,0x00,0x00,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0xe5,0xe5,0xe5,0xe5,
  0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe7,0xe5,0xe5,0x00,0x00,0x00,0xe5,0xe5,
  0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xa3,0x02,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xa3,0x01,0x01,0x39,0x02,0x02,0x39,0x02,
  0x39,0x39,0x39,0x39,0x39,0x39,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0xa3,0x01,0x02,
  0x9b,0x4c,0x4c,0x4c,0x4c,0x35,0x9b,0x35,0x35,0x9b,0x39,0x39,0x9b,0x9b,0x9b,0x9b,
  0x4c,0x39,0x9b,0x9b,0x9b,0x02,0x9b,0x39,0x02,0xa3,0xa3,0xa3,0x01,0x01,0x01,0x01,
  0x01,0xa3,0x00,0x01,0x00,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x01,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xe5,0xe5,0xe5,0xe5,0xe5,
  0xe5,0xe5,0xe7,0xe5,0xe7,0xe7,0xe7,0xe7,0xe5,0xe5,0xe5,0xe5,0x00,0x00,0x00,0xe5,
  0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0xa3,0x01,0x01,0x01,0xa3,0x01,0x01,0xa3,0xa3,0x01,0x01,0x01,0x01,0x02,0x01,
  0x01,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x01,0x01,0xa3,0x01,0x01,
  0x01,0x39,0x39,0x9b,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x02,0x02,0x02,0x01,0xa3,
  0xa3,0xa3,0xa3,0xa3,0x02,0xa3,0xa3,0x01,0x01,0x01,0x01,0x01,0xa3,0x01,0xa3,0x00,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0xa3,0x01,0x00,0x00,0x00,0x00,
  0x00,0xe5,0x00,0x00,0xe5,0xe5,0xe5,0xe5,0x00,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,
  0xe5,0xe5,0xe5,0xe5,0xe7,0xe7,0xe5,0xe7,0xe7,0xe5,0xe5,0xe5,0xe5,0xe5,0xe7,0x00,
  0x00,0x00,0xa3,0x00,0x00,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x02,
  0xa3,0x01,0x02,0x01,0x01,0xa3,0xa3,0x02,0x01,0xa3,0x01,0xa3,0xa3,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0xa3,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x01,0xa3,0x01,
  0x01,0x01,0x39,0x39,0x39,0x01,0x02,0x01,0x39,0x9b,0x01,0x01,0x01,0x01,0x01,0xa3,
  0x01,0x01,0x01,0x00,0xa3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe7,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,
  0xe5,0xe7,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x02,0xe0,
  0x01,0x01,0x02,0x02,0x02,0x02,0x4c,0x02,0x02,0x02,0x02,0x01,0xa3,0xa3,0xa3,0x01,
  0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0xa3,0x01,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0xa3,0x01,0x39,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xa3,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x39,0x39,0x01,0x01,0x01,0x39,0x39,0x39,0x39,
  0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xe5,0x00,0xe5,0x00,
  0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe7,0xe7,0xe5,0xe5,0xe5,0xbb,0xe5,0x00,0xe5,
  0xe5,0x00,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,
  0x00,0xe5,0xe5,0xe5,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,
  0x01,0x02,0x01,0x39,0x02,0x39,0x02,0x01,0x01,0x39,0x02,0x02,0x01,0x01,0x39,0x02,
  0x01,0x01,0x01,0x01,0xa3,0xa3,0x00,0x00,0x01,0x01,0xa3,0x01,0x01,0x00,0x01,0x01,
  0x00,0x00,0x00,0x00,0x01,0x01,0xa3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0xa3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x39,0x39,0x39,
  0x39,0x39,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xe5,0x00,
  0xbb,0xbb,0x01,0xbb,0x01,0x00,0x00,0xe5,0xbb,0xe5,0xe5,0xbb,0xbb,0xbb,0xbb,0xbb,
  0xbb,0xe5,0xe5,0xe5,0xe5,0xe5,0xf4,0xe5,0xf4,0xe5,0xe5,0xf4,0xe5,0xe7,0xe5,0xe5,
  0xe5,0xe5,0xe5,0x00,0xe5,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x39,0x01,0x01,0x39,0x39,0x39,0x39,0x39,0x39,
  0x01,0x39,0x39,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0xa3,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x00,0x00,0x01,0x01,0x00,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x39,
  0x01,0x01,0x39,0x01,0x01,0xbb,0x01,0xbb,0x01,0x01,0x01,0xbb,0x01,0xbb,0xbb,0xbb,
  0xe5,0x00,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xe5,0xbb,0xe5,0xe5,0xf4,0xe5,0xf4,
  0xda,0xda,0xda,0xe5,0xe5,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xf4,0xe5,0xe5,
  0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,
  0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x39,0x01,0x39,0x39,0x39,
  0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x01,0x39,0x01,0x39,0x39,0x39,0x39,
  0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x01,0xbb,0x01,0xbb,0xf3,0x00,0x01,
  0xbb,0x01,0xfc,0xfc,0xbb,0x00,0xbb,0xbb,0xbb,0x01,0x01,0xbb,0xbb,0xbb,0xbb,0xf0,
  0xbb,0xbb,0xbb,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xf4,0xda,0xda,0xda,
  0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xda,0xe7,
  0xf4,0xe7,0xe7,0xe7,0xe7,0xe5,0xe5,0xe5,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x39,0x01,0x39,0x01,0x39,0x39,
  0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x39,0x01,0x39,0x39,0x39,0x39,0x39,
  0x39,0x39,0x39,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x39,0x01,0x01,0x00,0x00,
  0x00,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0xbb,0xbb,0x00,0x00,0xbb,0xbb,0xbb,
  0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xf0,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xda,
  // 13: sky 2, part B
  0x9b,0xd6,0x9a,0xce,0xfc,0xce,0x1c,0xd7,0x1d,0xcf,0x3e,0xd7,0x00,0x00,0x08,0x00,
  0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,
  0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,
  0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,
  0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,
  0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,
  0x10,0x24,0x18,0x24,0x00,0x25,0x7e,0xdf,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,
  0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,
  0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,
  0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,
  0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,
  0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,
  0x08,0x60,0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,
  0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,
  0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,
  0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,
  0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,
  0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,
  0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,
  0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,
  0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,
  0x08,0xa2,0x7d,0xdf,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,
  0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x7a,0xce,0x10,0xa5,
  0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x3e,0xcf,0xbc,0xce,
  0x08,0xa7,0x10,0xa7,0xdb,0xd6,0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x18,0xc0,
  0x9e,0xdf,0x7e,0xe7,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,
  0xfd,0xd6,0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x18,0xc3,0x00,0xc4,
  0x08,0xc4,0xfb,0xce,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x10,0xc5,0x18,0xc5,
  0xdc,0xbe,0x00,0xc6,0x08,0xc6,0x10,0xc6,0x18,0xc6,0x00,0xc7,0x08,0xc7,0x10,0xc7,
  0x18,0xc7,0x7a,0xd6,0x00,0xe0,0x08,0xe0,0x10,0xe0,0x00,0xe1,0x10,0xe1,0xbc,0xd6,
  0x08,0xe2,0x18,0xe2,0xfe,0xce,0x08,0xe3,0x79,0xce,0x18,0xe3,0x10,0xe4,0x08,0xe5,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x3b,0x3b,0x05,0x05,0x05,0x3b,0x05,0x3b,0x05,
  0x05,0x05,0x05,0xfa,0x04,0xfa,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x04,0x02,0xe8,
  0x02,0xe8,0xe8,0xe8,0xe8,0xe8,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x00,0xc7,0xc7,
  0xc7,0xc7,0x00,0xc7,0x00,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xe8,0xc7,0xc7,
  0xc7,0xc7,0xc7,0xc7,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0x02,0xe8,0x02,0x02,0x02,
  0x04,0x02,0x02,0x04,0x04,0x02,0x02,0x02,0xc7,0x02,0x02,0xd8,0xd8,0xd8,0xd8,0xd8,
  0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x3b,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x3b,0xd0,0x05,0x3b,0x3b,0x05,0xd0,0x3b,0x05,0x05,0x05,0x3b,
  0x05,0x05,0x05,0xc6,0xc6,0xc6,0xfa,0xfa,0xfa,0xfa,0x04,0x04,0x04,0x04,0x04,0x02,
  0x04,0x04,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xc7,
  0xc7,0xc7,0xc7,0xe8,0xe8,0xe8,0xe8,0xc7,0xc7,0xc7,0xc7,0xc7,0xe8,0x02,0xe8,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0x04,0x04,
  0x02,0x04,0x02,0x02,0x04,0x04,0x04,0x04,0x02,0x04,0x02,0xd8,0xd8,0x02,0xd8,0xd8,
  0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0x05,0xd8,0xd8,0x05,0xd8,0x05,0x04,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x3b,0xd1,0xd0,0xd0,0xd0,0xd0,0xd0,0x3b,0xd0,0xd0,0xd0,
  0x3b,0xd0,0x05,0x05,0xc6,0xc6,0xfa,0xfa,0xfa,0xfa,0xfa,0x04,0x04,0x02,0x04,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0xe8,0xe8,0xe8,0x02,0xc7,0x02,0xc7,0xc7,0xe8,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x02,0xe8,0xe8,0xe8,0x02,0xc7,0x02,0x02,0x02,
  0x02,0x02,0x02,0x04,0x02,0x04,0x02,0x04,0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0xd8,0x04,0x04,
  0xd8,0xd8,0xd8,0xd8,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,0x3b,0x05,0x05,
  0x3b,0x05,0x05,0x05,0x05,0x05,0x3b,0x3b,0x05,0xd1,0xd1,0xd0,0x3b,0x05,0xd0,0xd0,
  0xd0,0xd1,0x3b,0x05,0xc6,0xc6,0xc6,0xc6,0xc6,0x04,0x04,0x04,0x04,0x04,0x04,0x02,
  0x04,0x04,0x02,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x02,0x02,0x02,0x02,0x02,
  0x02,0x04,0x04,0x04,0x04,0x04,0xfa,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,
  0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x3b,0x3b,0x05,0x3b,0xd1,
  0xd0,0xd0,0x3b,0x3b,0x05,0x05,0xc6,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x04,0x04,0x04,
  0x02,0x02,0x04,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0x04,0x04,0x04,0x02,0x02,0x02,
  0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,
  0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd0,
  0xd0,0xd1,0xd0,0x05,0x3b,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x04,
  0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x02,0x04,0x02,0x04,0x04,0x04,0x02,0x02,0x04,
  0x02,0x02,0x02,0x02,0x04,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x02,
  0x02,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x02,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0xc6,0x05,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0xc6,0x05,0xc6,0x05,0x05,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0xd0,0xd0,0x3b,0x3b,0x05,0x05,0xb1,0x05,0x05,0x05,0x04,0x04,0x02,0x04,
  0xe8,0xe8,0xe8,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0x04,0x02,0x02,0x02,0x02,
  0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x02,0x02,0x04,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x04,0x04,0x05,0x05,0x04,0x04,0x05,
  0x05,0x05,0x05,0x04,0x05,0xc6,0xc6,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,
  0x04,0x05,0x04,0x04,0xc6,0xc6,0x04,0x05,0xc6,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x3b,0x3b,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x02,0x04,0x02,
  0xe8,0xe8,0xc7,0xc7,0xe8,0x04,0xc7,0xe8,0xc7,0xe8,0xe8,0xc7,0xe8,0x02,0x02,0x02,
  0x02,0xe8,0xe8,0xe8,0xe8,0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x02,0x02,0x04,0x02,
  0x02,0x04,0x02,0x04,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0x05,0x05,0x04,0x05,0x05,
  0x05,0xc6,0xc6,0x04,0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0xfa,
  0x04,0xc6,0xc6,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0xd0,0x3b,0xd1,0x3b,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0xe8,0x02,0xc7,0x02,0xc7,0x04,0x02,0x02,0x02,0xe8,0x02,
  0x02,0x02,0xe8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x04,0x04,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xc6,0x04,0x04,0xc6,0x05,0x05,0x05,0x05,0xc6,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0xc6,0x05,0x05,
  0xc6,0xc6,0x05,0xc6,0xc6,0xc6,0xc6,0x05,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0xd0,0xd0,0xd0,0xd0,0xd0,0x05,0x05,0x05,0x05,0xc6,0xc6,0xc6,0xc6,0xfa,0x04,
  0x04,0x02,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0x02,0x02,0x02,
  0x02,0x04,0x04,0x02,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x02,0x02,0xe8,
  0xe8,0xc7,0x02,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x02,0x04,0x04,0x02,0x02,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x04,0x05,0x05,0x04,0x05,0x05,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0xc6,
  0xc6,0xc6,0x05,0x05,0x05,0x05,0xc6,0xc6,0x05,0x04,0xc6,0x05,0x05,0x05,0x05,0x3b,
  0x05,0xd0,0xd0,0xd0,0xd0,0xd1,0xd1,0xd0,0xd1,0xd0,0x05,0x05,0xc6,0xc6,0xfa,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x02,0x04,
  0x04,0xe8,0xe8,0xe8,0xc7,0xc7,0xc7,0xe8,0xe8,0x02,0x02,0x02,0x02,0xe8,0xe8,0xe8,
  0xc7,0xc7,0xe8,0x02,0xc7,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0xd8,0xd8,0x04,0x04,0x04,0x04,0x04,0x04,0x05,0x05,0x05,0x05,0xc6,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xc6,0x04,0x04,0x05,0xc6,0x05,0xc6,
  0x04,0x05,0xc6,0xc6,0x05,0x04,0x05,0x05,0x05,0x05,0xc6,0xc6,0x05,0x05,0x05,0x05,
  0x05,0x3b,0xd1,0xd0,0xd1,0xd1,0xd1,0xd0,0xd0,0xd0,0xd1,0x3b,0x05,0x05,0x04,0x04,
  0x04,0x04,0x04,0x04,0x02,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0xe8,0xe8,
  0xe8,0xe8,0xc7,0xc7,0xe8,0xe8,0xe8,0xc7,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xc7,
  0xe8,0x02,0x02,0x04,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0xd8,0x04,0x04,0x05,0xb1,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0xd8,
  0x05,0x05,0xc6,0x04,0xc6,0x05,0xc6,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0xd0,0xd0,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd0,0xd0,0x05,0x05,0x05,
  0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0xc7,0xe8,0xe8,
  0xe8,0xc7,0xc7,0xc7,0xc7,0xe8,0xc7,0xc7,0xe8,0xe8,0xe8,0xe8,0x02,0x02,0x02,0x02,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x04,
  0x04,0x04,0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x05,0xd8,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xd8,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,
  0x04,0x04,0xc6,0x05,0x05,0xc6,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x3b,0x05,0x05,
  0x05,0x3b,0x3b,0xd0,0xd0,0xd0,0xd1,0xd1,0xd1,0xd0,0xd0,0xd0,0xd0,0x3b,0x3b,0x05,
  0x3b,0x05,0x04,0x04,0x04,0x04,0x02,0x02,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,
  0xe8,0xe8,0xe8,0xe8,0x02,0x02,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xc7,0xe8,0x02,
  0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0xc6,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,
  0xb1,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x05,0x04,0x05,0xd8,0xd8,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x3b,0x3b,0xd0,0xd0,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0x3b,0xd0,0x05,0x3b,0x3b,0x05,
  0xb1,0x05,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0x02,0xe8,0xc7,0xc7,0xe8,0xe8,0x02,
  0xe8,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x02,0x02,0x02,0x04,0x02,0x02,0x02,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0xd8,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,0xb1,
  0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0x3b,0x05,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0xd1,0x3b,0x3b,0x3b,0x05,0xb1,
  0x05,0xb1,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x02,0x04,
  0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0xe8,0xe8,0xe8,0xe8,
  0xe8,0xe8,0xe8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x02,0x02,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xc6,0x05,0x05,0x05,0x05,0x05,0x05,
  0x04,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0xd8,0x05,0x05,0x04,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0xb1,0xb1,0x05,0x05,
  0x05,0x3b,0xd0,0xd1,0xd1,0xd1,0xd1,0x3b,0x3b,0x3b,0xd1,0xd1,0x3b,0x05,0xb1,0x05,
  0xb1,0x05,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0xe8,
  0xc7,0xe8,0x02,0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,
  0x02,0x02,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xc6,0x04,0x04,0x05,0x05,
  0x05,0x05,0x05,0x05,0xb1,0xb1,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0xb1,0x05,0x05,0x05,0x05,
  0xb1,0x05,0x05,0x3b,0x3b,0x3b,0xb1,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0x3b,0xb1,
  0x05,0x05,0x04,0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0x04,0xe8,0xe8,
  0xe8,0xe8,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0x04,0x05,0x04,0x05,
  0x04,0x04,0x04,0xd8,0x04,0x04,0x04,0x02,0x04,0x02,0x04,0x04,0x04,0x04,0x04,0x02,
  0x02,0x04,0x04,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0xc6,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0xb1,0xb1,0x05,0x3b,0x3b,0x05,0x05,0x3b,0x05,0x3b,0xd1,0xd1,0x3b,0x3b,0xb1,
  0x05,0x05,0x04,0x04,0x02,0x04,0x04,0x04,0xd8,0x02,0x04,0x04,0x04,0xd8,0x02,0x04,
  0x02,0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0xd8,0x02,0x04,0x04,0x04,0x02,0xe8,
  0xe8,0x02,0x02,0x04,0x04,0x04,0xd8,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x05,0x05,
  0x05,0x05,0xd8,0x04,0x04,0x04,0x04,0x02,0x02,0x04,0x02,0x04,0x02,0x02,0x04,0x04,
  0x02,0x02,0x02,0x02,0x04,0x04,0x04,0x04,0x04,0x02,0x04,0xd8,0xd8,0xd8,0xd8,0x05,
  0x05,0x05,0x05,0xb1,0xb1,0xb1,0xb1,0x05,0xd8,0xd8,0x04,0xd8,0xd8,0xd8,0xd8,0xd8,
  0x04,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,
  0x05,0xb1,0xb1,0xb1,0xb1,0x05,0x05,0x05,0xb1,0x3b,0x3b,0x3b,0xd1,0x3b,0x3b,0x3b,
  0x3b,0xb1,0xd8,0x04,0x02,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0xd8,0x02,0xd8,0x04,
  0x04,0x02,0x02,0x02,0x04,0x02,0x04,0x04,0x04,0x04,0x02,0x04,0x04,0x04,0x04,0x02,
  0xe8,0xe8,0xe8,0x02,0x02,0xd8,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x04,0xd8,0xd8,
  0x05,0x05,0x05,0x05,0xd8,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0xd8,0x02,0x02,0x04,0xd8,0xd8,0x05,0x05,
  0x05,0x04,0x05,0x05,0x05,0x05,0x05,0x05,0xd8,0x04,0x05,0x03,0xd8,0xd8,0x04,0x04,
  0x04,0xd8,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0xb1,0xb1,0x05,0xb1,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0x05,0x3b,0xb1,0x3b,0x3b,0xb1,0x3b,0x3b,0x05,
  0x3b,0xb1,0x05,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0x02,0x02,0xd8,0xd8,
  0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xd8,0x04,0x02,
  0xe8,0xe8,0x02,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xd8,0xd8,
  0xd8,0x05,0xd8,0x05,0x05,0x05,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x04,0x04,0x02,0x04,0x02,0x04,0x04,0x04,0x04,0xd8,0x04,0x04,
  0x04,0xd8,0x03,0xd8,0xd8,0x04,0x04,0x04,0x05,0xd8,0x05,0x03,0xd8,0x04,0xd8,0xd8,
  0x04,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0xb1,0xb1,0xb1,0x05,0x05,0x03,0x03,0x05,
  0x05,0xd8,0x05,0x05,0x05,0xb1,0x3b,0x3b,0x3b,0xb1,0x3b,0xb1,0xb1,0xb1,0xb1,0xb1,
  0xb1,0xb1,0xb1,0x05,0x04,0x02,0x02,0xe8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xd8,
  0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xd8,0x05,0xd8,
  0x02,0x02,0x02,0x02,0xd8,0x04,0x02,0x02,0x02,0x02,0x02,0x04,0x04,0xd8,0xd8,0xd8,
  0xd8,0x05,0x05,0xd8,0x04,0x04,0xd8,0xd8,0xd8,0x04,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0x04,0x04,0x02,0x04,0x04,
  0x02,0x04,0x02,0x02,0x04,0x02,0xd8,0xd8,0x04,0x04,0x02,0x02,0xd8,0x04,0x04,0xd8,
  0x02,0xd8,0xd8,0x04,0xd8,0x05,0xd8,0xd8,0x05,0x05,0x05,0x05,0x03,0x02,0xd8,0xd8,
  0xd8,0x05,0xd8,0x05,0x05,0xb1,0xb1,0x3b,0xd1,0x3b,0xb1,0xb1,0xb1,0x05,0x05,0xb1,
  0x05,0xb1,0xb1,0x05,0xd8,0x02,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0xd8,0x05,0x05,
  0x02,0x02,0x02,0x02,0x02,0x04,0x02,0xd8,0x04,0x02,0xd8,0x02,0xd8,0xd8,0x05,0xd8,
  0x05,0xd8,0xd8,0xd8,0x05,0x05,0xd8,0xd8,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x04,0x02,0x04,0x02,0x04,0x02,0x02,0x04,0x02,0x04,0x02,
  0xe1,0x02,0x04,0x02,0x02,0xe1,0xe1,0x02,0x02,0x02,0x02,0x04,0x04,0x02,0x02,0x02,
  0x02,0xd8,0x02,0x04,0x03,0xe1,0x03,0x03,0xd8,0xd8,0x05,0x05,0xd8,0x04,0xd8,0xd8,
  0xd8,0x05,0x05,0xb1,0x05,0xb1,0xb1,0x3b,0xb1,0xb1,0xb1,0x05,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0x02,0x02,0x02,0x02,0xd8,0x05,
  0xd8,0x04,0x02,0x02,0x04,0x04,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x04,0xd8,0xd8,
  0x05,0x03,0x03,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0x02,0x02,0x02,0x02,
  0x02,0x04,0x02,0x02,0x02,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x02,0xe1,0x02,0x02,0x02,0x02,0xe1,
  0x02,0x02,0x02,0x02,0x02,0xd8,0x02,0xd8,0x03,0x03,0xd8,0x03,0x03,0xd8,0x02,0x02,
  0xd8,0xb1,0x03,0x05,0xb1,0x03,0xb1,0xb1,0xb1,0xb1,0x05,0xd8,0xd8,0xd8,0xd8,0xd8,
  0xd8,0x02,0xd8,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0x02,0xc7,0x02,0x02,
  0x02,0x02,0xc7,0xc7,0x02,0x02,0x02,0xc7,0xc7,0x02,0x02,0x02,0x02,0x02,0x03,0x05,
  0x05,0x05,0xd8,0xd8,0xd8,0xd8,0xd8,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0xd8,0x02,0x02,0x02,0x02,0x02,0xd8,0xd8,0x03,0x03,0xd8,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe8,0x02,0x02,0x02,
  0x02,0xc7,0x02,0x02,0xc7,0x02,0x00,0x00,0x00,0xc7,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0xd8,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xb1,0xb1,0x03,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0xe1,0x02,0x02,0xe8,0x02,0x02,0x02,0x02,0x02,0xc7,0xc7,0xc7,0xc7,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x02,0xc7,0x02,0x02,0xe1,0xd8,0x02,
  0x02,0xd8,0xd8,0xd8,0xd8,0xd8,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0xe1,0x02,0x02,0x02,0x02,0x02,0x00,0x02,0xe8,0x02,0xe1,0xe1,0x02,
  0x00,0x00,0x00,0x00,0x00,0x00,0xc7,0x00,0x00,0xf7,0x00,0xc7,0xf1,0xf1,0xbe,0x00,
  0x00,0x00,0x00,0xbe,0x01,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0xca,0x00,0x00,0xe1,
  0x02,0x02,0x02,0x02,0x00,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x02,0x02,0x02,0x02,0x02,0x02,0xc7,0x02,0xc7,0x02,0x02,0xc7,0xc7,0xc7,0xc7,0xc7,
  0x00,0x00,0xc7,0xc7,0x00,0x00,0xc7,0xc7,0x00,0x00,0x01,0xc7,0x02,0x02,0xe1,0x02,
  0x02,0xe1,0xe1,0xe1,0xe1,0x02,0x02,0xe1,0x02,0x00,0x02,0x00,0xc7,0xc7,0xc7,0xc7,
  0xc7,0x02,0x02,0x02,0xc7,0xc7,0x02,0xe1,0x02,0x02,0xe1,0xe1,0xe1,0x02,0xe1,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe1,0xe1,0x00,0x02,0x02,0x00,0x00,0xc7,0x01,
  0xf1,0x00,0xca,0xf1,0xca,0xf1,0x00,0x00,0x00,0x00,0x00,0x00,0xf1,0x00,0x00,0x00,
  0x01,0xf1,0xbe,0xbe,0xfc,0xf1,0xbe,0xbe,0xbe,0xbe,0x00,0x00,0x00,0x00,0xbe,0x00,
  0xf1,0x00,0xf1,0x01,0xbe,0xbe,0xbe,0xbe,0xbe,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0xc7,0xc7,0xc7,0x00,0xc7,0x00,0xc7,0xc7,0x02,0xc7,0xc7,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x01,0x00,0x00,0x01,0xc7,
  0x01,0x00,0x00,0x00,0x02,0xe1,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x01,0x01,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x01,0x02,0xe1,0xe1,0x00,
  0xe1,0xe1,0x01,0xf1,0xf1,0xe1,0x00,0x00,0x02,0x00,0x01,0xf1,0xf1,0x00,0x00,0x01,
  0x00,0x00,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0xf1,0x00,0x00,0xf1,0x00,0x00,0xf1,
  0xf1,0x01,0x00,0xf1,0xf1,0xf1,0xf1,0xbe,0xf1,0xbe,0xbe,0xbe,0xbe,0xf1,0x01,0x01,
  0x00,0xf1,0xbe,0xbe,0xf1,0xf1,0xbe,0xbe,0xbe,0xbe,0xbe,0xf1,0xbe,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x01,0x00,0xf1,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xbe,0x00,0x00,0x00,0x00,0x01,0x01,0x00,
  0x01,0x01,0x00,0x01,0xca,0xe1,0xe1,0xe1,0x01,0x00,0x00,0xf1,0xf1,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,0x00,0x01,0x01,0xca,0xf1,0xca,0x01,0x00,0x00,0x00,0x00,0x02,
  0x01,0xf1,0xf1,0xca,0x01,0xc7,0x01,0xf1,0xf1,0xf1,0x00,0xca,0x01,0x01,0x00,0x00,
  0xf1,0xf1,0xf1,0xf1,0x00,0xf1,0xf1,0x00,0x00,0x00,0x00,0x00,0xf1,0xf1,0xf1,0x00,
  0x00,0x00,0x00,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xbe,0xbe,0xf1,0xbe,0xf1,0xf1,0x00,
  0xca,0xca,0x00,0x00,0x00,0xca,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
  0x00,0x00,0x01,0x00,0x00,0x00,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x01,
  0x01,0x00,0x00,0x00,0x00,0xca,0xca,0x00,0xca,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0xbe,0xbe,0xbe,0x00,0xf1,0xf1,0xf1,0x01,0xf1,0x00,0x00,0x00,0x00,0xbe,0xf1,0x00,
  0x00,0x00,0x00,0x00,0x01,0x00,0x00,0xf1,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0xf1,0xf1,0xf1,0xf1,0xf1,0x00,0xf1,0x01,0x01,0x00,0xf1,0xf1,0x00,0x01,0x01,0x00,
  0x01,0xf1,0x01,0xf1,0xbe,0x01,0x01,0xca,0x01,0xf1,0x00,0xca,0x00,0x00,0xca,0x00,
  0xca,0xca,0xca,0xca,0xf1,0xf1,0xca,0x00,0xf1,0xf1,0xf1,0x00,0xca,0xca,0x00,0x00,
  0x00,0xca,0xca,0xca,0xca,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0xf1,0x01,0x01,0x00,0xf1,0x01,0x01,0x01,0x01,0x01,0x01,0x00,
  0x01,0x01,0x00,0x01,0x01,0xca,0xca,0x00,0xca,0x00,0x00,0xf1,0xf1,0xbe,0xbe,0xbe,
  0xbe,0x01,0xf1,0xf1,0xf1,0xca,0xf1,0xf1,0xf1,0xf1,0xf1,0xbe,0xf1,0xf1,0xf1,0xf1,
  0xbe,0xbe,0x00,0xbe,0xbe,0x01,0x01,0xf1,0x00,0x01,0x01,0xf1,0xf1,0xf1,0xf1,0xf1,
  0xf1,0xf1,0x01,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xbe,0x01,0xf1,
  0xbe,0xbe,0xbe,0xbe,0x01,0x01,0xf1,0xf1,0xf1,0xf1,0xca,0xca,0xca,0x00,0x00,0xca,
  0xca,0xca,0xf1,0x00,0x00,0xf1,0xca,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0x00,0x00,0x00,
  0x00,0x00,0xca,0x00,0x00,0x00,0x00,0xca,0xbe,0x01,0x01,0xbe,0xf1,0xf1,0x00,0xf1,
  0xf1,0xf1,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0xf1,0x01,0xf1,0xf1,0x00,0xf1,0x01,
  0x00,0x01,0xf1,0x01,0xf1,0xf1,0xf1,0xca,0xf1,0x00,0x00,0xca,0x00,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0xf1,0xf1,0xf1,0x00,0x00,
  0xf1,0xf1,0xf1,0xbe,0xbe,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0x00,0x00,0xf1,
  // 14: sky 2, part C
  0x58,0xde,0x99,0xd6,0x9a,0xd6,0xda,0xde,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,
  0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,
  0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,
  0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,
  0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,
  0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,
  0x00,0x23,0x08,0x23,0x10,0x23,0x18,0xc6,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,
  0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,
  0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,
  0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,
  0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,
  0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x17,0xce,0x08,0x46,
  0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,
  0x10,0x60,0x18,0x60,0x00,0x61,0x7a,0xd6,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,
  0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,
  0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,
  0x78,0xe6,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,
  0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x58,0xf6,0x10,0x81,
  0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,
  0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,
  0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,
  0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,
  0x18,0xa1,0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,
  0x18,0xa3,0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0xf7,0xc5,
  0x10,0xa5,0x79,0xce,0x18,0xa5,0x99,0xe6,0xba,0xe6,0x00,0xa6,0x08,0xa6,0x10,0xa6,
  0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,
  0x18,0xc0,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,
  0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0xda,0xe6,0x38,0xf6,0x18,0xc3,0x00,0xc4,
  0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x10,0xc5,0x18,0xc5,0x00,0xc6,
  0x08,0xc6,0x10,0xc6,0x00,0xc7,0x9a,0xf6,0x08,0xc7,0x10,0xc7,0x18,0xc7,0x00,0xe0,
  0x08,0xe0,0x10,0xe0,0x18,0xe0,0x00,0xe1,0x08,0xe1,0x10,0xe1,0x99,0xf6,0x18,0xe1,
  0x00,0xe2,0x08,0xe2,0x10,0xe2,0xb6,0xbd,0x9a,0xce,0x5a,0xe6,0x00,0xe3,0x7a,0xce,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,
  0x01,0x6b,0xff,0x02,0x6b,0x02,0x02,0x02,0xfc,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,
  0x02,0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x6b,0xff,0x6b,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xfc,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0xfc,0xff,0xff,0xff,0x02,0xfc,0x02,0x02,0x02,0x03,0x02,0xff,0xff,
  0xfc,0xfc,0xfc,0x02,0xfc,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,
  0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x6b,0x02,
  0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x01,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,
  0x03,0x02,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,0xff,0x02,0x02,0x03,0x02,0x02,
  0x02,0x02,0x02,0x03,0x03,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x6b,0x02,0xfc,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x02,
  0x03,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x03,0x03,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x03,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x6b,0x02,0x6b,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x03,0x03,0x02,
  0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x02,0x02,0x02,0x02,
  0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x02,0x02,0x02,0x6b,0x6b,0x02,0x6b,0x02,0x02,0x02,0x6b,0x6b,0x02,0x02,0x6b,
  0x6b,0x6b,0x6b,0x02,0x6b,0x02,0x6b,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,0x03,0x03,0x03,0x03,0x02,0x03,
  0x02,0x03,0x03,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,
  0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x02,0x02,0x03,0x03,0x03,
  0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,0xfc,0x02,0x02,
  0x02,0x02,0x02,0x03,0x03,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xfc,
  0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x02,0x6b,0x02,0x6b,0x02,0x02,0x02,0x6b,0x02,0x6b,0x6b,0x02,0x6b,
  0x02,0x6b,0x6b,0x03,0x03,0x03,0xc4,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,
  0x02,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x02,0x03,0x03,0x02,
  0x02,0x02,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,0x6b,0x02,0x02,
  0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x03,0x02,0x6b,0x6b,0x02,0x6b,0x02,0x6b,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,
  0x02,0x02,0x02,0x6b,0x02,0x02,0x6b,0x6b,0x6b,0x02,0x02,0x02,0x01,0x02,0x6b,0x02,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x6b,0x02,0x02,0x02,0x02,0x6b,0x02,0x6b,
  0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,
  0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,
  0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x6b,0x03,0x03,0x03,0x6b,0x6b,
  0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x02,0x02,0x03,0x03,0x03,0x6b,0xc4,0x03,0x03,
  0xdc,0x03,0xc4,0xc4,0x6b,0x6b,0x02,0x02,0x02,0x6b,0x03,0x03,0x03,0x03,0x03,0x03,
  0xc4,0x03,0x03,0x03,0x03,0x02,0x02,0x6b,0x02,0x6b,0x6b,0x6b,0x01,0x02,0x02,0x02,
  0x6b,0x6b,0x02,0x6b,0x02,0x03,0x6b,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0xc4,0xc4,0xc4,0xc4,0xc4,0xc4,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,
  0x03,0x03,0x03,0x03,0x03,0x03,0x6b,0x6b,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x01,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0xdc,0xdc,
  0xdc,0xdc,0x03,0x02,0x6b,0x02,0x6b,0xc4,0xc4,0x03,0xc4,0xc4,0xc4,0xc4,0xc4,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xc4,0xc4,0x03,0x03,0x03,0xc4,0xc4,0x03,
  0x03,0xc4,0x03,0x03,0x03,0x02,0x03,0x03,0x03,0x02,0x02,0x6b,0x02,0x02,0x6b,0x02,
  0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x02,0x6b,0x6b,0x6b,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x6b,0x02,0x03,0x03,0x6b,0x02,0x02,0x03,0x03,0xc4,0xc4,0x03,
  0x03,0x03,0x03,0x03,0x03,0x6b,0x02,0xc4,0x6b,0x6b,0x02,0x02,0x6b,0x02,0x02,0x03,
  0x02,0x02,0x6b,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x6b,0x6b,0x6b,0x01,0x01,0x6b,0x02,0x02,0x02,0x6b,0x03,0x03,0x03,
  0xdc,0xdc,0xdc,0x03,0x03,0xdc,0x03,0x02,0x03,0x03,0x03,0xc4,0x03,0x03,0x03,0xdc,
  0xdc,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xdc,0x03,0x03,0x03,0x03,
  0x03,0x03,0xdc,0x03,0xc4,0x03,0x03,0x03,0x03,0xc4,0x02,0x02,0x02,0x02,0x02,0x02,
  0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,
  0x02,0x02,0x02,0x6b,0x03,0x03,0x03,0x02,0x03,0x03,0x02,0x6b,0x02,0x02,0x02,0xc4,
  0x03,0x03,0x03,0x03,0x03,0x02,0x6b,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x03,0xc4,
  0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x6b,0x6b,0x6b,
  0xc4,0x03,0x03,0x03,0x03,0xc4,0xc4,0x03,0x03,0x03,0x03,0xc4,0x03,0x03,0xdc,0xdc,
  0x03,0x03,0x03,0xdc,0xdc,0x03,0x03,0xdc,0x03,0x03,0x03,0xdc,0xdc,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,
  0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x02,0x02,
  0x02,0x02,0x02,0x6b,0x02,0x02,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x00,0x02,0x6b,0x03,0x03,0x03,0x03,
  0x02,0x03,0x03,0x03,0x03,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x02,0x02,
  0x6b,0x02,0x02,0x01,0x02,0x00,0x00,0x01,0x02,0x02,0x02,0x01,0x02,0x02,0x02,0x6b,
  0x02,0x02,0xc4,0x03,0xc4,0xc4,0xc4,0xdc,0xdc,0x03,0x03,0xc4,0xc4,0x03,0x03,0xc4,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x6b,0xc1,0x6b,0xc1,0x01,0x01,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x6b,0x02,0x02,0x02,0x03,0x03,0x03,
  0x03,0x03,0x02,0x02,0x02,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0xfc,0x02,
  0x02,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x01,0x01,0x02,0x02,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0xdc,0x03,0x03,0x03,0x02,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x03,0x03,0x03,0x02,0x03,0x03,0x03,
  0x03,0x02,0x02,0xc4,0x03,0x03,0xdc,0x03,0x03,0x03,0x02,0x02,0x02,0x03,0x6b,0x02,
  0x01,0x01,0x6b,0x6b,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0xc1,0x01,0x6b,0x01,0x01,
  0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x6b,0x02,0x02,0x02,0x02,0x02,
  0x03,0x02,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x6b,0xc1,0x01,0x01,
  0x6b,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0xdc,0x03,0x03,0x03,0x02,0x02,0x02,0x03,0x02,0x02,0x02,
  0x01,0x01,0x01,0x6b,0x6b,0x01,0x01,0x6b,0x6b,0x6b,0x01,0xc1,0xc1,0xc1,0xc1,0xc1,
  0xc1,0x01,0x01,0x6b,0x6b,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,
  0x01,0x01,0x02,0x02,0x01,0x02,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x01,0x02,0x6b,0x01,0x01,0x01,0x6b,0xc1,0x01,0x01,0x01,0x6b,
  0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x6b,0x01,0x01,0x01,0x01,0x01,0x01,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x01,0x6b,0xc1,0x6b,0x6b,0x01,0x01,0xc1,0xc1,0x6b,0xc1,0xc1,0xc1,0xc1,0xc1,0x01,
  0xc1,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x01,0x01,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x02,0x01,0x6b,0x6b,0x6b,0x01,
  0xc1,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x6b,0x01,0x01,
  0x01,0x02,0x02,0x02,0x03,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x01,0x6b,0x02,0x01,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x01,0x00,0x00,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,
  0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,
  0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xc1,0x01,0xc1,
  0xc1,0xc1,0x01,0x02,0x01,0x01,0x01,0x6b,0x01,0xc1,0x01,0x01,0xc1,0xc1,0x01,0x01,
  0x01,0x01,0x02,0x02,0x02,0x01,0x03,0x02,0x03,0x03,0x03,0x02,0x02,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0xdc,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x02,0x03,0xc4,0x03,0xdc,0xdc,0x03,0x03,0x03,0x03,0x03,0xc4,0xc4,0x01,0xc4,
  0xc4,0xc4,0x01,0xc4,0xc4,0xc4,0xc4,0xc3,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x01,0x01,0x02,0x02,
  0x01,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0xc1,0x01,0x01,0x01,0x02,0x01,
  0x01,0x01,0x01,0xc1,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0xc1,
  0x01,0x01,0x02,0x01,0x02,0x01,0xc1,0xc1,0xc1,0xc1,0x01,0x01,0xc1,0xc1,0xc1,0x01,
  0x01,0x02,0x02,0x02,0x02,0x01,0x02,0x02,0x01,0x02,0x03,0x01,0x03,0x03,0x03,0x03,
  0x02,0x03,0x02,0x02,0x01,0x01,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0xdc,0xdc,0x03,
  0xdc,0xdc,0xdc,0xdc,0x03,0xc4,0xc4,0xc4,0xc4,0xc4,0xc3,0xeb,0xeb,0xeb,0xc3,0xeb,
  0xc3,0xeb,0xc4,0xc3,0xc3,0xc4,0xc3,0x00,0x00,0x01,0x01,0x00,0x01,0x01,0x01,0x01,
  0x01,0x01,0xc1,0xc1,0x01,0x01,0x01,0xc1,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,
  0x01,0x01,0x02,0x01,0x01,0x01,0x01,0xc1,0x01,0x01,0x01,0xc1,0x01,0x01,0x01,0x01,
  0x01,0x01,0xc1,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x01,0x02,0x01,
  0x01,0x01,0x01,0x01,0x01,0xc1,0xc1,0xc1,0xc1,0x01,0xc1,0x01,0x01,0x01,0x01,0x02,
  0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x01,0x03,0x01,0x03,
  0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xc3,0xc3,0xc3,0xc4,0xc3,0xc3,0xc3,
  0xc4,0xeb,0xeb,0xf6,0xf6,0xf6,0xf6,0xf6,0x8e,0xf6,0x8e,0xf6,0x8e,0xf6,0xeb,0xeb,
  0xc3,0xc3,0xc3,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0xc1,
  0x01,0x01,0x01,0xc1,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x00,0x01,0x01,0xc1,0x01,0x01,0xc1,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0xc1,0xc1,0xc1,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x02,0x02,0x02,0x01,0x01,
  0x02,0x02,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0xc3,
  0xc3,0x01,0x01,0x01,0x02,0x02,0x01,0x02,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x01,
  0x02,0x02,0xfd,0xc3,0xfd,0xfd,0xfd,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xf6,0xf6,0xc3,0xeb,0xc3,0xc3,0xc3,0x01,
  0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0xc1,0xc1,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,
  0x01,0xc1,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x01,0x00,
  0xc3,0xc3,0x01,0xc3,0x8e,0xc3,0x8e,0x8e,0x8e,0x8e,0x8e,0xc3,0xc3,0xc3,0xc3,0xc3,
  0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x01,0x02,
  0xc3,0xc3,0xdd,0xdd,0x8e,0xdd,0x8e,0xdd,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x8e,0x8e,0x8e,0x8e,0x8e,0xf6,0xeb,0xeb,0xc3,0xc3,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x00,0x01,0x01,0x00,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x00,
  0x00,0x01,0x01,0x00,0x00,0x00,0x00,0xc3,0x80,0x80,0xc3,0xc3,0xc3,0xc3,0x8e,0xdd,
  0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xc3,0xc3,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xc3,0xc3,0xc3,0xfd,0xfd,
  0xdd,0x8e,0xdd,0xdd,0xdd,0xdd,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0xf6,0xf6,0xf6,0xeb,0xeb,0xc3,0xdc,0xdc,0xdc,0xdc,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x00,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x01,0x01,0x00,0x00,0xc1,0x01,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0xc3,
  0x8e,0x8e,0x8e,0x80,0xc3,0xc3,0xc3,0x8e,0xdd,0xdd,0x8e,0xdd,0xdd,0x8e,0x8e,0x8e,
  0x8e,0x8e,0x8e,0x8e,0x8e,0xc3,0xc3,0x01,0xc3,0xc3,0xc3,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xc3,0xc3,0xc3,0xc3,0x8e,0x8e,0x8e,0x8e,0x8e,
  0xdd,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xf6,0xeb,0xdc,
  0xc4,0xdc,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x00,0x80,0x8e,0x8e,
  0x8e,0xdd,0xdd,0x8e,0x8e,0x8e,0x8e,0xdd,0xdd,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x8e,0x8e,0xc3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xdc,0xdc,
  0x01,0x01,0x01,0x01,0x01,0x01,0xc3,0x00,0xc3,0xc3,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xeb,0xc3,0xc3,0xc3,0xdc,0xc3,0x01,0xdc,0x01,
  0x01,0xdc,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x00,0x00,0x80,0x80,0x80,0x80,0x80,0x8e,0x80,0x80,0x80,0x80,0x8e,
  0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xdd,0xdd,0xdd,0xdd,0x8e,0xc3,0x00,0xc3,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0xc3,0xc3,0xc3,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xdd,0xdd,0x8e,0x8e,
  0x8e,0x8e,0xc3,0xc3,0xc3,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xc3,
  0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x80,0x8e,0x80,0x80,0x80,0x80,0x8e,0x8e,0xdd,
  0xdd,0xdd,0xdd,0xdd,0xc3,0x80,0x80,0x00,0x80,0x80,0x00,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0xc3,0x00,0x01,0x01,0x00,0x8e,0xdd,0xdd,0xc3,0xc3,0xc3,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,
  0xc3,0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x00,0x00,0x00,
  0x00,0x00,0x80,0x8e,0x8e,0x80,0x80,0x8e,0x8e,0x8e,0xdd,0xdd,0xdd,0xdd,0xdd,0x8e,
  0x8e,0x80,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x01,0x01,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x80,0xc3,0x8e,
  0x8e,0x80,0x00,0x00,0x01,0x00,0x80,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x01,0x01,
  0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x80,0xc3,0x00,0x00,0x01,0x01,
  0x00,0x00,0x00,0x00,0x01,0x00,0x01,0x00,0x00,0x01,0x01,0x00,0x01,0x01,0x01,0x01,
  0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x00,0x00,
  0x00,0x00,0x80,0x8e,0x80,0x00,0x00,0x00,0x00,0x80,0x80,0x80,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,
  0x00,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x01,0x01,0x01,
  0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x01,
  0x00,0x00,0x00,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x80,0x80,0x00,0x00,0x00,0x00,0x80,0x8e,0x8e,0x80,0x00,0x00,
  0x00,0x00,0x00,0x00,0x33,0x33,0x33,0x33,0x33,0x00,0x33,0x33,0x33,0x33,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x33,0x33,0x33,
  0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x00,
  0x33,0x00,0x00,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
  0x33,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
  0x33,0x33,0x33,0x33,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x33,0x33,0x33,0x33,0x00,0x00,0x00,0x00,
  0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x33,0x00,0x00,0x33,0x00,0x00,0x33,0x33,0x33,0x00,0x33,0x33,
  0x33,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,
  0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x00,0x33,0x33,
  0x33,0x33,0x33,0x33,0x33,0x33,0x00,0x33,0x33,0x33,0x00,0x33,0x33,0x33,0x33,0x33,
  0x33,0x33,0x5e,0x33,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,0x33,0x33,0x33,0x33,
  0x5e,0x5e,0x5e,0x5e,0x33,0x33,0x33,0x33,0x33,0x5e,0x33,0x33,0x33,0x33,0x33,0x5e,
  0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x5e,0x5e,0x33,0x33,0x33,0x33,
  0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,0x33,
  0x33,0x33,0x33,0x33,0x33,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0xbf,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,0x5e,0x5e,0x33,0x5e,0x5e,0x33,0x5e,0x5e,0x5e,
  0x33,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,0x33,0x33,0x33,0x33,0x33,
  0x5e,0xbf,0x5e,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0x5e,0x5e,0x5e,0x5e,0x5e,
  0xfb,0xfb,0xfb,0xbf,0xbf,0xbf,0xfb,0xfb,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,
  0xbf,0xbf,0xbf,0xbf,0x5e,0x5e,0x33,0x5e,0x5e,0x33,0x33,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,
  0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,
  0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x5e,0x33,0x33,0x33,0x5e,0x33,0xbf,0xbf,0xbf,0xbf,
  0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,0xbf,
  // 15: sky 2, part D
  0x4d,0xac,0x52,0xb5,0xf1,0xac,0xd5,0xc5,0xd6,0xc5,0x00,0x00,0x08,0x00,0x10,0x00,
  0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,
  0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,
  0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,
  0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,
  0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,
  0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,
  0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,
  0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,
  0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,
  0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,
  0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,
  0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,
  0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,
  0x18,0x62,0x08,0x63,0x00,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,
  0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,
  0x18,0x66,0x0d,0x94,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x30,0x84,
  0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,
  0x08,0x82,0x2c,0xa4,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,
  0x00,0x84,0x12,0xad,0x08,0x84,0xb5,0xbd,0x18,0x84,0x31,0xbd,0x00,0x85,0x08,0x85,
  0x10,0x85,0x94,0xc5,0x18,0x85,0x00,0x86,0x12,0xb5,0x08,0x86,0x10,0x86,0x18,0x86,
  0xac,0x8b,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,
  0x18,0xa0,0x4e,0x9c,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x11,0xad,
  0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x6e,0xa4,0x08,0xa3,0x10,0xa3,0x18,0xa3,
  0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x18,0xa5,0x00,0xa6,
  0x08,0xa6,0xb0,0xa4,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0xb6,0xbd,
  0x18,0xa7,0x00,0xc0,0xf0,0xb4,0x08,0xc0,0x71,0x94,0x10,0xc0,0x18,0xc0,0x00,0xc1,
  0x54,0xbd,0x08,0xc1,0xaf,0xac,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,
  0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x18,0xc3,0xed,0x8b,0x00,0xc4,0x08,0xc4,
  0x10,0xc4,0x93,0xbd,0xb0,0x9c,0x37,0xce,0x18,0xc4,0x00,0xc5,0x4d,0x9c,0xe9,0x5a,
  0x08,0xc5,0xf2,0xac,0xed,0x93,0x93,0xc5,0x0a,0x73,0x72,0xc5,0x8b,0x83,0x2e,0x94,
  0x4a,0x7b,0x08,0xc6,0x38,0xce,0xd1,0x9c,0x88,0x6a,0x58,0xd6,0x2f,0x8c,0x98,0xde,
  0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0xeb,0xeb,0xeb,0x04,0x04,0x04,0xeb,0xeb,0xeb,0xeb,0xeb,0x04,
  0xeb,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xd8,0x03,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xeb,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0xeb,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xd8,0xcf,0xcf,0xcf,0xcf,
  0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0x04,0xcf,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0xcf,0x99,0x02,0x01,0x03,0x03,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xcf,0xcf,
  0xcf,0xcf,0xcf,0xcf,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x99,0x02,0x01,0xcf,0xcf,0xcf,
  0xcf,0xcf,0xcf,0xcf,0xcf,0x04,0xcf,0x04,0xcf,0xcf,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x03,0xcf,0x03,0x03,0x02,0xc9,0x02,0x02,0x01,0x03,0x03,
  0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0xeb,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xcf,0xcf,0xcf,
  0xcf,0xcf,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0x9b,0xf1,0xc9,0x02,0x99,0xe9,0x9b,0xcf,
  0x9b,0xcf,0xcf,0x03,0xcf,0x03,0x03,0x03,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x9b,0x02,0xc9,0xc9,0x02,0x02,0xa4,0x01,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x04,0x04,0x04,0x04,
  0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xeb,0xeb,0x04,
  0x04,0x04,0xeb,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xcf,0xcf,
  0xcf,0xcf,0xcf,0x9b,0x9b,0x9b,0x9b,0x9b,0x02,0xc9,0xc9,0x02,0x02,0x01,0x01,0x9b,
  0x9b,0x9b,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x04,
  0x04,0x03,0x04,0x04,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x04,0x04,0x03,0x03,0x04,
  0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x03,0x03,0x03,0x03,0x9b,0x9b,0xf1,0xc9,0xc9,0xc9,0xc9,0x02,0x02,0xa4,0xd8,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0xcf,
  0xcf,0xcf,0xcf,0x9b,0x9b,0x9b,0xa1,0x02,0xc9,0xc9,0xc9,0xc9,0x02,0x02,0xa4,0x01,
  0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x04,0xa1,0x04,0x03,0x04,0x03,0x04,0x03,
  0x03,0x03,0x03,0x03,0x9b,0x9b,0x01,0xc9,0xc9,0xea,0xc9,0x02,0x02,0x02,0xa4,0xa4,
  0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x03,0x04,0x04,0x04,0x03,
  0x03,0x03,0x04,0x03,0x03,0x04,0x03,0x04,0x03,0xa1,0x04,0x04,0x04,0x04,0x04,0x03,
  0x03,0x03,0x9b,0x9b,0x9b,0x9b,0xa4,0xc9,0xc9,0xea,0xc9,0x02,0x02,0x02,0xa4,0xa4,
  0x9d,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0xa1,0x01,0xa1,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x99,0xfb,0x01,0x03,0x03,0x04,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x01,0x02,0xc9,0xc9,0xc9,0xc9,0x02,0x02,0x02,0x02,0x9d,
  0x01,0xd8,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x99,0xfb,0x01,0x04,0x03,0x04,0x03,0x03,
  0x03,0x03,0x03,0x9b,0xa1,0x01,0x02,0xc9,0xc9,0xc9,0xc9,0x02,0x02,0x02,0x02,0x9d,
  0xa4,0x9d,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0xd8,0x01,0x03,0xeb,0xeb,0xff,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xd8,0xc9,0xfb,0xfb,0x01,0x03,0x03,0x03,0x03,
  0x03,0x9b,0x03,0x9b,0x01,0x99,0x02,0x02,0x02,0x02,0xc9,0x02,0x02,0x02,0x02,0xa4,
  0x01,0x9d,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xd8,0xc9,0xfb,0xfb,0x01,0x03,0x03,0x03,0x03,
  0x03,0xa1,0xa1,0xd8,0xa1,0x01,0x02,0x02,0x02,0x02,0xc9,0x02,0x02,0x02,0x02,0xa4,
  0xa4,0xa4,0x9d,0x01,0xa1,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xa1,0xfb,
  0xd4,0xf1,0xa1,0xa1,0x03,0xeb,0xeb,0xff,0xeb,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0xe9,0x02,0x02,0x02,0xfb,0x02,0xd8,0x03,0x9b,0x9b,
  0x9b,0xa1,0xa1,0x01,0xa4,0x99,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0xa4,0xa4,0x9d,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0xe9,0x02,0x02,0x02,0xfb,0x02,0xd8,0x03,0xa1,0x99,
  0x99,0xfb,0x01,0xa1,0xe9,0x03,0x03,0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0xa4,0x9d,0x9d,0x01,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xf1,0xfe,0xd4,
  0xea,0xc9,0xd8,0xe9,0xa1,0xeb,0x03,0xa1,0xeb,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x9b,0xa1,0x02,0x02,0x02,0x02,0xfb,0x02,0xfb,0x01,0x9b,0xa1,
  0xa1,0xa1,0x01,0x01,0xa4,0x99,0x02,0xd2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xd2,
  0xa4,0x9d,0x01,0x9d,0x01,0xa1,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0xcf,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0xa1,0x02,0x02,0x02,0x02,0xc9,0xc9,0xfb,0xf1,0xd4,0xd4,
  0xfb,0xfe,0x99,0x01,0x03,0x03,0xa1,0xa1,0xa1,0x01,0x99,0x02,0x02,0x02,0xd2,0xd2,
  0x02,0xf1,0x99,0xd8,0xa1,0xd8,0x01,0xa1,0x03,0xa1,0x01,0xea,0xe5,0xf6,0xe5,0xe5,
  0xd4,0xf1,0xd8,0x01,0x99,0x03,0xd8,0x03,0xeb,0x03,0x03,0x03,0x9b,0x03,0x03,0x03,
  0x03,0x03,0x03,0x9b,0xa1,0x99,0x02,0xfb,0xfb,0x02,0xfb,0xfb,0xfb,0x02,0x01,0xa1,
  0xe9,0xe9,0xd8,0x01,0xa4,0xa4,0xa4,0x02,0xd2,0x02,0x02,0xd2,0xd2,0x02,0xd2,0x02,
  0xd2,0x01,0x9d,0x9d,0xf5,0x01,0xa1,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0xd8,0x03,0x03,0x03,0x03,0x03,0x03,0xa1,0xd4,0xf1,0xe5,0xa1,0x03,
  0x03,0x03,0x03,0x9b,0xa1,0x99,0x02,0xfb,0xfb,0x02,0xf1,0xe9,0x01,0xe9,0x01,0x99,
  0xfe,0x99,0xd8,0xa1,0x01,0x01,0xd8,0x03,0xf3,0xa1,0x03,0x01,0x01,0xf1,0x99,0x02,
  0x01,0x99,0x99,0x99,0x01,0xfb,0xfe,0xd4,0xfe,0xfe,0xe5,0xd4,0xfe,0xf8,0xf6,0xf6,
  0xf2,0xfe,0xf1,0xfb,0xa1,0xd8,0xe9,0x03,0x03,0xe9,0x03,0x03,0xe9,0x01,0xa1,0xe9,
  0x03,0x03,0x03,0xa1,0x9d,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x01,
  0xe9,0x01,0xf5,0xf5,0x01,0xa4,0xa4,0xb7,0x02,0xd2,0x02,0xd2,0xd2,0x02,0xd2,0xd2,
  0xd2,0xa4,0x9d,0x01,0x01,0xf5,0x01,0xa1,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x99,0x03,0x03,0x03,0x03,0x03,0x99,0xfb,0xf4,0xf4,0xe5,0xe5,0xfb,
  0x03,0x03,0x03,0xa1,0x01,0x02,0x02,0x02,0xfb,0xfb,0x01,0xd8,0xe9,0xa1,0xa1,0x99,
  0xb1,0x99,0x01,0x03,0x01,0x02,0xfb,0xfb,0xf7,0xea,0xe9,0xe9,0xc9,0xf1,0xfb,0x01,
  0x01,0xfb,0xfe,0xb1,0xe5,0xe5,0xd4,0xd4,0xd4,0xc9,0x02,0xe5,0xea,0xf8,0xf8,0xa8,
  0xf6,0xf2,0xd4,0xfb,0xfb,0xfb,0x01,0xf1,0x03,0x03,0x04,0x03,0xeb,0xeb,0xa1,0x9b,
  0x01,0xe9,0xa1,0x01,0x99,0x02,0xb7,0x02,0x02,0x02,0x02,0xfb,0x02,0x02,0x02,0x02,
  0x01,0x01,0xf5,0xf5,0x01,0x01,0xa4,0xa4,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0xb7,
  0xd2,0xa4,0x9d,0x9d,0xf5,0x01,0x01,0xf3,0xf3,0xa1,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0xf1,0x03,0x03,0x03,0x03,0x03,0x02,0xfe,0xf6,0xf4,0xf6,0xe5,0x99,
  0x01,0x03,0xa1,0x01,0x99,0xa4,0x9d,0xd2,0xf5,0x02,0xf1,0xf1,0xfb,0xf1,0x99,0x99,
  0xa1,0xa1,0xd8,0xa1,0x01,0xa1,0x03,0xb1,0xfe,0xd4,0xf7,0xf6,0xe5,0xf6,0xe5,0xd4,
  0x01,0xd4,0xfb,0x01,0x01,0x01,0xc9,0xa8,0xf7,0xd4,0xea,0xd4,0xf7,0xf7,0xf2,0xa8,
  0xf6,0xa8,0x81,0xea,0xfb,0x02,0xfb,0xd4,0x01,0xd8,0x03,0xa1,0xeb,0x04,0x03,0xa1,
  0xa1,0xa1,0xa1,0xd8,0x99,0x99,0xb7,0x99,0x99,0xb7,0x02,0x02,0x02,0x02,0x02,0x02,
  0x99,0xf5,0xf5,0xf5,0xf5,0x9d,0x01,0x9d,0x9d,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,0xa4,
  0xa4,0xa4,0x9d,0x9d,0x01,0x01,0xf5,0x01,0xf3,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0xf1,0xeb,0xeb,0x03,0x03,0xfb,0xd4,0xe5,0xf6,0xf6,0xe5,0xe5,0xf4,
  0xfe,0xe9,0x01,0x99,0x01,0xf3,0xa4,0x01,0x03,0xa4,0xc9,0x02,0x02,0x02,0x02,0xd8,
  0x02,0xfb,0xfb,0xfb,0xf1,0x01,0xf5,0x03,0x01,0xa1,0xa1,0x02,0xf1,0xd4,0xfe,0xf1,
  0x02,0x99,0x01,0x01,0xa1,0x01,0xf1,0x02,0xa8,0xf7,0xf7,0xf2,0xe5,0xf7,0xd4,0xf7,
  0xe5,0xf6,0xb1,0xb1,0xf7,0xd4,0xd4,0xc9,0x99,0x02,0x02,0x01,0x03,0xa1,0x03,0x03,
  0x03,0x03,0xe9,0xa1,0xa4,0x99,0xa4,0xa4,0x99,0xa4,0xa4,0x99,0xb7,0xb7,0x99,0xa4,
  0xa4,0xa4,0xf5,0xf5,0xf5,0xf5,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,0x9d,
  0x9d,0xa4,0x9d,0x9d,0x9d,0x01,0x01,0x01,0x01,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,
  0xf3,0xf3,0xf3,0xeb,0xfd,0xeb,0xeb,0x03,0x02,0xd4,0xfe,0xe5,0xfc,0xe5,0xd4,0xfb,
  0xfb,0xf3,0x01,0xf3,0x03,0x03,0xf3,0x03,0x03,0x03,0x9d,0xa4,0xd8,0x99,0x02,0x01,
  0x02,0x01,0x02,0xd4,0xf1,0x01,0x01,0x03,0x03,0xd2,0x01,0xa1,0x01,0x01,0xd8,0x99,
  0xc9,0x02,0xfb,0x02,0xfb,0x02,0xfb,0xc9,0xa8,0xe5,0xe5,0xa8,0xa8,0xf8,0xa8,0xf7,
  0xc9,0xe5,0xf7,0xb1,0xb1,0xb1,0xb1,0xea,0x02,0x99,0x01,0xa1,0xe9,0xa1,0xe9,0xa1,
  0xd8,0x9b,0xd8,0x01,0xa1,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xa4,0x01,0xa4,0xa4,
  0x01,0x01,0x01,0x9d,0x9d,0x9d,0x9d,0x9d,0xd2,0xd2,0xa4,0x01,0x9d,0x9d,0x9d,0x9d,
  0x9d,0xa4,0xa4,0x9d,0x9d,0xa4,0x9d,0x9d,0x9d,0x9d,0xf5,0xd2,0xe5,0xc9,0xe5,0xb7,
  0xf5,0x99,0xf3,0xcf,0xfb,0xd4,0x99,0x01,0x02,0xf3,0xf3,0xea,0xef,0xb7,0xf3,0xf5,
  0x01,0xf5,0xa1,0x03,0xeb,0x03,0x03,0xe9,0xa1,0xa1,0x01,0x03,0x03,0x01,0xe9,0xd8,
  0x01,0xd8,0x02,0x99,0x01,0x01,0x02,0x99,0x99,0xe9,0xe9,0x01,0xa4,0xbc,0x01,0x02,
  0x02,0xf7,0xea,0xfb,0xb1,0xb1,0xb1,0xa8,0xa8,0xa8,0xf6,0xe5,0xe5,0xf4,0xf4,0xf4,
  0xf6,0xf8,0xf8,0xa8,0xe5,0xe5,0xf7,0xb1,0xb1,0x99,0x01,0xa1,0x01,0x01,0x9b,0x01,
  0xa1,0xf1,0x01,0xa1,0x01,0xa1,0xfb,0xfb,0xa4,0xf5,0xf5,0xf5,0xf5,0x01,0x01,0x01,
  0x01,0x01,0x9d,0x9d,0x01,0x01,0xa4,0xa4,0xd2,0xd2,0xd2,0xa4,0xd2,0x99,0x99,0xfb,
  0xd4,0x02,0xf5,0xf5,0xf5,0xf5,0xf5,0xf5,0xf5,0x01,0xf5,0xc9,0xfc,0xe5,0xf4,0xbc,
  0xf3,0x02,0xd8,0x04,0xf1,0xe5,0xe5,0x99,0x02,0xf5,0x9d,0x02,0xf4,0x01,0x01,0xf5,
  0xf5,0xa1,0x03,0xeb,0x03,0x04,0xa1,0xeb,0xff,0xff,0xff,0xeb,0x03,0xa1,0x01,0xf3,
  0xf3,0xa4,0xf7,0x99,0xa4,0xb7,0xd8,0xd8,0x02,0xda,0xa4,0x02,0xd2,0xbc,0x02,0xa4,
  0xbc,0xc9,0x81,0xa8,0xa8,0xa8,0xa8,0xf6,0xf6,0xa8,0xa8,0xf6,0xf6,0xf8,0xf4,0xf4,
  0xf4,0xf8,0xa8,0xf6,0xf6,0xa8,0xe5,0xe5,0xe5,0xd4,0xf1,0x02,0x02,0x99,0xd8,0x99,
  0x01,0xd8,0x99,0xfb,0xa1,0xfb,0x87,0xfb,0xfe,0xa4,0x01,0x01,0x01,0x01,0x01,0x01,
  0xa4,0xa4,0xb7,0xd2,0xd2,0xd2,0xd2,0x9d,0x9d,0xd2,0xd2,0xa4,0xd2,0x03,0xd8,0xfb,
  0xe5,0x01,0x99,0xd4,0xf5,0x9d,0xf5,0xf5,0xf5,0xf5,0x9d,0xbc,0xef,0xf7,0xef,0x99,
  0xeb,0xfd,0xfd,0xff,0xfa,0x9b,0x9b,0x04,0xfd,0xa1,0x9d,0x02,0x71,0xa4,0x01,0x01,
  0x03,0x03,0x03,0x03,0x03,0x03,0xe9,0xe9,0xa1,0xf3,0xa1,0x03,0x03,0x03,0x03,0xa1,
  0x01,0xe9,0xc9,0xb1,0xe5,0x02,0xd8,0xe9,0xa4,0x02,0xda,0xda,0xee,0xda,0xbc,0xbc,
  0xbc,0xb1,0xbc,0x81,0xa8,0xf6,0xf6,0xf6,0xa8,0xa8,0xa8,0xf6,0xf6,0xf8,0xf8,0xf6,
  0xf8,0xf6,0xa8,0xa8,0xe5,0xe5,0xf6,0xf6,0xf6,0xe5,0xfb,0xfb,0xd4,0xb1,0xd8,0xd8,
  0x99,0x99,0x99,0xfe,0xd4,0xd4,0x01,0xd4,0xd4,0xd8,0x01,0x99,0xa4,0x99,0xa4,0xb7,
  0xa4,0xb7,0xa4,0xd2,0xd2,0xd2,0xd2,0xd2,0x9d,0x9d,0xd8,0xe5,0xd8,0xd8,0x03,0x87,
  0xb1,0x03,0x01,0x02,0xfb,0xe9,0xeb,0x03,0x03,0xf7,0x71,0x4e,0x4e,0x4e,0xfc,0xf6,
  0xd4,0xb1,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0x99,0x99,0xb1,0xfc,0xf4,0xb7,0x01,
  0xf5,0xf5,0xa4,0xa1,0xa1,0x03,0xfb,0x02,0xe9,0xe9,0xeb,0xeb,0x03,0x03,0x03,0x03,
  0x03,0xeb,0xa4,0xe9,0x01,0x01,0x02,0xc9,0x02,0xa4,0x02,0x02,0xc9,0xbc,0xb1,0xf7,
  0xb1,0xb1,0xbc,0x02,0x02,0xbc,0x81,0x81,0xa8,0xa8,0xf6,0xa8,0xf6,0xf6,0xf6,0xa8,
  0xa8,0xa8,0xf6,0xa8,0xe5,0xf6,0xf6,0xe5,0xa8,0xe5,0xd4,0xb1,0xf7,0x81,0xf7,0xf6,
  0xf6,0xfb,0x87,0xe5,0xfb,0x99,0x01,0xfb,0xfb,0xfb,0x99,0xd8,0xf1,0x99,0xf1,0x99,
  0x02,0x02,0xf1,0x02,0xd2,0xd2,0xd2,0xd2,0xd2,0xf3,0xa1,0xd8,0x02,0xe9,0xa1,0xd4,
  0xe5,0xf7,0xf7,0x01,0xa1,0xfb,0xa1,0x03,0xa1,0xd8,0xe5,0x4e,0x4e,0x4e,0x4e,0xf4,
  0x81,0xfa,0xff,0xd4,0xf4,0xa8,0xd4,0xe5,0xf4,0xf6,0xe5,0xe5,0xfc,0xfc,0x02,0x01,
  0x9d,0x01,0x01,0x02,0xfb,0x99,0xfb,0x02,0xe9,0x01,0xd8,0x03,0xf3,0x9d,0x03,0x01,
  0xf3,0xa1,0xa1,0x02,0x01,0x03,0xa1,0xe9,0xa4,0xbc,0xa4,0x02,0x02,0xbc,0xd4,0xb1,
  0xc9,0xb1,0xb1,0xf7,0xbc,0x02,0xb1,0xbc,0xbc,0xbc,0xbc,0x81,0xf2,0xa8,0xa8,0xf6,
  0xf6,0xa8,0xa8,0xa8,0xf6,0xa8,0xe5,0xa8,0xa8,0xf6,0xe5,0xa8,0xe5,0xf6,0xf6,0xd4,
  0x87,0x99,0x87,0xd4,0xd4,0xf6,0x01,0xfb,0xf1,0x87,0x87,0xfb,0x99,0x04,0x04,0xfb,
  0xb1,0x02,0xd4,0xf1,0xf3,0x03,0xfb,0xf3,0xfb,0x03,0x99,0xa1,0xfb,0xf1,0x99,0xea,
  0x02,0xf6,0xfe,0x03,0xe9,0xfb,0x87,0x01,0x99,0x99,0x99,0xfc,0xfc,0x4e,0x4e,0xef,
  0xfc,0xfc,0xe5,0xe5,0xfc,0x4e,0xfc,0xf6,0xe5,0xe5,0xf4,0xe5,0xef,0x02,0x99,0xfb,
  0x02,0xea,0xfe,0xf7,0xe5,0xfe,0x01,0x99,0x01,0xfb,0x02,0xa1,0x03,0xf3,0xe9,0x01,
  0xf5,0xd8,0x01,0x01,0xf3,0xf5,0x03,0x01,0xe9,0xd2,0xbc,0x01,0xda,0xda,0xda,0xc9,
  0xbc,0xbc,0xbc,0xda,0xbc,0xbc,0x81,0xee,0xda,0xbc,0x02,0xb1,0xb1,0x81,0x81,0x81,
  0xf2,0xf2,0xf2,0xa8,0xa8,0xe5,0xe5,0xa8,0xe5,0xd4,0xd4,0xfe,0x87,0xd4,0x01,0x99,
  0xfb,0xf6,0xd4,0x99,0x99,0xfb,0x01,0x01,0xd8,0xa1,0xa1,0x01,0xf6,0xf4,0xf4,0xf6,
  0xe5,0xef,0xef,0xe5,0xf7,0xd2,0x03,0x9d,0xf6,0xf4,0x87,0x01,0xfb,0xfb,0x01,0xfb,
  0xd4,0x99,0x99,0xd4,0xfb,0x9b,0x99,0x03,0x03,0xa1,0x03,0xfc,0x4e,0x4e,0x4e,0xef,
  0xfc,0xf4,0xa8,0xe5,0xe5,0xf4,0xfc,0xe5,0xe5,0xf7,0xf6,0xa8,0xf4,0xe5,0xd4,0xd4,
  0xfb,0xd8,0x01,0xf1,0xfb,0xfb,0x01,0xe9,0xa1,0x01,0x9b,0x03,0x03,0x03,0x01,0x02,
  0xd8,0x01,0xa4,0xf3,0x03,0xf3,0x01,0xa4,0xa4,0x02,0xb1,0xb1,0xa4,0xbc,0x91,0x91,
  0xbc,0xbc,0xbc,0xbc,0xda,0x02,0x02,0x02,0xbc,0xb1,0xbc,0x02,0x02,0xbc,0xc9,0x02,
  0xbc,0x9d,0xd2,0xa4,0xbc,0xc9,0xc9,0xf1,0xfb,0xfb,0xfb,0x87,0xe5,0xe5,0x87,0xfe,
  0xf6,0xf6,0x87,0x99,0x9b,0xa1,0xfb,0xd4,0xfb,0xe5,0x71,0xf6,0xef,0xef,0xef,0xef,
  0xef,0xef,0xef,0xfc,0xef,0xef,0xfc,0xef,0xf4,0xf4,0xe5,0xe5,0xf1,0x99,0xf1,0x02,
  0xe9,0xfb,0xa4,0x02,0xc9,0xd2,0xd2,0xa4,0xd2,0xd2,0x02,0xbc,0xf6,0xef,0xfc,0xef,
  0xf4,0xe5,0xe5,0xe5,0xe5,0xe5,0xf7,0xf7,0xe5,0xe5,0x71,0xfc,0xfc,0xf4,0xf6,0xf8,
  0xf6,0xfc,0xd8,0xeb,0x04,0xa1,0xfb,0xd4,0xd4,0xfe,0xfe,0xd4,0x99,0x02,0xd4,0xb1,
  0xf7,0xc9,0xd2,0xf5,0x9d,0x01,0xf5,0x01,0xd2,0xf5,0x9d,0x01,0xf5,0xda,0x02,0x02,
  0xc9,0xb1,0xe5,0xf6,0xb1,0xbc,0x02,0x02,0xd2,0x02,0xd2,0x9d,0x01,0xd8,0xa1,0xe9,
  0xd8,0xf5,0x01,0xa4,0xf5,0xa4,0x9b,0x03,0xd8,0x01,0xd8,0x03,0xf1,0x99,0x01,0xfb,
  0xd4,0xfb,0x02,0xfb,0xf1,0x02,0xef,0x71,0xfc,0x71,0xf6,0xf4,0xf6,0xe5,0xf6,0xef,
  0x87,0xef,0xf4,0xef,0xf4,0xf4,0xfc,0xfc,0xef,0xf6,0xd4,0xd4,0xa4,0xd2,0xd2,0x02,
  0x02,0xbc,0xbc,0xbc,0xd2,0x02,0xd2,0x9d,0x01,0x9d,0x01,0xe9,0xea,0xfe,0xe9,0xe9,
  0x03,0x03,0xa1,0x01,0x03,0x03,0xff,0xff,0x03,0xeb,0x9b,0xeb,0x01,0xe9,0xf1,0xd4,
  0xfb,0xe5,0xc9,0xfb,0x99,0x01,0xd8,0xa1,0x03,0xa1,0x01,0x9b,0xeb,0xa1,0xfb,0x99,
  0xea,0xfb,0x02,0x02,0xd2,0xbc,0xc9,0x02,0x9d,0xd8,0xa4,0xa4,0xa4,0x9d,0xa4,0x02,
  0xa4,0x9d,0x9d,0xf7,0xa8,0xe5,0xf6,0xe5,0x81,0x81,0xf2,0xb1,0x02,0x9d,0xa4,0x01,
  0xa1,0xf5,0x03,0x01,0x01,0xf1,0xa1,0xa1,0x01,0x99,0x02,0x01,0xd8,0x03,0x9b,0x03,
  0x9b,0xa1,0xfb,0xf1,0x01,0x01,0xd4,0xfb,0xa1,0x99,0xfb,0x87,0xf6,0xf4,0xe5,0xd4,
  0x87,0xe5,0x87,0xe5,0xf7,0xf6,0xf6,0xf6,0xf6,0xe5,0x02,0xfb,0xf2,0xee,0x02,0xa4,
  0xd2,0x9d,0x9d,0xf2,0xf6,0xf8,0xfc,0xf4,0xe5,0xf2,0x81,0xc9,0xf5,0x03,0xa1,0x03,
  0xff,0x03,0xeb,0x03,0x03,0xd8,0xa1,0xa1,0x01,0x01,0x01,0xd8,0x9b,0xeb,0x03,0xeb,
  0x03,0xa1,0xf1,0x01,0xa1,0xd8,0x99,0x99,0x04,0x9b,0x01,0x01,0xfb,0xeb,0xeb,0x03,
  0x03,0x01,0x99,0xe9,0x01,0xe5,0xf6,0xf4,0xf4,0xf8,0xf6,0xe5,0xc9,0xbc,0xa4,0xf5,
  0xa4,0x02,0x9d,0x9d,0xa4,0xd2,0xbc,0xa8,0xf6,0xe5,0xb1,0x02,0xc9,0xd2,0x02,0xbc,
  0xe5,0xb1,0x99,0xea,0x03,0xe9,0xe9,0xe9,0x02,0x03,0xd8,0xa1,0x03,0x03,0xe9,0x03,
  0xd8,0x01,0x03,0x9b,0xeb,0x03,0xfd,0xeb,0xeb,0xa1,0xa1,0xa1,0x9b,0xf1,0xf1,0xd8,
  0xd8,0x02,0x01,0x02,0xc9,0xb1,0xee,0x91,0xbc,0xbc,0xbc,0xa8,0xf2,0xda,0x9d,0xf5,
  0xf3,0xd2,0xf5,0xf5,0xd2,0x02,0xee,0xa8,0xf8,0xf2,0xb1,0xa4,0xa4,0x9d,0x02,0x02,
  0x81,0xb1,0x01,0xc9,0x03,0xa1,0xa1,0xa1,0x02,0xeb,0x03,0x03,0xeb,0x03,0xe9,0xeb,
  0xd8,0x01,0x03,0x03,0xfa,0xeb,0xfd,0xeb,0xeb,0xa1,0xa1,0xa1,0x03,0xf1,0x99,0xd8,
  0x04,0xa1,0x9b,0x02,0xb1,0xb1,0xbc,0xbc,0xee,0x81,0xbc,0xb1,0xf2,0x02,0xa4,0x9d,
  0xbc,0x02,0xa4,0xa4,0x02,0xda,0xbc,0x81,0xf2,0xf8,0xf2,0xee,0x91,0x81,0xf6,0xf4,
  0xf8,0xe5,0xf2,0xf8,0xf2,0xf7,0xf2,0xe5,0xe5,0xc9,0xb1,0xb1,0x02,0xb1,0xf7,0x01,
  0xd8,0xc9,0xc9,0xb1,0xfb,0xd8,0x01,0x03,0xfa,0xeb,0x01,0x99,0x02,0x99,0xa4,0x01,
  0xd4,0xfe,0xfb,0x01,0x02,0xbc,0x91,0x81,0x81,0x91,0x91,0x81,0x00,0x00,0x91,0xbc,
  0xda,0xda,0x9d,0x9d,0xd2,0xda,0xbc,0x81,0xf2,0xf4,0xa8,0x91,0x91,0x81,0xf6,0xfc,
  0xf8,0xf2,0xa8,0xf4,0xf2,0x81,0xa8,0xa8,0xe5,0x02,0xb1,0xb1,0xd2,0xbc,0xf7,0xe9,
  0xd8,0xc9,0xc9,0xb1,0xfb,0xd8,0x01,0x03,0xfa,0xeb,0x01,0x99,0x02,0x99,0x99,0x99,
  0xd4,0xfe,0xfb,0x01,0x02,0xbc,0x91,0x81,0x81,0x91,0xbc,0xee,0xbc,0xbc,0xbc,0xda,
  0x91,0xbc,0x81,0xbc,0xbc,0x91,0xbc,0xbc,0x91,0x81,0x81,0x81,0x81,0x81,0xb1,0x81,
  0xb1,0x81,0x81,0x81,0xf2,0x81,0x81,0xee,0x81,0xf2,0xf2,0x81,0xf2,0x81,0xbc,0xc9,
  0xe9,0x02,0xf7,0xbc,0x02,0x02,0xbc,0xf2,0xc9,0x01,0xa1,0xb1,0xe9,0xd8,0xf1,0x99,
  0xa4,0xc9,0x01,0x02,0x01,0xc9,0xbc,0x81,0x81,0x81,0x91,0x81,0x91,0x91,0xbc,0x81,
  0x81,0x91,0x81,0xbc,0xbc,0x91,0xbc,0xbc,0x91,0x81,0x81,0x81,0xee,0x81,0xb1,0xee,
  0x81,0x81,0x81,0xf2,0xf2,0x81,0x81,0xee,0x81,0xf2,0x81,0x81,0xf2,0x81,0xbc,0xc9,
  0xe9,0x02,0xf7,0xbc,0x02,0x02,0xbc,0xf2,0xc9,0x01,0xa1,0xb1,0xe9,0xd8,0xf1,0x02,
  0x02,0xc9,0x01,0x02,0x01,0xc9,0xbc,0x81,0x81,0x81,0x91,0x81,0x91,0x91,0xbc,0x81,
  0x91,0x81,0x91,0x81,0x81,0x81,0x81,0x91,0x81,0x91,0x91,0x91,0x91,0x81,0x81,0x91,
  0x91,0x91,0x81,0x91,0x91,0x91,0x91,0x91,0x81,0x91,0x81,0x81,0x81,0x81,0x81,0x81,
  0xee,0xbc,0xb1,0x81,0x81,0xa8,0x81,0x81,0x81,0x81,0xbc,0x81,0x81,0x81,0xf2,0x81,
  0x81,0x81,0xee,0x81,0x81,0x81,0x91,0x91,0x91,0x00,0x81,0x81,0x81,0x81,0x81,0xda,
  0x91,0x81,0x91,0x81,0x81,0x81,0x81,0x91,0x81,0x91,0x91,0x91,0x91,0x81,0x81,0x91,
  0x91,0x91,0x81,0x91,0x91,0x91,0x91,0x91,0x81,0x91,0x81,0x81,0x81,0x81,0x81,0x81,
  0xee,0xbc,0xb1,0x81,0x81,0xa8,0x81,0x81,0x81,0x81,0xbc,0x81,0x81,0x81,0xf2,0x81,
  0x81,0x81,0xee,0x81,0x81,0x81,0x91,0x91,0x91,0x00,0x81,0x81,0x81,0x81,0x81,0xda,
  0x00,0x91,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x91,0x00,0x00,0x00,0x00,0xbc,0xbc,
  0x00,0x00,0x91,0x00,0x00,0x00,0x91,0x00,0x91,0x91,0x91,0x91,0x00,0x91,0xbc,0x00,
  0x91,0x91,0x91,0xbc,0x91,0xbc,0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x00,0xbc,
  0x00,0x00,0x00,0x00,0x00,0xbc,0x91,0x00,0x00,0x00,0x91,0x00,0x91,0x91,0x91,0x00,
  0x00,0x91,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x91,0x00,0x91,0x00,0x00,0xbc,0x00,
  0x00,0x00,0x91,0x00,0x00,0x00,0x91,0x00,0x91,0x91,0x91,0x91,0x00,0x00,0x00,0x00,
  0x00,0x00,0x91,0xbc,0x91,0xbc,0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x91,0x00,0xbc,
  0x00,0x00,0x00,0x00,0x00,0xbc,0x00,0x00,0x00,0x00,0x91,0x00,0x91,0x91,0x91,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  // 16: sky 3, part A
  0xf9,0xad,0xfa,0xad,0xb9,0xad,0x1b,0xb6,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,
  0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,
  0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,
  0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,
  0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,
  0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,
  0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,
  0x00,0x25,0x08,0x25,0xfb,0xb5,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,
  0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,
  0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,
  0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,
  0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,
  0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,
  0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,
  0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,
  0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,
  0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,
  0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,
  0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x5c,0xbe,
  0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,0x08,0x86,
  0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,
  0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,
  0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,0x08,0xa4,
  0x10,0xa4,0x18,0xa4,0x00,0xa5,0x5b,0xbe,0x08,0xa5,0x10,0xa5,0x18,0xa5,0x00,0xa6,
  0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0x00,0xc0,
  0x08,0xc0,0x10,0xc0,0x18,0xc0,0xd8,0xb5,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x18,0xc1,
  0x00,0xc2,0x08,0xc2,0x19,0xb6,0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,
  0x18,0xc3,0x00,0xc4,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x10,0xc5,
  0x18,0xc5,0x00,0xc6,0x08,0xc6,0x10,0xc6,0x18,0xc6,0x00,0xc7,0x08,0xc7,0x7d,0xc6,
  0x10,0xc7,0x18,0xc7,0x00,0xe0,0x08,0xe0,0x10,0xe0,0x18,0xe0,0x00,0xe1,0x08,0xe1,
  0x10,0xe1,0x18,0xe1,0x1c,0xae,0x00,0xe2,0x08,0xe2,0x10,0xe2,0x3c,0xb6,0x08,0xe3,
  0x10,0xe3,0x00,0xe4,0x1a,0xae,0x18,0xe4,0x00,0xe5,0x08,0xe5,0x00,0xe6,0x08,0xe7,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0x02,0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,
  0x02,0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0xc2,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0xc2,0xc2,0xc2,0xc2,
  0x02,0x02,0x02,0xc2,0xc2,0xc2,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,
  0x02,0x02,0xc2,0xc2,0x02,0x02,0x02,0x02,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0xc2,0xc2,0x02,0x02,0xc2,0x02,0xc2,0x02,0x02,
  0x02,0x02,0x02,0xc2,0x02,0x02,0xc2,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,
  0x02,0x02,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0xc2,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,
  0x02,0x02,0x02,0x02,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,
  0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,
  0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0xc2,0x02,0xc2,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,
  0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x02,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0xc2,0xc2,0x02,0xc2,
  0xc2,0xc2,0xc2,0x00,0x00,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x00,
  0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x02,
  0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x02,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0xc2,0xc2,
  0xc2,0xc2,0x00,0x00,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,
  0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,
  0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,
  0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,
  0x02,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0xc2,0xc2,
  0xc2,0xc2,0xc2,0xc2,0x02,0xc2,0x02,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x02,0x02,0x00,0x00,0x02,0x02,0x02,0x00,0x02,0x02,0x00,0x02,0x02,0xc2,
  0xc2,0xc2,0xc2,0xc2,0x02,0xc2,0x02,0x02,0x00,0x02,0x00,0x02,0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,
  0xc2,0xc2,0xc2,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x00,
  0x00,0x00,0x00,0xc2,0x00,0x00,0x00,0x02,0x00,0x00,0xcb,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x01,0x01,0x00,0xfa,0x00,0x00,0x00,0x01,0x01,0x01,0x00,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x02,0x02,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x02,0x00,
  0x01,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0xc2,
  0xc2,0x02,0xc2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0xcb,0x00,0x02,0x00,0x00,0x00,
  0x00,0xd2,0x00,0x00,0x00,0x01,0xfa,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xfa,0x01,
  0x01,0x01,0xd2,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x3a,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0xc2,0x02,0xc2,0xc2,0xc2,0x02,0x02,0x02,0x00,0x00,0x02,0x00,0x02,0x00,0x00,0x00,
  0x00,0x02,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0xc2,
  0x02,0xc2,0xc2,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0xcb,0x00,0x00,0x00,0x00,0x00,0xcb,0xcb,0x00,0x02,0x02,0x02,0x00,
  0x00,0x00,0x00,0x00,0xd2,0xfa,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x03,0x01,0x01,0xd2,0x01,0x01,0x01,0x03,0x03,0x01,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,
  0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x02,0x02,
  0xc2,0x02,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0xcb,0x00,0x00,0x02,0x00,0x00,
  0xcb,0xd2,0x00,0x01,0x01,0xfa,0x01,0x01,0x01,0xfa,0x03,0x03,0xd2,0x01,0x01,0xd2,
  0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xbb,0xbb,0xbb,
  0xbb,0xbb,0x03,0x03,0x03,0x03,0x3a,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x00,0x02,0x02,0x02,0xc2,0x02,0x02,0x00,0x00,0x02,0x02,0x01,0x00,0x02,0x02,
  0x01,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0xc2,0x02,
  0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xcb,0x00,0x00,
  0x00,0x00,0x00,0x00,0xcb,0x02,0x02,0xc2,0x02,0xcb,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x01,0xd2,0x01,0x01,0x01,0x01,0x01,0x03,0x01,0x03,0x03,0x03,0x03,0xd2,
  0x01,0x03,0x03,0xbb,0xbb,0x03,0x03,0xbb,0xbb,0xbb,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,
  0xbb,0x97,0x03,0x03,0x03,0x03,0x03,0x03,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x00,
  0x02,0x00,0x00,0xc2,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x01,0x02,0x01,0x01,0x02,
  0x02,0x01,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x02,0x02,0x02,
  0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0xcb,0x00,0x02,0x02,0xcb,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,
  0x00,0x00,0x00,0xd2,0x01,0x01,0x01,0x01,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0xbb,0xbb,0x03,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0x03,0xbb,0xbb,
  0xbb,0x97,0x97,0x97,0x97,0xbb,0xbb,0x03,0x03,0x01,0x01,0x01,0x01,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x00,0x02,0x00,0x02,
  0x02,0x02,0xc2,0x02,0x02,0x02,0x00,0x01,0x01,0x02,0x00,0x02,0x02,0x02,0x01,0x01,
  0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xcb,
  0x00,0x00,0x00,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0xcb,0x02,0xcb,0x00,0x00,
  0x01,0x01,0xd2,0x00,0x01,0xd2,0xd2,0x03,0x03,0x03,0x01,0x01,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0x03,0xbb,0xbb,
  0x97,0x97,0xbb,0xbb,0x97,0x97,0x03,0xbb,0x03,0x03,0x03,0x03,0x3a,0x01,0x01,0x01,
  0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0x02,0x00,
  0x02,0x02,0x02,0x02,0x00,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x02,
  0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0xc2,0x02,0xc2,0x02,0xc2,0x02,0xc2,0x02,0x02,0x02,0x00,0x00,0x00,
  0x01,0x01,0x01,0x01,0xd2,0xd2,0x03,0x03,0x03,0x03,0x01,0x03,0x03,0x03,0x03,0x01,
  0x03,0x03,0x03,0x03,0x03,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0x97,
  0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xbb,0xbb,0xbb,0x03,0x03,0x3a,0x01,0x01,
  0x01,0x01,0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x02,0x02,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x02,0x02,0x02,
  0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x01,0x01,0x01,0x02,0x00,0x00,0x02,
  0xc2,0x02,0x02,0x00,0x02,0x00,0x02,0x02,0x00,0x02,0x02,0x00,0x00,0x00,0x00,0x00,
  0xcb,0xcb,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0xcb,0x00,
  0xd2,0x01,0x01,0xd2,0xd2,0xd2,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x01,
  0x01,0x01,0x3a,0x03,0x03,0x03,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0x97,0x97,0xbb,
  0x97,0x97,0xe7,0xe7,0x97,0x97,0xbb,0x97,0x03,0x03,0x03,0x03,0x03,0x03,0x3a,0x01,
  0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x00,0x02,0x00,0x02,0x02,0xc2,
  0x02,0x02,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x01,
  0x02,0x01,0x02,0x01,0x01,0x01,0x02,0x02,0x01,0x01,0x00,0x01,0x00,0x02,0x00,0x02,
  0x00,0x02,0x02,0x00,0x00,0x02,0x02,0x02,0x00,0x02,0x02,0x02,0x02,0x00,0x00,0x00,
  0xcb,0x00,0xcb,0x02,0x02,0x02,0xc2,0x02,0xc2,0x02,0x02,0x02,0x02,0xcb,0x00,0xcb,
  0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0x03,0xd2,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x01,0x01,0x01,0x01,0x03,0x03,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0x03,0xbb,0x97,
  0x97,0xe7,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xbb,0xbb,0xbb,0x03,0x03,0x01,0x01,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x00,
  0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x02,0x02,0x02,0x01,0x01,0x02,0x00,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
  0x02,0x02,0x02,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x02,0x00,0x00,0x00,
  0x00,0xcb,0xcb,0x00,0xc2,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xcb,0xcb,0xcb,
  0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0x01,0x03,0xd2,0x03,0x03,0x03,0x03,0x03,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x03,0x03,0x03,0xbb,0x03,0x97,0xbb,0xbb,0xbb,0x97,0xbb,
  0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xbb,0x03,0x03,0x03,0x03,0x3a,0x01,
  0x01,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x00,0x00,0x02,0x02,
  0x02,0x02,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,
  0x02,0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x00,0x00,0x01,0x00,
  0x00,0x00,0x02,0x00,0x00,0x02,0x02,0x02,0x00,0x02,0x02,0xcb,0x00,0x02,0x02,0x02,
  0x02,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0xcb,
  0xcb,0xd2,0x00,0xd2,0x00,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0x03,0x03,0x01,0x03,0x01,
  0x3a,0x01,0x01,0x00,0x01,0x01,0x03,0xbb,0xbb,0xbb,0x03,0x03,0x03,0x03,0x97,0x97,
  0x97,0x97,0x97,0xe7,0xe7,0xe7,0x97,0x97,0x97,0x97,0x03,0x03,0xbb,0x03,0x03,0x03,
  0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x00,0x00,0x00,0x00,0x00,0x02,0x02,
  0x00,0x00,0x01,0x01,0x01,0x01,0x3a,0x03,0x01,0x3a,0x01,0x01,0x01,0x01,0x02,0x02,
  0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x00,0x00,
  0x00,0x00,0x02,0x02,0x02,0x00,0x02,0x02,0x00,0x00,0x00,0x02,0x02,0x00,0xc2,0x02,
  0x02,0x02,0xcb,0x02,0x02,0x02,0x02,0xc2,0xc2,0x02,0x02,0xc2,0x02,0xcb,0xcb,0xcb,
  0xcb,0xcb,0xd2,0xd2,0xd2,0x01,0xd2,0xd2,0xfa,0xd2,0x01,0x01,0xd2,0x03,0x03,0x01,
  0x01,0x00,0x00,0x01,0x01,0x03,0xbb,0xbb,0x03,0xbb,0x03,0xbb,0x03,0x97,0x03,0xbb,
  0x97,0x97,0xe7,0xe7,0xe7,0xe7,0x97,0x97,0x97,0x97,0x97,0x03,0x03,0x03,0x03,0x3a,
  0x03,0x01,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x01,0x00,0x00,0x02,
  0x00,0x01,0x01,0x01,0x01,0x03,0x03,0x03,0x03,0x3a,0x03,0x01,0x01,0x01,0x02,0x02,
  0x02,0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x00,0x00,0x02,0x00,0x00,0xc2,
  0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,0x02,0x02,0xc2,0xc2,0x02,0x02,
  0xcb,0xcb,0x00,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x00,0x01,0x03,0x03,0xbb,0xbb,0xbb,0xbb,0x97,0xbb,0x97,0xbb,0x97,0x97,
  0x97,0x97,0x97,0xe7,0xe7,0xe7,0x97,0x97,0x97,0x97,0x97,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x01,0x01,0x00,0x00,0x00,0x00,0x01,0x00,0x01,0x00,0x00,0x01,0x01,
  0x00,0x00,0x01,0x01,0x03,0x03,0xf6,0x03,0x03,0x03,0xf2,0x3a,0x01,0x01,0x02,0x02,
  0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x3a,0x3a,0x01,0x03,0x03,0x3a,0x01,
  0x01,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,
  0x02,0x02,0x02,0xc2,0x02,0xcb,0xcb,0x02,0x00,0x02,0x02,0x02,0xc2,0xc2,0x02,0x02,
  0xcb,0xcb,0x00,0x00,0xd2,0xd2,0xd2,0xd2,0xd2,0xd2,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x3a,0x01,0x00,0x01,0x03,0xbb,0xbb,0xbb,0xbb,0x97,0x97,0xbb,0x97,0x97,0xbb,
  0x97,0x97,0x97,0xe7,0xe7,0xe7,0x97,0x97,0x97,0x97,0x03,0x03,0x03,0x03,0xbb,0x97,
  0x03,0xbb,0x03,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x01,0x00,0x01,0x00,
  0x01,0x01,0x3a,0x3a,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x3a,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x3a,0x03,0xf2,0x03,0x03,0x03,0x03,0x03,0x3a,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x02,
  0x02,0x02,0x02,0x00,0xcb,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0xc2,0x02,0x02,
  0xcb,0xcb,0xcb,0x00,0x00,0xd2,0xd2,0xd2,0xd2,0xd2,0x01,0x01,0x01,0x01,0x01,0x03,
  0x01,0x01,0x01,0xfa,0x01,0x03,0xbb,0xbb,0xbb,0xbb,0xbb,0xe7,0xe7,0xe7,0xe7,0x97,
  0x97,0xe7,0x97,0xe7,0xe7,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,
  0x97,0x97,0x03,0x03,0x03,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,
  0x01,0x01,0x3a,0x03,0x03,0x97,0xf6,0xf6,0x97,0x03,0x03,0x03,0x3a,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0xf2,0xf2,0x03,0xf6,0xf6,0xf6,0xf6,0x03,0x03,0x03,0x03,
  0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xcb,0x00,0x00,0x00,0x02,
  // 17: sky 3, part B
  0x7d,0xc6,0x9d,0xc6,0x5c,0xbe,0x3b,0xbe,0xdd,0xce,0x5b,0xbe,0x00,0x00,0x08,0x00,
  0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,
  0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x1e,0xcf,0x00,0x06,
  0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,
  0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,
  0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,
  0x08,0x24,0x9e,0xc6,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,
  0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,
  0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,
  0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,
  0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,
  0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x3e,0xdf,0x00,0x47,0x08,0x47,0x10,0x47,
  0x18,0x47,0x00,0x60,0x08,0x60,0xd9,0xad,0x10,0x60,0x18,0x60,0x00,0x61,0x5e,0xd7,
  0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,
  0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,
  0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,
  0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x9e,0xbe,0x08,0x80,0x10,0x80,0x18,0x80,
  0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,0x08,0x82,0x10,0x82,0x18,0x82,
  0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x19,0xb6,0x00,0x84,0x08,0x84,0x10,0x84,
  0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x18,0x85,0x00,0x86,0x08,0x86,0x10,0x86,
  0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,0xfb,0xb5,
  0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,
  0x10,0xa2,0x18,0xa2,0xfa,0xad,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,
  0x08,0xa4,0xfe,0xd6,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,0x9f,0xe7,
  0xfe,0xc6,0x18,0xa5,0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,
  0x10,0xa7,0xf9,0xb5,0x18,0xa7,0x1d,0xd7,0x00,0xc0,0x08,0xc0,0x7e,0xe7,0x10,0xc0,
  0x18,0xc0,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,0x10,0xc2,
  0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x3c,0xb6,0x18,0xc3,0x00,0xc4,0x08,0xc4,
  0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x10,0xc5,0x18,0xc5,0x00,0xc6,0x08,0xc6,
  0x10,0xc6,0x18,0xc6,0x00,0xc7,0x08,0xc7,0x1c,0xae,0x00,0xe0,0xb9,0xad,0x08,0xe0,
  0x1a,0xae,0x10,0xe0,0xbd,0xbe,0x00,0xe1,0x08,0xe1,0x18,0xe1,0x00,0xe2,0x18,0xe3,
  0xf6,0xf6,0x6b,0xd1,0xd1,0x6b,0x6b,0x6b,0xf6,0x6b,0xf6,0xf6,0xf6,0xf6,0xf6,0xf6,
  0x6b,0xd1,0xd1,0x6b,0x6b,0xd1,0x9c,0x03,0x03,0x9c,0xaf,0x9c,0x9c,0xaf,0xaf,0xaf,
  0xaf,0x6b,0x6b,0xaf,0xaf,0x9c,0x03,0x05,0x05,0x05,0x00,0x00,0x00,0x02,0x00,0x02,
  0x02,0x02,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe4,0x02,0x02,0x02,
  0x02,0x02,0x03,0x03,0xaf,0x9c,0xaf,0xba,0xaf,0x6b,0x6b,0x6b,0x6b,0x6b,0xba,0x6b,
  0xba,0xaf,0xaf,0x03,0x02,0xe4,0x02,0xe4,0xe4,0xe4,0xe4,0xaf,0xaf,0xba,0xba,0xba,
  0xba,0xba,0xba,0xba,0xba,0xaf,0xaf,0xe4,0xe4,0xe4,0xe4,0x02,0xe4,0x02,0x03,0xaf,
  0xaf,0xba,0xba,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0xf6,
  0x6b,0x6b,0x6b,0x6b,0xd1,0x6b,0xd1,0x6b,0xf6,0xf6,0xf6,0xf6,0xf6,0xf6,0xf6,0x6b,
  0x6b,0x6b,0xd1,0x6b,0xd1,0x9c,0x9c,0x03,0x9c,0x03,0xaf,0xaf,0x9c,0xaf,0xaf,0x6b,
  0x6b,0x6b,0x6b,0xaf,0xaf,0x9c,0x03,0x05,0x05,0x00,0x00,0x01,0x01,0x00,0x02,0x00,
  0x00,0x00,0x00,0x00,0x02,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x02,
  0x00,0x02,0x02,0x02,0x03,0x03,0x9c,0xaf,0xaf,0xba,0xba,0x6b,0x6b,0x6b,0x6b,0xba,
  0xba,0xaf,0xaf,0x03,0x02,0xe4,0xe4,0xe4,0xe4,0x02,0x03,0x03,0xaf,0xaf,0xaf,0xba,
  0xba,0xba,0xba,0xba,0xba,0xaf,0xe4,0xe4,0xe4,0xe4,0x00,0x00,0x00,0x02,0x02,0x03,
  0xaf,0xaf,0xba,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,
  0xf6,0xf6,0xd1,0x6b,0xd1,0x6b,0x6b,0xd1,0x6b,0x6b,0x6b,0xf6,0xf6,0xcd,0xf6,0xf6,
  0x6b,0xd1,0xd1,0xd1,0xd1,0x9c,0x9c,0x03,0x03,0x03,0x9c,0x9c,0x9c,0x9c,0xaf,0x6b,
  0x6b,0x6b,0xaf,0xaf,0xaf,0x03,0x03,0x05,0x02,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
  0x00,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x00,0x00,0x01,0x00,
  0x00,0x00,0x02,0x02,0x02,0x03,0x03,0xaf,0x9c,0xaf,0xaf,0xba,0xba,0xba,0xba,0xaf,
  0xaf,0xaf,0x03,0xe4,0x02,0xe4,0xe4,0xe4,0xe4,0x02,0x02,0xe4,0xe4,0x03,0xf4,0xaf,
  0xba,0xba,0xba,0xba,0xaf,0x03,0xe4,0x00,0x00,0x8c,0x8c,0x8c,0x00,0x00,0x02,0x03,
  0x03,0xaf,0xaf,0xba,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,
  0x6b,0x6b,0x6b,0xd1,0x6b,0xd1,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,
  0x6b,0x6b,0x6b,0x6b,0xd1,0x9c,0x03,0x03,0x03,0x03,0x03,0x03,0x9c,0x9c,0x9c,0x9c,
  0x6b,0x6b,0x6b,0xaf,0x9c,0x9c,0x05,0x02,0x02,0x02,0x00,0x00,0x01,0x01,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x02,0x03,0x03,0xaf,0xaf,0xaf,0xba,0xba,0xba,0xba,0xaf,0xaf,
  0xaf,0x03,0x03,0x02,0x02,0x00,0xe4,0xe4,0xe4,0x02,0xe4,0xe4,0xe4,0xe4,0x02,0x03,
  0xaf,0xba,0xba,0xba,0xaf,0xe4,0x00,0x8c,0x00,0x01,0x39,0x00,0x00,0x02,0x02,0x02,
  0x05,0x03,0x9c,0xaf,0xba,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0x6b,0xba,0x6b,0x6b,
  0x6b,0x6b,0x6b,0xd1,0x6b,0x6b,0x9c,0x9c,0xf8,0xaf,0x9c,0xd1,0x6b,0x6b,0x6b,0x6b,
  0xd1,0x6b,0xd1,0xd1,0xd1,0x9c,0x9c,0x03,0x03,0x03,0x03,0x9c,0x9c,0x9c,0x9c,0x9c,
  0x9c,0xaf,0xaf,0x9c,0x9c,0x03,0x02,0x05,0x05,0x00,0x00,0x00,0x00,0x00,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x02,0x05,0xaf,0xaf,0xaf,0xaf,0xba,0xba,0xba,0xba,0xaf,0xaf,
  0xf4,0x03,0xe4,0x00,0x00,0x00,0xe4,0x02,0x02,0x02,0x00,0xe4,0xe4,0x02,0x02,0x03,
  0xaf,0xaf,0xaf,0xaf,0xaf,0x03,0x02,0x01,0xfa,0xfa,0xfa,0x01,0x8c,0x02,0x02,0x02,
  0x03,0x03,0x9c,0xaf,0x6b,0xba,0x6b,0xba,0x6b,0xba,0xba,0xba,0x6b,0x6b,0x6b,0x6b,
  0xd1,0x6b,0xaf,0xaf,0x9c,0xaf,0xaf,0x9c,0x03,0x9c,0x9c,0xaf,0xd1,0x6b,0x6b,0x6b,
  0xd1,0x6b,0xd1,0xd1,0xd1,0x9c,0x9c,0x9c,0x9c,0x03,0x9c,0x03,0x9c,0x03,0x9c,0x9c,
  0x03,0x9c,0xaf,0xaf,0x9c,0x05,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,
  0x00,0x00,0x00,0x05,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x01,0x00,0x00,0x00,0x00,
  0x02,0x02,0x00,0x00,0x02,0x03,0xe4,0xaf,0xaf,0xf4,0xba,0xba,0xba,0xba,0xba,0xaf,
  0xf4,0xaf,0xe4,0x02,0xe4,0x00,0x00,0xe4,0x02,0x00,0x02,0x02,0x00,0x00,0x02,0x02,
  0x03,0xaf,0xaf,0xaf,0x03,0x02,0x00,0x01,0x04,0x04,0x01,0x00,0x8c,0x02,0x02,0x02,
  0x03,0x03,0x9c,0x9c,0xaf,0xf8,0x6b,0x6b,0x6b,0xaf,0x6b,0x6b,0xf8,0xd1,0xaf,0xd1,
  0xaf,0x9c,0x9c,0xf8,0x03,0x9c,0x03,0x05,0x05,0x03,0x03,0x03,0x9c,0x6b,0x6b,0xd1,
  0x6b,0xd1,0x6b,0xd1,0x6b,0x9c,0xf8,0x9c,0x9c,0x03,0x03,0x9c,0x9c,0x03,0x9c,0x9c,
  0x03,0x9c,0x9c,0x9c,0x03,0x05,0x05,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
  0x00,0x00,0x00,0x05,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x01,0x00,0x00,0x00,
  0x02,0x02,0x02,0x02,0x02,0x03,0xaf,0xaf,0xaf,0xaf,0xba,0xba,0xba,0xba,0xaf,0xaf,
  0xf4,0xaf,0xe4,0x02,0xe4,0x00,0x00,0x00,0x02,0x8c,0x00,0x00,0x00,0x8c,0x00,0x00,
  0xe4,0x03,0xaf,0xaf,0x02,0x00,0x8c,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0xe4,0x03,
  0x03,0x03,0x9c,0x9c,0xaf,0xaf,0xaf,0x6b,0x6b,0x6b,0x6b,0xaf,0xaf,0xaf,0xaf,0xaf,
  0x03,0x03,0x03,0x03,0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x03,0x9c,0x9c,0xd1,0x6b,
  0x6b,0xd1,0xd1,0xd1,0x6b,0x9c,0x9c,0x9c,0x9c,0x9c,0x03,0x9c,0x03,0x9c,0x9c,0x9c,
  0x9c,0x9c,0x03,0x03,0x05,0x05,0x00,0x00,0x00,0x01,0x00,0x01,0x01,0x00,0x00,0x00,
  0x00,0x00,0x00,0x05,0x02,0x02,0x02,0x02,0x02,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
  0x00,0x02,0x02,0x02,0x02,0x02,0x03,0x03,0xaf,0xaf,0xaf,0xba,0xba,0xba,0xaf,0xaf,
  0xf4,0xaf,0xaf,0xe4,0xe4,0x8c,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x02,
  0x03,0xaf,0xaf,0xaf,0x02,0x8c,0x01,0x8c,0x01,0x04,0x04,0x01,0x00,0x00,0x02,0x02,
  0x05,0x05,0x03,0x9c,0xaf,0xf8,0xaf,0x9c,0xaf,0xaf,0xaf,0xaf,0x9c,0xaf,0x9c,0x9c,
  0x03,0x03,0x05,0x05,0x05,0x05,0x00,0x05,0x05,0x05,0x05,0x05,0x03,0x9c,0x9c,0xd1,
  0xd1,0xd1,0xd1,0x6b,0x6b,0x9c,0x9c,0x9c,0x03,0x03,0x9c,0x03,0x9c,0x9c,0x9c,0x9c,
  0x9c,0x9c,0x03,0x03,0x05,0x05,0x00,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x00,0x01,0x04,0x04,0x04,
  0x01,0x00,0x02,0x02,0xe4,0x02,0x02,0x03,0x03,0xaf,0xaf,0xaf,0xaf,0xf4,0xaf,0xe4,
  0xe4,0xe4,0xe4,0xe4,0xe4,0x8c,0x00,0x00,0x8c,0x00,0x01,0x01,0x01,0x8c,0x02,0x02,
  0xaf,0xaf,0xaf,0xaf,0xe4,0x00,0x8c,0x01,0x01,0x04,0x04,0x01,0x00,0x02,0x02,0x02,
  0x05,0x03,0x03,0x9c,0x9c,0xaf,0xf8,0xf8,0x9c,0xf8,0xaf,0x9c,0xf8,0x9c,0x03,0x9c,
  0x05,0x05,0x05,0x05,0x05,0x00,0x01,0x00,0x00,0x00,0x05,0x05,0x03,0x03,0x9c,0xd1,
  0xd1,0x6b,0xd1,0xd1,0xd1,0xd1,0x9c,0xaf,0x9c,0x9c,0x03,0x03,0x03,0x03,0x9c,0x9c,
  0x9c,0x9c,0x9c,0x9c,0x03,0x05,0x00,0x00,0x01,0x01,0x00,0x00,0x05,0x05,0x02,0x00,
  0x00,0x00,0x00,0x00,0x05,0x02,0x02,0x00,0x01,0x00,0x00,0x00,0x01,0x04,0x04,0x04,
  0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x02,0x02,
  0xe4,0x03,0x03,0x03,0xe4,0x00,0x00,0x00,0x00,0x8c,0x00,0x00,0x00,0x00,0x02,0x03,
  0x9c,0xaf,0xaf,0xaf,0x03,0x02,0x8c,0x01,0x04,0x04,0x04,0xfa,0x01,0x00,0x02,0x02,
  0x02,0x05,0x03,0x03,0x9c,0xaf,0xaf,0x9c,0xf8,0xf8,0xaf,0x9c,0x9c,0xf8,0x03,0x03,
  0x03,0x05,0x05,0x05,0x01,0x01,0x01,0x01,0x01,0x00,0x05,0x03,0x05,0x03,0x9c,0x6b,
  0xd1,0x6b,0x6b,0xd1,0xd1,0xd1,0x9c,0x9c,0x9c,0x9c,0x03,0x03,0x03,0x9c,0x9c,0x9c,
  0x9c,0x9c,0x9c,0x9c,0x03,0x05,0x05,0x02,0x01,0x01,0x00,0x00,0x05,0x05,0x05,0x02,
  0x00,0x00,0x05,0x05,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x01,0x00,0x00,0x01,0x02,0x02,0x05,0x02,0x02,0x02,0xe4,0xe4,
  0x02,0x02,0x03,0x03,0xe4,0xe4,0x00,0x02,0x02,0x8c,0x00,0x00,0x00,0x02,0x02,0x03,
  0xaf,0xaf,0xaf,0xaf,0xaf,0x03,0xe4,0x01,0x04,0x04,0x04,0xfa,0x01,0x00,0x02,0x02,
  0x02,0x03,0x03,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0xaf,0xaf,0x9c,0x03,0x9c,0x03,0x03,
  0x05,0x05,0x05,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x05,0x05,0x03,0xd1,
  0x6b,0xd1,0xd1,0xd1,0xd1,0x9c,0x9c,0x03,0x03,0x05,0x03,0x03,0x05,0x9c,0x9c,0x9c,
  0x9c,0x9c,0x9c,0x03,0x03,0x05,0x05,0x02,0x00,0x01,0x00,0x05,0x05,0x05,0x05,0x00,
  0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x01,0x00,0x01,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x00,0x02,0x02,0x00,0x00,0x01,0x00,0x02,
  0xe4,0xe4,0x02,0x02,0x02,0x02,0x02,0x02,0xe4,0x00,0x00,0x00,0xe4,0x03,0x03,0xaf,
  0xaf,0xaf,0xaf,0xaf,0xf4,0xaf,0xe4,0x8c,0x8c,0x01,0xfa,0x04,0x01,0x00,0x02,0x02,
  0x05,0x03,0x03,0x03,0x9c,0xf8,0xf8,0x9c,0x9c,0x9c,0xf8,0x9c,0x03,0x03,0x05,0x05,
  0x05,0x05,0x00,0x00,0x01,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x05,0x03,0x9c,
  0x9c,0xd1,0xd1,0xd1,0xd1,0x9c,0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x03,0x9c,0x03,
  0x9c,0x9c,0x03,0x03,0x03,0x05,0x05,0x05,0x01,0x00,0x00,0x05,0x05,0x05,0x00,0x02,
  0x01,0x01,0x00,0x00,0x00,0x02,0x01,0x00,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x1e,
  0xc1,0x1e,0x1e,0xc1,0xc1,0x04,0x04,0x04,0x04,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
  0x02,0x02,0x02,0x02,0x02,0xe4,0x02,0x02,0x02,0xe4,0x02,0xe4,0x02,0xe4,0xaf,0xaf,
  0xaf,0xaf,0xaf,0xaf,0xaf,0xaf,0xe4,0xe4,0x00,0x8c,0x01,0xfa,0x01,0x8c,0x02,0x02,
  0x02,0x03,0x03,0x9c,0x9c,0x9c,0xaf,0x9c,0xaf,0xaf,0x9c,0x9c,0x03,0x05,0x05,0x05,
  0x00,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x05,0x9c,
  0x9c,0xd1,0xd1,0xd1,0xd1,0x9c,0x9c,0x03,0x03,0x05,0x05,0x05,0x05,0x03,0x9c,0x9c,
  0x03,0x03,0x9c,0x05,0x05,0x05,0x05,0x00,0x00,0x02,0x05,0x05,0x05,0x05,0x02,0x00,
  0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0xc1,0x1e,
  0x1e,0x1e,0x1e,0x6f,0x1e,0x04,0x04,0x04,0x04,0x01,0x00,0x01,0x01,0x01,0x01,0x00,
  0x00,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe4,0x00,0x00,0x00,0x00,0x02,0x03,0xaf,
  0xaf,0xf4,0xaf,0xf4,0xf4,0xf4,0xe4,0xe4,0xe4,0x00,0x8c,0x01,0x8c,0x00,0x02,0x02,
  0x03,0x03,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x9c,0x03,0x05,0x05,0x05,0x05,
  0x01,0x01,0x04,0x04,0x04,0x01,0x04,0x01,0x01,0x01,0x04,0x01,0x01,0x00,0x05,0x03,
  0x9c,0x9c,0xd1,0x6b,0x9c,0x9c,0x9c,0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x03,0x03,
  0x03,0x03,0x03,0x05,0x05,0x05,0x05,0x02,0x02,0x00,0x00,0x05,0x05,0x05,0x00,0x01,
  0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0xc1,0x1e,0xc1,
  0x1e,0x1e,0x1e,0x1e,0x04,0x04,0x01,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x04,0x01,
  0x00,0x02,0xe4,0x02,0x02,0x02,0xe4,0x02,0x02,0x8c,0x00,0x01,0x8c,0x00,0xe4,0xaf,
  0xf4,0xaf,0xaf,0xaf,0xf4,0xf4,0xe4,0xe4,0xe4,0xe4,0x00,0x8c,0x00,0xe4,0x02,0x02,
  0x03,0x03,0x03,0x9c,0x03,0x03,0x9c,0x9c,0x9c,0x9c,0x03,0x03,0x05,0x05,0x05,0x00,
  0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x05,0x05,
  0x9c,0x9c,0x6b,0x6b,0x9c,0x9c,0x03,0x05,0x03,0x05,0x05,0x05,0x05,0x05,0x03,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x01,0x01,0x01,0x00,0x00,0x02,0x02,0x00,0x01,
  0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x1e,0x1e,0x1e,0xc1,0x1e,
  0xc1,0xc1,0x1e,0xc1,0x04,0x04,0x01,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0x02,0x02,0x02,0xe4,0x00,0x00,0x39,0x39,0x39,0x8c,0x00,0xe4,
  0xe4,0xe4,0xe4,0xaf,0x03,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x05,0x05,0x05,0x00,0x01,
  0x01,0x04,0x04,0x1e,0xc1,0x1e,0x1e,0x04,0x04,0x04,0x04,0x04,0x04,0x00,0x05,0x05,
  0x03,0x9c,0xd1,0xd1,0x9c,0x03,0x05,0x05,0x05,0x05,0x00,0x00,0x05,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x00,0x00,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x01,
  0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x04,0x04,0xc1,0x1e,0x1e,0xc1,0x04,0xc1,
  0x1e,0x6f,0x6f,0x1e,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x39,0x04,0x39,0x04,0x39,0x8c,0xe4,
  0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0xe4,0x02,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x05,0x03,0x03,0x03,0x03,0x05,0x00,0x01,0x01,
  0x04,0x04,0x04,0xd3,0x6f,0xd3,0x1e,0xd3,0xd3,0x1e,0x1e,0x04,0x04,0x01,0x01,0x00,
  0x03,0x9c,0xd1,0x9c,0x03,0x03,0x03,0x05,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
  0x05,0x05,0x05,0x05,0x05,0x05,0x02,0x00,0x01,0x04,0x04,0x01,0x00,0x00,0x01,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x04,0x04,0x1e,0x1e,0x1e,0xc1,0x04,0x1e,
  0x6f,0x6f,0x1e,0x6f,0x04,0x04,0x04,0x04,0x1e,0xc1,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x01,0x04,0x01,0x00,0x8c,0x8c,0x39,0x04,0x04,0x04,0x04,0x04,0x8c,0xe4,
  0xe4,0xe4,0xe4,0x02,0xe4,0xe4,0xe4,0x00,0x00,0x00,0xe4,0xe4,0xe4,0x02,0x03,0x03,
  0xaf,0x03,0x03,0x03,0x05,0x03,0x05,0x05,0x05,0x03,0x03,0x05,0x00,0x01,0x01,0x01,
  0x04,0x04,0xc1,0xd3,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x04,0x04,0x04,0x00,
  0x03,0x03,0xf8,0x9c,0x03,0x03,0x05,0x05,0x05,0x05,0x00,0x01,0x01,0x01,0x00,0x00,
  0x05,0x02,0x05,0x05,0x05,0x05,0x00,0x01,0x04,0x04,0x04,0x01,0x01,0x01,0x04,0x04,
  0x04,0x04,0x04,0x1e,0x04,0x04,0x04,0x04,0x04,0xc1,0x1e,0x6f,0x6f,0x6f,0xc1,0x1e,
  0x6f,0x6f,0x6f,0x1e,0x04,0x04,0xc1,0x1e,0x1e,0x04,0x04,0x04,0xc1,0xc1,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x01,0x00,0x01,0x04,0x04,0xc8,0x1e,0xc8,0x04,0x8c,0xe4,
  0xe4,0xe4,0xe4,0xe4,0xe4,0x00,0x00,0x8c,0x8c,0x8c,0x8c,0x00,0xe4,0x02,0x03,0x03,
  0xaf,0x03,0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x05,0x00,0x01,0x01,0x01,
  0x04,0x04,0xc1,0x64,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x04,0x04,0x04,0x00,
  0x05,0x03,0x9c,0x03,0x03,0x03,0x03,0x05,0x05,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x00,0x05,0x05,0x05,0x00,0x00,0x01,0x04,0xc1,0x1e,0x04,0x04,0x04,0x04,0x04,
  0x04,0x1e,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xc1,0x1e,0x6f,0x6f,0x6f,0x6f,0x6f,
  0xc7,0x6f,0x64,0x6f,0x1e,0x1e,0x6f,0x6f,0xc1,0x04,0x1e,0xc1,0xc1,0x6f,0x1e,0x1e,
  0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x04,0x04,0xc8,0x04,0x1e,0x1e,0x04,0x8c,0x00,
  0xe4,0xe4,0xe4,0xe4,0xe4,0x8c,0x8c,0x8c,0x8c,0x39,0x01,0x00,0xe4,0x02,0x02,0x03,
  0x03,0x03,0x03,0x03,0x03,0x05,0x05,0x02,0x05,0x05,0x05,0x05,0x00,0x01,0x01,0x04,
  0x04,0x04,0xd3,0xd3,0xd3,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x1e,0x04,0x04,0x01,
  0x05,0x05,0x03,0x03,0x03,0x03,0x03,0x05,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x01,0x00,0x05,0x02,0x00,0x01,0x04,0x1e,0xd3,0xd3,0x1e,0x04,0x04,0x1e,0xd3,
  0x6f,0x1e,0x04,0x1e,0xc1,0x04,0x04,0x04,0x04,0x04,0xc1,0x1e,0x6f,0xc7,0xc7,0xc7,
  0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,
  0x1e,0x04,0x04,0x04,0xc8,0x04,0xc8,0x04,0x04,0x04,0x1e,0x1e,0x1e,0x04,0x00,0xe4,
  0xe4,0xe4,0xe4,0xe4,0x00,0x8c,0x8c,0x8c,0x39,0x39,0x8c,0x00,0xe4,0xe4,0x02,0x03,
  0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x02,0x00,0x05,0x05,0x05,0x00,0x01,0x01,0x04,
  0x04,0xc1,0xd3,0xd3,0xd3,0x6f,0xd3,0x6f,0x6f,0x6f,0xd3,0x1e,0x1e,0x1e,0x04,0x01,
  0x00,0x05,0x03,0x03,0x03,0x03,0x05,0x05,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x00,0x02,0x00,0x01,0x04,0x1e,0x1e,0xd3,0xd3,0x1e,0xd3,0xd3,0x1e,
  0xc1,0x04,0x04,0x1e,0x1e,0x6f,0xc1,0x04,0x04,0x04,0xc1,0x6f,0xc7,0xc7,0xc7,0xc7,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0xc7,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,
  0x1e,0xc1,0x04,0x1e,0x1e,0x1e,0x1e,0xc1,0x1e,0x1e,0x1e,0x1e,0x1e,0x39,0x8c,0xe4,
  0xe4,0xe4,0xe4,0xe4,0x00,0x00,0x8c,0x8c,0x39,0x39,0x8c,0x00,0xe4,0xe4,0x02,0x03,
  0x03,0x03,0x03,0x05,0x05,0x05,0x05,0x05,0x02,0x00,0x05,0x05,0x00,0x01,0x04,0x04,
  0x04,0x1e,0xd3,0xd3,0xd3,0xd3,0x6f,0x6f,0x6f,0x1e,0xc1,0x1e,0xd3,0xd3,0x04,0x01,
  0x01,0x05,0x03,0x05,0x03,0x05,0x05,0x00,0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x04,
  0x01,0x01,0x01,0x00,0x05,0x00,0x00,0x04,0xd3,0x6f,0x6f,0x1e,0x1e,0xd3,0x64,0x64,
  0x6f,0xc1,0x1e,0x1e,0x1e,0x1e,0x1e,0x04,0x04,0x04,0xc1,0x6f,0x6f,0xc7,0xc7,0xc7,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,
  0x6f,0x6f,0x1e,0xc1,0x1e,0x1e,0x1e,0x6f,0x1e,0x1e,0x1e,0x1e,0x1e,0x04,0x04,0x8c,
  0xe4,0x00,0xe4,0xe4,0xe4,0x00,0x8c,0x8c,0x39,0x39,0x8c,0xe4,0xe4,0xe4,0xe4,0x03,
  0x03,0xf4,0x05,0x03,0x05,0x05,0x00,0x02,0x05,0x00,0x00,0x05,0x00,0x01,0x01,0x04,
  0x1e,0x1e,0xd3,0xd3,0x1e,0xd3,0x6f,0x6f,0x6f,0x6f,0x04,0x04,0xc1,0x04,0x04,0x01,
  0x00,0x05,0x05,0x03,0x03,0x03,0x05,0x00,0x01,0x04,0x04,0x01,0x01,0x04,0x04,0x04,
  0x04,0x01,0x04,0x01,0x01,0x00,0x01,0x04,0x6f,0x6f,0x64,0x64,0x6f,0x6f,0x6f,0x64,
  0x1e,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0xc1,0x04,0x04,0x1e,0x6f,0x6f,0x6f,0x6f,0xc7,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,0x6f,
  0x6f,0x6f,0x1e,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x1e,0x1e,0xc8,0xc8,0x39,
  0x8c,0x00,0x00,0xe4,0xe4,0x00,0x8c,0x39,0x39,0x39,0x8c,0xe4,0xe4,0x02,0x02,0x03,
  0x03,0xf4,0x03,0x03,0x05,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x04,
  0x1e,0xd3,0xd3,0x1e,0xd3,0x1e,0xd3,0x6f,0xd3,0xd3,0xc1,0x04,0x04,0x04,0x01,0x01,
  0x00,0x05,0x05,0x03,0x03,0x05,0x05,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x1e,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0xc7,0x64,
  0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0x04,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0xc7,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,
  0x6f,0x1e,0x04,0x04,0x1e,0xc1,0x1e,0x6f,0x6f,0x1e,0x1e,0x1e,0xc8,0x04,0x04,0x39,
  0x8c,0x8c,0x8c,0x8c,0x00,0x8c,0x39,0x39,0x39,0x8c,0x00,0xe4,0xe4,0xe4,0xe4,0x03,
  0xaf,0xaf,0x03,0x03,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x04,
  0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x04,0x04,0x04,0x01,0x01,
  0x00,0x00,0x05,0x03,0x03,0x05,0x05,0x01,0x01,0x04,0x04,0x04,0xc1,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x6f,0x6f,0x1e,0xd3,0x6f,0xc7,0xc7,0x6f,
  0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x64,0x6f,0xc1,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0x6f,
  0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,
  0x6f,0x6f,0x1e,0xc1,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x04,0x04,0x04,0x39,0x01,0x8c,
  0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x39,0x39,0x8c,0x8c,0xe4,0xe4,0xe4,0xe4,0x03,0x03,
  0xaf,0xaf,0xaf,0x03,0x05,0x02,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x01,0x01,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x00,0x01,0x02,
  0x00,0x05,0x05,0x05,0x05,0x05,0x00,0x01,0x01,0x04,0x04,0x04,0xc1,0x04,0xd3,0x04,
  0xd3,0x1e,0xc1,0xd3,0x6f,0x04,0x04,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0xc7,0xc7,0x6f,
  0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0x64,
  0x6f,0x6f,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,
  0x6f,0x1e,0x6f,0x1e,0x1e,0x1e,0x1e,0xc8,0x04,0x04,0x01,0x8c,0x8c,0x00,0x00,0x00,
  0x8c,0x8c,0x8c,0x8c,0x00,0x8c,0x8c,0x8c,0x8c,0x00,0xe4,0xe4,0xe4,0xaf,0xaf,0xaf,
  0xf4,0xaf,0xaf,0xaf,0x03,0x05,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x04,0x04,0x01,0x01,0x02,0x00,0x02,
  0x02,0x05,0x02,0x05,0x05,0x05,0x05,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x1e,
  0xd3,0xd3,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,0x6f,0xc7,0x6f,0x6f,0xc7,0xc7,0x6f,0x6f,
  0x6f,0x6f,0x6f,0x6f,0x6f,0xc7,0x6f,0x6f,0x6f,0x1e,0x1e,0x6f,0x6f,0x6f,0xc1,0x6f,
  0x6f,0x6f,0x6f,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,0x6f,0x6f,
  0x6f,0x6f,0x1e,0x1e,0x1e,0x1e,0xc8,0x04,0x39,0x39,0x8c,0x00,0x00,0xe4,0xe4,0x00,
  0x8c,0x8c,0x39,0x8c,0x8c,0x8c,0x8c,0x8c,0x00,0xe4,0xe4,0xe4,0xaf,0xaf,0xf4,0xaf,
  0xaf,0xaf,0xaf,0x9c,0x03,0x00,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
  0x04,0x04,0x01,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x02,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x02,0x00,0x00,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0xc1,
  0xd3,0x64,0xc7,0xc7,0xc7,0xc7,0x6f,0xd3,0x6f,0x6f,0x6f,0x6f,0x6f,0xc7,0x6f,0x1e,
  0x6f,0x6f,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0x6f,0xc1,0xc1,0x1e,0x6f,0x6f,0x6f,0x6f,
  0x6f,0x1e,0x6f,0x6f,0x6f,0xc7,0xc7,0xc7,0xc7,0x6f,0x6f,0x6f,0x6f,0x6f,0x1e,0x1e,
  0x1e,0x1e,0x1e,0xc1,0x1e,0x04,0x04,0x04,0x39,0x8c,0x8c,0x00,0xe4,0xe4,0xe4,0x00,
  0x00,0x8c,0x8c,0x39,0x39,0x8c,0x8c,0x00,0xe4,0xe4,0xe4,0xf4,0xaf,0xaf,0xaf,0xba,
  0xba,0xba,0xba,0xaf,0xaf,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x04,
  0x04,0x04,0x01,0x00,0x00,0x02,0x02,0x00,0x02,0x05,0x05,0x02,0x02,0x05,0x05,0x05,
  0x05,0x05,0x05,0x05,0x05,0x02,0x00,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x1e,
  0x6f,0x64,0xd6,0xc7,0xd6,0xc7,0x6f,0x6f,0xd3,0x6f,0x6f,0x1e,0x1e,0x6f,0x6f,0x1e,
  0x1e,0x1e,0x1e,0x04,0x1e,0x6f,0x6f,0x6f,0xc1,0x04,0xc1,0x1e,0xd3,0x1e,0xc1,0x1e,
  0x64,0x1e,0x1e,0x6f,0x6f,0x6f,0x6f,0x6f,0x64,0x1e,0x1e,0x6f,0x1e,0x6f,0x1e,0x1e,
  0xc8,0x1e,0x04,0x04,0x04,0x01,0x39,0x01,0x39,0x8c,0x8c,0x00,0x00,0xe4,0xe4,0x00,
  0x00,0x00,0x8c,0x39,0x8c,0x8c,0xe4,0xe4,0xe4,0x03,0xe4,0xaf,0xaf,0xba,0xba,0xba,
  0xba,0xba,0xba,0xaf,0xaf,0x02,0x02,0x01,0x04,0x04,0x01,0x00,0x00,0x01,0x04,0x04,
  0x01,0x04,0x04,0x01,0x01,0x00,0x02,0x05,0x05,0x05,0x05,0x05,0x05,0x9c,0x9c,0x03,
  0x03,0x03,0x05,0x05,0x05,0x00,0x02,0x00,0x00,0x01,0x00,0x01,0x04,0x04,0x1e,0xd3,
  0x6f,0xd6,0xd6,0xd6,0xd6,0xd6,0x6f,0x6f,0xd3,0xd3,0x1e,0x04,0x04,0x1e,0x1e,0x1e,
  0x04,0x04,0x04,0x04,0x04,0x1e,0x1e,0x1e,0x1e,0x04,0xc1,0x04,0xc1,0x04,0x04,0x04,
  0x1e,0xc8,0x04,0xc1,0x1e,0x1e,0x1e,0x1e,0x1e,0x04,0x04,0x04,0x1e,0xc1,0x1e,0x1e,
  0x1e,0x04,0x04,0x04,0x39,0x8c,0x39,0x8c,0x8c,0x8c,0x8c,0x8c,0x00,0xe4,0xe4,0xe4,
  0xe4,0x00,0x8c,0x8c,0x8c,0x8c,0xe4,0xe4,0xe4,0xe4,0xaf,0xaf,0xba,0xba,0xba,0xba,
  0xba,0xba,0xba,0xaf,0xaf,0x02,0x02,0x00,0x01,0x01,0x00,0x00,0x00,0x01,0x04,0x04,
  0x00,0x01,0x04,0x01,0x02,0x05,0x03,0x05,0x03,0x05,0x05,0x03,0x9c,0xf8,0xaf,0xf8,
  0x9c,0x03,0x03,0x05,0x05,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x04,0x04,0xd3,0x6f,
  0xc7,0xd6,0xd6,0xd6,0xc7,0xc7,0x6f,0x6f,0x6f,0x64,0xd3,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x1e,0x1e,0x1e,0x04,0xc8,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x04,0x04,0xc8,
  0xc8,0x04,0x04,0x39,0x8c,0x8c,0x8c,0x8c,0x8c,0x8c,0x00,0x8c,0x00,0x00,0xe4,0xe4,
  0xe4,0xe4,0xe4,0x00,0x00,0xe4,0xe4,0xe4,0xe4,0xe4,0xaf,0xba,0xba,0xba,0xba,0xba,
  0xba,0x6b,0xaf,0x9c,0x03,0x02,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,
  // 18: sky 3, part C
  0x98,0xa5,0xd9,0xad,0x1a,0xb6,0x5c,0xbe,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,
  0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x58,0x95,
  0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,
  0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,
  0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,
  0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0xfa,0xa5,0x18,0x21,0x00,0x22,0x08,0x22,
  0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,
  0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,
  0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,
  0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,
  0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,
  0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,
  0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,
  0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,
  0x99,0x9d,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,
  0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,
  0x08,0x66,0xdd,0xce,0x17,0x8d,0x9c,0xc6,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,
  0x10,0x67,0x18,0x67,0x00,0x80,0x38,0x8d,0x7c,0xc6,0xf6,0x8c,0xb6,0x7c,0x08,0x80,
  0x10,0x80,0x18,0x80,0x7e,0xe7,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,
  0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,
  0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x00,0x86,0x08,0x86,
  0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,
  0x10,0xa0,0x18,0xa0,0x00,0xa1,0xbe,0xce,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x5e,0xd7,
  0x00,0xa2,0x08,0xa2,0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x1d,0xd7,
  0x18,0xa3,0x00,0xa4,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,
  0x57,0x95,0xfa,0xb5,0x95,0x7c,0x18,0xa5,0x00,0xa6,0x08,0xa6,0x75,0x74,0x55,0x6c,
  0x10,0xa6,0x1e,0xcf,0x18,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x1b,0xae,0x18,0xa7,
  0x00,0xc0,0xf3,0x6b,0x08,0xc0,0x10,0xc0,0x18,0xc0,0x14,0x6c,0x00,0xc1,0x9e,0xef,
  0xd6,0x7c,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x5b,0xbe,0xb3,0x5b,0x08,0xc2,
  0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x35,0x74,0x10,0xc3,0x18,0xc3,0x00,0xc4,
  0x37,0x95,0xf4,0x5b,0x10,0xc4,0x18,0xc4,0x00,0xc5,0xf7,0x94,0x17,0x7d,0x13,0x6c,
  0xd4,0x63,0xd8,0xb5,0x19,0xb6,0x78,0xa5,0xb5,0x84,0xbd,0xbe,0xb6,0x6c,0x18,0xe0,
  0x83,0x83,0x8c,0x8c,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0x02,0x02,0xc9,0x01,0x2c,0xc9,
  0x02,0x02,0x02,0x02,0xe5,0x8c,0xfd,0x83,0xfd,0xfd,0x81,0xfd,0x81,0xbf,0xb7,0xb7,
  0x92,0x92,0x92,0x92,0x92,0x92,0xb7,0xb7,0xb7,0x92,0xb7,0x81,0xb3,0xb3,0xb3,0x83,
  0xb3,0x81,0xb3,0xb3,0xb3,0xd1,0x81,0x81,0xb3,0x81,0xb3,0xfd,0xfd,0xb3,0xb3,0xfd,
  0x81,0xb3,0xfd,0x83,0x83,0xb3,0x81,0xb3,0xb3,0xb3,0x83,0x83,0x8c,0x83,0xfd,0xb3,
  0xb3,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0x03,0x03,
  0x03,0x03,0x03,0xfd,0x03,0x03,0x03,0x03,0xd6,0xd6,0xc9,0x2c,0x2c,0x2c,0x2c,0x2c,
  0x2c,0x2c,0xd6,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x8c,0x03,0x8c,0x8c,
  0x03,0x8c,0x8c,0x03,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0x02,0x02,0x02,0x01,0x01,0x01,
  0xc9,0xc9,0xc9,0x02,0xe5,0x03,0x8c,0x83,0xfd,0x81,0x81,0x81,0xbf,0xb7,0xd1,0xbf,
  0xb7,0x92,0x92,0x92,0x92,0xb7,0xb7,0xb7,0xb7,0xb7,0xbf,0xd1,0x81,0x81,0xfd,0x8c,
  0xb3,0xfd,0x83,0xb3,0x81,0xd1,0xd1,0xb7,0xb3,0x81,0x83,0x8c,0x8c,0x83,0x8c,0x8c,
  0x83,0x83,0x8c,0x03,0x83,0xb3,0x81,0xb3,0xb3,0xb3,0xfd,0x81,0x8c,0x8c,0xfd,0x8c,
  0xfd,0xfd,0x03,0x03,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xd6,0xd6,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,
  0x2c,0x2c,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0xe5,0xe5,0xe5,0x02,0x02,0x02,0x02,0xc9,0x01,0x01,0x01,0x01,
  0x01,0x2c,0xc9,0x02,0xe5,0xe5,0x8c,0x81,0x81,0x81,0xd1,0xbf,0xb7,0x92,0xbf,0xd1,
  0xd1,0xb7,0xb7,0x92,0x92,0xb7,0xb7,0xb7,0xd1,0xd1,0xd1,0x81,0x81,0xb3,0x83,0x83,
  0xb3,0x83,0x8c,0xb3,0x81,0xd1,0xb7,0x92,0x81,0xb3,0x8c,0x8c,0x8c,0x83,0x8c,0x8c,
  0x03,0x03,0x03,0x03,0x8c,0xb3,0xb3,0xb3,0xb3,0xb3,0xb3,0x83,0x8c,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xfd,0xfd,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0xd6,0xd6,0x2c,0x2c,0x2c,0x2c,0x2c,0x01,0x2c,0x2c,
  0x2c,0x2c,0xd6,0xd6,0xd6,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0xe5,0xe5,0x02,0xe5,0x02,0xe5,0x02,0x02,0xc9,0x02,0x02,0x2c,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0xc9,0x02,0xe5,0x03,0xfd,0x81,0x81,0xd1,0xd1,0xb7,0xbf,0xd1,0xd1,
  0xd1,0xbf,0xb7,0xb7,0x92,0xbf,0xd1,0xd1,0x81,0x81,0x81,0x81,0xb3,0x83,0x83,0x83,
  0x8c,0x83,0x8c,0x83,0xb3,0x81,0xd1,0x92,0xb7,0xb3,0x8c,0x03,0x8c,0x83,0x8c,0x03,
  0x03,0x03,0x03,0x03,0x8c,0x8c,0x8c,0x83,0x83,0xb3,0xb7,0xb7,0xb7,0xfd,0x02,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0xd6,0x2c,0x2c,0x2c,0x00,0x00,0x00,0x2c,0x2c,0x2c,
  0x2c,0x2c,0xc9,0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x8c,0x03,0xe5,0x02,
  0x02,0x02,0xc9,0x01,0xc9,0x2c,0xc9,0xc9,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0xc9,0x02,0xe5,0x8c,0x81,0xd1,0xbf,0xbf,0xbf,0x81,0x81,0xb3,
  0xd1,0xd1,0xb7,0xb7,0xbf,0xd1,0x81,0x81,0xb3,0xb3,0xfd,0x83,0x83,0x8c,0x8c,0x8c,
  0x8c,0x03,0x8c,0x83,0xfd,0xb3,0x81,0x92,0x92,0x81,0x8c,0x8c,0xfd,0x8c,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xfd,0x03,0x83,0xd1,0xd1,0xbf,0xbf,0x83,0x02,
  0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xd6,0x03,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x2c,0x2c,0x00,0x00,0x00,0x00,0x2c,0x2c,0x2c,
  0x2c,0x02,0x02,0x02,0xd6,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0xe5,0xd6,
  0x02,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0xc9,0x02,0x03,0x83,0x81,0xbf,0x92,0xb7,0xbf,0xb3,0xb3,0x81,
  0x81,0x81,0xb7,0xd1,0xd1,0x81,0x81,0xb3,0x8c,0x8c,0x8c,0x03,0x03,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x8c,0xb3,0xd1,0x92,0x92,0xbf,0x81,0x83,0x03,0x03,0xfd,0x03,
  0x03,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0xb3,0xb3,0xd1,0xb7,0xb7,0xbf,0x03,
  0x03,0x02,0x02,0x02,0xd6,0x02,0x03,0x03,0xd6,0xd6,0xd6,0xd6,0xd6,0x03,0xd6,0x03,
  0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x2c,0x00,0x00,0x00,0x00,0x2c,0x2c,0x2c,0x2c,
  0x2c,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x03,0x03,0x03,0x02,0x02,0x02,0x02,0x02,
  0x2c,0x01,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x2c,0xc9,0x02,0x03,0x83,0x81,0xb7,0xb7,0x81,0xb3,0x83,0xfd,
  0xb3,0x81,0x81,0x81,0x81,0xb3,0x8c,0x8c,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x02,
  0xe5,0x02,0x02,0x03,0x03,0x83,0xbf,0x92,0x92,0xb7,0xd1,0xb3,0x03,0x03,0x03,0x03,
  0x03,0x03,0x02,0x03,0x03,0x03,0x03,0x03,0x02,0x03,0x83,0x81,0xd1,0xd1,0xbf,0xbf,
  0xb7,0x03,0x03,0x03,0x02,0x02,0xd6,0xd6,0xd6,0xd6,0xd6,0x2c,0x2c,0xd6,0xd6,0xd6,
  0xd6,0x02,0x02,0x03,0x03,0x03,0x02,0xc9,0x00,0x70,0x00,0x2c,0x01,0x2c,0x2c,0xc9,
  0xc9,0x03,0x83,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0xe5,0x02,0x02,0xc9,
  0x01,0x01,0x01,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x01,0x00,0x01,0x01,0x01,0x01,0xe5,0x03,0x81,0xbf,0xd1,0x81,0x83,0x83,0x8c,
  0xfd,0xfd,0xb3,0xfd,0xfd,0x8c,0x03,0x03,0x03,0x03,0xe5,0x02,0x02,0x02,0xe5,0x02,
  0x02,0x02,0x02,0x02,0x03,0x83,0xd1,0x92,0x92,0xd1,0x81,0x81,0x8c,0x03,0x03,0x03,
  0x03,0x02,0x02,0x03,0x03,0x8c,0x03,0x02,0x02,0xfd,0x83,0x81,0xb3,0xbf,0xb7,0x92,
  0x92,0x81,0x03,0x02,0xd6,0x03,0x92,0xb7,0xfd,0xd6,0x02,0xfd,0x81,0x83,0x02,0x2c,
  0xd6,0x2c,0x03,0x03,0x8c,0x03,0x02,0x02,0x00,0x70,0x00,0x2c,0x2c,0x2c,0x02,0x02,
  0x03,0x8c,0x8c,0x03,0x03,0x02,0x02,0xd6,0x02,0xd6,0xd6,0x02,0xc9,0xc9,0x01,0x2c,
  0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,
  0x00,0x01,0x00,0x00,0x01,0x01,0x01,0x02,0xe5,0x8c,0x83,0x8c,0x8c,0x8c,0x03,0x8c,
  0x8c,0x8c,0x8c,0x8c,0xfd,0x03,0x03,0xe5,0x02,0xc9,0xc9,0x02,0x2c,0x2c,0xc9,0xc9,
  0xc9,0x01,0xc9,0x02,0x03,0x8c,0x81,0xbf,0xb7,0xd1,0xb3,0xb3,0x8c,0x03,0x03,0x03,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x83,0xb3,0x81,0xbf,0xb7,0x92,
  0x92,0xb7,0xc9,0x2c,0x2c,0x81,0x92,0x92,0x92,0x83,0x81,0xb7,0xb7,0xd1,0xb3,0x8c,
  0x2c,0x02,0x8c,0xb3,0x81,0xb3,0x03,0x02,0x2c,0x70,0x70,0x70,0x2c,0xd6,0x02,0x03,
  0x03,0x03,0x03,0x03,0x03,0x02,0x02,0x02,0xd6,0xd6,0x02,0x01,0xc9,0x2c,0x01,0x01,
  0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x70,0x00,0x70,0x00,0x01,0x01,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0xc9,0xe5,0xe5,0x03,0x03,0x03,0x8c,
  0xe5,0xe5,0x03,0x03,0xe5,0x03,0x03,0x02,0xc9,0x01,0x01,0x01,0x01,0x01,0x2c,0x2c,
  0x01,0x01,0x01,0xc9,0x02,0x8c,0x81,0xb7,0x92,0xb7,0x81,0xfd,0x8c,0x03,0x03,0x03,
  0x02,0x02,0x02,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x8c,0xb3,0xd1,0xb7,0xb7,0x92,
  0x92,0x92,0x01,0x00,0xe5,0x92,0x92,0x92,0x92,0xbf,0xb7,0xb7,0xb7,0xb7,0x81,0xb3,
  0x83,0x03,0x83,0x81,0x81,0x81,0x81,0x03,0x00,0x70,0x70,0x70,0x00,0x02,0x02,0x03,
  0x02,0x02,0x03,0x02,0x03,0x02,0x02,0x2c,0x2c,0x2c,0x2c,0x01,0x01,0x2c,0x01,0x01,
  0x00,0x01,0x01,0x01,0x01,0x00,0x70,0x00,0xfb,0xfb,0x00,0xfb,0x70,0x00,0x01,0x01,
  0x01,0x00,0x00,0xfb,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x02,0x02,0x02,0x02,0xe5,
  0xe5,0xe5,0x02,0x02,0xe5,0xe5,0x03,0x02,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x01,
  0x01,0x01,0x01,0xc9,0x02,0x03,0xb3,0xb7,0xb7,0x81,0x83,0xfd,0x03,0x03,0x03,0x03,
  0x03,0x02,0x02,0x03,0x02,0x02,0x02,0x03,0x02,0x03,0x03,0x83,0x81,0xbf,0xb7,0x92,
  0x92,0x92,0xbf,0xfd,0xbf,0x92,0x92,0x92,0x92,0x92,0xb7,0xbf,0xbf,0xbf,0xd1,0x81,
  0xd1,0x81,0x81,0xbf,0x81,0xd1,0xbf,0xd1,0x02,0x70,0x70,0x70,0x2c,0x02,0xc9,0x02,
  0x02,0xd6,0x02,0x02,0x02,0xc9,0x2c,0x2c,0x2c,0x2c,0x01,0x01,0x01,0x00,0x00,0x00,
  0x01,0x00,0x01,0x01,0x00,0xfb,0xfb,0xfb,0xfb,0xfb,0xc8,0xc8,0x0f,0x01,0x01,0x01,
  0x01,0x01,0x01,0x70,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0xc9,0xc9,0xc9,
  0xc9,0xc9,0x01,0xc9,0xc9,0x02,0x02,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x01,0x01,0x01,0xc9,0x02,0x03,0xb3,0xd1,0xd1,0x8c,0x03,0x03,0x03,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0xe5,0x02,0x02,0x02,0x02,0x03,0xfd,0xb3,0x81,0xbf,0xb7,
  0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0xb7,0xb7,0xb7,0xb7,0xbf,
  0x92,0x92,0xbf,0xbf,0x81,0xbf,0xb7,0xb7,0x81,0x03,0x2c,0x2c,0x2c,0x02,0x02,0x2c,
  0x2c,0x02,0xd6,0x02,0x2c,0x2c,0x2c,0x2c,0x01,0x01,0x01,0x01,0x00,0x01,0x01,0x00,
  0x01,0x01,0x00,0x00,0x70,0xfb,0xfb,0xc8,0x0f,0xc8,0xc8,0xc8,0x00,0x01,0x01,0x01,
  0x01,0x00,0x00,0x70,0x01,0x01,0x2c,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x02,0xc9,0x00,0x00,0x00,0x00,0x00,0x00,0xc8,0xfb,0xfb,
  0x00,0x00,0x01,0x2c,0x02,0xe5,0x83,0xb3,0x83,0x8c,0x03,0x03,0x03,0x02,0x02,0x02,
  0x02,0x02,0x02,0x02,0x02,0x02,0x83,0x83,0x03,0x03,0x03,0x8c,0xb3,0xb3,0xd1,0xb7,
  0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0x92,0xb7,0xb7,
  0xb7,0xb7,0xbf,0x81,0x81,0xd1,0xbf,0xbf,0xbf,0x81,0x02,0x2c,0x2c,0xd6,0x02,0x02,
  0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x01,0x2c,0x01,0x00,0x00,0x70,0x00,0x01,
  0x70,0x00,0x70,0x70,0x70,0x0f,0xfb,0xc8,0x0f,0x0f,0xc8,0xfb,0x00,0x01,0xc9,0xc9,
  0x01,0x01,0x01,0x00,0x01,0xc9,0xc9,0x01,0x00,0x0f,0x00,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x02,0x01,0x00,0xfb,0x00,0x00,0x00,0x0f,0x0f,0x0f,0x0f,
  0xfb,0x01,0x01,0x01,0xc9,0x02,0x03,0x03,0x8c,0x8c,0x03,0x03,0x02,0x02,0x02,0xd6,
  0x02,0xd6,0x02,0x03,0x03,0x03,0xb3,0xd1,0x81,0x8c,0x03,0xfd,0xfd,0xd1,0xd1,0xb7,
  0x92,0x92,0xdf,0xdf,0x92,0xdf,0x92,0x92,0x92,0xb7,0xb7,0x92,0x92,0x92,0x92,0x92,
  0xb7,0xd1,0xd1,0x81,0xb3,0x81,0xd1,0xbf,0xd1,0x83,0x02,0xd6,0xc9,0xd6,0x03,0x81,
  0xfd,0x02,0x2c,0x2c,0x2c,0x2c,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x70,0x00,0x00,
  0x00,0x70,0x70,0x0f,0x0f,0x0f,0x0f,0x00,0x00,0x0f,0xfb,0xfb,0x00,0x01,0x02,0xc9,
  0x2c,0x01,0x01,0x2c,0x01,0x01,0x01,0x01,0x01,0x00,0xfb,0x01,0x01,0x00,0x00,0x01,
  0x00,0x00,0x01,0x01,0x01,0xc9,0x01,0x00,0x00,0x00,0xfb,0x00,0x0f,0x0f,0x0f,0x0f,
  0x00,0x01,0x01,0x2c,0x2c,0x02,0x02,0x02,0x03,0x03,0x02,0x02,0x02,0x02,0x02,0x02,
  0x02,0x03,0x03,0x03,0x03,0x83,0xfd,0x81,0xd1,0xfd,0x8c,0x8c,0x83,0xb3,0x81,0xb7,
  0xb7,0xdf,0xdf,0x92,0xdf,0xdf,0xdf,0x92,0x92,0xbf,0xd1,0xb7,0x92,0xb7,0x92,0xdf,
  0x92,0xb7,0xbf,0x81,0x83,0x83,0x81,0xd1,0x83,0x02,0x2c,0x2c,0x2c,0xd6,0x03,0x8c,
  0x03,0x2c,0x2c,0x2c,0x2c,0xc9,0x01,0x01,0x00,0x00,0x00,0x00,0x70,0x70,0x00,0x00,
  0x00,0x00,0x70,0x70,0x0f,0xfb,0xfb,0x00,0x00,0x70,0xfb,0x0f,0x00,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x0f,0xc8,0xfb,0x00,0xfb,0x00,
  0x00,0x00,0x00,0x01,0xc9,0x02,0x01,0xfb,0x0f,0x0f,0x0f,0x0f,0xc8,0x0f,0x00,0x00,
  0x01,0x00,0x2c,0x2c,0x2c,0xc9,0xd6,0x02,0x02,0x02,0x02,0x02,0x02,0xd6,0x02,0x02,
  0x03,0x03,0x03,0x03,0x03,0xfd,0x83,0xb3,0xfd,0xfd,0xfd,0x8c,0x83,0xb3,0x81,0xd1,
  0xb7,0x92,0xdf,0xdf,0xdf,0xdf,0xdf,0x92,0xb7,0xb7,0xd1,0xb7,0xb7,0xbf,0xb7,0xdf,
  0xdf,0x92,0xbf,0x81,0x8c,0x8c,0x83,0xfd,0x02,0x2c,0x00,0x00,0x01,0x2c,0x02,0x03,
  0x03,0x03,0x02,0xc9,0x01,0x03,0x03,0x2c,0x2c,0x00,0x70,0x70,0x00,0x00,0x00,0x00,
  0x00,0x00,0x70,0x00,0x00,0x01,0x01,0x00,0x00,0xfb,0x70,0xfb,0x70,0xfb,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x70,0x0f,0x82,0xc8,0xc8,0xc8,0x00,
  0xfb,0x00,0x00,0x01,0xc9,0xe5,0x01,0xfb,0xc8,0xc8,0x0f,0x0f,0x0f,0xfb,0xfb,0x01,
  0x01,0x01,0x01,0x01,0x2c,0x2c,0x2c,0xd6,0xd6,0xd6,0x02,0x02,0x02,0xd6,0xd6,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0xfd,0x8c,0xfd,0x03,0xfd,0x8c,0x83,0xb3,0x81,0xd1,
  0xb7,0xdf,0xdf,0xdf,0xdf,0x92,0x92,0xb7,0xbf,0xd1,0x81,0x81,0xd1,0x81,0x81,0xb7,
  0xb7,0xb7,0x81,0x83,0x83,0x8c,0x03,0x70,0x70,0x70,0x70,0x70,0x70,0x2c,0x2c,0x02,
  0x03,0x8c,0x83,0xfd,0x83,0xbf,0xb7,0xfd,0x02,0x02,0x00,0x70,0x2c,0x2c,0x01,0x00,
  0x70,0x70,0x00,0x70,0x01,0xc9,0x00,0x70,0x00,0x00,0x00,0x00,0xfb,0x00,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x70,0x0f,0x82,0x82,0xc8,0xc8,0xc8,
  0x00,0xfb,0x00,0x01,0x01,0x01,0x00,0x0f,0x0f,0x0f,0x0f,0x0f,0xc8,0x0f,0xfb,0x2c,
  0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0xc9,0xd6,0xd6,0x02,0x02,0x02,0x03,
  0x03,0x03,0x8c,0x03,0xc9,0xd6,0x03,0x03,0x03,0x83,0x8c,0xfd,0xb7,0xb7,0xd1,0xd1,
  0xb7,0x92,0xdf,0xdf,0xdf,0x92,0xb7,0x81,0x81,0x81,0x81,0xb3,0x81,0x83,0x8c,0x83,
  0xd1,0xbf,0xfd,0x83,0x81,0x83,0x02,0x0f,0x0f,0x0f,0x0f,0x0f,0x70,0x70,0x00,0x2c,
  0x03,0x8c,0x83,0x81,0x81,0xbf,0xb7,0xbf,0x03,0xc9,0x00,0x00,0x02,0x2c,0x01,0x01,
  0x00,0xfb,0x00,0xfb,0x01,0x00,0x70,0x00,0x00,0x70,0xfb,0x00,0x70,0x00,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x0f,0x0f,0x0f,0x82,0x82,0xc8,0xc8,
  0xc8,0x00,0x00,0x00,0x00,0x00,0x0f,0x0f,0x0f,0x82,0xc8,0x0f,0xc8,0x0f,0x00,0xc9,
  0x01,0x01,0xc9,0x2c,0x2c,0x2c,0xc9,0x2c,0x2c,0x2c,0xc9,0xc9,0xd6,0x02,0x03,0x03,
  0x8c,0x8c,0x8c,0x03,0x02,0x2c,0xd6,0x02,0x81,0xb7,0x81,0x81,0xb7,0xb7,0xb7,0xb7,
  0x92,0x92,0xdf,0xdf,0xdf,0x92,0xbf,0x81,0xb3,0x81,0xb3,0x8c,0x8c,0x8c,0x8c,0x03,
  0x03,0x83,0xfd,0x8c,0x81,0xd1,0xd1,0x00,0x8b,0x0f,0x8b,0x8b,0x0f,0x0f,0x70,0x2c,
  0x02,0x03,0x81,0x81,0x81,0xb7,0xbf,0x81,0x83,0xe5,0x02,0x02,0xc9,0x00,0x00,0x01,
  0x70,0x0f,0xfb,0x0f,0x00,0x00,0x00,0x00,0x00,0xfb,0x0f,0x00,0x00,0x00,0x00,0x01,
  0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x70,0x0f,0x0f,0x0f,0xf0,0x82,0xf0,0xc8,
  0xc8,0xc8,0x70,0xfb,0xfb,0x0f,0xc8,0x0f,0x82,0x82,0x82,0x0f,0xc8,0x0f,0x01,0x8c,
  0x02,0x01,0x2c,0x2c,0x2c,0xc9,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0xd6,0x02,0x03,0xfd,
  0x8c,0x8c,0xfd,0x8c,0x03,0x2c,0x8c,0xfd,0xb7,0xdf,0xdf,0x92,0x92,0x92,0xb7,0x92,
  0x92,0x92,0xb7,0xb7,0xb7,0xbf,0x81,0xfd,0x81,0x81,0x83,0x8c,0x03,0x03,0x03,0x03,
  0x02,0xd6,0xd6,0x03,0xbf,0x92,0xb7,0x03,0x8b,0x0f,0x0f,0x0f,0x0f,0x8b,0x0f,0x70,
  0x2c,0x03,0xfd,0x81,0xb7,0xbf,0xbf,0x81,0x83,0xe5,0x02,0x01,0x00,0x00,0x00,0x00,
  0x0f,0x0f,0x0f,0x82,0x0f,0xfb,0x00,0xfb,0xfb,0xfb,0x0f,0xfb,0x00,0x00,0x70,0x01,
  0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x70,0x0f,0xfb,0x0f,0x0f,0x82,0x82,0x82,0xf0,
  0x0f,0xfb,0x0f,0x0f,0x0f,0x0f,0xc8,0x00,0x82,0x82,0xc8,0x0f,0xc8,0x00,0x01,0xe5,
  0x02,0x2c,0x01,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0x2c,0xd6,0x03,0x03,
  0x03,0x8c,0x8c,0x8c,0x03,0x02,0xb7,0xb7,0xb7,0xdf,0xdf,0xdf,0x92,0xdf,0x92,0xb7,
  0x92,0x92,0xbf,0x81,0x81,0xfd,0xb3,0x83,0xb3,0xb3,0xfd,0x03,0x03,0x03,0x03,0x03,
  0x02,0x2c,0xd6,0x8c,0xb7,0x92,0x92,0x81,0x00,0x8b,0x8b,0x0f,0x0f,0x0f,0x8b,0x70,
  0x01,0x02,0x03,0xfd,0x81,0x81,0xfd,0xbf,0x83,0xe5,0x02,0x00,0x00,0x00,0x70,0x70,
  0x82,0x82,0x82,0x82,0x0f,0x0f,0x70,0x00,0x00,0x0f,0x82,0x0f,0xc8,0x00,0x0f,0xfb,
  0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x70,0x0f,0x0f,0x0f,0x0f,0x82,0xf0,0x82,0x82,
  0xf0,0x0f,0x0f,0x82,0x0f,0x0f,0x00,0x0f,0x82,0x82,0x70,0x00,0x00,0x01,0x02,0x02,
  0x02,0xe5,0x2c,0x2c,0x2c,0x2c,0x01,0x01,0x01,0x2c,0x01,0x2c,0x2c,0xd6,0x02,0x02,
  0xd6,0x02,0xd6,0xd6,0x02,0x02,0xd1,0xd1,0xb7,0x92,0xdf,0xdf,0xdf,0xdf,0xdf,0xb7,
  0xb7,0xb7,0x81,0x83,0x83,0x8c,0xfd,0xfd,0xb3,0xb3,0x81,0x81,0x81,0x81,0x8c,0x03,
  0x2c,0x2c,0xd6,0x03,0x83,0x81,0xd1,0x81,0x00,0x8b,0x8b,0x8b,0x0f,0x0f,0x8b,0x0f,
  0x00,0xe5,0x03,0x8c,0xfd,0x92,0x92,0x81,0xfd,0x8c,0x8c,0x02,0x01,0x70,0x0f,0x0f,
  0xe0,0xe0,0x82,0x82,0x82,0x82,0x0f,0x0f,0x70,0x00,0xc8,0x82,0x82,0x82,0x0f,0x0f,
  0x00,0x00,0x01,0x01,0x01,0x00,0x00,0x70,0x70,0x0f,0x0f,0x0f,0x82,0x8d,0x8d,0x8d,
  0x8d,0x8d,0x82,0x82,0x0f,0xc8,0xc8,0x0f,0x82,0xf0,0xfb,0x01,0x01,0x01,0xc9,0xc9,
  0x2c,0x2c,0x2c,0x00,0x01,0x01,0x00,0x00,0x00,0x70,0x70,0x70,0x2c,0x2c,0x01,0x2c,
  0x2c,0x2c,0x2c,0xd6,0xd6,0x03,0x03,0xfd,0xbf,0xb7,0x92,0x92,0x92,0x92,0xb7,0xb7,
  0xd1,0x81,0x83,0x8c,0xb7,0x81,0x8c,0x81,0x8c,0x81,0x92,0x92,0xbf,0x81,0x03,0x02,
  0x02,0x02,0x2c,0x02,0x03,0x8c,0x81,0x81,0x00,0x8b,0x8b,0x82,0x8b,0x8b,0x0f,0x0f,
  0x02,0x81,0x83,0x83,0x92,0xdf,0xdf,0x92,0xbf,0x83,0x83,0xe5,0xc9,0x0f,0x82,0x8d,
  0x8e,0x82,0x82,0x82,0x8d,0x8d,0x82,0x82,0x82,0x0f,0x0f,0x82,0x8d,0x82,0x82,0xc8,
  0xfb,0x70,0x00,0x70,0x70,0x70,0xfb,0x0f,0xc8,0x0f,0x0f,0x82,0x8d,0xe0,0xfc,0x8e,
  0xfc,0xfc,0xe0,0x8d,0x82,0x82,0x0f,0xc8,0xc8,0x00,0x00,0x01,0x01,0x01,0x01,0x00,
  0x01,0x00,0x70,0x70,0x70,0x00,0x70,0x70,0x0f,0x0f,0x0f,0x70,0x70,0x00,0x00,0x00,
  0x70,0x70,0x2c,0x02,0x03,0x02,0x02,0x83,0x81,0x81,0xbf,0xb7,0xb7,0x81,0x81,0x81,
  0xfd,0x02,0x03,0x81,0xdf,0x92,0x81,0xb7,0xd1,0x81,0x81,0xb7,0x81,0xfd,0x03,0x03,
  0x02,0x02,0x02,0x2c,0x02,0x03,0x83,0x03,0x0f,0x82,0x8b,0x8b,0x82,0x8b,0x8b,0x2c,
  0xbf,0x92,0x92,0x92,0x92,0x92,0x92,0xb7,0x83,0xe5,0x83,0x02,0x00,0x01,0x82,0x8e,
  0x8e,0xf0,0x82,0x82,0xe0,0xe0,0x8d,0x8d,0xe0,0x8d,0x8d,0x8e,0xe0,0x8d,0x82,0x82,
  0x82,0x0f,0x0f,0x0f,0x0f,0x82,0x0f,0x0f,0x00,0x0f,0x82,0x8d,0xe0,0x8e,0xce,0xce,
  0xce,0xce,0xce,0xfc,0x8e,0xe0,0x82,0x0f,0x00,0x2c,0x00,0x01,0x01,0x01,0x01,0x00,
  0x00,0x00,0x0f,0x0f,0x0f,0x0f,0x0f,0x82,0x82,0x82,0x0f,0x0f,0x0f,0x70,0x70,0x2c,
  0xc9,0xe5,0x02,0x81,0x81,0x03,0x8c,0x83,0x03,0x03,0x83,0xfd,0xfd,0x8c,0xd6,0x03,
  0x02,0x2c,0x02,0xb7,0xdf,0xdf,0x92,0xbf,0x83,0x8c,0x03,0x8c,0x8c,0x8c,0xe5,0x02,
  0x02,0xc9,0xd6,0x02,0xd6,0x02,0x03,0xe5,0x0f,0x8b,0x8b,0x8b,0x82,0x8b,0x0f,0x02,
  0x81,0xb7,0xb7,0xbf,0xbf,0x92,0xbf,0x92,0xe5,0xe5,0xe5,0xe5,0xc9,0x01,0x8e,0xce,
  0xe0,0x82,0x8d,0xe0,0x8e,0xe0,0xe0,0xe0,0x8e,0xe0,0x8e,0x8e,0x8e,0x8e,0xe0,0x8d,
  0x8d,0x82,0x82,0x82,0x82,0x82,0x82,0x82,0xfb,0x0f,0x82,0x8e,0xca,0xce,0xce,0xce,
  0xcf,0xce,0xce,0xce,0x8e,0x8d,0x82,0x0f,0xfa,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x70,0x0f,0x0f,0x82,0x82,0xf5,0x82,0x82,0x82,0x82,0x0f,0x0f,0x70,0x2c,0xbf,
  0xb7,0xbf,0x81,0x81,0x81,0xbf,0x81,0x8c,0x03,0x03,0xe5,0x02,0x2c,0x01,0x01,0x00,
  0x00,0x00,0x2c,0xbf,0xdf,0xdf,0xbf,0x02,0x02,0x02,0x02,0x02,0x8c,0x83,0x8c,0x03,
  0x02,0x2c,0x2c,0x2c,0xd6,0x02,0xfd,0x81,0x70,0x8b,0x8b,0xf6,0x82,0x70,0x01,0x03,
  0x83,0x81,0xbf,0xbf,0xbf,0xb7,0xbf,0x92,0x92,0x8c,0xe5,0x02,0xc9,0x01,0x8e,0xce,
  0xcf,0xcf,0xce,0x8e,0x8e,0x8e,0x8e,0x8e,0xce,0xce,0xce,0xce,0xce,0xce,0xca,0xfc,
  0x8e,0xe0,0xe0,0xe0,0xe0,0x8d,0xe0,0x82,0x82,0x0f,0xe0,0x8e,0xce,0xce,0xfc,0xe0,
  0x8d,0x82,0x8d,0x82,0xf0,0xf0,0x82,0x0f,0xf0,0xfb,0xf9,0x01,0x01,0x01,0x01,0x01,
  0x01,0x0f,0x82,0x82,0x82,0x82,0xe0,0xe0,0xe0,0x82,0x82,0x0f,0x0f,0x70,0xe5,0x81,
  0xfd,0xfd,0x83,0xfd,0x81,0xbf,0x81,0x81,0x03,0x02,0x00,0x00,0x70,0x70,0xfb,0x0f,
  0x0f,0x02,0xe5,0xfd,0xbf,0xbf,0x8c,0x01,0x00,0x70,0x00,0x00,0x01,0x2c,0x2c,0x01,
  0x2c,0x2c,0x01,0x01,0x00,0x00,0x01,0x2c,0x70,0x8b,0x82,0xf6,0xe5,0x92,0x8c,0x03,
  0x03,0x83,0x83,0x83,0xfd,0x83,0xfd,0xbf,0xfd,0xe5,0x00,0x0f,0x00,0xc8,0x8e,0xec,
  0xcf,0xcf,0xcf,0xcf,0xcf,0xce,0xce,0xce,0xcf,0xec,0xdd,0xdd,0xcf,0xcf,0xcf,0xce,
  0xce,0xce,0x8e,0x8e,0x8e,0x8e,0x8e,0xe0,0x82,0x82,0x8d,0xe0,0xe0,0xe0,0x8d,0x82,
  0x82,0x82,0x8d,0x82,0x82,0x8d,0xe0,0xe0,0xe0,0x8d,0x00,0x00,0xfb,0x0f,0xc8,0x0f,
  0x0f,0x82,0xe0,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xe0,0x82,0x8b,0x82,0x0f,0x00,0x02,
  0x02,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0x01,0x0f,0x82,0x82,0x82,0x82,0x82,0x82,
  0x0f,0xe5,0x8c,0x02,0xd6,0x02,0x2c,0x70,0x0f,0x0f,0x0f,0x0f,0x82,0x0f,0x0f,0x70,
  0x00,0x01,0x2c,0x02,0x02,0xd6,0x70,0x8b,0xf6,0xe0,0xe0,0xe0,0x02,0xbf,0x8c,0x02,
  0xe5,0xe5,0xe5,0xe5,0x02,0xe5,0xe5,0x8c,0xe5,0x01,0x00,0xc8,0x00,0xc8,0xc8,0x8e,
  0x8d,0xce,0xcf,0xcf,0xdd,0xcf,0xdd,0xdd,0xd9,0xd9,0xd9,0xd9,0xd9,0xdd,0xd9,0xdd,
  0xdd,0xcf,0xce,0xcf,0xce,0xcf,0xcf,0xce,0x8e,0xe0,0xe0,0xe0,0x8d,0xe0,0x8e,0xe0,
  0xe0,0xe0,0xe0,0xe0,0x8e,0x8e,0x8e,0xce,0xce,0xce,0x8e,0x8e,0x8e,0xe0,0xe0,0x8e,
  0x8e,0xce,0x8e,0xce,0xce,0xce,0xce,0xce,0x8e,0xce,0x8e,0xe0,0xe0,0x82,0x82,0x0f,
  0x0f,0x70,0x70,0x0f,0x0f,0x0f,0x8b,0x82,0xe0,0x8e,0x8e,0x8e,0x8e,0x8e,0xe0,0x82,
  0x0f,0x0f,0x0f,0x82,0x82,0x82,0x82,0xe0,0xe0,0xe0,0xe0,0xe0,0x8d,0x82,0xf6,0x0f,
  0x70,0x70,0x2c,0x2c,0xe5,0x02,0xd6,0xf6,0xe0,0xe0,0xfe,0xe0,0x70,0xd6,0xd6,0xc9,
  0xc9,0x02,0x02,0x02,0xc9,0x83,0xc9,0x02,0x01,0x01,0x01,0xf9,0x01,0x01,0xfb,0xf0,
  0xe0,0xce,0xcf,0xd9,0xd9,0xf8,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xf8,0xf8,0xf8,0xf8,
  0xf8,0xd9,0xd9,0xdd,0xf7,0xdd,0xdd,0xcf,0xcf,0xce,0xce,0xce,0xce,0xce,0xce,0xce,
  0x8e,0xfc,0x8e,0x8e,0xce,0xce,0xcf,0xec,0xec,0xec,0xcf,0xcf,0xcf,0xce,0xce,0xcf,
  0xcf,0xcf,0xcf,0xcf,0xec,0xec,0xcf,0xcf,0xcf,0xce,0x8e,0x8e,0x8e,0x8e,0xe0,0x8e,
  0xe0,0xe0,0xe0,0xe0,0xe0,0x8e,0x8e,0xce,0xce,0xcf,0xce,0xce,0xcf,0xce,0x8e,0x8e,
  0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xce,0x8e,0xe0,0xe0,0xf6,
  0x82,0x82,0x0f,0x0f,0x70,0x70,0x8b,0x82,0x82,0xfe,0x8e,0x8e,0x82,0x0f,0x01,0x01,
  0x01,0x01,0x00,0x01,0xfa,0x8c,0xfa,0x01,0xf0,0xe0,0x82,0x00,0xc8,0xfb,0x00,0xfb,
  0xfc,0xe0,0xcf,0xd9,0xdd,0xf8,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,
  0xe6,0xe6,0xe6,0xe6,0xf8,0xe6,0xd9,0xf7,0xf7,0xdd,0xdd,0xec,0xcf,0xcf,0xcf,0xec,
  0xce,0xce,0x8e,0xce,0xcf,0xdd,0xdd,0xdd,0xd9,0xd9,0xd9,0xd9,0xd9,0xdd,0xdd,0xdd,
  0xdd,0xdd,0xdd,0xdd,0xdd,0xdd,0xf8,0xdd,0xdd,0xdd,0xec,0xcf,0xcf,0xce,0xce,0xce,
  0xce,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x8e,0x8e,0xce,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xcf,0xce,0xce,0x8e,0x8e,0x8e,
  0x8e,0x8e,0xe0,0x82,0x82,0x82,0x8e,0x8b,0x82,0x8e,0x8e,0x82,0x0f,0x70,0x00,0x00,
  0x00,0x0f,0x0f,0xc8,0x0f,0x00,0xc8,0xc8,0x8d,0xcf,0xce,0xf5,0xc8,0xfb,0xfb,0x0f,
  0x82,0x8d,0xce,0x8e,0xca,0xfc,0x8d,0xe0,0xdd,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,
  0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xe6,0xf8,0xf8,0xd9,0xd9,0xcf,0x8e,0xce,0xec,0xdd,
  0xec,0xec,0xec,0xdd,0xd9,0xd9,0xd9,0xd9,0xd9,0xf8,0xf8,0xf1,0xf8,0xd9,0xf1,0xf8,
  0xd9,0xf1,0xd9,0xd9,0xd9,0xd9,0xd9,0xf8,0xd9,0xf1,0xd9,0xd9,0xf7,0xdd,0xdd,0xcf,
  0xcf,0xcf,0xce,0x8e,0xce,0x8e,0x8e,0x8e,0xe0,0x8e,0x8e,0xe0,0x8e,0x8e,0x8e,0x8e,
  0xe0,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xce,0xcf,0xdd,0xdd,0xdd,0xce,0xce,0x8e,0x8e,
  0x8e,0xfe,0x8e,0x8e,0x82,0xf6,0xe0,0x82,0x8e,0xe0,0xe0,0x82,0x82,0x0f,0x70,0x0f,
  0x82,0x82,0x8d,0x8d,0x82,0x82,0x8e,0x8e,0xdd,0xf8,0xd9,0xca,0xf5,0x0f,0x0f,0xf0,
  // 19: sky 3, part D
  0x58,0xc6,0x38,0xc6,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,
  0xf7,0xbd,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x00,0x03,
  0x08,0x03,0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0xb7,0xbd,0x18,0x04,
  0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,
  0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,
  0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,
  0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,
  0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,
  0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,
  0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x08,0x42,0x10,0x42,0x18,0x42,
  0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,
  0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,
  0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,
  0x00,0x61,0x08,0x61,0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,
  0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,0xf8,0xc5,0x10,0x64,
  0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,
  0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,0x18,0x67,0x00,0x80,0x08,0x80,0xf6,0x8c,
  0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x96,0xad,0x00,0x82,
  0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x9a,0xd6,0x08,0x83,0x10,0x83,0xb6,0x7c,
  0x18,0x83,0x00,0x84,0x08,0x84,0x10,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,
  0x18,0x85,0x00,0x86,0x77,0x9d,0x08,0x86,0x10,0x86,0x18,0x86,0x00,0x87,0x08,0x87,
  0x10,0x87,0xb8,0xad,0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,0x37,0x9d,
  0x00,0xa1,0x08,0xa1,0x18,0xbe,0x10,0xa1,0x18,0xa1,0xd8,0xbd,0x00,0xa2,0x08,0xa2,
  0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x75,0xad,0x00,0xa4,
  0x08,0xa4,0x10,0xa4,0x18,0xa4,0xdb,0xd6,0x00,0xa5,0x08,0xa5,0x10,0xa5,0x18,0xa5,
  0x00,0xa6,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x34,0xa5,0xb3,0x5b,0xd8,0xad,0x00,0xa7,
  0x08,0xa7,0x10,0xa7,0x18,0xa7,0x00,0xc0,0x08,0xc0,0x10,0xc0,0x95,0x7c,0x18,0xc0,
  0x50,0x84,0x76,0xa5,0x00,0xc1,0x08,0xc1,0x7a,0xd6,0x10,0xc1,0x18,0xc1,0x00,0xc2,
  0x08,0xc2,0x17,0x8d,0x10,0xc2,0x17,0xc6,0x18,0xc2,0xf3,0x6b,0x00,0xc3,0x08,0xc3,
  0x3d,0xe7,0xd3,0x5b,0x10,0xc3,0xef,0x7b,0x18,0xc3,0x00,0xc4,0x5a,0xc6,0xd6,0x7c,
  0x2c,0x63,0x08,0xc4,0x9f,0xef,0x99,0xce,0x9a,0xce,0xf3,0x9c,0x94,0xbd,0xf5,0xcd,
  0x79,0xd6,0xf7,0xcd,0xae,0x73,0xf4,0x9c,0xb2,0x94,0x48,0x42,0x24,0x21,0xaa,0x52,
  0x87,0xef,0x87,0x87,0x87,0x87,0xa9,0xa9,0xa2,0xd6,0xd6,0xe5,0xcd,0xcd,0xcd,0xcd,
  0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xe9,0xe5,0xe5,0xe5,0xd6,0x97,0xd6,0xe5,
  0xe5,0xe5,0xe5,0xe5,0xe5,0xe9,0xe9,0xe9,0xe9,0xe9,0xcd,0xe9,0xcd,0xe9,0xcd,0xe9,
  0xe9,0xe5,0xe5,0xcd,0xe9,0xcd,0xcd,0xcd,0xe9,0xcd,0xe5,0xcd,0xe9,0xe9,0xe5,0xe5,
  0xe5,0xe5,0xd6,0xd6,0xd6,0xef,0xe1,0x87,0x87,0xef,0xef,0xef,0x97,0x97,0x97,0xef,
  0xef,0xef,0xef,0x97,0xef,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xef,0x87,0xa0,0xef,
  0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xef,0xef,0xef,0xef,0x97,0xa0,0xa2,0xa2,0xe1,
  0x87,0xd6,0xef,0x97,0x97,0xef,0xe5,0xe5,0xe5,0xcd,0xe5,0xd6,0xd6,0x87,0x87,0xe1,
  0x97,0xef,0x87,0xe1,0x87,0xc7,0xa9,0xa9,0xa9,0xa2,0xc7,0xd6,0xe9,0xcd,0xcd,0xe9,
  0xcd,0xe9,0xe9,0xe9,0xe9,0xcd,0xcd,0xe9,0xcd,0xe9,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,
  0xe5,0xe9,0xe9,0xe9,0xe9,0xe9,0xe9,0xcd,0xe9,0xe9,0xcd,0xe9,0xe9,0xcd,0xcd,0xe9,
  0xcd,0xe9,0xe9,0xe9,0xe9,0xcd,0xe9,0xcd,0xcd,0xe9,0xcd,0xcd,0xcd,0xcd,0xe9,0xe5,
  0xe9,0xe5,0xd6,0xd6,0x97,0x97,0xa0,0xe1,0xe1,0xef,0x97,0x97,0x97,0x97,0x97,0x97,
  0xe1,0xe1,0xe1,0xa0,0x87,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0xef,0x87,0xef,
  0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xef,0xef,0x97,0xef,0xa0,0xe1,0x97,
  0xd6,0xd6,0xd6,0xd6,0xd6,0x97,0xd6,0x97,0xd6,0xe5,0xe5,0xd6,0xaf,0xe1,0xef,0xef,
  0x87,0xe1,0x87,0xe1,0x87,0xa2,0xb5,0xb2,0xee,0xb5,0xce,0xa2,0xa2,0xa2,0xaf,0x87,
  0x87,0xa2,0xc7,0x8e,0xa2,0xaf,0x87,0x87,0xd6,0xd6,0xd6,0x87,0x87,0xd6,0xd6,0xd6,
  0xd6,0xd6,0x87,0xd6,0xef,0xe5,0xe9,0xe9,0xe9,0xe9,0xcd,0xe9,0xe9,0xcd,0xe9,0xe9,
  0xe9,0xe9,0xe9,0xe9,0xcd,0xe9,0xe9,0xe9,0xcd,0xe9,0xe9,0xe9,0xcd,0xcd,0xcd,0xcd,
  0xe9,0xe9,0xe5,0xd6,0xd6,0xd6,0xd6,0x97,0xef,0x97,0xd6,0xd6,0x97,0x97,0xd6,0x97,
  0xe1,0xe1,0xa2,0x87,0xe1,0xa0,0x87,0xef,0xef,0xa0,0xef,0xef,0x87,0xef,0xef,0xef,
  0xef,0xef,0xef,0xef,0xef,0xef,0xef,0x97,0x97,0xa0,0xef,0x97,0x97,0x97,0xd6,0xd6,
  0xd6,0xe5,0xd6,0xe5,0xd6,0x97,0xd6,0xd6,0xd6,0xe5,0x87,0xef,0xaf,0x8e,0xaf,0xa2,
  0x87,0xe1,0xef,0x87,0xe1,0xaf,0xa9,0xa9,0xce,0xee,0xb2,0xb2,0xb2,0xb2,0xb2,0xb2,
  0xa9,0x8e,0xa9,0xc7,0xa9,0xa9,0xa9,0xa9,0xa9,0x8e,0xa2,0x8e,0xa9,0xb5,0xa9,0xa9,
  0x8e,0xa9,0xa9,0xa9,0xa9,0xe1,0xe5,0xe5,0xd6,0xe1,0xd6,0xd6,0xe5,0xe5,0xcd,0xe9,
  0xcd,0xe9,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xe5,0xd6,
  0xd6,0xd6,0xe5,0xe5,0xe5,0xd6,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xd6,0x97,0x97,
  0x87,0xa2,0xa2,0xa2,0xe1,0x87,0xef,0x87,0xe1,0x87,0x87,0xa0,0xa0,0x87,0xef,0xef,
  0x87,0xa2,0xa2,0x87,0xef,0x97,0x97,0x97,0x97,0xef,0xef,0x97,0x97,0xd6,0xd6,0xd6,
  0xe5,0xe9,0xe5,0xe5,0xe5,0xe5,0xe5,0xd6,0xd6,0xd6,0x87,0xc7,0x87,0xaf,0x87,0x87,
  0x87,0xe1,0xaf,0xa2,0xa2,0xc7,0xa2,0xa2,0xa2,0xb5,0xb2,0xb5,0xa9,0xa9,0xb5,0xb5,
  0xb5,0xb2,0xa9,0xa9,0xa9,0xce,0xce,0xb5,0xa2,0xaf,0xc7,0xa9,0xce,0xb2,0xb2,0xb2,
  0xb5,0xb2,0xce,0xb5,0xce,0xa9,0xa2,0xa9,0xa9,0xce,0xce,0xa9,0xa2,0xa2,0xe1,0xd6,
  0x87,0x87,0x87,0xd6,0xe5,0xe9,0xe5,0xe5,0xe5,0xe5,0xe5,0xd6,0xd6,0x87,0xa2,0xa9,
  0xa9,0xa9,0xc7,0xa2,0xa2,0xc7,0xa2,0xa2,0xa2,0xe1,0xef,0xef,0x97,0xd6,0xd6,0x87,
  0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xe1,0xe1,0xe1,0xe1,0xe1,0xe1,0xa0,0xa0,
  0xa2,0xa2,0xe1,0xa0,0x87,0x97,0x97,0x97,0x97,0x97,0xef,0xef,0x97,0xd6,0xd6,0xe5,
  0xe5,0xe9,0xcd,0xcd,0xe9,0xe5,0xe5,0xd6,0xd6,0xd6,0xef,0x87,0x87,0xe1,0xef,0x87,
  0xe1,0xa2,0xc7,0xc7,0xc7,0xa9,0xc7,0xa2,0xa2,0x8e,0x8e,0xa2,0xa2,0xc7,0xc7,0x8e,
  0xa9,0xb5,0xa9,0xa2,0xc7,0xc7,0xc7,0xa2,0xaf,0x87,0x87,0x87,0x8e,0xa9,0xb2,0xb2,
  0xb5,0xb2,0xb2,0xb5,0xce,0xce,0xce,0xce,0xce,0xce,0xce,0xce,0xce,0xce,0xb2,0xce,
  0xce,0xa9,0xa9,0xb2,0xb2,0xa2,0xa2,0x8e,0xa2,0xa9,0xa9,0xa9,0xa9,0xa9,0xce,0xa9,
  0xa9,0xb5,0xa9,0xa9,0xb5,0xa9,0xa9,0xa2,0xa2,0xe1,0xef,0x97,0x97,0xd6,0x97,0x87,
  0xa2,0xc7,0xa2,0xa2,0xa2,0xc7,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xe1,0xe1,
  0xa2,0xe1,0xe1,0xe1,0xef,0x97,0x87,0xe1,0x97,0xd6,0xd6,0xa0,0xd6,0x97,0xef,0xd6,
  0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xe5,0xd6,0xd6,0xd6,0xef,0x87,0xe1,0xe1,0xef,0x97,
  0x87,0xa2,0xa2,0xc7,0xa2,0xa9,0x8e,0x8e,0xa9,0xa9,0xa9,0xc7,0xc7,0xc7,0xa2,0xa2,
  0x8e,0xa9,0x8e,0xa2,0xa2,0x87,0xa2,0xaf,0x87,0x87,0xaf,0xc7,0xa9,0xce,0xb5,0xce,
  0xce,0xb2,0xb5,0xb5,0x00,0xee,0xb2,0xb2,0xce,0xce,0xb5,0xb2,0xce,0xce,0xce,0xce,
  0xce,0xce,0xa9,0xb2,0x00,0xb2,0xce,0xce,0xb5,0xb5,0xce,0xce,0xa9,0xb5,0xb5,0xce,
  0xb5,0xce,0xce,0xce,0xb5,0xce,0xa2,0xe1,0xef,0x87,0xef,0x97,0xd6,0x97,0xe1,0x87,
  0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
  0x87,0xe1,0xe1,0xe1,0x97,0xd6,0xef,0xe1,0xef,0x97,0xd6,0x97,0x97,0xef,0xaf,0xef,
  0xd6,0x87,0x87,0xef,0xef,0xd6,0xef,0xd6,0x97,0x97,0xd6,0xef,0xd6,0xd6,0xd6,0x87,
  0x87,0xa2,0xc7,0xc7,0xa2,0xa2,0xa2,0x8e,0xa9,0xa9,0xb5,0xa9,0x8e,0x8e,0x8e,0xa2,
  0xa2,0xa2,0x8e,0xfc,0xce,0xa9,0xa9,0xa9,0xa2,0xa2,0xa2,0xa2,0xa9,0xb5,0xce,0xb5,
  0xb2,0xb2,0xb2,0xb2,0x94,0xc3,0x00,0xc3,0x00,0xe8,0xc3,0x94,0xb5,0xb2,0xb2,0xb2,
  0xce,0xb2,0xb2,0xb2,0x01,0x01,0xce,0xb2,0xb5,0xb5,0xce,0xce,0xce,0xce,0xce,0xb5,
  0xb5,0xa9,0xa9,0xce,0xce,0xb5,0xa2,0xef,0xd6,0x97,0x97,0xd6,0xd6,0xef,0xe1,0x87,
  0xa2,0xa2,0x87,0xa2,0xc7,0xa2,0xa2,0xa2,0xa2,0xa2,0xe1,0xef,0x87,0xa2,0xa2,0xa2,
  0xe1,0x87,0xe1,0xe1,0xe1,0xa2,0x87,0xa2,0xa2,0xd6,0xd6,0xd6,0xd6,0x97,0x97,0xd6,
  0x97,0xd6,0xd6,0xd6,0xd6,0x97,0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0x87,
  0x87,0xa2,0xaf,0xaf,0xa2,0xa2,0xa2,0x8e,0x8e,0xa2,0xc7,0xa2,0xc7,0xa2,0xa2,0xaf,
  0xc7,0xa9,0xd6,0xff,0x8e,0xa2,0xa2,0xa9,0xa2,0xa2,0xa2,0xa2,0xa2,0xc7,0xa2,0xa9,
  0xce,0xb5,0xce,0xb5,0xb2,0x94,0xf3,0xf2,0xe8,0xc3,0xc3,0xc3,0xf3,0xee,0xb2,0xb2,
  0xb2,0xb2,0xce,0xb2,0xb2,0xee,0xb2,0xb2,0xb2,0xb2,0x8e,0xa9,0xce,0xb5,0xce,0xb5,
  0xce,0xb5,0xb5,0xb5,0xb5,0xa9,0x87,0xef,0xd6,0x97,0xd6,0xd6,0xd6,0xe1,0x87,0xa2,
  0x87,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,0xc7,0xa2,0xa2,0xaf,0xa2,0xa2,0xa2,0xa2,0xa2,
  0x87,0xe1,0x87,0x87,0x87,0xa2,0xb2,0x8e,0x87,0xef,0xd6,0xd6,0x97,0xef,0x97,0x97,
  0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0x97,0xd6,0xd6,0x97,0x97,0xd6,0xd6,0xd6,0xd6,0x97,
  0xaf,0xe1,0x87,0xef,0x87,0x87,0xe1,0xd9,0xd9,0x87,0xef,0x97,0xef,0x87,0x87,0xd6,
  0x87,0xa2,0x9b,0xfd,0xfc,0xd8,0xa9,0xc7,0xa2,0xa2,0xa2,0xa2,0xa9,0xa2,0xa9,0xa9,
  0xa9,0xa9,0xa9,0x00,0xf3,0xf3,0xf3,0xf3,0xc3,0x94,0xe3,0x8e,0x08,0x08,0xb2,0xce,
  0xb2,0xce,0xb5,0xb2,0xb2,0xb2,0xb5,0xce,0xb5,0xa2,0xa2,0x8e,0xa9,0x8e,0xa9,0xa9,
  0x8e,0x8e,0xa9,0xa2,0xa2,0xa2,0x87,0x97,0xd6,0xd6,0xd6,0xd6,0xd6,0x87,0xa2,0xa2,
  0xa2,0xa2,0xc7,0xa2,0xa2,0xa2,0xa2,0xa2,0xb5,0xb2,0xa2,0xa2,0xa2,0xa2,0xa2,0xa2,
  0x87,0x87,0xe1,0xe1,0x87,0x8e,0xb5,0xb2,0x8e,0xe1,0xe1,0x87,0xef,0xa2,0xaf,0x87,
  0x87,0x87,0x8e,0xa2,0xa2,0xa2,0x87,0xd6,0xd6,0xd6,0xd6,0xd6,0xd6,0x87,0xfc,0x87,
  0xcc,0xaf,0x87,0x87,0x87,0xa2,0xd6,0x8e,0xe3,0x8e,0xaf,0xa2,0xaf,0x87,0xd6,0xd8,
  0xd6,0x87,0xff,0xff,0x9b,0xf0,0xa2,0x8e,0xa9,0x8e,0xce,0xa9,0xce,0xb5,0xb5,0xb5,
  0xb5,0xd9,0xc3,0xc3,0x94,0x01,0xc3,0xdc,0x94,0x00,0xbe,0xcc,0xf5,0xcc,0x8e,0x00,
  0xb2,0xb2,0xb2,0xb5,0x00,0x01,0xa9,0xb2,0xce,0x8e,0xa2,0xa2,0xa2,0x8e,0x8e,0xa2,
  0xa2,0xa9,0x8e,0xa9,0x8e,0xa9,0x87,0xef,0xd6,0xe5,0xe5,0xe5,0xe5,0x87,0xa2,0xa2,
  0xa2,0xa2,0x87,0xe5,0xa2,0xa2,0xc7,0x8e,0x8e,0xb5,0x8e,0xa2,0xa2,0xa2,0xa2,0xa2,
  0xaf,0x87,0x87,0xe1,0xa2,0xaf,0xa2,0xaf,0xa2,0xa2,0x87,0x87,0xa2,0x8e,0xa2,0xa2,
  0xa2,0xa2,0x08,0x08,0xa2,0xaf,0xa2,0xa2,0xa2,0xa2,0xa2,0xb5,0xb2,0xb2,0x8e,0xcc,
  0xd8,0xaf,0xa2,0xa2,0xc7,0xa2,0xfb,0xf0,0x01,0x08,0xcc,0x8e,0x8e,0x8e,0xfc,0x9b,
  0x9b,0xd8,0xf0,0xfd,0xff,0xff,0xfb,0xce,0xa9,0xa9,0xa2,0xa9,0xa2,0x8e,0xa2,0x00,
  0xf3,0xe8,0xc3,0xc3,0xc3,0x08,0x94,0xc3,0x16,0xcc,0x8e,0xf5,0x8e,0x08,0xbe,0x8e,
  0xf3,0xb2,0xb5,0xb2,0xb2,0xb2,0x01,0x94,0xf3,0xb5,0xce,0xce,0xa9,0xb5,0xce,0xce,
  0xb5,0xce,0xb5,0x8e,0xaf,0xa2,0x87,0xd6,0xd6,0xe5,0xd6,0xd6,0xd6,0xa2,0xa2,0xfc,
  0xa2,0x01,0x87,0xe5,0x87,0xa2,0xa2,0xa2,0x08,0xb2,0xb5,0xa2,0xa2,0x87,0x87,0x87,
  0x87,0xef,0xe1,0xa2,0x8e,0xa2,0xa2,0xa2,0xa2,0xa2,0x8e,0xaf,0xaf,0x08,0x76,0x08,
  0x8e,0xa2,0xaf,0xa2,0xb2,0x94,0x94,0xf4,0xa2,0xa2,0x8e,0xb5,0x08,0x08,0xcc,0xfc,
  0xfc,0xbe,0xaf,0xa2,0xf4,0xc3,0xbe,0xeb,0xcc,0xc3,0x00,0xcc,0xbe,0x08,0xd8,0xf0,
  0xfa,0xf0,0xfd,0xff,0xfd,0xfd,0xd8,0xa2,0xa2,0x8e,0x8e,0x8e,0xa2,0x00,0x94,0x94,
  0xf2,0xe8,0xe8,0xc3,0x01,0xf3,0x00,0xf3,0xfb,0xfb,0xcc,0x08,0x08,0x01,0x8e,0xbe,
  0xbe,0xb2,0x8e,0xc7,0xa9,0xf3,0xf3,0xf3,0x00,0xa9,0xa2,0xaf,0x8e,0xaf,0xa9,0x8e,
  0x8e,0xb2,0x00,0xa2,0x87,0xd6,0xd6,0xd6,0xe5,0xe5,0xe5,0xd6,0xd6,0xa2,0xa2,0xf0,
  0xfc,0x76,0xaf,0xd6,0x87,0xb5,0xb5,0x08,0x8e,0xbe,0xf5,0xfb,0xcc,0xa2,0xaf,0xaf,
  0xaf,0xef,0x87,0xa2,0xb5,0x08,0x8e,0xa2,0x87,0x87,0x87,0x87,0xaf,0x08,0x08,0xd9,
  0xaf,0xa2,0xaf,0xb2,0x8e,0x16,0xf3,0xdc,0xe3,0x08,0x08,0x8e,0xbe,0xbe,0xd8,0xd8,
  0xf0,0xfc,0x08,0xe8,0xe8,0xe8,0x00,0xff,0xfb,0x08,0xe8,0xf3,0xbe,0xcc,0xeb,0xf0,
  0xfa,0xff,0xfd,0x4d,0xfd,0xff,0xfa,0xa2,0xa2,0xb5,0xb2,0xb2,0x94,0xe8,0xc3,0xe8,
  0xe8,0xdc,0xe8,0xfb,0xe3,0x01,0x08,0x76,0xcc,0xf5,0xcc,0x8e,0x8e,0x08,0xbe,0x8e,
  0x01,0xb2,0x01,0xb2,0xf3,0xf3,0xb2,0xb5,0xb5,0x87,0xfc,0xfc,0xfb,0xaf,0x8e,0x8e,
  0xf3,0xf3,0xb2,0xd6,0xd6,0xe5,0xe5,0xe5,0xe5,0xd6,0xe5,0x87,0xaf,0xd8,0xfc,0xfd,
  0xfb,0xe8,0xe8,0xc3,0xc3,0x01,0x08,0x16,0xbe,0xcc,0xfb,0xfc,0xf5,0x87,0x87,0x87,
  0x87,0xd6,0xd6,0x87,0x87,0xb2,0x8e,0x87,0x87,0x87,0x87,0x87,0xaf,0x8e,0xa2,0x87,
  0x8e,0x8e,0x8e,0xf5,0xfc,0xbe,0x8e,0xf3,0xf3,0xe3,0x01,0x76,0xcc,0xfa,0xfd,0xf0,
  0x94,0xe8,0xe8,0xe8,0xe8,0xe8,0xf2,0x8e,0xcc,0xf5,0xf3,0xc3,0xf3,0x01,0xe8,0x08,
  0xcc,0x8e,0xfd,0xfd,0x4d,0x2d,0xcc,0xee,0xf4,0xc3,0xc3,0xe8,0xe8,0xc3,0xc3,0x00,
  0xc3,0xc3,0xf3,0x16,0x08,0x9b,0xfa,0xf3,0x8e,0x01,0x8e,0xf5,0x01,0xe3,0xdc,0x00,
  0x94,0xf3,0x00,0x01,0x76,0x76,0x8e,0xa2,0xb2,0xd9,0xcc,0xd8,0xd8,0xfb,0xf9,0x01,
  0x76,0xb2,0xf3,0xaf,0xe5,0xfc,0xfb,0xa2,0x00,0xc3,0x94,0xb5,0x8e,0xfb,0xb2,0xf5,
  0xe8,0xc3,0x94,0xdc,0xf3,0x76,0x08,0xbe,0xcc,0xcc,0xf5,0xf5,0xd8,0xd8,0xd6,0xd6,
  0xd6,0xd6,0xd6,0x87,0x87,0xe8,0xf2,0xf2,0xe8,0xee,0xce,0xa2,0xa2,0xa9,0x01,0x94,
  0xc3,0x94,0x76,0x08,0xbe,0xfb,0xfc,0x08,0xb2,0x76,0xc3,0xc3,0xe8,0xe8,0xc3,0xdc,
  0xc3,0xc3,0x94,0xc3,0xe8,0xe8,0xc3,0x94,0xdc,0xdc,0x76,0x76,0xf4,0xe8,0xe8,0xe8,
  0xc3,0xe8,0xe8,0xf3,0x01,0xb2,0xe8,0xe8,0xe8,0xc3,0xe8,0xc3,0xe8,0xe8,0xc3,0xe8,
  0x16,0xc3,0x8e,0x08,0xfc,0xf0,0xff,0xbe,0xcc,0x8e,0xcc,0xbe,0xe3,0xe3,0x16,0xdc,
  0xe3,0xdc,0xf3,0x01,0x08,0xe3,0xb2,0x8e,0xaf,0xe3,0x8e,0xd8,0xd8,0xcc,0xbe,0x08,
  0xe3,0xe3,0x08,0xfc,0xfb,0xf5,0xfc,0xc3,0xc3,0xc3,0xc3,0xc3,0xf3,0x94,0x94,0xc3,
  0xc3,0xdc,0xc3,0x94,0xdc,0x00,0x16,0xbe,0xcc,0xfb,0xf5,0xfc,0xd8,0x9b,0xd8,0x8e,
  0x16,0x08,0xaf,0x87,0x87,0xfb,0xc3,0xc3,0xe8,0xe8,0xc3,0x94,0x01,0xf3,0xdc,0x94,
  0x94,0x94,0xdc,0xdc,0x94,0x94,0x94,0x94,0x94,0x94,0x94,0x94,0xc3,0xe8,0xe8,0xe8,
  0xe8,0xc3,0xdc,0xdc,0xc3,0xc3,0xc3,0xc3,0x94,0x94,0x94,0xc3,0xc3,0x94,0x94,0xc3,
  0xc3,0xc3,0xe8,0xe8,0xc3,0xc3,0xc3,0xc3,0xe8,0xe8,0xe8,0xf2,0xe8,0xdc,0x94,0x08,
  0x94,0xdc,0xe3,0xd8,0xeb,0xf0,0xfd,0xd8,0xfc,0x8e,0x8e,0xf3,0xf3,0x94,0x8e,0x01,
  0x01,0xcc,0xf3,0x00,0xb2,0x08,0x8e,0x76,0xf3,0xdc,0xe3,0xcc,0xff,0xff,0xeb,0xf5,
  0xf5,0xfb,0xfc,0xd8,0xd8,0xf0,0xd8,0xf5,0xdc,0xc3,0xe8,0xe8,0x94,0xf3,0x08,0xf3,
  0x08,0xbe,0x94,0xdc,0x00,0xf3,0xf3,0x76,0x08,0x08,0xbe,0xbe,0xcc,0xbe,0x16,0x08,
  0x16,0x08,0x76,0x76,0x76,0xf5,0xeb,0xeb,0xd8,0xe3,0x01,0x00,0xf3,0x08,0x00,0x00,
  0xf3,0xf3,0xdc,0xdc,0x94,0xdc,0xf4,0x94,0x94,0x94,0xdc,0xf3,0xdc,0xdc,0xc3,0xe8,
  0xf3,0x94,0xc3,0xc3,0xc3,0xe8,0xe8,0xc3,0xc3,0x94,0xf3,0xdc,0xf3,0xdc,0x94,0xc3,
  0xc3,0xe8,0xe8,0xe8,0xe8,0xe8,0xc3,0xc3,0xf3,0xc3,0xe8,0xe8,0xe3,0xe8,0xf3,0x8e,
  0x76,0x8e,0x08,0x9b,0xfd,0x2d,0xfd,0xf0,0xf0,0xd8,0xf5,0xd8,0xcc,0x8e,0xf3,0xbe,
  0xfc,0xeb,0xdc,0xf3,0x00,0x00,0xb2,0x01,0xf3,0xf3,0xdc,0x94,0xc3,0x94,0x01,0x08,
  0x8e,0x08,0x01,0x00,0x8e,0xf0,0xf0,0xf5,0xc3,0x94,0x94,0xdc,0x01,0xbe,0xcc,0x08,
  0x76,0x08,0x94,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x76,0x76,0x08,0x16,0x08,
  0x16,0x08,0x76,0x08,0x16,0x8e,0xcc,0xcc,0xcc,0xfc,0xbe,0x8e,0x16,0x08,0x08,0x76,
  0x76,0xf3,0x94,0x94,0x94,0x00,0xf3,0xf3,0x94,0x94,0x08,0xb2,0xd9,0xc3,0x94,0x01,
  0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0x00,0xf3,0xf3,0x94,0xc3,0xe8,
  0xe8,0xe8,0xe8,0xe8,0xe8,0xbe,0xc3,0xf3,0x94,0xf3,0x94,0x94,0xf3,0x16,0x9b,0xd8,
  0xfc,0xf5,0x9b,0xff,0xfe,0x2d,0xfd,0x2d,0xf0,0xf0,0xfa,0xf5,0xd8,0xfc,0xf0,0xf0,
  0xf0,0xcc,0xdc,0xdc,0xdc,0xdc,0xf3,0x01,0x01,0x01,0xf3,0xf3,0x00,0xf3,0x08,0xf3,
  0xf8,0xf3,0xdc,0x94,0x8e,0xfc,0x8e,0x94,0xdc,0xdc,0xf3,0x00,0xdc,0x94,0x94,0x01,
  0x16,0x01,0x94,0xdc,0xf3,0xf3,0xdc,0xdc,0xf3,0xf3,0xf3,0x76,0x76,0x76,0x76,0x76,
  0x08,0x08,0x08,0x08,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x16,0x16,0xe3,0x00,0x00,0x01,0xf3,0x76,0x01,0x8e,0xee,0xee,0xee,0xb2,0x01,0x01,
  0xc3,0xe3,0x01,0x00,0x01,0x00,0xf3,0xf3,0x01,0xf3,0xf3,0xf3,0xdc,0x94,0x94,0x94,
  0xc3,0xc3,0xc3,0xc3,0x08,0xbe,0x16,0xf3,0xf3,0x76,0xc3,0xf3,0xf3,0x8e,0xeb,0xfa,
  0xd8,0xd8,0xfa,0xfd,0xfd,0x2d,0x2d,0xfd,0xf0,0xeb,0xd8,0xfc,0xf5,0xf5,0xf0,0xd8,
  0xfa,0x08,0xdc,0xf3,0xf3,0xf3,0x94,0xf3,0x01,0xb2,0xb2,0x01,0x01,0xcc,0xf3,0x01,
  0xc3,0x00,0x01,0x01,0xf3,0xf4,0xf3,0xdc,0xdc,0xf3,0xf3,0x01,0xf3,0xf3,0x00,0xf3,
  0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,0x01,0x01,0x76,0xb2,0x08,0x76,0x76,0x08,0x16,
  0x16,0x16,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xbe,0x8e,0x8e,0x8e,
  0x8e,0x08,0x08,0x01,0x01,0x76,0x01,0xc3,0xc3,0xe8,0xc3,0xc3,0xc3,0xb2,0x8e,0xee,
  0xbe,0x76,0x01,0xe3,0xf3,0x01,0x08,0x08,0xe3,0x00,0xf3,0xf3,0xf3,0xf3,0xf3,0xf3,
  0xdc,0xf3,0xf3,0x94,0xf5,0x08,0xe3,0x08,0xe3,0x01,0x08,0x01,0x94,0xc3,0xbe,0xcc,
  0xcc,0xcc,0xcc,0xfe,0xfe,0x2d,0xfe,0xfe,0xf5,0xbe,0x8e,0xbe,0xeb,0xeb,0xeb,0xd8,
  0x8e,0x94,0xf3,0xf3,0x00,0x76,0x01,0xf3,0xf3,0xf3,0x01,0x76,0xbe,0xbe,0xcc,0x08,
  0x01,0x08,0xe3,0xe8,0x01,0xc3,0x01,0x00,0xf3,0xf3,0x76,0x01,0x76,0x76,0x01,0xf3,
  0xf3,0xf3,0x00,0x01,0x76,0x76,0x08,0x76,0x76,0xb2,0xb2,0x76,0x16,0xe3,0xf3,0x94,
  0xc3,0xc3,0xe8,0xf3,0x16,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x16,0x08,0x08,
  0x8e,0x8e,0x08,0x08,0xf3,0xb2,0xc3,0xc3,0xee,0xf4,0xe8,0xee,0xe8,0x00,0x01,0x00,
  0x16,0x08,0x01,0x00,0xe8,0xc3,0xc3,0xf8,0x01,0x01,0x01,0x01,0x01,0x01,0x00,0x00,
  0xf3,0xf3,0x00,0x00,0xcc,0xf3,0x8e,0xe3,0x94,0x94,0x08,0xf3,0x00,0x01,0xf3,0xf3,
  0xf3,0x00,0x00,0x8e,0xeb,0xbe,0xbe,0x16,0xf3,0x01,0x16,0x08,0x08,0x01,0xf3,0x01,
  0xdc,0x00,0xf3,0xf3,0x01,0xf3,0x00,0xf3,0xf3,0x01,0x01,0xf5,0xd8,0xd8,0xf5,0x8e,
  0x76,0x01,0x8e,0xbe,0x08,0x08,0x94,0xf3,0xf3,0x00,0x01,0x08,0x76,0x76,0x76,0x76,
  0x01,0x76,0x08,0x76,0x76,0x76,0x08,0x01,0x01,0x76,0x8e,0xbe,0xbe,0xbe,0xbe,0x8e,
  0x8e,0x01,0xf3,0xc3,0xe8,0x16,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x08,0x08,0x08,0x08,
  0x08,0x08,0xbe,0xcc,0xcc,0xbe,0xf4,0xee,0xee,0xee,0xb2,0xc3,0x00,0xc3,0xc3,0xb2,
  0xc3,0xf3,0x08,0x01,0xf3,0xf3,0x00,0xf8,0xe3,0xe3,0x01,0xe3,0x01,0xe3,0x76,0x01,
  0xe3,0x08,0x8e,0x08,0xf5,0xf3,0x08,0xf3,0x16,0x08,0x01,0x00,0x00,0x08,0xf8,0x00,
  0x01,0xf3,0xf3,0xf3,0xf3,0xf3,0x94,0xf3,0x00,0x08,0xb2,0x01,0xf3,0xf3,0xf3,0x94,
  0x94,0xdc,0xf3,0xf3,0x76,0x01,0x00,0xf3,0xf3,0xf3,0x01,0xfc,0xbe,0xf5,0xfc,0x94,
  0xbe,0xbe,0x01,0xf3,0x08,0x08,0xf9,0x01,0xf3,0xdc,0x94,0x01,0x01,0x76,0x76,0x08,
  0x76,0x08,0x76,0x76,0x16,0x08,0x01,0x01,0x76,0x76,0x08,0x08,0x08,0x8e,0x8e,0x8e,
  0x8e,0x8e,0x76,0x08,0xbe,0x08,0x08,0x16,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x16,0xf5,0xf5,0xfc,0x76,0x94,0xc3,0xf4,0xb2,0xee,0xee,0xee,0x01,0xf4,0xe8,0x08,
  0xe8,0xc3,0x00,0x01,0x01,0x01,0x01,0x01,0xcc,0x08,0x08,0x08,0xe3,0x08,0x08,0x08,
  0x8e,0x8e,0x8e,0x8e,0x8e,0xf5,0xd8,0xd8,0xd8,0xd8,0xeb,0xf5,0xf5,0x08,0x01,0xf3,
  0xe3,0xf3,0xf3,0xf3,0xf3,0x01,0x01,0xe3,0x76,0x08,0x8e,0x16,0x01,0x00,0x00,0xf3,
  0xf3,0x00,0xf3,0xf3,0x08,0xb2,0x76,0x01,0x01,0xf3,0x00,0xcc,0x8e,0x8e,0x8e,0x8e,
  0x08,0x00,0xbe,0x08,0x08,0x16,0x08,0xf3,0xf3,0xf3,0x94,0xf3,0x01,0xf8,0x8e,0x8e,
  0x08,0x08,0x08,0x16,0x08,0x01,0x01,0x76,0x08,0x08,0x16,0x08,0x08,0x16,0x08,0x8e,
  0x8e,0x8e,0x8e,0xbe,0x8e,0x08,0x76,0x08,0x76,0x08,0x16,0x8e,0x8e,0x8e,0x08,0x8e,
  0x8e,0xf5,0xd8,0xd8,0xfb,0xfb,0xee,0x94,0xdc,0x8e,0xbe,0xe3,0xf4,0xf4,0xc3,0xc3,
  0xc3,0x08,0xb2,0x94,0xf3,0xf3,0xf4,0x16,0xcc,0x8e,0x08,0x08,0x16,0x08,0x76,0x01,
  0x08,0x08,0x08,0x16,0x8e,0x8e,0x8e,0x08,0x08,0x16,0x08,0x08,0x16,0x76,0x76,0x01,
  0x01,0x00,0xf3,0xf3,0xf3,0x00,0x08,0x8e,0x08,0x08,0x8e,0x8e,0x01,0x01,0xe3,0x76,
  0x76,0x76,0x76,0x16,0x08,0x08,0xb2,0x76,0x76,0xf3,0xf3,0x8e,0xcc,0xf5,0xf5,0x01,
  0x08,0x8e,0xcc,0x76,0x01,0xe3,0x08,0xe3,0xf3,0xf3,0xf3,0xf8,0xf9,0xbe,0x16,0x08,
  0x08,0x16,0x16,0x08,0x08,0xf3,0x01,0x08,0x16,0x16,0x08,0x16,0x16,0x16,0x16,0x8e,
  0x08,0x16,0x08,0x16,0x16,0x16,0x16,0x08,0x08,0x08,0x16,0x8e,0x8e,0x8e,0x8e,0x8e,
  0x16,0x16,0xd8,0xbe,0xcc,0xd8,0xfb,0xee,0x8e,0xbe,0xb2,0x8e,0x8e,0xc3,0xf4,0xf2,
  0xfe,0xfc,0x08,0xb2,0xf3,0x00,0xf3,0xbe,0xcc,0xcc,0xe3,0x08,0x01,0x01,0x01,0xe3,
  0x08,0x01,0x01,0xe3,0x76,0x76,0x08,0xe3,0xe3,0x08,0x76,0x76,0x01,0xe3,0x76,0x08,
  0xe3,0x76,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0xe3,0x01,0x01,0x16,0x08,0x08,
  0x08,0x08,0x08,0x76,0xf3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x16,0xcc,0xeb,0xf0,
  0xeb,0xd8,0xfc,0xd8,0xfc,0xf5,0xcc,0xcc,0xf3,0xf3,0xf7,0xf6,0x01,0xf3,0x16,0x16,
  0x08,0x16,0x16,0x8e,0x76,0x08,0x8e,0x76,0x16,0x08,0x16,0x16,0x8e,0x16,0x16,0x16,
  0x08,0x08,0x76,0x76,0x08,0x08,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0x16,0x16,
  0x08,0x08,0xd8,0xcc,0x8e,0xbe,0xfa,0xd8,0xcc,0x8e,0x8e,0xb2,0x8e,0xfc,0xfb,0xfc,
  0xeb,0xff,0xfc,0x8e,0x8e,0xf3,0x16,0xcc,0xf5,0x8e,0xe3,0x01,0x01,0x01,0x01,0x01,
  0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xe3,0x01,0x01,0x01,0xe3,0x08,
  0x08,0x08,0x08,0x76,0x08,0x76,0xe3,0x76,0x76,0x01,0x08,0x08,0x08,0x16,0x8e,0x16,
  0x08,0x16,0x8e,0x8e,0x08,0x08,0x08,0x76,0x76,0x08,0x01,0x01,0xf3,0x76,0x08,0x16,
  0x08,0x8e,0xbe,0xbe,0xbe,0xcc,0xcc,0x16,0xf3,0x94,0x01,0xf3,0xf3,0xf3,0x01,0x08,
  0x08,0x16,0x16,0x76,0x08,0x08,0x01,0x01,0x76,0x76,0x08,0x08,0x08,0x08,0x16,0x08,
  0x16,0x08,0x76,0x76,0x08,0x08,0x08,0x08,0x08,0x76,0x08,0x16,0x16,0x16,0x08,0x16,
  0x08,0x08,0xcc,0x9b,0xfc,0x08,0xb2,0x8e,0xf5,0xcc,0xf5,0xd8,0xfc,0xfc,0xd8,0xd8,
  0xcc,0xd8,0xfe,0xfd,0xf0,0xf5,0xcc,0xf5,0xf5,0x08,0x01,0x01,0x00,0xf3,0xf3,0xf3,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x76,0xe3,0x01,0x01,0x01,0x08,
  0x08,0x08,0x08,0xe3,0x01,0x01,0x01,0x76,0x08,0x76,0x16,0x16,0x8e,0x8e,0xbe,0x8e,
  0x16,0x76,0x01,0x08,0x01,0xe3,0x76,0x76,0xb2,0x08,0xb2,0x76,0x01,0x01,0x01,0x01,
  0x00,0x00,0x76,0x08,0x08,0x16,0x08,0xf8,0xbe,0x08,0xf3,0x00,0x00,0x01,0x01,0x01,
  0x76,0x76,0x08,0x8e,0xbe,0x01,0x01,0x01,0x01,0x76,0xe3,0x08,0x16,0x16,0x16,0x16,
  0x08,0x08,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,0x76,
  0x76,0x08,0x8e,0xbe,0xcc,0xfb,0xcc,0xf5,0xcc,0x8e,0x8e,0x8e,0x8e,0x8e,0x8e,0xbe,
  0x08,0x08,0xf5,0xf0,0xf0,0xfa,0xd8,0xfc,0xbe,0x76,0x01,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xe3,
  0x08,0xe3,0xe3,0x01,0x00,0x00,0x01,0x01,0x01,0x01,0x76,0x76,0x08,0x08,0x08,0x76,
  0x01,0x76,0x76,0x08,0x08,0x76,0x76,0x76,0x76,0xe3,0x01,0x01,0x01,0x01,0xe3,0x01,
  0x01,0x01,0x01,0x01,0x76,0x8e,0xf9,0x8e,0xf3,0x00,0x01,0x01,0x01,0x00,0x00,0x00,
  0x00,0x01,0x8e,0xcc,0x00,0x01,0x01,0x01,0x01,0x01,0xe3,0xe3,0x08,0x08,0x76,0x76,
  0x76,0x76,0x01,0x01,0x01,0x00,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x00,0x01,0x01,0x01,0xe3,0xe3,0x01,0x01,0x01,0x01,0x01,0xe3,0x76,0x76,0x76,
  0x00,0x00,0x01,0x08,0x08,0x16,0x16,0x08,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x76,0xe3,0x76,0x01,
  0x01,0x01,0x01,0x01,0xe3,0xe3,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x00,0xf3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0xe3,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x00,0x00,
  0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  // 20: car texture
  0xaf,0x73,0x86,0x31,0x08,0x42,0x04,0x21,0x65,0x29,0x00,0x00,0x08,0x00,0x10,0x00,
  0x18,0x00,0x00,0x01,0x08,0x01,0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,
  0x18,0x02,0x00,0x03,0x08,0x03,0x10,0x03,0x18,0x03,0x53,0x9c,0x00,0x04,0x08,0x04,
  0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,
  0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,
  0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,
  0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,
  0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,
  0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,
  0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,0x10,0x41,0x18,0x41,0x00,0x42,0x10,0x42,
  0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,
  0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,
  0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,
  0x18,0x60,0x00,0x61,0x08,0x61,0x10,0x61,0x10,0x8c,0x18,0x61,0x00,0x62,0x08,0x62,
  0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,0x10,0x63,0x18,0x63,0x00,0x64,0x08,0x64,
  0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,0x18,0x65,0x00,0x66,0x08,0x66,
  0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x18,0x85,0x10,0x67,0x18,0x67,0x00,0x80,
  0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,0x18,0x81,0x00,0x82,
  0x08,0x82,0x10,0x82,0x18,0x82,0x00,0x83,0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,
  0x08,0x84,0x18,0x84,0x00,0x85,0x08,0x85,0x10,0x85,0x00,0x86,0x08,0x86,0x10,0x86,
  0x18,0x86,0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0xcf,0x7b,0x00,0xa0,0x08,0xa0,
  0x10,0xa0,0x18,0xa0,0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x00,0xa2,0x08,0xa2,
  0x10,0xa2,0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x8a,0x52,0x00,0xa4,
  0x4d,0x6b,0x08,0xa4,0x10,0xa4,0x18,0xa4,0x00,0xa5,0x08,0xa5,0x10,0xa5,0x18,0xa5,
  0x00,0xa6,0xe8,0x41,0xcb,0x5a,0x08,0xa6,0x10,0xa6,0x18,0xa6,0x00,0xa7,0x08,0xa7,
  0x10,0xa7,0x18,0xa7,0x00,0xc0,0x08,0xc0,0xc7,0x39,0x32,0x94,0x0c,0x63,0x10,0xc0,
  0x18,0xc0,0x00,0xc1,0xf0,0x83,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,0x08,0xc2,
  0x10,0xc2,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0xb0,0x7b,0x18,0xc3,0x6a,0x52,
  0x00,0xc4,0x08,0xc4,0x10,0xc4,0x18,0xc4,0x00,0xc5,0xed,0x52,0x08,0xc5,0x10,0xc5,
  0x18,0xc5,0xc3,0x18,0x00,0xc6,0x08,0xc6,0x6e,0x73,0x10,0xc6,0x18,0xc6,0xaf,0x83,
  0x00,0xc7,0x32,0x53,0x6a,0x4a,0x66,0x29,0x16,0x6c,0xd0,0x7b,0x28,0x5a,0x32,0x8c,
  0x68,0x6a,0xeb,0x62,0x31,0x94,0x52,0x8c,0xc7,0x41,0x16,0x74,0x13,0x7c,0x49,0x4a,
  0x04,0x01,0xb8,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0xe9,
  0xa5,0xec,0x6c,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,
  0xcd,0xcd,0xcd,0xcd,0x15,0xcd,0xcd,0xcd,0xcd,0x15,0x15,0x15,0xcd,0xcd,0xcd,0xcd,
  0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xfb,0xcd,0xfb,0xfb,0xd2,0xec,
  0x01,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x03,
  0xb8,0x00,0x00,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0x15,0x15,
  0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0xcd,
  0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xfb,0x6c,0xec,0xec,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0xff,0xa5,0xec,0xf7,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0x15,0xcd,0xcd,0xcd,0x00,
  0x00,0xfe,0xfe,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x00,0xfe,0x00,0xfe,0xf5,0x15,
  0x15,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xfb,0xf7,0xec,0xec,0xec,
  0x04,0x01,0xce,0xc2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x03,0x00,0x00,0xa5,0x15,0xcd,0xcd,0x15,0x15,0xcd,0x15,0xcd,0x15,0xcd,0xfb,0xfe,
  0xfe,0x84,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0xfd,0x84,0x84,0x84,0x00,0xfb,0x84,0xf5,
  0xfb,0x15,0x15,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xfb,0xec,0xec,0xec,0x00,
  0x04,0x01,0xce,0xc2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x04,0xc2,0xa5,0xec,0xf7,0x15,0xcd,0x15,0x15,0xcd,0x15,0x15,0x15,0xf5,0x84,0xfb,
  0xfe,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0xfe,0xfb,0x84,0x84,
  0xf5,0xfb,0x15,0xfb,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xfb,0x00,0x00,0xec,0x00,0xce,
  0x04,0x01,0xce,0xc2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0xa5,0x00,0xd2,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0xf5,0x84,0x84,0xfe,
  0xfe,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,0xfd,0xd2,0x84,0x84,
  0x84,0xfd,0xf5,0x15,0xcd,0xcd,0xcd,0xcd,0xcd,0x6c,0x00,0x00,0x00,0x00,0xce,0x03,
  0x04,0x01,0xce,0xb6,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0xb6,0xa5,0x00,0x6c,0x15,0x15,0x15,0x15,0x15,0xfb,0xfd,0x84,0x84,0xfe,
  0xfe,0x84,0x84,0x79,0x79,0x79,0x79,0x79,0x84,0x84,0x84,0x84,0xfd,0xd2,0x79,0x84,
  0x79,0x79,0xfd,0xf5,0x15,0xcd,0xcd,0x15,0xd2,0x00,0x00,0x00,0xa5,0xb6,0x04,0x04,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0x03,0xec,0x00,0xec,0xcd,0x15,0xcd,0x15,0x15,0xf5,0xfd,0xfd,0x99,0x00,
  0xfe,0x99,0xfd,0x99,0xfd,0xfd,0xf4,0xfd,0xfd,0xfd,0xfd,0xfd,0xf4,0x6c,0xf4,0xf4,
  0xf4,0xf4,0xf4,0xe5,0xf5,0x15,0xcd,0xa5,0x00,0x00,0xa5,0xce,0xcc,0x04,0x04,0x04,
  0x04,0x01,0xce,0xc2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0x04,0xcc,0xa5,0x00,0x00,0x15,0x15,0x15,0xd2,0xf4,0xfd,0xfd,0xfd,0x00,
  0x00,0xfd,0xfd,0xfd,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xd2,0xf4,0xf4,
  0xf4,0xf1,0xe5,0xfd,0x15,0xf7,0x00,0x00,0xa5,0xb8,0xb6,0xcc,0x04,0x04,0x04,0x04,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0x04,0x04,0xdf,0xa5,0x00,0xa5,0x15,0x15,0xfe,0xe5,0xe5,0xf1,0xf1,0x00,
  0x00,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xf4,0xfe,0xf4,0xd2,0xf1,0xf1,
  0xf1,0xfb,0x15,0xfb,0x00,0x00,0xa5,0xb8,0xb6,0xc2,0xb6,0xfc,0x04,0x04,0x04,0x01,
  0x04,0x01,0xce,0xb6,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x01,0x04,0x04,0x04,0x04,0xce,0x00,0x00,0x00,0xa5,0xcd,0x15,0x15,0xfb,0xfb,0x00,
  0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf1,0xf7,0x15,0x15,
  0x15,0xf7,0xec,0xa5,0xd2,0xb8,0xcc,0x03,0xcc,0xb6,0xc2,0xff,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xb6,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0x04,0x04,0xcc,0xb6,0x00,0xa5,0xec,0xec,0xb8,0xb8,0xa5,0xd2,0xf7,0x00,
  0xcd,0xcd,0x15,0x15,0x15,0x15,0xf7,0x15,0x15,0x15,0xcd,0xcd,0xf7,0xef,0xa5,0x00,
  0xec,0x00,0xf7,0x00,0xc1,0x03,0x04,0x04,0x04,0x01,0xff,0xb6,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xb6,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0x04,0x04,0xfc,0xc2,0xc2,0xb6,0xce,0xec,0xa5,0xa5,0xa5,0xa5,0xa5,0xec,
  0xd2,0xa5,0x00,0xa5,0xd2,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0xa5,0x00,0xa5,
  0xd2,0xa5,0xff,0xe9,0x03,0x04,0x04,0x01,0x01,0xff,0x02,0x01,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xb6,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0xf3,0x04,
  0x01,0x01,0x01,0x04,0x04,0xff,0xc2,0xdf,0x04,0x03,0x02,0xce,0xa5,0xf7,0xd2,0xec,
  0xa5,0xa5,0xa5,0xec,0xce,0xce,0xec,0xa5,0xa5,0xa5,0xd2,0xd2,0xa5,0xa5,0xec,0xa5,
  0xdf,0xe9,0x03,0x03,0x04,0x01,0x01,0x01,0x01,0x02,0xff,0xfc,0x04,0x01,0x01,0x01,
  0x04,0x01,0xce,0xc2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x04,0x01,0x01,0x01,0x04,0x01,0xb6,0xb6,0x01,0x04,0x04,0x03,0xe9,0xf3,0xc2,0xb8,
  0xc2,0xce,0xb8,0xb8,0xa5,0xec,0xb8,0xce,0xc2,0xc2,0xb6,0xb6,0xff,0xff,0xc1,0x03,
  0x04,0x03,0x04,0x04,0x01,0x01,0xcc,0xcc,0x01,0x02,0xff,0xfc,0x01,0x01,0x01,0x04,
  0x04,0x01,0xce,0xc2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0xf3,0x04,
  0x01,0x04,0x01,0x04,0x01,0xff,0x01,0xff,0x01,0x01,0x01,0x04,0x03,0x03,0x03,0xdf,
  0xc1,0xcc,0xfc,0xc1,0x02,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x02,0xc1,0xcc,0x03,
  0x04,0x03,0x04,0x01,0xcc,0x01,0x01,0x01,0x01,0x02,0xff,0x02,0x04,0x04,0x04,0x04,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x01,0x04,0x04,0x04,0x02,0xff,0xcc,0x04,0x01,0x01,0x01,0x01,0x04,0x03,0x03,0xff,
  0xc1,0xc1,0xcc,0x02,0x02,0x02,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfc,0xff,0x03,
  0x04,0x04,0x04,0x01,0x01,0xcc,0xcc,0x01,0x01,0xcc,0x03,0xe9,0xe9,0x04,0x04,0x01,
  0x04,0x01,0xb8,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x01,0x04,0x04,0x04,0xff,0x02,0xfc,0x04,0x01,0x01,0x01,0x01,0x04,0x04,0x03,0xb6,
  0xff,0x02,0xc1,0x02,0xff,0xff,0xdf,0xdf,0xdf,0xdf,0xdf,0xf2,0xdf,0xcc,0xc2,0x04,
  0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x03,0x04,0x04,0x04,0x01,
  0x01,0x01,0xb8,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x04,0x04,0x04,0x02,0xf6,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x03,0xb6,
  0xc2,0xff,0xc1,0xff,0xdf,0xdf,0xc2,0xdf,0xc2,0xb6,0xb6,0xdf,0xdf,0xc1,0xb6,0x04,
  0x01,0x01,0x04,0x01,0x01,0x01,0x01,0xcc,0xcc,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0xb8,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x01,0x01,0x01,0x01,0x03,0xe9,0xe9,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x03,0xdf,
  0xce,0xff,0xcc,0xff,0xb6,0xb6,0xdf,0xc2,0xdf,0xb6,0xb6,0xb6,0xf2,0xc1,0xc2,0x04,
  0x04,0x04,0x04,0x01,0x01,0xcc,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x01,0x04,0x04,
  0x01,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x03,0xff,
  0xce,0xff,0xc1,0xff,0xdf,0xdf,0xb6,0xb6,0xb6,0xb6,0xdf,0xdf,0xff,0xff,0xc2,0x04,
  0x04,0x04,0x04,0x01,0x01,0xcc,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xcc,0x01,0x01,0x04,0x02,
  0xc2,0xff,0xc1,0xff,0xb6,0xb6,0xc2,0xc2,0xb6,0xb6,0xdf,0xdf,0xc1,0xdf,0xc2,0x01,
  0x01,0x01,0x01,0x01,0x04,0x01,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,
  0x04,0x01,0xb8,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x04,0x04,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0xcc,
  0xc2,0xff,0x02,0xff,0xff,0xdf,0xf2,0xff,0xff,0xff,0xff,0xff,0xcc,0xdf,0xc2,0x04,
  0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x01,0x01,0x01,0x01,
  0x01,0x01,0xb8,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x01,
  0xc2,0xdf,0x02,0xc1,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x02,0x02,0xc2,0xc2,0x04,
  0x04,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x04,0x04,
  0xb6,0xf2,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x02,0xff,0xff,0xc2,0xc2,0x01,
  0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x04,
  0xcc,0xcc,0xc1,0xcc,0xcc,0xc1,0xc1,0xc1,0xc1,0x02,0x02,0xc1,0x02,0xff,0xff,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0xf3,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x04,0x04,0x04,0x01,0x01,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x01,0x04,0x04,0x04,0x04,0x01,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x04,0x04,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x01,0x01,0x01,0x01,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x03,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x01,0x01,0x04,0x01,0x01,0x01,0x04,0x01,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,
  0x04,0x01,0xce,0xc2,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xce,0xc2,0x01,0x04,
  0x04,0x01,0x01,0x01,0x04,0x04,0x01,0x01,0x01,0x04,0x01,0x01,0x01,0x01,0x01,0x01,
  0x04,0x03,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x01,0x01,0x01,0x04,0x04,0x01,0x04,0x04,0x01,0x04,0x04,0x04,0x01,0x04,
  0x04,0x01,0xc2,0xf2,0x01,0x03,0x04,0x04,0x04,0x04,0x04,0xf3,0xc2,0xb6,0x04,0x04,
  0x01,0x01,0x03,0x03,0x03,0x04,0x01,0x01,0x01,0x01,0x04,0x01,0x01,0x01,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x04,0x04,0x04,0x01,0x04,0x04,0x04,0x01,0x01,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x01,0x03,0x04,0x04,0x04,0x03,0x03,0x01,0x01,0x01,0x01,0x04,0x04,0x04,0x04,0x01,
  0x01,0x01,0x01,0x01,0xcc,0x04,0x01,0x01,0x04,0x04,0x04,0x04,0x01,0xcc,0x01,0x01,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x01,0x04,0x04,0x01,0x01,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x01,0x03,0x04,0x01,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x03,0x04,0x01,
  0x01,0x01,0x01,0x01,0x04,0x03,0x03,0x04,0x04,0x04,0x03,0x03,0x01,0x01,0x04,0x04,
  0x01,0x01,0x01,0x03,0x03,0x04,0x01,0x01,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x03,
  0x03,0x03,0x03,0x03,0x03,0x03,0x02,0xff,0xff,0x02,0x03,0x03,0x03,0x03,0x03,0x03,
  0x04,0x03,0x04,0x01,0x03,0x03,0x04,0x04,0x03,0x04,0x01,0xcc,0x01,0x04,0x04,0x04,
  0x01,0x01,0x01,0x03,0x03,0x03,0x03,0x03,0xe9,0x03,0x03,0x03,0x03,0x04,0x04,0x01,
  0x04,0x01,0x04,0x04,0x04,0x01,0x01,0x01,0x04,0x04,0x04,0x01,0x04,0x03,0x03,0x04,
  0x03,0x03,0x03,0xe9,0xcc,0xc2,0xc2,0xc2,0xce,0xc2,0xce,0xcc,0x03,0x03,0x03,0x03,
  0x04,0x03,0x04,0x04,0x03,0x03,0x03,0x01,0x04,0x03,0x01,0xcc,0x01,0x01,0x04,0x04,
  0x01,0x04,0x03,0x01,0xcc,0x01,0x04,0x04,0x03,0x03,0x04,0x04,0x01,0x04,0x03,0x04,
  0x01,0x01,0x04,0x04,0x01,0xcc,0xcc,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x04,0x01,
  0x03,0x03,0x03,0x01,0xc2,0xc2,0xce,0xb8,0xb8,0xce,0xce,0xce,0x01,0x03,0x03,0x03,
  0x01,0x03,0x04,0x04,0x03,0x03,0x03,0x04,0x03,0x03,0x03,0x03,0xe9,0xe9,0xe9,0x03,
  0x03,0x03,0x03,0x01,0xcc,0x01,0x04,0x01,0x03,0x03,0x04,0x01,0xcc,0xcc,0x03,0xe9,
  0xe9,0xe9,0xe9,0xe9,0x03,0x03,0x03,0xe9,0xe9,0x03,0xe9,0xe9,0x03,0x04,0x01,0x01,
  0x03,0x03,0xe9,0xc2,0xce,0xc2,0x00,0x00,0xa5,0x00,0xce,0xb8,0xf9,0xe9,0x03,0x03,
  0x04,0x03,0x04,0x04,0x03,0x03,0x03,0x01,0x04,0x04,0x04,0x01,0x03,0x04,0x03,0x03,
  0x04,0x03,0x03,0x01,0xcc,0x01,0x03,0x04,0x03,0x03,0x04,0x01,0x01,0x01,0x03,0x03,
  0x03,0x03,0x03,0x03,0x04,0x01,0x04,0x03,0xe9,0x03,0xe9,0xe9,0x03,0x04,0x04,0x01,
  0x03,0x03,0x01,0xc2,0xce,0x00,0xcd,0xbf,0xbe,0xcd,0x00,0xce,0xce,0x01,0x03,0x03,
  0x03,0x03,0x04,0x04,0x03,0x03,0x03,0x01,0x03,0x01,0xcc,0xcc,0x01,0x01,0x04,0x04,
  0x04,0x01,0x04,0x04,0xcc,0x04,0x04,0x03,0x03,0x03,0x04,0x01,0x01,0x03,0x03,0x01,
  0x04,0x01,0x01,0x04,0x04,0xcc,0xcc,0x01,0x01,0x04,0x03,0x04,0x04,0x03,0x04,0x01,
  0x03,0x03,0x02,0xce,0xce,0xd2,0xbe,0xd2,0x00,0x6c,0xa5,0xb8,0xce,0x02,0x03,0x03,
  0x01,0x03,0x04,0x01,0x04,0x03,0x04,0x03,0x04,0x01,0x01,0x01,0x04,0x01,0x04,0x01,
  0x01,0x04,0x04,0x03,0x03,0x03,0xe9,0x03,0xe9,0x03,0xe9,0x03,0x03,0x01,0x04,0x01,
  0x01,0x01,0x01,0x04,0x04,0x01,0x01,0x01,0x04,0x03,0x04,0x04,0x04,0x03,0x03,0x03,
  0x03,0x03,0x02,0xce,0xce,0xd2,0x15,0x00,0xce,0xec,0xa5,0xb8,0xce,0xcc,0x03,0x03,
  0x01,0x03,0x04,0x01,0x04,0x04,0x03,0x04,0x01,0x01,0x01,0x04,0x04,0x01,0x04,0x01,
  0x04,0x01,0x01,0x01,0xcc,0x04,0x01,0x01,0x01,0x01,0x01,0x04,0x01,0xcc,0x04,0x01,
  0x01,0x01,0x04,0x04,0x04,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x04,
  0x03,0x03,0x04,0xce,0xce,0xd2,0xd2,0xcd,0xb8,0x00,0xa5,0xce,0xce,0x04,0x03,0x03,
  0x01,0x04,0x03,0x03,0x03,0x03,0x04,0x01,0xcc,0x01,0x01,0x01,0x04,0x03,0x04,0x04,
  0x01,0x04,0x04,0x01,0xcc,0x04,0x04,0x04,0x01,0x01,0x04,0x04,0x01,0x01,0x04,0x04,
  0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x04,0x04,0x04,0x04,0x01,0x04,0x01,0x01,
  0x03,0x03,0xe9,0xc2,0xb8,0xce,0xa5,0xa5,0xa5,0x6c,0xce,0xb8,0xb6,0xe9,0x03,0x03,
  0x01,0x01,0x01,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x01,0x04,0x04,
  0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,
  0x03,0x03,0x03,0x04,0xce,0xec,0xce,0x00,0x00,0xce,0xb8,0xce,0x04,0x03,0x03,0x03,
  0x01,0x01,0x01,0x01,0x04,0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x04,
  0x04,0x03,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x04,0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0x01,0x01,0x01,0x04,0xf3,
  0x03,0x03,0x03,0x03,0x01,0xce,0xb8,0xb8,0xce,0xce,0xc2,0x01,0xe9,0x03,0x03,0x03,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x04,
  0x04,0x04,0x03,0x04,0x04,0x03,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x03,0x03,0x03,0x03,0x03,0xe9,0xc1,0xff,0xff,0xcc,0xe9,0x03,0x03,0x03,0x03,0x03,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
  0x04,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x04,0x04,0x04,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
  0x01,0x01,0x01,0x04,0x04,0x04,0x01,0x01,0x04,0x04,0x04,0x03,0x04,0x04,0x04,0x04,
  0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x03,0x03,0x04,0x04,0x04,0x04,
  0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
  0x01,0x04,0x04,0x04,0x04,0x04,0x04,0x01,0xcc,0xcc,0xc1,0x02,0xc1,0x02,0xdf,0xff,
  0xff,0xdf,0xdf,0xb6,0xb6,0xb6,0xb6,0xb6,0xc2,0xce,0xce,0xce,0xec,0xec,0xec,0xb8,
  0xce,0xce,0xff,0xff,0xb6,0xb8,0xec,0x00,0xec,0xec,0x00,0xa5,0xf3,0xcc,0x02,0x02,
  0xc1,0x02,0x02,0x02,0xfc,0xc2,0xef,0x6c,0xd2,0xd2,0xd2,0xa5,0xce,0xce,0xec,0xf7,
  0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0xd2,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
  0x15,0x15,0x15,0xcd,0x15,0x15,0x15,0xf5,0xfb,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,
  0x00,0xce,0xb6,0xc2,0x00,0x00,0xa5,0xa5,0xec,0xec,0xec,0x00,0xf3,0xcc,0xc1,0xc1,
  0x02,0x02,0x02,0x02,0xfc,0xf9,0x6f,0xf8,0xd2,0x00,0x00,0xa5,0xec,0xb6,0xdf,0xdd,
  0xcd,0xcd,0xcd,0xfb,0xcd,0xcd,0xfa,0xd2,0xec,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,
  0xcd,0xfb,0xcd,0xfb,0xfb,0xfb,0xcd,0xdd,0xf5,0xd2,0xd2,0x6c,0xf7,0xcd,0xcd,0xcd,
  0xc2,0xb6,0xce,0x00,0x00,0x00,0x00,0xa5,0xec,0xc1,0xdf,0xa5,0xf3,0xcc,0xc1,0xc1,
  0x02,0xcc,0xcc,0xcc,0xcc,0xc2,0xf8,0xf8,0xd2,0xa5,0x00,0xa5,0xa5,0xb8,0xdf,0xd2,
  0xfe,0x74,0xf1,0xf4,0xfd,0xfd,0xfd,0x78,0xf5,0xcd,0x15,0xcd,0xcd,0x00,0xf5,0xcd,
  0xcd,0xd2,0xec,0x6c,0xfb,0xcd,0x15,0xdd,0x84,0x84,0x79,0xf4,0xf4,0xf1,0xfe,0xcd,
  0x02,0xce,0x00,0x00,0xec,0x00,0xa5,0xa5,0xce,0x6e,0xf8,0xec,0xf3,0xcc,0xc1,0xc1,
  0x02,0xcc,0x00,0x00,0xdf,0xc2,0x6f,0xf8,0xec,0xb8,0xb8,0xb8,0xb8,0xec,0xce,0xcd,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xdd,0xfe,0xd2,0xd2,0x6c,0xa5,0xec,0xec,0xa5,
  0xa5,0xec,0xec,0xec,0xd2,0xd2,0xdd,0xdd,0x84,0x84,0x84,0xfd,0xf4,0xfe,0xf1,0xcd,
  0xc2,0xec,0xec,0xa5,0xa5,0xa5,0xa5,0xa5,0xec,0x69,0xf8,0xec,0xc2,0xc2,0xc1,0xc1,
  0xff,0x02,0xf5,0xd2,0xb6,0xb6,0xb8,0xb8,0xb8,0xa5,0xa5,0xec,0xec,0xb8,0xff,0xcd,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xdd,0xfe,0xd2,0xd2,0xa5,0xa5,0xec,0xa5,0xec,
  0x00,0xa5,0xec,0xa5,0xef,0xd2,0xdd,0xdd,0xfd,0x84,0x84,0xfd,0xf4,0xfe,0xe5,0xd2,
  0xff,0xb8,0x00,0xd2,0xa5,0xa5,0xa5,0xa5,0xc2,0xc2,0xc2,0xec,0xb8,0xf5,0xc1,0xc1,
  0xcc,0x03,0xf5,0xf7,0xb6,0xb6,0xa5,0xd2,0xd2,0xa5,0xec,0xec,0xec,0xb8,0xb6,0xfa,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xdd,0xfe,0xf7,0x6c,0xef,0xa5,0xec,0xa5,0x00,
  0x00,0xa5,0xec,0xa5,0xef,0x6c,0x6c,0xdd,0xfd,0x84,0x84,0xfd,0xf4,0xfe,0xe5,0xd2,
  0xc2,0xec,0xec,0xd2,0xd2,0xa5,0xa5,0xa5,0xc2,0xec,0xb8,0xec,0xb8,0xf5,0xc1,0xc1,
  0x02,0x01,0xec,0xb8,0xb6,0xb8,0xa5,0xd2,0xd2,0xa5,0xec,0xec,0xec,0xec,0xf2,0xfa,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xdd,0xfe,0xa5,0x00,0xef,0xec,0xec,0xec,0x00,
  0x00,0xec,0xec,0xec,0xd2,0xa5,0xa5,0xdd,0xfd,0x84,0x84,0xfd,0xf4,0xfe,0xe5,0xd2,
  0xc2,0xec,0x00,0xef,0xd2,0xd2,0xa5,0xa5,0xce,0xec,0xb8,0xec,0xc2,0x00,0xc1,0xc1,
  0x02,0x01,0x00,0x00,0xb6,0xa5,0xa5,0xd2,0xd2,0xec,0xec,0xec,0xec,0xec,0xff,0x15,
  0xe5,0xf1,0xfe,0xfe,0xfd,0x84,0x84,0xf5,0xd2,0x6c,0x6c,0xef,0xec,0xec,0xec,0x00,
  0x00,0xec,0xec,0xec,0xd2,0x6c,0x6c,0xdd,0xfd,0x84,0x84,0xfd,0xf4,0xfe,0xe5,0xd2,
  0xb6,0xb8,0xec,0xef,0xd2,0xa5,0xa5,0xa5,0xce,0xec,0xb8,0xec,0xc2,0x00,0xc1,0xc1,
  0x02,0x01,0xec,0x00,0xb6,0xb8,0xa5,0xd2,0xd2,0xec,0xec,0xec,0xec,0xec,0xf2,0x15,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xf5,0xd2,0xdd,0xdd,0xef,0xec,0xec,0xec,0x00,
  0x00,0xec,0xec,0xec,0xd2,0xdd,0xdd,0xdd,0xfd,0x84,0x84,0xfd,0xf4,0xf4,0xe5,0x6c,
  0xf2,0xec,0x00,0xef,0xa5,0xd2,0xa5,0xa5,0xce,0xb8,0xb8,0xec,0xc2,0xec,0xc1,0xc1,
  0xff,0x02,0x00,0xa5,0xb6,0xb6,0xa5,0xd2,0xd2,0xec,0xec,0xec,0xec,0xec,0xdf,0xcd,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xf5,0xd2,0xef,0x00,0xef,0xec,0xec,0xec,0x00,
  0x00,0xec,0xec,0xec,0xd2,0xdd,0xec,0xdd,0x79,0x84,0x84,0xfd,0xf4,0xf4,0xe5,0xd2,
  0xdf,0xec,0x00,0xa5,0xa5,0xd2,0xa5,0xa5,0xce,0xec,0xb8,0xec,0xb8,0x00,0xc1,0xc1,
  0xc1,0xe9,0x00,0xd2,0xb6,0xdf,0xa5,0xd2,0xd2,0xa5,0xec,0xec,0xec,0xb8,0xff,0xfa,
  0xe5,0xf1,0xfe,0xf4,0xfd,0x84,0x84,0xf5,0xfe,0xf7,0x6c,0xef,0xf5,0xec,0xf5,0x00,
  0x00,0xf5,0xec,0xf5,0xd2,0x6c,0x6c,0xdd,0x79,0x84,0x84,0xfd,0xf4,0xf4,0xe5,0xd2,
  0x02,0xb8,0xec,0xa5,0xa5,0xa5,0xa5,0xa5,0xc2,0xc2,0xc2,0xec,0xb8,0xf5,0xc1,0xc1,
  0xcc,0xf3,0x00,0xd2,0xb6,0xf9,0xb8,0xb8,0xb8,0xa5,0xa5,0xec,0xec,0xb8,0xc2,0xcd,
  0xe5,0xf1,0xf4,0xf4,0xfd,0x84,0x84,0xf5,0xfe,0xec,0x00,0xef,0xec,0xec,0xec,0x00,
  0x00,0xec,0x00,0x00,0xd2,0xec,0xb8,0xdd,0x84,0x84,0x84,0xfd,0xf4,0xf4,0xe5,0xd2,
  0xce,0xec,0xec,0xa5,0xa5,0xa5,0xa5,0xa5,0xec,0xf9,0xf9,0xec,0xb8,0xf5,0xc1,0xc1,
  0x02,0xcc,0x00,0x00,0xc1,0xc2,0x6f,0xf8,0xec,0xb8,0xb8,0xb8,0xb8,0xb8,0xb6,0xcd,
  0xe5,0xf1,0xf4,0xfd,0xfd,0x84,0x84,0xf5,0xfe,0x6c,0xf5,0x6c,0xd2,0xb8,0xb8,0xef,
  0xef,0xec,0xec,0xec,0x6c,0x6c,0xf5,0xdd,0x84,0x84,0x84,0xfd,0xfd,0xf4,0xf1,0xec,
  0xff,0xb8,0xec,0xa5,0xa5,0xa5,0xa5,0xa5,0xb8,0x6e,0xf8,0xec,0x02,0x02,0xc1,0xc1,
  0x02,0xcc,0xcc,0xcc,0xcc,0xc2,0xf8,0xf8,0xd2,0xec,0xec,0x00,0xec,0xce,0xdf,0xd2,
  0xfe,0xf1,0xf1,0xf4,0xfd,0xfd,0x84,0x78,0xf5,0x15,0xcd,0xcd,0xcd,0x6c,0xcd,0xcd,
  0xcd,0x6c,0x6c,0xcd,0xcd,0x15,0xcd,0xdd,0x84,0x84,0x79,0xf4,0xf4,0xf1,0xfe,0xcd,
  0xff,0xdf,0xb8,0xec,0xec,0x00,0xa5,0xa5,0xce,0x01,0xf6,0xec,0xf3,0xcc,0xc1,0xc1,
  0x02,0x02,0x02,0x02,0xfc,0xf9,0x6f,0xf8,0xd2,0x00,0x00,0xa5,0xec,0xdf,0xc2,0x6c,
  0xcd,0xcd,0xcd,0xfb,0xcd,0xcd,0xfa,0xd2,0xec,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,
  0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xa5,0xf5,0xd2,0xd2,0x6c,0xf7,0xcd,0xcd,0xcd,
  0xec,0xdf,0xdf,0xb8,0x00,0x00,0xa5,0xa5,0xec,0xf9,0xb8,0x00,0x01,0xcc,0xc1,0xc1,
  0x02,0x02,0x02,0xc1,0xcc,0xb6,0x00,0x00,0xd2,0x00,0x00,0x00,0xc2,0xc2,0x00,0x6c,
  0x15,0x15,0x15,0x15,0x15,0x15,0x15,0x15,0xf5,0x15,0x15,0x15,0x15,0x15,0x15,0x15,
  0x15,0x15,0x15,0x15,0x15,0x15,0x15,0xa5,0xf7,0xcd,0xcd,0xcd,0xcd,0xcd,0xcd,0xfb,
  0x00,0xec,0xb6,0xb6,0xce,0x00,0xa5,0xa5,0x00,0x00,0xec,0xa5,0x01,0xcc,0xc1,0xc1,
  // 21: logo
  0xff,0xff,0x28,0x42,0x00,0x00,0x08,0x00,0x10,0x00,0x18,0x00,0x00,0x01,0x08,0x01,
  0x10,0x01,0x18,0x01,0x00,0x02,0x08,0x02,0x10,0x02,0x18,0x02,0x00,0x03,0x08,0x03,
  0x10,0x03,0x18,0x03,0x00,0x04,0x08,0x04,0x10,0x04,0x18,0x04,0x00,0x05,0x08,0x05,
  0x10,0x05,0x18,0x05,0x00,0x06,0x08,0x06,0x10,0x06,0x18,0x06,0x00,0x07,0x08,0x07,
  0x10,0x07,0x18,0x07,0x00,0x20,0x08,0x20,0x10,0x20,0x18,0x20,0x00,0x21,0x08,0x21,
  0x10,0x21,0x18,0x21,0x00,0x22,0x08,0x22,0x10,0x22,0x18,0x22,0x00,0x23,0x08,0x23,
  0x10,0x23,0x18,0x23,0x00,0x24,0x08,0x24,0x10,0x24,0x18,0x24,0x00,0x25,0x08,0x25,
  0x10,0x25,0x18,0x25,0x00,0x26,0x08,0x26,0x10,0x26,0x18,0x26,0x00,0x27,0x08,0x27,
  0x10,0x27,0x18,0x27,0x00,0x40,0x08,0x40,0x10,0x40,0x18,0x40,0x00,0x41,0x08,0x41,
  0x10,0x41,0x18,0x41,0x00,0x42,0x10,0x42,0x18,0x42,0x00,0x43,0x08,0x43,0x10,0x43,
  0x18,0x43,0x00,0x44,0x08,0x44,0x10,0x44,0x18,0x44,0x00,0x45,0x08,0x45,0x10,0x45,
  0x18,0x45,0x00,0x46,0x08,0x46,0x10,0x46,0x18,0x46,0x00,0x47,0x08,0x47,0x10,0x47,
  0x18,0x47,0x00,0x60,0x08,0x60,0x10,0x60,0x18,0x60,0x00,0x61,0x08,0x61,0x11,0x63,
  0x10,0x61,0x18,0x61,0x00,0x62,0x08,0x62,0x10,0x62,0x18,0x62,0x00,0x63,0x08,0x63,
  0x18,0x63,0x00,0x64,0x08,0x64,0x10,0x64,0x18,0x64,0x00,0x65,0x08,0x65,0x10,0x65,
  0x18,0x65,0x00,0x66,0x08,0x66,0x10,0x66,0x18,0x66,0x00,0x67,0x08,0x67,0x10,0x67,
  0x18,0x67,0x00,0x80,0x08,0x80,0x10,0x80,0x18,0x80,0x00,0x81,0x08,0x81,0x10,0x81,
  0x18,0x81,0x00,0x82,0x10,0x84,0x08,0x82,0xf7,0xbd,0x10,0x82,0x18,0x82,0x00,0x83,
  0x08,0x83,0x10,0x83,0x18,0x83,0x00,0x84,0x08,0x84,0x18,0x84,0x00,0x85,0x8a,0x52,
  0x38,0xc6,0x38,0x8d,0x08,0x85,0x10,0x85,0x00,0x86,0x08,0x86,0x10,0x86,0x18,0x86,
  0x00,0x87,0x08,0x87,0x10,0x87,0x18,0x87,0x00,0xa0,0x08,0xa0,0x10,0xa0,0x18,0xa0,
  0x00,0xa1,0x08,0xa1,0x10,0xa1,0x18,0xa1,0x86,0x31,0x00,0xa2,0x08,0xa2,0x10,0xa2,
  0x18,0xa2,0x00,0xa3,0x08,0xa3,0x10,0xa3,0x18,0xa3,0x00,0xa4,0x08,0xa4,0x10,0xa4,
  0x18,0xa4,0x00,0xa5,0x08,0xa5,0x45,0x31,0x10,0xa5,0x18,0xa5,0x69,0x4a,0x00,0xa6,
  0x08,0xa6,0x19,0xae,0x10,0xa6,0x00,0xa7,0x08,0xa7,0x10,0xa7,0x18,0xa7,0x00,0xc0,
  0x08,0xc0,0x10,0xc0,0x18,0xc0,0x00,0xc1,0x08,0xc1,0x10,0xc1,0x18,0xc1,0x00,0xc2,
  0x08,0xc2,0x10,0xc2,0x96,0xb5,0x18,0xc2,0x00,0xc3,0x08,0xc3,0x10,0xc3,0x18,0xc3,
  0x00,0xc4,0x08,0xc4,0x44,0x29,0x10,0xc4,0x18,0xc4,0x00,0xc5,0x08,0xc5,0x10,0xc5,
  0x12,0x63,0x18,0xc5,0xc7,0x39,0x00,0xc6,0x0c,0x63,0x08,0xc6,0x10,0xc6,0x00,0xc7,
  0x4d,0x4a,0x79,0xce,0x0c,0x5b,0x75,0xad,0x08,0xc7,0x4d,0x6b,0xce,0x52,0x6a,0x4a,
  0x95,0xad,0xcb,0x5a,0x36,0x7c,0x14,0x7c,0xf8,0xbd,0x6e,0x73,0xae,0x73,0x18,0xbe,
  0xe7,0x41,0xb2,0x94,0x17,0xbe,0x6d,0x73,0xec,0x5a,0xeb,0x39,0x69,0x52,0xb6,0xc5,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x8c,0x8c,0xd2,0x8c,0xd2,
  0x98,0xf9,0xea,0x97,0xea,0xd2,0xd2,0x98,0xd2,0x8c,0xd2,0x98,0xf7,0xd2,0x98,0x8c,
  0xeb,0x8c,0x8c,0xeb,0x8c,0xeb,0x8c,0xeb,0x8c,0xeb,0x98,0xeb,0x98,0xd2,0x8c,0xd2,
  0x98,0x8c,0x8c,0xd2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x98,0x8c,0xd2,0xd2,0x8c,0xd2,0x8c,
  0xd2,0xea,0x67,0x99,0x67,0xea,0x8c,0x8c,0xeb,0x98,0x8c,0xd2,0x98,0x98,0x8c,0xeb,
  0x8c,0x98,0xd2,0x98,0xeb,0x8c,0xd2,0x8c,0x98,0x8c,0xeb,0x8c,0xd2,0xeb,0x98,0x8c,
  0x8c,0x98,0x8c,0x8c,0x8c,0xe9,0xd2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0xf3,0xf3,0xf3,0x00,0x00,0x00,0x8c,0x98,0x98,0xf9,0xe4,0x97,0xef,0x97,0x97,
  0x01,0xe8,0x99,0x8c,0x99,0xe8,0x97,0x97,0x97,0xbe,0x97,0x97,0x97,0x97,0x97,0x97,
  0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0x97,0xef,0x97,
  0x97,0x97,0x01,0xe4,0xf9,0x98,0x8c,0xf7,0xd2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0xf3,0xf2,0xf3,0x00,0x00,0x8c,0x8c,0xf9,0xe4,0xbe,0xf6,0xf6,0xf9,0xf9,0xf9,
  0x8a,0xea,0xf9,0xeb,0xf3,0xea,0xf9,0xf9,0x8a,0xf9,0x8a,0x8a,0xf9,0xf9,0xf9,0xf9,
  0xf9,0x8a,0x8a,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0xf9,0x8a,0xf9,0xf9,
  0x8a,0xf9,0x8a,0x8a,0xe4,0xf1,0xf9,0x98,0xf7,0xe9,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0xf3,0xf2,0xf3,0x00,0x98,0x8c,0xf9,0x97,0xf6,0x8a,0xfe,0xda,0xbb,0xbb,0xbb,
  0xbb,0x01,0x67,0xf3,0x67,0xac,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0x01,
  0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,
  0xbb,0xbb,0xbb,0xfe,0x8a,0x8a,0x97,0xf6,0x8c,0x8c,0xf7,0x00,0x00,0x00,0x00,0x00,
  0x00,0xf2,0xf2,0xf2,0xeb,0x8c,0xf9,0x97,0x8a,0xe4,0xbb,0xbb,0xac,0xac,0xac,0xac,
  0xac,0xac,0x01,0xef,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xbb,0xf8,
  0xac,0xbb,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xac,0xbb,0xbb,0xbb,
  0xbb,0xbb,0xbb,0xbb,0xda,0xfe,0x8a,0xf1,0xed,0x98,0x98,0xff,0x00,0x00,0x00,0x00,
  0x00,0xf2,0x99,0xf2,0xeb,0xf9,0xf1,0x8a,0xbe,0xbb,0xac,0xac,0xf8,0x01,0x01,0x01,
  0xe2,0xac,0xe2,0x01,0xe2,0xe2,0x01,0x01,0xf8,0x01,0x01,0xe2,0xac,0x01,0xfc,0xea,
  0xef,0xe2,0xe2,0xf8,0xf8,0xf8,0x01,0x01,0x01,0x01,0xf8,0xac,0x01,0xfc,0xe8,0xee,
  0xe8,0xee,0xee,0xe8,0xef,0xbb,0xda,0xf3,0xf3,0xf6,0x8c,0x98,0xff,0x00,0x00,0x00,
  0x00,0xf2,0x99,0x99,0xeb,0xed,0x8a,0xfe,0xbb,0xac,0xe2,0x01,0xbe,0xef,0x97,0x97,
  0xbe,0x01,0x01,0x01,0xef,0xbe,0xef,0xbe,0xbe,0x97,0xbe,0x01,0x73,0x99,0xf2,0x99,
  0x99,0x99,0x73,0x01,0xef,0xbe,0xef,0x97,0x97,0xbe,0x01,0xf3,0x99,0x99,0x99,0x99,
  0x99,0x99,0x99,0x99,0x99,0xac,0xda,0xf2,0x99,0xf6,0xf6,0xf9,0x99,0x99,0x99,0x00,
  0x00,0x99,0x99,0x99,0xf9,0xf1,0xf6,0xbb,0xac,0xf8,0xef,0x97,0x97,0xea,0xea,0x97,
  0xf3,0xbd,0x99,0x99,0x99,0x99,0x97,0xea,0xfc,0x97,0xef,0x99,0x99,0x99,0x99,0x99,
  0x99,0x99,0x99,0x99,0xef,0x97,0xea,0xf1,0xf1,0xef,0x99,0x99,0x99,0x99,0x99,0x99,
  0x99,0x99,0x99,0xbd,0x99,0xe2,0xbb,0x99,0x99,0xf9,0x99,0x99,0x99,0x99,0x99,0x00,
  0x00,0xeb,0xc1,0x99,0xf1,0x8a,0xda,0xac,0xf8,0xef,0x97,0xf1,0xea,0xea,0xe4,0xea,
  0xf3,0xc1,0xbd,0xbd,0xbd,0xbd,0xf1,0xe4,0xea,0x97,0xeb,0xbd,0xc1,0xc1,0x99,0xbd,
  0xbd,0xc1,0xc1,0xbd,0xf2,0x97,0xea,0xea,0x97,0xbd,0xc1,0xbd,0xc1,0xc1,0xc1,0xc1,
  0xc1,0xc1,0xc1,0xc1,0x99,0xbe,0xe2,0x99,0xc1,0xc1,0xc1,0xc1,0xbd,0x99,0x99,0x00,
  0x00,0xd2,0xc1,0x99,0xbe,0xf6,0xbb,0xac,0xbe,0x97,0xea,0xea,0xea,0xfc,0xfc,0xf1,
  0xf3,0xbd,0xc1,0xc1,0xc1,0xc1,0xf1,0xea,0x97,0x73,0xf7,0xc1,0xf9,0xea,0x01,0x97,
  0x01,0xf5,0xf7,0xc1,0xc1,0xf2,0xfc,0x97,0xfc,0xc1,0xf7,0xc1,0x99,0xee,0xef,0xef,
  0xe8,0xe8,0xc1,0xc1,0xd2,0xef,0x01,0xbd,0xc1,0xc1,0xc1,0xbd,0xf9,0xf9,0x00,0x00,
  0x00,0x8c,0xe9,0xeb,0xf6,0xbe,0xac,0xf8,0xef,0xf1,0xea,0xfc,0x97,0x97,0x97,0x97,
  0xef,0xe2,0xfa,0xf7,0x73,0x01,0x97,0x97,0x01,0xeb,0x98,0xf9,0xfc,0xbe,0x97,0xf1,
  0x97,0x01,0xea,0x99,0xf3,0xf1,0x97,0xbe,0x8c,0xe9,0xe9,0x73,0x01,0xbe,0xbe,0xef,
  0xbe,0xe2,0x8c,0xe9,0x8c,0x97,0x01,0xd2,0x98,0xe9,0x8c,0xf9,0xf5,0xd2,0x00,0x00,
  0x00,0x98,0xe9,0xeb,0xf6,0xbb,0xac,0x01,0x97,0xea,0xfc,0x97,0xef,0x01,0x01,0xbe,
  0x01,0xe2,0x98,0xe9,0x73,0x01,0xbe,0xbe,0xef,0x98,0xe9,0xea,0xf8,0x01,0xbe,0x97,
  0x01,0x01,0xe2,0x01,0xe2,0x01,0x01,0xe2,0xe9,0xe9,0x99,0xe2,0x01,0xbe,0x01,0x01,
  0x01,0xac,0xfa,0xe9,0xfa,0x97,0xbe,0xd2,0xe9,0xe9,0xf9,0x01,0xd2,0x8c,0x98,0x00,
  0x00,0xfa,0xe9,0xeb,0xed,0xbb,0xe2,0xef,0xf1,0xea,0xf1,0xbe,0x01,0xe2,0xe2,0xe2,
  0xe2,0xac,0x98,0xe9,0x73,0xe2,0xf8,0xe2,0x67,0xe9,0xe9,0x01,0xe2,0xf8,0x01,0xbe,
  0x01,0xf8,0xe2,0xe2,0xf8,0xe2,0xf8,0xfc,0xe9,0xe9,0x73,0xac,0xe2,0xe2,0xf8,0xe2,
  0xf8,0xac,0x98,0xe9,0xff,0x97,0xef,0xff,0xe9,0x8c,0x01,0xf5,0xf9,0xf4,0x98,0x00,
  0x00,0x8c,0xe9,0xeb,0xe4,0xbb,0x01,0x97,0xea,0xfc,0x97,0x01,0xe2,0xf8,0xf8,0x01,
  0x01,0xe2,0x98,0xe9,0xf6,0xf8,0x01,0xf8,0xf3,0xe9,0x98,0xac,0x01,0x01,0x01,0xfe,
  0x01,0x01,0x01,0x01,0x01,0x01,0xf8,0x67,0xe9,0xe9,0x01,0xf8,0xf8,0x01,0x01,0xf8,
  0xf8,0xac,0x8c,0xe9,0x8c,0xef,0xef,0xd2,0xe9,0x99,0xda,0xe4,0x8a,0x98,0x8c,0x00,
  0x00,0xeb,0xf7,0xeb,0x01,0xbb,0x01,0x97,0xea,0xfc,0x97,0xf8,0xe2,0x01,0x01,0xbe,
  0xbe,0x01,0xf4,0xf4,0x67,0xbe,0xbe,0x01,0xf3,0xf4,0xf4,0x01,0xbe,0xbe,0xbe,0xf1,
  0x97,0x97,0xbe,0xbe,0xfe,0xfe,0xbe,0xf3,0xf4,0x98,0x01,0xbe,0xbe,0xef,0xbe,0xbe,
  0xbe,0xe2,0xd2,0xe9,0xd2,0xbe,0xef,0xeb,0x98,0xf2,0xda,0xe4,0xf6,0x8c,0xd2,0x00,
  0x00,0xbd,0xc1,0x99,0xf8,0xac,0x01,0x97,0xea,0x97,0xef,0xf8,0xe2,0x01,0xeb,0xeb,
  0xf9,0xf9,0xbd,0xc1,0xf3,0xf9,0xf9,0xf9,0xe0,0xc1,0xc1,0xf9,0xf9,0xe9,0xe9,0xff,
  0xeb,0xe9,0x8c,0xf9,0xf9,0xf9,0xeb,0xf3,0xc1,0xc1,0xfc,0xeb,0xf9,0xbc,0xbc,0xf9,
  0xeb,0xe4,0xbd,0xf4,0xeb,0xbe,0xef,0xbd,0xc1,0xf3,0xda,0xe4,0xf5,0x8c,0x8c,0x00,
  0x00,0x99,0xc1,0xf9,0x01,0xbb,0xe2,0xef,0xf1,0xfc,0x97,0xf8,0xac,0xf8,0x01,0xbe,
  0xbe,0xe2,0x99,0xbd,0x67,0x01,0xbe,0x01,0xef,0xc1,0xbd,0xee,0xed,0x98,0xe9,0x01,
  0xed,0xe9,0xd2,0x67,0xfd,0xe2,0x01,0xee,0xc1,0xbd,0x67,0x01,0x01,0x01,0x01,0x01,
  0x01,0xe2,0x99,0xbd,0x99,0xbe,0xef,0x99,0xbd,0xe0,0xda,0xe4,0xf6,0x8c,0x98,0x00,
  0x00,0x99,0xbd,0xf2,0xfe,0xda,0xbb,0x01,0xbe,0xbe,0x01,0xe2,0xac,0xac,0xe2,0xac,
  0xac,0xac,0x99,0x99,0x67,0xac,0xe2,0xe2,0xac,0x99,0x99,0x99,0xe4,0xe4,0xe4,0xd2,
  0xeb,0xf1,0xf3,0x99,0x99,0x67,0xac,0xfd,0x99,0xbd,0x99,0xac,0xac,0xe2,0xac,0xac,
  0xac,0xda,0xf2,0x99,0xf2,0xef,0xef,0xf2,0x99,0x67,0xda,0xe4,0x8a,0x98,0x8c,0x00,
  0x00,0xf2,0x99,0xf2,0xf3,0xf3,0xe0,0xe0,0xe0,0xe0,0xf3,0x74,0xf3,0xac,0xac,0xfd,
  0xfd,0xfd,0xf2,0x95,0x67,0xfd,0xfd,0xe2,0xe2,0x4f,0x95,0xf2,0x99,0x67,0xe2,0xeb,
  0xf9,0xf2,0x95,0x95,0x95,0xe2,0xe2,0xac,0xf2,0x95,0x99,0xf2,0xe0,0xe0,0xe0,0xe0,
  0xe0,0x67,0x95,0x99,0xf2,0x97,0xef,0xf2,0x95,0x67,0x01,0xed,0xf9,0x8c,0x98,0x00,
  0x00,0xf3,0xf2,0xf2,0xf2,0xf2,0xf2,0xf2,0xf2,0xf2,0xf2,0x95,0xf2,0xe2,0xf8,0xe0,
  0xf2,0xf2,0xf2,0xf2,0xf2,0xf2,0xf2,0xe8,0x01,0xe2,0x70,0xf3,0xf2,0xf2,0x95,0x95,
  0xf2,0xf2,0x95,0xf2,0x4b,0xf8,0xbe,0x01,0xe8,0x95,0x95,0xf2,0xf2,0xf2,0xf2,0xf2,
  0xf2,0xf2,0xf2,0xf2,0xf2,0x97,0xbe,0xf3,0xf2,0x4f,0xf1,0xe4,0xd2,0x98,0x8c,0x00,
  0x00,0xf3,0xf2,0xf2,0xf2,0xf2,0xf3,0xf2,0xf2,0xf2,0xf3,0xf3,0xf2,0x01,0x01,0x70,
  0xf2,0xf3,0xf3,0xf3,0xf3,0xf2,0xf3,0xe8,0x97,0x97,0x01,0x70,0xf3,0xf3,0xf3,0xf3,
  0xf2,0xf2,0xf2,0xe8,0xbe,0x97,0x97,0x97,0xbe,0x70,0xf2,0xf2,0xf3,0xf2,0xf2,0xf2,
  0xf2,0xf2,0xf3,0xf3,0xf2,0x97,0x01,0xe0,0xf2,0x4f,0xe4,0x01,0x98,0x8c,0x00,0x00,
  0x00,0x74,0x74,0x74,0x74,0x74,0x74,0x74,0x74,0x74,0x74,0x74,0x74,0xbe,0x97,0xe0,
  0xf3,0xf2,0xf3,0xf3,0xf2,0xf3,0xf2,0xe8,0xf1,0xf1,0xf1,0xf1,0x67,0x70,0xf2,0xf2,
  0xf2,0x70,0xed,0xe4,0xed,0xea,0xf1,0xf1,0x97,0xef,0x4f,0xf2,0xf2,0xf2,0xf2,0xf2,
  0xf2,0xf2,0xf2,0xf2,0x74,0xbe,0xf8,0xe0,0xf3,0x4f,0xe4,0x01,0x8c,0xf7,0x00,0x00,
  0x00,0x00,0xf9,0xf9,0x97,0x01,0xf8,0xbb,0xfd,0xac,0xe8,0x01,0xee,0x67,0x67,0x67,
  0xee,0xee,0xee,0x4f,0xee,0xee,0xee,0x67,0xe0,0xf3,0xf3,0xe9,0xe9,0x67,0x70,0x70,
  0x70,0x67,0xf3,0xe9,0xe9,0xf3,0xf3,0xf3,0xf3,0xe0,0x67,0x67,0x4f,0xee,0xee,0xee,
  0xee,0xee,0xee,0xee,0xe8,0xe8,0x01,0xfd,0xfd,0xfd,0x01,0xf6,0x8c,0x00,0x00,0x00,
  0x00,0x00,0xeb,0x8c,0xeb,0x01,0xf1,0xda,0xac,0xfd,0xef,0xee,0xee,0xe0,0xe0,0xe0,
  0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe9,0xe9,0xf2,0xf3,0xe9,
  0xe9,0xf3,0xf2,0xe9,0xe9,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,0xe0,
  0xe0,0x67,0x67,0x67,0xee,0xe8,0xac,0xbb,0x01,0xe4,0x01,0xf9,0x8c,0x00,0x00,0x00,
  0x00,0x00,0x00,0x8c,0x98,0xf6,0x97,0xed,0xac,0xfd,0xfd,0xee,0x67,0x67,0xe0,0xe0,
  0xe0,0xe0,0x67,0x67,0x67,0xe0,0xe0,0xe0,0xe0,0x67,0x67,0x67,0xe0,0xe9,0xe9,0xe0,
  0xf3,0xe9,0xe9,0xe0,0xe0,0xe0,0x67,0x67,0x67,0x67,0xe0,0xe0,0xe0,0xe0,0xe0,0x67,
  0x67,0x67,0x67,0x67,0xe8,0xfd,0xbb,0xbb,0xf6,0x01,0xf9,0x98,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0xff,0x98,0x8c,0x97,0xf1,0xed,0xac,0xac,0xfd,0xe8,0xe8,0xe8,0xe8,
  0xe8,0xee,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xee,0xe8,0xe8,0xe8,0xe8,0xe9,0xe9,0xe8,
  0x67,0xe9,0xe9,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xe8,0xee,0xee,0xe8,0xe8,0xe8,0xe8,
  0xe8,0xe8,0xe8,0xfd,0xfd,0xac,0xac,0x8a,0xbb,0xf9,0x8c,0x8c,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x8c,0xd2,0xd2,0x01,0xf1,0xed,0xfe,0xbb,0xbb,0xbb,0xbb,0xac,
  0xac,0xac,0xac,0xac,0xbb,0xac,0xac,0xac,0xac,0xac,0xac,0xff,0x8c,0x97,0x01,0x8c,
  0xff,0x01,0xe4,0xff,0xff,0xac,0xac,0xac,0xbb,0xac,0xac,0xbb,0xac,0xac,0xac,0xbb,
  0xac,0xbb,0xbb,0xbb,0xbb,0x97,0x8a,0xac,0x8a,0x8c,0xd2,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x8c,0xf7,0x8c,0xf1,0xe2,0xed,0xe4,0xf8,0xda,0xda,0xda,
  0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xff,0x8c,0x01,0xda,0x8c,
  0xff,0xbb,0xe4,0x8c,0xff,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xbb,0xda,
  0xda,0xda,0xfe,0xe4,0xf6,0xfb,0x01,0xf9,0xf7,0xf4,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x8c,0x98,0xfa,0xf9,0xe2,0x01,0xe4,0xf5,0xf5,0xe4,
  0xf5,0xfb,0xfb,0xf5,0xed,0xf6,0xed,0xf5,0xed,0xe4,0xe4,0xed,0xf5,0xe4,0xe4,0xed,
  0xed,0xe4,0xed,0x8a,0xf6,0xf5,0xe4,0xe4,0xed,0xed,0xf6,0xf6,0xf6,0xed,0xed,0xe4,
  0xe4,0xed,0xf6,0x97,0x01,0xe4,0x98,0x8c,0xe9,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xd2,0xf7,0x8c,0x98,0xeb,0xf6,0xf1,0xf1,0x97,
  0xed,0x97,0xe4,0x97,0x97,0xe4,0x97,0xe4,0x97,0x97,0x97,0x97,0xef,0xef,0x97,0x97,
  0xbe,0xf1,0x97,0xfc,0x97,0xe4,0x97,0x97,0xe4,0x97,0xe4,0x97,0xe4,0xe4,0xe4,0xe4,
  0xe4,0xf1,0xe4,0xf9,0x98,0x98,0x98,0xd2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x98,0x8c,0x8c,0x8c,0xd2,0x8c,0x8c,
  0xe9,0xeb,0x98,0xe9,0x98,0xeb,0x8c,0xe9,0xeb,0x8c,0x98,0xe9,0x8c,0x8c,0xe9,0xeb,
  0x8c,0xeb,0x98,0xe9,0xf7,0xe9,0xeb,0x98,0xeb,0x98,0xeb,0x98,0xe9,0x8c,0x98,0xe9,
  0x8c,0x8c,0x8c,0x8c,0xd2,0x8c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x8c,0x8c,0x98,0x98,0xf0,
  0x98,0x98,0xeb,0xf7,0xe9,0x98,0xeb,0x98,0x98,0x8c,0x98,0xe9,0x98,0x8c,0x98,0x98,
  0xd2,0x98,0xeb,0x98,0xe9,0x98,0x98,0xd2,0x98,0xe9,0x98,0xf4,0x98,0xe9,0x8c,0x98,
  0xe9,0x8c,0x8c,0xd2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};
#endif // LCR_SETTING_POTATO_GRAPHICS

void LCR_loadImage(unsigned int index)
{
#if !LCR_SETTING_POTATO_GRAPHICS
  LCR_currentImage.image = LCR_images + index * LCR_IMAGE_STORE_SIZE;

  for (int i = 0; i < 256; ++i)
  {
    LCR_currentImage.palette[i] = *LCR_currentImage.image;
    LCR_currentImage.image++;
    LCR_currentImage.palette[i] |= ((uint16_t) (*LCR_currentImage.image)) << 8;

    LCR_currentImage.palette[i] =
      LCR_CONVERT_COLOR(LCR_currentImage.palette[i]);

    LCR_currentImage.image++;
  }

  LCR_currentImage.pixel = LCR_currentImage.image;
#endif
}

/**
  Changes currently loaded image's brightness either up or down by modifying
  its palette. This can be used for shading.
*/
void LCR_imageChangeBrightness(int up)
{
  if (up)
    for (int i = 0; i < 256; ++i)
      LCR_currentImage.palette[i] |=
#if LCR_SETTING_332_COLOR 
        0x0024;
#else
        0x18e3;
#endif
  else
    for (int i = 0; i < 256; ++i)
      LCR_currentImage.palette[i] =
#if LCR_SETTING_332_COLOR
        LCR_currentImage.palette[i] & 0x00db;
#else
        ((LCR_currentImage.palette[i] >> 1) & 0x7bef);
#endif
}

/**
  Samples currently loaded image at given pixels coordinates (with wrapping).
  This is slower than reading the image pixel by pixel.
*/
uint_fast16_t LCR_sampleImage(int_fast32_t x, int_fast32_t y)
{
  // bottleneck here, optimization will increase rendering performance
  x = (y % LCR_IMAGE_SIZE) * LCR_IMAGE_SIZE + (x % LCR_IMAGE_SIZE);
  x += (x < 0) * (LCR_IMAGE_SIZE * LCR_IMAGE_SIZE);
  return LCR_currentImage.palette[LCR_currentImage.image[x]];
}

/**
  Gets the next pixel of currently loaded image. This is faster than sampling
  the image by pixel coordinates.
*/
uint_fast16_t LCR_getNextImagePixel(void)
{
  uint16_t r = LCR_currentImage.palette[*LCR_currentImage.pixel];
  LCR_currentImage.pixel++;
  return r;
}

#if LCR_SETTING_CAR_SHADOW

#define LCR_SHADOW_VERTEX_COUNT 4
#define LCR_SHADOW_TRIANGLE_COUNT 2

static const int32_t LCR_shadowVertices[] =
{
  -130,16,-175, 130,16,-175, -130,16, 155, 130,16, 155
};

static const uint16_t LCR_shadowTriangles[] =
{
  0,1,2,2,1,3
};

#endif

#define LCR_CAR_VERTEX_COUNT 124
#define LCR_CAR_TRIANGLE_COUNT 228

static const int32_t LCR_carVertices[] =
{
  698, 202, 472, 698, -278, 671, 698, 3, 390, 698, 
  285, 671, 498, -231, 906, 698, 3, 953, 498, 3, 
  1004, 698, 202, 871, 498, 238, 906, 287, 197, 865, 
  287, 3, 945, 287, -190, 865, 287, -270, 671, 498, 
  -328, 671, 698, -196, 472, 498, -231, 436, 287, -190, 
  478, 287, 3, 397, 498, 3, 339, 287, 197, 478, 
  498, 238, 436, 498, 335, 671, 287, 277, 671, 698, 
  -196, 871, -700, 202, 871, -700, -278, 671, -700, 3, 
  953, -700, 285, 671, -500, -231, 436, -700, 3, 390, 
  -500, 3, 339, -700, 202, 472, -500, 238, 436, -289, 
  197, 478, -289, 3, 397, -289, -190, 478, -289, -270, 
  671, -500, -328, 671, -700, -196, 871, -500, -231, 906, 
  -289, -190, 865, -289, 3, 945, -500, 3, 1004, -289, 
  197, 865, -500, 238, 906, -500, 335, 671, -289, 277, 
  671, -700, -196, 472, -357, -92, 227, -422, 174, 279, 
  -479, 143, -323, -201, 526, 202, -236, 495, -266, -234, 
  -17, 741, -254, -84, 315, 240, -17, 741, 278, 255, 
  497, -245, 229, 706, 251, 229, 706, -212, 216, -585, 
  207, 526, 202, -272, 255, 497, 242, 495, -266, -222, 
  171, -763, 228, 171, -763, 238, -61, -811, -232, -61, 
  -811, -235, -144, -356, -384, -142, -257, 218, 216, -585, 
  241, -144, -356, 260, -84, 315, 363, -92, 227, 390, 
  -142, -257, 485, 143, -323, 428, 174, 279, -700, 202, 
  -504, -700, -278, -703, -700, 3, -421, -700, 285, -703, 
  -500, -231, -938, -700, 3, -985, -500, 3, -1035, -700, 
  202, -902, -500, 238, -938, -289, 197, -897, -289, 3, 
  -977, -289, -190, -897, -289, -270, -703, -500, -328, -703, 
  -700, -196, -504, -500, -231, -468, -289, -190, -509, -289, 
  3, -429, -500, 3, -371, -289, 197, -509, -500, 238, 
  -468, -500, 335, -703, -289, 277, -703, -700, -196, -902, 
  698, 202, -902, 698, -278, -703, 698, 3, -985, 698, 
  285, -703, 498, -231, -468, 698, 3, -421, 498, 3, 
  -371, 698, 202, -504, 498, 238, -468, 287, 197, -509, 
  287, 3, -429, 287, -190, -509, 287, -270, -703, 498, 
  -328, -703, 698, -196, -902, 498, -231, -938, 287, -190, 
  -897, 287, 3, -977, 498, 3, -1035, 287, 197, -897, 
  498, 238, -938, 498, 335, -703, 287, 277, -703, 698, 
  -196, -504
};

static const uint16_t LCR_carTriangles[] =
{
  24, 26, 25, 25, 27, 24, 28, 30, 29, 30, 32, 31, 30, 
  34, 33, 30, 28, 35, 28, 37, 36, 25, 37, 28, 38, 39, 
  37, 37, 39, 40, 39, 42, 41, 26, 42, 39, 42, 44, 43, 
  24, 44, 42, 27, 45, 44, 27, 31, 32, 46, 45, 32, 45, 
  46, 43, 25, 26, 38, 27, 25, 31, 31, 25, 29, 29, 25, 
  47, 43, 46, 36, 43, 36, 41, 36, 40, 41, 46, 33, 36, 
  33, 34, 36, 35, 36, 34, 28, 29, 47, 30, 31, 29, 30, 
  33, 32, 30, 35, 34, 28, 36, 35, 25, 28, 47, 38, 37, 
  25, 37, 40, 36, 39, 41, 40, 26, 39, 38, 42, 43, 41, 
  24, 42, 26, 27, 44, 24, 27, 32, 45, 46, 32, 33, 45, 
  43, 44, 48, 50, 49, 49, 52, 51, 53, 55, 54, 56, 58, 
  57, 52, 50, 59, 56, 61, 60, 55, 53, 58, 51, 61, 49, 
  60, 51, 62, 63, 64, 59, 63, 66, 65, 54, 61, 53, 59, 
  50, 67, 50, 68, 67, 67, 68, 54, 63, 59, 66, 67, 66, 
  59, 61, 54, 49, 49, 54, 48, 57, 53, 61, 59, 69, 52, 
  70, 65, 66, 71, 70, 67, 48, 54, 68, 72, 74, 73, 75, 
  62, 74, 54, 55, 71, 57, 61, 56, 62, 69, 74, 51, 60, 
  61, 58, 53, 57, 60, 75, 56, 52, 62, 51, 64, 69, 59, 
  65, 64, 63, 71, 55, 56, 69, 70, 74, 74, 70, 73, 70, 
  71, 73, 64, 65, 69, 70, 69, 65, 56, 75, 71, 75, 72, 
  71, 58, 56, 55, 62, 52, 69, 54, 71, 67, 70, 66, 67, 
  72, 73, 71, 48, 68, 50, 49, 50, 52, 72, 75, 74, 75, 
  60, 62, 0, 2, 1, 1, 3, 0, 4, 6, 5, 6, 8, 
  7, 6, 10, 9, 6, 4, 11, 4, 13, 12, 1, 13, 4, 
  14, 15, 13, 13, 15, 16, 15, 18, 17, 2, 18, 15, 18, 
  20, 19, 0, 20, 18, 3, 21, 20, 3, 7, 8, 22, 21, 
  8, 21, 22, 19, 1, 2, 14, 3, 1, 7, 7, 1, 5, 
  5, 1, 23, 19, 22, 12, 19, 12, 17, 12, 16, 17, 22, 
  9, 12, 9, 10, 12, 11, 12, 10, 4, 5, 23, 6, 7, 
  5, 6, 9, 8, 6, 11, 10, 4, 12, 11, 1, 4, 23, 
  14, 13, 1, 13, 16, 12, 15, 17, 16, 2, 15, 14, 18, 
  19, 17, 0, 18, 2, 3, 20, 0, 3, 8, 21, 22, 8, 
  9, 21, 19, 20, 76, 78, 77, 77, 79, 76, 80, 82, 81, 
  82, 84, 83, 82, 86, 85, 82, 80, 87, 80, 89, 88, 77, 
  89, 80, 90, 91, 89, 89, 91, 92, 91, 94, 93, 78, 94, 
  91, 94, 96, 95, 76, 96, 94, 79, 97, 96, 79, 83, 84, 
  98, 97, 84, 97, 98, 95, 77, 78, 90, 79, 77, 83, 83, 
  77, 81, 81, 77, 99, 95, 98, 88, 95, 88, 93, 88, 92, 
  93, 98, 85, 88, 85, 86, 88, 87, 88, 86, 80, 81, 99, 
  82, 83, 81, 82, 85, 84, 82, 87, 86, 80, 88, 87, 77, 
  80, 99, 90, 89, 77, 89, 92, 88, 91, 93, 92, 78, 91, 
  90, 94, 95, 93, 76, 94, 78, 79, 96, 76, 79, 84, 97, 
  98, 84, 85, 97, 95, 96, 100, 102, 101, 101, 103, 100, 104, 
  106, 105, 106, 108, 107, 106, 110, 109, 106, 104, 111, 104, 113, 
  112, 101, 113, 104, 114, 115, 113, 113, 115, 116, 115, 118, 117, 
  102, 118, 115, 118, 120, 119, 100, 120, 118, 103, 121, 120, 103, 
  107, 108, 122, 121, 108, 121, 122, 119, 101, 102, 114, 103, 101, 
  107, 107, 101, 105, 105, 101, 123, 119, 122, 112, 119, 112, 117, 
  112, 116, 117, 122, 109, 112, 109, 110, 112, 111, 112, 110, 104, 
  105, 123, 106, 107, 105, 106, 109, 108, 106, 111, 110, 104, 112, 
  111, 101, 104, 123, 114, 113, 101, 113, 116, 112, 115, 117, 116, 
  102, 115, 114, 118, 119, 117, 100, 118, 102, 103, 120, 100, 103, 
  108, 121, 122, 108, 109, 121, 119, 120
};

static const uint16_t LCR_carUvs[] =
{
  102, 283, 59, 382, 118, 324, 61, 265, 58, 224, 
  119, 193, 58, 193, 119, 163, 58, 163, 0, 163, 
  0, 193, 0, 224, 0, 256, 58, 256, 119, 256, 
  119, 39, 58, 7, 58, 39, 0, 39, 0, 70, 
  58, 70, 119, 70, 0, 101, 58, 101, 119, 101, 
  119, 132, 58, 132, 0, 132, 100, 365, 19, 281, 
  1, 322, 18, 364, 19, 281, 59, 382, 61, 265, 
  1, 322, 18, 364, 102, 283, 118, 325, 101, 366, 
  119, 224, 119, 7, 0, 7, 249, 201, 243, 118, 
  369, 119, 247, 15, 361, 15, 500, 246, 397, 242, 
  500, 350, 103, 397, 57, 499, 57, 402, 444, 80, 
  200, 400, 103, 505, 3, 404, 3, 497, 181, 101, 
  302, 393, 200, 502, 432, 502, 393, 500, 432, 400, 
  497, 403, 497, 499, 229, 199, 126, 176, 406, 204, 
  368, 198, 234, 244, 258, 213, 484, 88, 500, 161, 
  130, 113, 302, 509, 393, 402, 234, 352, 126, 245, 
  126, 351, 397, 354, 376, 219, 250, 201, 369, 198, 
  368, 117, 244, 117, 361, 14, 444, 80, 247, 14, 
  181, 101, 229, 199, 125, 176, 407, 203, 258, 383, 
  485, 87, 500, 161, 129, 113, 376, 377, 102, 283, 
  59, 382, 118, 324, 61, 265, 58, 224, 119, 193, 
  58, 193, 119, 163, 58, 163, 0, 163, 0, 193, 
  0, 224, 0, 256, 58, 256, 119, 256, 119, 39, 
  58, 7, 58, 39, 0, 39, 0, 70, 58, 70, 
  119, 70, 0, 101, 58, 101, 119, 101, 119, 132, 
  58, 132, 0, 132, 100, 365, 19, 281, 1, 322, 
  18, 364, 19, 281, 59, 382, 61, 265, 1, 322, 
  18, 364, 102, 283, 118, 325, 101, 366, 119, 224, 
  119, 7, 0, 7, 102, 283, 59, 382, 118, 324, 
  61, 265, 58, 224, 119, 193, 58, 193, 119, 163, 
  58, 163, 0, 163, 0, 193, 0, 224, 0, 256, 
  58, 256, 119, 256, 119, 39, 58, 7, 58, 39, 
  0, 39, 0, 70, 58, 70, 119, 70, 0, 101, 
  58, 101, 119, 101, 119, 132, 58, 132, 0, 132, 
  100, 365, 19, 281, 1, 322, 18, 364, 19, 281, 
  59, 382, 61, 265, 1, 322, 18, 364, 102, 283, 
  118, 325, 101, 366, 119, 224, 119, 7, 0, 7, 
  102, 283, 59, 382, 118, 324, 61, 265, 58, 224, 
  119, 193, 58, 193, 119, 163, 58, 163, 0, 163, 
  0, 193, 0, 224, 0, 256, 58, 256, 119, 256, 
  119, 39, 58, 7, 58, 39, 0, 39, 0, 70, 
  58, 70, 119, 70, 0, 101, 58, 101, 119, 101, 
  119, 132, 58, 132, 0, 132, 100, 365, 19, 281, 
  1, 322, 18, 364, 19, 281, 59, 382, 61, 265, 
  1, 322, 18, 364, 102, 283, 118, 325, 101, 366, 
  119, 224, 119, 7, 0, 7
};

static const uint16_t LCR_carTriangleUvs[] =
{
  0, 2, 1, 1, 3, 0, 4, 6, 5, 6, 8, 7, 6, 
  10, 9, 6, 4, 11, 4, 13, 12, 14, 13, 4, 15, 17, 
  16, 16, 17, 18, 17, 20, 19, 21, 20, 17, 20, 23, 22, 
  24, 23, 20, 25, 26, 23, 25, 7, 8, 27, 26, 8, 26, 
  27, 22, 1, 2, 28, 3, 1, 29, 29, 1, 30, 30, 1, 
  31, 32, 34, 33, 32, 33, 35, 33, 36, 35, 34, 37, 33, 
  37, 38, 33, 39, 33, 38, 4, 5, 40, 6, 7, 5, 6, 
  9, 8, 6, 11, 10, 4, 12, 11, 14, 4, 40, 15, 16, 
  41, 16, 18, 42, 17, 19, 18, 21, 17, 15, 20, 22, 19, 
  24, 20, 21, 25, 23, 24, 25, 8, 26, 27, 8, 9, 26, 
  22, 23, 43, 45, 44, 44, 47, 46, 48, 50, 49, 51, 53, 
  52, 47, 45, 54, 51, 56, 55, 57, 58, 53, 46, 59, 44, 
  55, 61, 60, 62, 64, 63, 62, 66, 65, 67, 59, 68, 54, 
  45, 69, 45, 70, 69, 71, 72, 49, 73, 54, 74, 69, 74, 
  54, 59, 67, 44, 44, 67, 43, 75, 68, 59, 63, 77, 76, 
  78, 80, 79, 81, 78, 71, 82, 49, 72, 83, 85, 84, 86, 
  87, 85, 49, 50, 81, 52, 56, 51, 87, 88, 85, 61, 55, 
  56, 53, 58, 52, 89, 86, 90, 76, 60, 61, 64, 77, 63, 
  65, 64, 62, 91, 92, 90, 88, 93, 85, 85, 93, 84, 78, 
  81, 94, 95, 96, 88, 93, 88, 96, 90, 86, 91, 86, 83, 
  91, 97, 90, 92, 60, 76, 77, 49, 81, 71, 78, 79, 71, 
  98, 94, 81, 43, 70, 45, 44, 45, 47, 83, 86, 85, 86, 
  89, 87, 99, 101, 100, 100, 102, 99, 103, 105, 104, 105, 107, 
  106, 105, 109, 108, 105, 103, 110, 103, 112, 111, 113, 112, 103, 
  114, 116, 115, 115, 116, 117, 116, 119, 118, 120, 119, 116, 119, 
  122, 121, 123, 122, 119, 124, 125, 122, 124, 106, 107, 126, 125, 
  107, 125, 126, 121, 100, 101, 127, 102, 100, 128, 128, 100, 129, 
  129, 100, 130, 131, 133, 132, 131, 132, 134, 132, 135, 134, 133, 
  136, 132, 136, 137, 132, 138, 132, 137, 103, 104, 139, 105, 106, 
  104, 105, 108, 107, 105, 110, 109, 103, 111, 110, 113, 103, 139, 
  114, 115, 140, 115, 117, 141, 116, 118, 117, 120, 116, 114, 119, 
  121, 118, 123, 119, 120, 124, 122, 123, 124, 107, 125, 126, 107, 
  108, 125, 121, 122, 142, 144, 143, 143, 145, 142, 146, 148, 147, 
  148, 150, 149, 148, 152, 151, 148, 146, 153, 146, 155, 154, 156, 
  155, 146, 157, 159, 158, 158, 159, 160, 159, 162, 161, 163, 162, 
  159, 162, 165, 164, 166, 165, 162, 167, 168, 165, 167, 149, 150, 
  169, 168, 150, 168, 169, 164, 143, 144, 170, 145, 143, 171, 171, 
  143, 172, 172, 143, 173, 174, 176, 175, 174, 175, 177, 175, 178, 
  177, 176, 179, 175, 179, 180, 175, 181, 175, 180, 146, 147, 182, 
  148, 149, 147, 148, 151, 150, 148, 153, 152, 146, 154, 153, 156, 
  146, 182, 157, 158, 183, 158, 160, 184, 159, 161, 160, 163, 159, 
  157, 162, 164, 161, 166, 162, 163, 167, 165, 166, 167, 150, 168, 
  169, 150, 151, 168, 164, 165, 185, 187, 186, 186, 188, 185, 189, 
  191, 190, 191, 193, 192, 191, 195, 194, 191, 189, 196, 189, 198, 
  197, 199, 198, 189, 200, 202, 201, 201, 202, 203, 202, 205, 204, 
  206, 205, 202, 205, 208, 207, 209, 208, 205, 210, 211, 208, 210, 
  192, 193, 212, 211, 193, 211, 212, 207, 186, 187, 213, 188, 186, 
  214, 214, 186, 215, 215, 186, 216, 217, 219, 218, 217, 218, 220, 
  218, 221, 220, 219, 222, 218, 222, 223, 218, 224, 218, 223, 189, 
  190, 225, 191, 192, 190, 191, 194, 193, 191, 196, 195, 189, 197, 
  196, 199, 189, 225, 200, 201, 226, 201, 203, 227, 202, 204, 203, 
  206, 202, 200, 205, 207, 204, 209, 205, 206, 210, 208, 209, 210, 
  193, 211, 212, 193, 194, 211, 207, 208
};

static const uint8_t LCR_carVertexTypes[] =
{
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
  3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
  0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
};

#endif
